/***************************************************************************
                          Digikam.cpp  -  description
                             -------------------
    begin                : Thu Dec 20 12:40:20 CST 2001
    copyright            : (C) 2001 by Renchi Raju
    email                : renchi@green.tam.uiuc.edu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "digikam.h"

#include <qiconview.h>
#include <qdir.h>
#include <qfileinfo.h>
#include <qfile.h>
#include <qimage.h>
#include <qpixmap.h>
#include <qlayout.h>

#include <kmessagebox.h>
#include <kcombobox.h>
#include <kfiledialog.h>
#include <klocale.h>
#include <kstddirs.h>
#include <kio/job.h>
#include <kurl.h>

#include "setup.h"
#include "fileOp.h"

QMap<Camera *, Digikam *> Digikam::my_cameraMap;

Digikam::Digikam(QWidget *parent, const char *name) : Digikam_MainWindow(parent, name), 
						      my_camera(NULL)
{

  statusBar = new KStatusBar(this);
  statusBar->insertFixedItem("Downloading Image XXXXXXX", 0, true);
  statusBar->setItemAlignment(0, AlignLeft | AlignVCenter);
  Digikam_MainWindowLayout->addMultiCellWidget( statusBar, 5, 7, 0, 2);

  progressBar=new KProgress(this);
  progressBar->setBarStyle(KProgress::Blocked);
  //progressBar->setTextEnabled(true);
  Digikam_MainWindowLayout->addMultiCellWidget( progressBar, 5, 7, 3, 6);

  setStatusMsg("Ready.");

  IconView->setGridX(95);
  IconView->setGridY(95);
  IconView->setAutoArrange(true);
  IconView->setSorting(true);
  IconView->setResizeMode(QIconView::Adjust);
  IconView->setWordWrapIconText(true);
  IconView->setSelectionMode(QIconView::Extended);
  IconView->setItemsMovable(false);


  dirBox->insertItem(QDir::currentDirPath(),0);
  dirBox->insertItem(QDir::homeDirPath());  

  initGlobals();


}

Digikam::~Digikam()
{
  if (my_camera) 
    gp_camera_free(my_camera);
  gp_exit();
  delete IconView;
  
  delete statusBar;
  delete progressBar;

}



void Digikam::initGlobals() {


  gp_debug_set_level(GP_DEBUG_NONE);

  gp_frontend_register(
		       frontendCameraStatus,
		       frontendCameraProgress,
		       0, // CameraMessage
		       0, // CameraConfirm
		       0  // CameraPrompt
		       );

  my_camera = 0;

  globDir = QDir::homeDirPath()+"/tmp/digikam";

  QDir dir(globDir);

  if (!dir.exists()) {
    if (KStandardDirs::makeDir(globDir))
      qWarning("Created dir");
    else
      qWarning("Failed to create dir");
  }
  else {

    const QFileInfoList *fileList;

    fileList = dir.entryInfoList();

    QFileInfoListIterator it(*fileList);
    QFileInfo *fi;
    QString fileNameWithPath;
    
    while(fi = it.current()){
      if(fi->fileName() == "." || fi->fileName() == "..")
	;
      else {
	fileNameWithPath = dir.path()+"/"+fi->fileName();
	qWarning(fileNameWithPath);
	dir.remove(fileNameWithPath);
      }
      ++it;
    }
  }

  my_cameraInitialised=false;
    
  qWarning("Digikam good to go");

}



void Digikam::slot_close() 
{
  close();
}

void Digikam::slot_setup()
{
	Setup *setup=new Setup();
	setup->exec();
	my_cameraInitialised=false;
	slot_connectToCamera();
}


void Digikam::slot_connectToCamera()
{
  
  if (!(my_cameraInitialised)) {
    if (initCamera()!=0)  {
			setStatusMsg("Abort.");	
      return;
    }
  }

  getThumbNails();

  updateThumbNails();

}


int Digikam::initCamera() {

  const char *my_cameraModel, *my_cameraPort, *my_cameraPortPath;

  setStatusMsg("Initializing camera.");

  config = new KSimpleConfig("digikamrc");
  config->setGroup("Camera Settings");

  if ((config->readEntry("Camera Model") == NULL) ||
      (config->readEntry("Camera Port Name") == NULL)) {
    KMessageBox::error(this,"Camera Model or Port Setting\n"\
		       "not specified correctly!\n"\
		       "Please, run Setup");
    return 1;
  }


  my_cameraModel=config->readEntry("Camera Model");
  my_cameraPort=config->readEntry("Camera Port Name");
  my_cameraPortPath=config->readEntry("Camera Port Path");

  gp_camera_new(&my_camera);
  gp_camera_set_model(my_camera,my_cameraModel);
  gp_camera_set_port_name(my_camera,my_cameraPort);

  if (gp_camera_init(my_camera)<0) {
    KMessageBox::error(this,"Failed to initialize camera!");
    return 1;
  }

  if (!(my_cameraMap.contains(my_camera))) {
    my_cameraMap[my_camera]=this;
    qWarning("Inserted camera into cameraMap");
  }
  else {
    qWarning("Camera already present in map. not inserting");
  }

  setStatusMsg("Camera initialized.");

  delete config;
  my_cameraInitialised=true;

  return 0;

}


void Digikam::getThumbNails() {

  const char *my_cameraFolder, *my_cameraFile;
  CameraFile *cameraFile;
  QString saveFile;
 
  setStatusMsg("Getting ThumbNail images.");

  my_cameraFolder = "/";

  gp_list_new(&my_cameraListFiles);
  gp_camera_folder_list_files(my_camera, my_cameraFolder, my_cameraListFiles);
  float count = gp_list_count (my_cameraListFiles);

  for (int i = 0; i < count; i++) {
    gp_list_get_name  (my_cameraListFiles, i, &my_cameraFile);
    qWarning(my_cameraFile);
    gp_file_new(&cameraFile);
    gp_camera_file_get (my_camera, my_cameraFolder, my_cameraFile, 
    			GP_FILE_TYPE_PREVIEW, cameraFile);
    saveFile=globDir+"/thumb_"+my_cameraFile;
    qWarning(saveFile);
    if (gp_file_save(cameraFile,saveFile)<0) {
      KMessageBox::error(this,"Failed to save thumbnails!");
      return;
    }
    gp_file_free(cameraFile);
  }

  gp_list_free (my_cameraListFiles);
  

}


void Digikam::updateThumbNails() {

  QDir dir(globDir);
  const QFileInfoList *fileList;
  fileList = dir.entryInfoList();

  IconView->clear();

  QFileInfoListIterator it(*fileList);
  QFileInfo *fi;
  QString fileNameWithPath;
  QString actualImageFileName;

  while(fi = it.current()){
    QIconViewItem *i = NULL;

    if(fi->fileName() == "." || fi->fileName() == "..")
      ;
    else {
      fileNameWithPath = dir.path()+"/"+fi->fileName();
      qWarning(fileNameWithPath);
      QImage *thumbnail = new QImage(fileNameWithPath);
      QPixmap thumbnail_pixmap;
      thumbnail_pixmap.convertFromImage(*thumbnail);

      actualImageFileName=(fi->fileName()).remove(0,6);

      i = new QIconViewItem(IconView, actualImageFileName, thumbnail_pixmap);
      i->setDropEnabled(false);
      i->setDragEnabled(false);
      delete thumbnail;

    }

    ++it;
    } 
}


void Digikam::slot_selectAll() {
  IconView->selectAll(true);
}

void Digikam::slot_downloadImages() {

  const char *my_cameraFolder, *my_cameraFile;
  CameraFile *cameraFile;
  QString saveFile;
  //QString saveFileTemp;
  QString saveDir;
  bool overwriteallmode=false;
  bool skipallmode=false;
  bool savefilemode=true;

  int j=0;

  my_cameraFolder = "/";

  saveDir=dirBox->currentText();
  QDir dir(saveDir);
  if (!dir.exists()) {
    KMessageBox::error(this,saveDir+" does not exit!");
    return;
  }

  for ( QIconViewItem *i = IconView->firstItem(); i; i = i->nextItem() ) 
    if (i->isSelected()) 
      ++j;
  if (j == 0) {
    KMessageBox::error(this,"No images selected!");
    return;
  }

  gp_list_new(&my_cameraListFiles);
  gp_camera_folder_list_files(my_camera, my_cameraFolder, my_cameraListFiles);

  j = 0;

  setStatusMsg("Starting Downloads.");


  for ( QIconViewItem *i = IconView->firstItem(); i; i = i->nextItem() ) {
    if (i->isSelected()) {

      if (skipallmode) break;

      gp_list_get_name  (my_cameraListFiles, j, &my_cameraFile);
      qWarning(my_cameraFile);

      gp_file_new(&cameraFile);
      gp_camera_file_get(my_camera, my_cameraFolder, my_cameraFile,
    			GP_FILE_TYPE_NORMAL, cameraFile);
      saveFile=saveDir+"/"+my_cameraFile;
      qWarning(saveFile);

      while (QFile::exists(saveFile) && !(overwriteallmode)) {

	FileOpDialog *dlg = new FileOpDialog(my_cameraFile, saveFile, NULL, NULL);
	
	if (dlg->exec()== QDialog::Rejected){
	  setStatusMsg("Abort.");
	  delete dlg;
	  return;
	}

	if (dlg->fileOpSelection() == Skip) {
	  qWarning("Skipped.");
	  delete dlg;
	  savefilemode=false;
	  break;
	} 
	else if (dlg->fileOpSelection() == SkipAll) {
	  qWarning("Skipped All.");
	  skipallmode=true;
	  savefilemode=false;
	  delete dlg;
	  break;
	} 
	else if (dlg->fileOpSelection() == Overwrite) {
	  qWarning("Overwrite.");	  
	  delete dlg;
	  savefilemode=true;
	  break;
	}
	else if (dlg->fileOpSelection() == OverwriteAll) {
	  qWarning("Overwrite All.");	  
	  delete dlg;
	  savefilemode=true;
	  overwriteallmode=true;
	  break; 
	}
	else if (dlg->fileOpSelection() == Rename) {
	  qWarning("Rename.");
	  savefilemode=true;
	  saveFile = saveDir+"/"+dlg->renameString();
	  delete dlg;
	  qWarning(saveFile+" New Name");
	}

      }

      if (savefilemode) {
	if (gp_file_save(cameraFile,saveFile)<0) {
	  KMessageBox::error(this,"Failed to save images!");
	  gp_file_free(cameraFile);
	  return;
	}
      }
      savefilemode=true;
    	
      gp_file_free(cameraFile);
      
    }
    ++j;
  }
  gp_list_free (my_cameraListFiles);

  setStatusMsg("Finished Downloads.");

}


void Digikam::slot_browse() {
  
  QString result = 
    KFileDialog::getExistingDirectory( dirBox->text(dirBox->currentItem()),
				       this );
  if (!result.isEmpty()) {
    for (int i = 0; i < dirBox->count(); i++)
      if (result == dirBox->text(i)) {
	dirBox->setCurrentItem(i);
	return;
      }
    dirBox->insertItem(result, 0);
    dirBox->setCurrentItem(0);
  }
  
}


int Digikam::frontendCameraStatus(Camera *camera, char *status) {

  if (my_cameraMap.contains(camera)) {
    Digikam *object=my_cameraMap[camera];
    object->setStatusMsg(status);
  }
  return 0;
}

int Digikam::frontendCameraProgress(Camera *camera, CameraFile *file, float progress) {


  if (my_cameraMap.contains(camera)) {
    Digikam *object=my_cameraMap[camera];
    if (progress >= 0) {
      object->setProgressValue(int(progress));
    }
  }
  return 0;
}
  
void Digikam::setStatusMsg(const QString &msg) {
  statusBar->changeItem(msg, 0);
}


void Digikam::setProgressValue(int progress) {
  progressBar->setValue(progress);
}
