#include <iostream.h>
#include <errno.h>
#include <string>

#include "exif_parser.h"
#include "ifd.h"
#include "exif_tags.h"
#include "tagFunctions.h"



Image_Section::Image_Section()
{ 
  size=0; 
  type=0; 
  data=NULL;

}

Image_Section::Image_Section(const Image_Section& section)
{
  size = section.size;
  type = section.type;
  
  data = new unsigned char[size];
  for (unsigned int i=0; i<size; i++)
    data[i] = section.data[i];

}

Image_Section& Image_Section::operator=(const Image_Section& section)
{
  if (this != &section) {

    if (data) 
      delete [] data;

    size = section.size;
    type = section.type;
  
    data = new unsigned char[size];
    for (unsigned int i=0; i<size; i++)
      data[i] = section.data[i];

  }

  return *this;
}


// ------------------------------------------------------------------------


int Exif_Parser::motorolaOrder=0;


Exif_Parser::Exif_Parser(const char* fileName) {

  exifError = EXIF_OK;
  imageFileName = fileName;

  exifData = NULL;
  exifDataSize = 0;
  commentData = NULL;
  commentDataSize = 0;
  thumbnailPtr = NULL;
  thumbnailSize = 0;
  imageData = NULL;
  imageDataSize = 0;
    
  IFD0 = NULL;
  IFD1 = NULL;
  subIFD = NULL;
  interIFD = NULL;
  makerIFD = NULL;


}


Exif_Parser::~Exif_Parser() {

  for (unsigned int i=0; i<imageSections.size(); i++) 
    delete [] imageSections[i].data;

  imageSections.clear();
  
  if (imageData)
  delete [] imageData;

  if (IFD0)
    delete IFD0;
  
  if (IFD1)
    delete IFD1;

  if (subIFD)
    delete subIFD;

  if (interIFD)
    delete interIFD;

  if (makerIFD)
    delete makerIFD;
}


EXIF_ERROR Exif_Parser::processFile() {

  fileStream.open(imageFileName, ios::binary || ios::in);
  if (!fileStream.good()) {
    exifError = EXIF_OPEN_FAILED;
    return exifError;
  }


  exifError = hasExif();
  if (exifError != EXIF_OK) 
    return exifError;

    
  exifError = getByteOrder();
  if (exifError != EXIF_OK) 
    return exifError;
  
  if (Get_UShort(exifData+10) != 0x2a
      || Get_ULong(exifData+12) != 0x08) {
    exifError = EXIF_INVALID_MARKER;
    return exifError;
  }

  IFD0 = new IFD(exifData+16, exifData+8, exifDataSize-6);

  processIFD(IFD0, IFD0_tags);

  if (subIFD)
    processIFD(subIFD, subIFD_tags);

  if (IFD1)
    processIFD(IFD1, IFD1_tags);

  if (interIFD)
    processIFD(interIFD, interIFD_tags);


  if (makerIFD)
    processIFD(makerIFD, maker_tags);

  if (commentData)
    processComment();

  return EXIF_OK;

}

string Exif_Parser::getErrorName() {

  switch(exifError) {
  case EXIF_OK:                  return string("Exif OK"); break;
  case EXIF_OPEN_FAILED:         return string("File Open Failed"); break;
  case EXIF_READ_FAILED:         return string("File Read Failed"); break;
  case EXIF_WRITE_FAILED:        return string("File Write Failed"); break;
  case EXIF_NOT_JPEG:            return string("File is not a JPEG File"); break;
  case EXIF_EXTRA_PADDING_BYTES: return string("Too many Padding Bytes Found"); break;
  case EXIF_INVALID_MARKER:      return string("Invalid Marker Found"); break;
  case EXIF_PREMATURE_FILE_END:  return string("Premature End of File"); break;
  case EXIF_NO_EXIF:             return string("No Exif Section Found"); break;
  case EXIF_UNKNOWN_BYTE_ORDER:  return string("Unknown Byte Ordering"); break;
  default:                       return string("Unknown Error"); break;
  }
}  

int Exif_Parser::getIFD0(vector<string> &vector_IFD0_name, vector<string> &vector_IFD0_value) {

  if (IFD0) {
    vector_IFD0_name = IFD0->vector_name; 
    vector_IFD0_value = IFD0->vector_value;   
    return 0;
  }
  return -1;

}

int Exif_Parser::getIFD1(vector<string> &vector_IFD1_name, vector<string> &vector_IFD1_value) {

  if (IFD1) {
    vector_IFD1_name = IFD1->vector_name; 
    vector_IFD1_value = IFD1->vector_value;   
    return 0;
  }
  return -1;

}

int Exif_Parser::getsubIFD(vector<string> &vector_subIFD_name, vector<string> &vector_subIFD_value) {

  if (subIFD) {
    vector_subIFD_name = subIFD->vector_name; 
    vector_subIFD_value = subIFD->vector_value;   
    return 0;
  }
  return -1;

}


int Exif_Parser::getinterIFD(vector<string> &vector_interIFD_name, vector<string> &vector_interIFD_value) {

  if (interIFD) {
    vector_interIFD_name = interIFD->vector_name; 
    vector_interIFD_value = interIFD->vector_value;   
    return 0;
  }
  return -1;

}

int Exif_Parser::getmakerIFD(vector<string> &vector_makerIFD_name, vector<string> &vector_makerIFD_value) {

  if (makerIFD) {
    vector_makerIFD_name = makerIFD->vector_name; 
    vector_makerIFD_value = makerIFD->vector_value;   
    return 0;
  }
  return -1;

}

int Exif_Parser::getThumbnail(unsigned char* &thumbPtr, unsigned int &thumbSize) {

  if (thumbnailPtr && thumbnailSize) {
    thumbPtr = thumbnailPtr;
    thumbSize = thumbnailSize;
    return 0;
  }
  return -1;

}

void Exif_Parser::getComments(string &comments) {

  if (!commentString.empty())
    comments = commentString;

}

void Exif_Parser::changeComments(const string &newComments) {
  
  if (commentString == newComments)
    return;

  commentString = newComments;

  unsigned int j = 0;
  for (unsigned int i=0; i<imageSections.size(); i++) {
    if (imageSections[i].type == M_COM) break;
    ++j;
  }


  if (j < imageSections.size()) {
    // Found comments - delete them
    delete imageSections[j].data;
  }
  else {

    // No comments found - Insert new section 
    // immediately after exif section (at position 2)
    Image_Section section;
    section.type = M_COM;
    vector<Image_Section>::iterator iter = imageSections.begin();
    iter += 2;
    imageSections.insert(iter, 1, section);
    j = 2;
  }

  imageSections[j].size = 2 + commentString.length();
  imageSections[j].data = new unsigned char[imageSections[j].size];

  commentData = imageSections[j].data;
  commentDataSize = imageSections[j].size;

  commentData[0] = (unsigned char)(imageSections[j].size >> 8);
  commentData[1] = (unsigned char)(imageSections[j].size);

  for (int i=0; i< commentDataSize; i++) 
    commentData[2+i] = commentString[i];

  imageSections[j].data =  commentData;


}
  


EXIF_ERROR Exif_Parser::writeImageFile(const char* fileName) {

  ofstream outStream (fileName, ios::binary || ios::out);

  if (!outStream) {
    exifError = EXIF_OPEN_FAILED;
    return exifError;
  }

  outStream.put(0xff);
  outStream.put(0xd8);

  for (unsigned int i=0; i<imageSections.size(); i++) {
    outStream.put(0xff);
    outStream.put(imageSections[i].type);
    outStream.write(imageSections[i].data, imageSections[i].size);
  }

  if (!outStream.write(imageData, imageDataSize)) {
    exifError = EXIF_WRITE_FAILED;
    return exifError;
  }
  outStream.close();

  return EXIF_OK;
}


EXIF_ERROR Exif_Parser::hasExif() {

  unsigned char header[2];
  fileStream.read(header, 2);
  if (fileStream.gcount() != 2) {
    int err = errno;
    cerr << "Read failed: " << strerror(err) << endl;
    return EXIF_READ_FAILED;
  }


  if ((header[0] == 0xFF) && (header[1] == M_SOI)) {
    

    bool found_exif=false;
    bool found_SOSorEOI=false;
    unsigned char marker;
    
    while (!fileStream.eof()) {

      for (int i=0; i<7; i++){
	fileStream.read(&marker, 1);
	if (marker != 0xFF) break;
	if (i >= 6){
	  // 0xff is legal padding, but if we get that many, something's wrong.
	  return EXIF_EXTRA_PADDING_BYTES;
	}
      }

      Image_Section section;

      // read data size
      unsigned char lh, ll;
      fileStream.read(&lh, 1);    
      fileStream.read(&ll, 1);
      section.size = (lh << 8) | ll;
      if (section.size < 2) {
	return EXIF_INVALID_MARKER;
      }

      section.data = new unsigned char[section.size];
      section.data[0] =  lh;
      section.data[1] =  ll;
      fileStream.read(&(section.data[2]), section.size-2);
      if (fileStream.gcount() != (unsigned)section.size-2) {
	return EXIF_PREMATURE_FILE_END;
      }

      section.type = marker;
      imageSections.push_back(section);

      switch (marker) {
	
      case M_SOS: 

	int currPos, endPos;
	found_SOSorEOI=true;

	currPos = fileStream.tellg();
	fileStream.seekg(0,ios::end);
	endPos = fileStream.tellg();
	fileStream.seekg(currPos);

	imageDataSize = endPos - currPos;	
	imageData = new unsigned char[imageDataSize];
	
	fileStream.read(imageData, imageDataSize);

	break;

      case M_EOI:
	found_SOSorEOI=true;
	break;

      case M_COM:
	commentDataSize = section.size;
	commentData = section.data;
	break;

      case M_JFIF:
	break;
      
      case M_EXIF:
	if (section.data[2] == 'E' && section.data[3] == 'x' &&
	    section.data[4] == 'i' && section.data[5] == 'f') {
	  found_exif = true;
	}
	exifDataSize = section.size;
	exifData = section.data;
	break;

      default:
	break;

      }


      if (found_SOSorEOI) { 
	if (found_exif) 
	  return EXIF_OK;
	else 
	  return EXIF_NO_EXIF;
      }
			  

    }

    return EXIF_NO_EXIF;

  }
  else {
    return EXIF_NOT_JPEG;
  }
    

}




void Exif_Parser::processIFD(IFD *ifd, knowntag *tagTable) {

  for (int i=0; i<ifd->numEntries; i++) {

    unsigned short tag = ifd->ifdTable[i]->getTag();

    for  (int j=0; ; j++) {

      if (tag == 0x8769) {
	unsigned char* subDirStart = ifd->offsetBase + Get_ULong(ifd->ifdTable[i]->getValuePtr());
	subIFD = new IFD(subDirStart, ifd->offsetBase, ifd->exifLength);
	break;
      }

      if (tag == 0x927c) {
	if  (strncmp(makerName.data(), "NIKON", 5) == 0) {
	  unsigned char* makerIFDStart = ifd->ifdTable[i]->getValuePtr();
	  makerIFD = new IFD(makerIFDStart, ifd->offsetBase, ifd->exifLength);
	  maker_tags = nikon_tags;
	}
	else if (strncmp(makerName.data(), "OLYMP", 5) == 0) {
	  unsigned char* makerIFDStart = ifd->ifdTable[i]->getValuePtr()+ 0x08;
	  makerIFD = new IFD(makerIFDStart, ifd->offsetBase, ifd->exifLength);
	  maker_tags = olympus_tags;
	}
	break;
      }

      if (tag == 0xa005) {
	unsigned char* interIFDStart = ifd->offsetBase + Get_ULong(ifd->ifdTable[i]->getValuePtr());
	interIFD = new IFD(interIFDStart, ifd->offsetBase, ifd->exifLength);
	break;
      }


      if (ifd==IFD1 && tag == 0x0201) {
	unsigned thumbnailOffset = Get_ULong(ifd->ifdTable[i]->getValuePtr());
	thumbnailPtr = ifd->offsetBase + thumbnailOffset;
	break;
      }

      if (ifd==IFD1 && tag == 0x0202) {
	thumbnailSize = Get_ULong(ifd->ifdTable[i]->getValuePtr());
	break;
      }


      if (tagTable[j].tag == tag) {

	if (tagTable[j].func) {
	  string str = tagTable[j].func(tagTable[j].name, ifd->ifdTable[i], tagTable[j].units);
	  if (!str.empty()) {

	    if (string(tagTable[j].name) == "Make")
	      makerName = str;

	    ifd->vector_name.push_back(string(tagTable[j].name));
	    ifd->vector_value.push_back(str);
	  }
	}
	break;
      } 

      else if (tagTable[j].tag == 0) {
	//cout << "Unknown tag 0x0" << hex << tag;
	//cout <<  " With format " << ifd->ifdTable[i]->getFormat() << endl;
	break;
      }
    }


  }

  // Thumbnail IFD (IFD1)
  if (ifd == IFD0) {
    unsigned char* IFD1DirStart = ifd->offsetBase + Get_ULong(ifd->dirStart + 2 + 12*ifd->numEntries);
    if (IFD1DirStart < (ifd->offsetBase +ifd->exifLength)) {
      IFD1 = new IFD(IFD1DirStart, ifd->offsetBase, ifd->exifLength);
    }
  }

}


void Exif_Parser::processComment() {

  int ch;

  for (int i=2; i<commentDataSize; i++){
    ch = commentData[i];
    if (ch == '\r' && commentData[i+1] == '\n') continue; // Remove cr followed by lf.

    if (isprint(ch) || ch == '\n' || ch == '\t'){
      commentString += (char)ch;
    }else{
      commentString += '?';
    }
  }

}


EXIF_ERROR Exif_Parser::getByteOrder() {

  if (exifData[8] == 'I' && exifData[9] == 'I') {
    motorolaOrder = 0;
    return EXIF_OK;
  }
  else if (exifData[8] == 'M' && exifData[9] == 'M') {
    motorolaOrder = 1;
    return EXIF_OK;
  }
  else {
    exifError = EXIF_UNKNOWN_BYTE_ORDER;
    return exifError;
  }
    
}


unsigned short Exif_Parser::Get_UShort(void *Short)
{
    if (motorolaOrder){
        return (((unsigned char *)Short)[0] << 8) | ((unsigned char *)Short)[1];
    }else{
        return (((unsigned char *)Short)[1] << 8) | ((unsigned char *)Short)[0];
    }
}

unsigned long Exif_Parser::Get_ULong(void *Long) 
{
  return (unsigned)Get_SLong(Long) & 0xffffffff;
}


signed long Exif_Parser::Get_SLong(void *Long)
{
  if (motorolaOrder){
    return  ((( char *)Long)[0] << 24) | (((unsigned char *)Long)[1] << 16)
      | (((unsigned char *)Long)[2] << 8 ) | (((unsigned char *)Long)[3] << 0 );
  }else{
    return  ((( char *)Long)[3] << 24) | (((unsigned char *)Long)[2] << 16)
      | (((unsigned char *)Long)[1] << 8 ) | (((unsigned char *)Long)[0] << 0 );
  }
}


