#include <qstring.h>
#include <qstringlist.h>
#include <qevent.h>
#include <qpainter.h>
#include <qdir.h>
#include <qfile.h>
#include <qpopmenu.h>
#include <qpoint.h>
#include <qcursor.h>

#include <kapp.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kservice.h>
#include <krun.h>
#include <kurl.h>
#include <kstandarddirs.h>
#include <kiconloader.h>
#include <klineeditdlg.h>
#include <kurldrag.h>

#include "cameraiconview.h"
#include "gphoto_controller.h"
#include "gphoto_events.h"
#include "mtlist.h"
#include "camerafileiteminfo.h"
#include "camerainfodialog.h"
#include "savefiledialog.h"
#include "imageviewer.h"
#include "cameraiconitem.h"
#include "camerafileitem.h"
#include "camerafolderitem.h"
#include "camerafoldertree.h"


CameraFolderTree::CameraFolderTree(QWidget* parent)
    : KListView(parent)
{
    mIconView = 0;
    mGpController = 0;
    mRootItem = 0;
    mCameraGlobalPath = "/";
    mBusy = false;
    mCamItemServicePtr = 0;
    mHighlightedItem = 0;

    addColumn(i18n("Camera Folders"));
    setFullWidth(true);

    setDragEnabled(false);
    setDropVisualizer(false);
    setDropHighlighter(false);
    setAcceptDrops(true);

    // -- Setup Connections -----------------------------------

    connect(this, SIGNAL(executed(QListViewItem*)),
            this, SLOT(slot_executed(QListViewItem*)));

    connect(this, SIGNAL(clicked(QListViewItem*)),
            this, SIGNAL(signal_cameraFolderClicked()));

    connect(this, SIGNAL(selectionChanged(QListViewItem*)),
            this, SLOT(slot_selectionChanged(QListViewItem*)));

    connect(this, SIGNAL(contextMenu(KListView*, QListViewItem*,
                                     const QPoint&)),
            this, SLOT(slot_contextMenu(KListView*, QListViewItem*,
                                        const QPoint&)));

    // -- Create a new camera ---------------------------------

    camera_new();
}

CameraFolderTree::~CameraFolderTree()
{
    if (mGpController)
        delete mGpController;
}

void CameraFolderTree::setIconView(CameraIconView*
                                         camView)
{
    mIconView = camView;

    // -- Setup Connections -----------------------------------

    connect(mIconView, SIGNAL(signal_openItem(CameraIconItem*,KService::Ptr)),
            this, SLOT(slot_openItem(CameraIconItem*,KService::Ptr)));
    connect(mIconView, SIGNAL(signal_resizeThumbnails()),
            this, SLOT(slot_iconViewInsertItems()));
}


void CameraFolderTree::slot_executed(QListViewItem* it)
{
    if (!it) return;
    CameraFolderItem* item =
        static_cast<CameraFolderItem *>(it);

    if (item == mRootItem) {
        camera_connect();
    }

    if (item->isOpen())
        item->setOpen(false);
    else
        item->setOpen(true);
}


void CameraFolderTree::slot_selectionChanged(QListViewItem* it)
{
    if (!mGpController->isCameraInitialised())
        return;

    if (!it) return;

    slot_iconViewInsertItems();

}


void CameraFolderTree::slot_iconViewInsertItems()
{
    if (!mGpController->isCameraInitialised())
        return;

    if (!currentItem())
        setCurrentItem(mRootItem);

    CameraFolderItem* folderItem =
        static_cast<CameraFolderItem *>(currentItem());

    mIconView->clear();

    if (folderItem->isRoot()) {

        QListViewItemIterator it(this);
        for ( ; it.current(); ++it ) {

            CameraFolderItem* camFolder =
                static_cast<CameraFolderItem*>(it.current());

            QPtrListIterator<CameraFileItem>
                iter(camFolder->fileItemList);

            CameraFileItem* fileItem;
            while ( (fileItem = iter.current()) != 0 ) {
                ++iter;
                (void) new CameraIconItem(mIconView,
                                          fileItem,
                                          mIconView->getThumbnailSize());
            }
        }

    }
    else {

        QPtrListIterator<CameraFileItem>
                iter(folderItem->fileItemList);

        CameraFileItem* fileItem;
        while ( (fileItem = iter.current()) != 0 ) {
            ++iter;
            (void) new CameraIconItem(mIconView,
                                      fileItem,
                                      mIconView->getThumbnailSize());
        }
    }
}


void CameraFolderTree::slot_openItem(CameraIconItem* camItem,
                                     KService::Ptr servicePtr)
{
    if (!mGpController->isCameraInitialised())
        return;

    if (!camItem) return;

    mCamItemServicePtr = servicePtr;

    mGpController->openImage(camItem->fileInfo()->getFolder(),
                             camItem->fileInfo()->getName());
}


void CameraFolderTree::camera_new()
{
    clear();

    if (mGpController)
        delete mGpController;

    mGpController = new Gphoto_Controller(this);
    mGpController->start();

    mCameraModel = QString(mGpController->cameraModel().latin1());
    mCameraPort  = QString(mGpController->cameraPort().latin1());
    mCameraGlobalPath = QString(mGpController->cameraGlobalPath().latin1());

    QString camModel = mCameraModel;

    if (mCameraModel == QString("Directory Browse"))
        camModel = QString("USB MassStorage Camera");

    mRootItem = new CameraFolderItem(this, camModel,
                                     QString::null,true);
    mRootItem->setOpen(false);

    mNumThumbnailsRequested  = 0;
    mNumThumbnailsDownloaded = 0;
}

void CameraFolderTree::camera_change(const QString& model,
                                     const QString& port,
                                     const QString& path)
{
    if (model != mCameraModel ||
        port  != mCameraPort ||
        path  != mCameraGlobalPath) {
        camera_disconnect();
        camera_new();
    }
}


void CameraFolderTree::camera_connect()
{
    if (!(mGpController->isCameraInitialised())) {
        mGpController->initializeCamera();
    }
}


void CameraFolderTree::camera_disconnect()
{
    if (!mGpController) return;

    delete mGpController;
    mGpController = NULL;

    mIconView->clear();
    clear();

    camera_new();
    emit signal_cameraConnected(false);
}

void CameraFolderTree::camera_cancel()
{
    mGpController->stopOperations();
}


void CameraFolderTree::camera_download(const QString& downloadDir)
{
    if (!mGpController->isCameraInitialised()) return;

    QDir dir(downloadDir);
    if (!dir.exists()) {
        camera_cancel();
        KMessageBox::error(0, downloadDir + " Directory does not exist");
        return;
    }


    int count = 0;
    for ( QIconViewItem *i = mIconView->firstItem(); i;
          i=i->nextItem() ) {
        if (i->isSelected()) ++count;
    }
    if (count == 0) return;

    bool proceed = true;
    bool overwriteAll = false;

    for (QIconViewItem *i = mIconView->firstItem(); i;
         i=i->nextItem()) {

        if (i->isSelected()) {
            CameraIconItem *item =
                static_cast<CameraIconItem*>(i);
            if (!item) continue;
            downloadOneImage(item->fileInfo()->getName(),
                             item->fileInfo()->getFolder(),
                             downloadDir,
                             proceed, overwriteAll);
            if (!proceed) return;
        }
    }

}


void CameraFolderTree::camera_download(const QStringList& nameList,
                                       const QStringList& folderList,
                                       const QString& downloadDir)
{
    if (!mGpController->isCameraInitialised()) return;

    QDir dir(downloadDir);
    if (!dir.exists()) {
        camera_cancel();
        KMessageBox::error(0, downloadDir + i18n(" Directory does not exist"));
        return;
    }

    bool proceed = true;
    bool overwriteAll = false;

    QStringList::ConstIterator it = nameList.begin();
    QStringList::ConstIterator it2 = folderList.begin();

    for ( ;it != nameList.end(); ++it, ++it2 ) {
        QString name   = *it;
        QString folder = *it2;
        downloadOneImage(name, folder, downloadDir,
                         proceed, overwriteAll);
        if (!proceed) return;
    }
}


void CameraFolderTree::camera_downloadAll(const QString&
                                          downloadDir)
{
    if (!mGpController->isCameraInitialised()) return;

    QDir dir(downloadDir);
    if (!dir.exists()) {
        camera_cancel();
        KMessageBox::error(0, downloadDir +
                           i18n(" Directory does not exist"));
        return;
    }

    bool proceed = true;
    bool overwriteAll = false;

    QListViewItemIterator it(this);
    for ( ; it.current(); ++it ) {

        CameraFolderItem* camFolder =
            static_cast<CameraFolderItem*>(it.current());

        QPtrListIterator<CameraFileItem>
            iter(camFolder->fileItemList);

        CameraFileItem* fileItem;
        while ( (fileItem = iter.current()) != 0 ) {
            ++iter;
            downloadOneImage(fileItem->fileInfo()->getName(),
                             fileItem->fileInfo()->getFolder(),
                             downloadDir,
                             proceed, overwriteAll);
            if (!proceed) return;
        }
    }

}

void CameraFolderTree::downloadOneImage(const QString& name,
                                        const QString& folder,
                                        const QString& downloadDir,
                                        bool& proceedFurther,
                                        bool& overwriteAll)
{
    proceedFurther = true;

    QString saveFile;
    saveFile=downloadDir + "/" + name;
    while (QFile::exists(saveFile) && !(overwriteAll)) {

    bool overwrite=false;

    SavefileDialog *dlg = new SavefileDialog(saveFile);
    if (dlg->exec()== QDialog::Rejected) {
        delete dlg;
        camera_cancel();
        proceedFurther = false;
        return;
    }

    switch(dlg->saveFileOperation()) {
    case (SavefileDialog::Skip): {
        delete dlg;
        return;
    }
    case (SavefileDialog::SkipAll): {
        delete dlg;
        proceedFurther = false;
        return;
    }
    case (SavefileDialog::Overwrite): {
        overwrite = true;
        delete dlg;
        break;
    }
    case (SavefileDialog::OverwriteAll): {
        overwriteAll = true;
        delete dlg;
        break;
    }
    case (SavefileDialog::Rename): {
        saveFile = downloadDir+"/"+dlg->renameFile();
        delete dlg;
        break;
    }
    default:  {
        delete dlg;
        proceedFurther = false;
        camera_cancel();
        return;
    }

    }

    if (overwrite) break;
    }

    mGpController->downloadImage(folder, name, saveFile);

}

void CameraFolderTree::camera_delete()
{
    if (!mGpController->isCameraInitialised()) return;

    int count = 0;
    QStringList deleteList;

    for ( QIconViewItem *i = mIconView->firstItem(); i;
          i=i->nextItem() ) {
        if (i->isSelected()) {
            ++count;
            deleteList.append(i->text());
        }
    }

    if (count == 0) return;

    QString warnMsg(i18n("About to delete these Image(s)\nAre you sure?"));
    if (KMessageBox::warningContinueCancelList(this, warnMsg,
                                               deleteList,
                                               i18n("Warning"),
                                               i18n("Delete"))
        ==  KMessageBox::Continue) {

        CameraIconItem *item =
            static_cast<CameraIconItem*>(mIconView->firstItem());

        while(item) {

            CameraIconItem *nextItem =
                static_cast<CameraIconItem *>(item->nextItem());

            if (item->isSelected())
                mGpController->deleteImage(item->fileInfo()->getFolder(),
                                           item->fileInfo()->getName());

            item = nextItem;

        }
    }

}

void CameraFolderTree::camera_deleteAll()
{
    if (!mGpController->isCameraInitialised()) return;

    QString warnMsg(i18n("About to delete All Image(s) in the Camera\nAre you sure?"));
    if (KMessageBox::warningContinueCancel (this, warnMsg, i18n("Warning"),
					  i18n("Delete"))
        == KMessageBox::Continue) {

        QListViewItemIterator it(this);
        for ( ; it.current(); ++it ) {

            CameraFolderItem* camFolder =
                static_cast<CameraFolderItem*>(it.current());

            QPtrListIterator<CameraFileItem>
                iter(camFolder->fileItemList);

            CameraFileItem* fileItem;
            while ( (fileItem = iter.current()) != 0 ) {
                ++iter;
                mGpController->deleteImage(
                    fileItem->fileInfo()->getFolder(),
                    fileItem->fileInfo()->getName());
            }

        }

    }
}

void CameraFolderTree::camera_upload(const QString& localImagePath)
{
    if (!mGpController->isCameraInitialised()) return;

    QString msg(i18n("Please Select a Folder on Camera to Upload"));

    if (!selectedItem()) {
        KMessageBox::error(0,msg);
        return;
    }

    CameraFolderItem* folderItem =
        static_cast<CameraFolderItem *>(selectedItem());

    if (!folderItem || folderItem->isRoot()) {
         KMessageBox::error(0,msg);
         return;
    }

    QString uploadName = KURL(localImagePath).fileName();

    bool ok;

    while (folderItem->containsItem(uploadName)) {
        QString msg(i18n("Camera Folder \'"));
        msg += folderItem->getFolderName();
        msg += i18n("\' Contains item ") + uploadName;
        msg += i18n("\n Enter New Name");
        uploadName = KLineEditDlg::getText(msg,uploadName,&ok,this);
        if (!ok) return;
    }

    mGpController->uploadImage(folderItem->getFolderPath(),
                               localImagePath,
                               uploadName);

}

void CameraFolderTree::camera_information()
{
    if (!mGpController->isCameraInitialised())
        return;

    mGpController->getCameraInformation();
}

bool CameraFolderTree::cameraReadyForUpload(QString& reason)
{
    bool result = false;

    if (!mGpController->isCameraInitialised()) {
        reason = i18n("Camera Not Initialised");
        return result;
    }

    if (!mGpController->cameraSupportsUpload()) {
        reason = i18n("Camera does not support Uploads");
        return result;
    }

    if (!selectedItem()) {
        reason = i18n("Please Select a Folder on Camera to Upload");
        return result;
    }

    CameraFolderItem* folderItem =
        static_cast<CameraFolderItem *>(selectedItem());

    if (!folderItem || folderItem->isRoot()) {
        reason = i18n("Please Select a Folder on Camera to Upload");
        return result;
    }

    result = true;
    return result;

}

void CameraFolderTree::customEvent(QCustomEvent *event)
{
    if (event->type() >= QCustomEvent::User) {

        switch (event->type()) {

        case Gphoto_InitializedT:
            eventCameraInitialized(event);
            break;

        case Gphoto_NewSubFoldersT:
            eventNewSubFolders(event);
            break;

        case Gphoto_NewItemsT:
            eventNewItems(event);
            break;

        case Gphoto_ThumbNailReadyT:
            eventThumbNailReady(event);
            break;

        case Gphoto_DownloadImageT:
            eventDownloadImage(event);
            break;

        case Gphoto_DeleteImageT:
            eventDeleteImage(event);
            break;

        case Gphoto_OpenImageT:
            eventOpenImage(event);
            break;

        case Gphoto_InformationT:
            eventCameraInformation(event);
            break;

        case Gphoto_StatusMsgT:
            eventCameraStatusMsg(event);
            break;

        case Gphoto_ProgressT:
            eventCameraProgress(event);
            break;

        case Gphoto_ErrorMsgT:
            eventCameraErrorMsg(event);
            break;

        case Gphoto_BusyT:
            eventCameraBusy(event);
            break;

        default:
            break;

        }

    }
}

void CameraFolderTree::eventCameraInitialized(QCustomEvent *event)
{
    Gphoto_Event_InitializedCamera*
        ev(static_cast<Gphoto_Event_InitializedCamera *>(event));

    QString cameraModel, cameraPort;

    cameraModel = QString(ev->cameraModel().latin1());
    cameraPort  = QString(ev->cameraPort().latin1());
    mCameraGlobalPath = QString(ev->cameraPath().latin1());

    mGpController->getSubFolders(mCameraGlobalPath);
    mGpController->getImagesInfo(mCameraGlobalPath);

    (void) new CameraFolderItem(mRootItem,
                                mCameraGlobalPath,
                                mCameraGlobalPath);

    mRootItem->setOpen(true);

    emit signal_cameraConnected(true);

}

void CameraFolderTree::eventNewSubFolders(QCustomEvent *event)
{

    Gphoto_Event_NewSubFolders*
        ev(static_cast<Gphoto_Event_NewSubFolders *>(event));

    QString parentFolder(ev->parentFolder());
    QStringList subFolderList(ev->subFolderList());
    QStringList subFolderNameList(ev->subFolderNameList());

    QListViewItemIterator it(this);
    for ( ; it.current(); ++it ) {

      CameraFolderItem* camFolder =
          static_cast<CameraFolderItem*>(it.current());
      if (!camFolder) return;

      if (parentFolder == camFolder->getFolderPath()) {

	QStringList::Iterator subFolderNameIter
            = subFolderNameList.begin();
        QStringList::Iterator subFolderIter
                 = subFolderList.begin();

	for ( ; subFolderIter != subFolderList.end();
              subFolderIter++ ) {

            (void) new CameraFolderItem(camFolder,
                                        *subFolderNameIter,
                                        *subFolderIter);
            mGpController->getImagesInfo(*subFolderIter);

            subFolderNameIter++;

	}

	break;

      }

    }

}



void CameraFolderTree::eventNewItems(QCustomEvent *event)
{
    Gphoto_Event_NewItems*
        ev(static_cast<Gphoto_Event_NewItems *>(event));

    QString parentFolder(ev->parentFolder());
    MTList<CameraFileItemInfo> infoList(ev->itemsInfoList());

    int numItems = infoList.count();

    QListViewItemIterator it(this);
    for ( ; it.current(); ++it ) {

        CameraFolderItem* camFolder =
            static_cast<CameraFolderItem*>(it.current());
        if (!camFolder) continue;

        if (camFolder->getFolderPath() == parentFolder) {

            camFolder->incrementNumItems(numItems);
            mRootItem->incrementNumItems(numItems);
            camFolder->showNumItems();
            mRootItem->showNumItems();

            while( !(infoList.isEmpty()) ) {

                CameraFileItemInfo info( infoList.first() );
                infoList.pop_front();

                camFolder->fileItemList.append(
                    new CameraFileItem(info));
                mGpController->getThumbnail(parentFolder,
                                            info.getName());

                mNumThumbnailsRequested++;
            }
            break;
        }
    }

    slot_iconViewInsertItems();
}


void CameraFolderTree::eventThumbNailReady(QCustomEvent *event)
{

    Gphoto_Event_ThumbNailReady*
        ev(static_cast<Gphoto_Event_ThumbNailReady *>(event));

    mNumThumbnailsDownloaded++;
    int progress = int(double(mNumThumbnailsDownloaded)/
                       double(mNumThumbnailsRequested)*100);
    emit signal_cameraProgress(progress);

    QListViewItemIterator it(this);
    for ( ; it.current(); ++it ) {

        CameraFolderItem* camFolder =
            static_cast<CameraFolderItem*>(it.current());
        if (!camFolder) continue;

        if (camFolder->getFolderPath() == ev->parentFolder()) {

            QPtrListIterator<CameraFileItem>
                iter(camFolder->fileItemList);

            CameraFileItem* fileItem;
            while ( (fileItem = iter.current()) != 0 ) {
                ++iter;

                if (fileItem->fileInfo()->getName() == ev->imageName() &&
                    fileItem->fileInfo()->getFolder() == ev->parentFolder()) {

                    if (ev->success()) {
                        QPixmap thumbnail;
                        ev->fillPixmap(thumbnail);
                        fileItem->setPixmap(thumbnail);
                        return;
                    }
                }
            }

        }

    }

}


void CameraFolderTree::eventDownloadImage(QCustomEvent *event)
{

    Gphoto_Event_DownloadImage*
        ev(static_cast<Gphoto_Event_DownloadImage *>(event));

    QListViewItemIterator it(this);
    for ( ; it.current(); ++it ) {

        CameraFolderItem* camFolder =
            static_cast<CameraFolderItem*>(it.current());
        if (!camFolder) continue;

        if (camFolder->getFolderPath() == ev->parentFolder()) {

            QPtrListIterator<CameraFileItem>
                iter(camFolder->fileItemList);

            CameraFileItem* fileItem;
            while ( (fileItem = iter.current()) != 0 ) {
                ++iter;

                if (fileItem->fileInfo()->getName() == ev->imageName() &&
                    fileItem->fileInfo()->getFolder() == ev->parentFolder()) {
                    fileItem->setDownloaded(true);
                    return;
                }
            }
        }
    }
}

void CameraFolderTree::eventDeleteImage(QCustomEvent *event)
{

    Gphoto_Event_DeleteImage*
        ev(static_cast<Gphoto_Event_DeleteImage *>(event));

    QListViewItemIterator it(this);
    for ( ; it.current(); ++it ) {

        CameraFolderItem* camFolder =
            static_cast<CameraFolderItem*>(it.current());
        if (!camFolder) continue;

        if (camFolder->getFolderPath() == ev->parentFolder()) {

            QPtrListIterator<CameraFileItem>
                iter(camFolder->fileItemList);

            CameraFileItem* fileItem;
            while ( (fileItem = iter.current()) != 0 ) {
                ++iter;

                if (fileItem->fileInfo()->getName() == ev->imageName() &&
                    fileItem->fileInfo()->getFolder() == ev->parentFolder()) {

                    camFolder->fileItemList.remove(fileItem);
                    mRootItem->incrementNumItems(-1);
                    camFolder->incrementNumItems(-1);
                    camFolder->showNumItems();
                    mRootItem->showNumItems();
                    return;
                }
            }
        }
    }

}


void CameraFolderTree::eventOpenImage(QCustomEvent *event)
{
    Gphoto_Event_OpenImage*
        ev(static_cast<Gphoto_Event_OpenImage *>(event));

    QString savedFile = locateLocal("tmp", ev->imageName());

    if (mCamItemServicePtr) {
        KRun::run(*mCamItemServicePtr, KURL(savedFile));
    }
    else {
        ImageViewer* imgViewer = new ImageViewer;
        imgViewer->loadImage(savedFile);
        imgViewer->show();
    }
}

void CameraFolderTree::eventCameraInformation(QCustomEvent *event)
{
    Gphoto_Event_Information*
        ev(static_cast<Gphoto_Event_Information *>(event));

    QString summary(ev->getSummary());
    QString manual(ev->getManual());
    QString about(ev->getAbout());

    CameraInfoDialog *infoDlg = new CameraInfoDialog(summary,
                                                     manual, about);
    infoDlg->show();

}

void CameraFolderTree::eventCameraStatusMsg(QCustomEvent *event)
{
    Gphoto_Event_CameraStatusMsg*
        ev(static_cast<Gphoto_Event_CameraStatusMsg *>(event));

    emit signal_cameraStatusMsg(ev->statusMsg());
}

void CameraFolderTree::eventCameraProgress(QCustomEvent *event)
{
    Gphoto_Event_CameraProgress*
    ev(static_cast<Gphoto_Event_CameraProgress *>(event));

    emit signal_cameraProgress(ev->val());
}

void CameraFolderTree::eventCameraErrorMsg(QCustomEvent *event)
{
    Gphoto_Event_CameraErrorMsg*
        ev(static_cast<Gphoto_Event_CameraErrorMsg *>(event));

    QString errorMsg(i18n("Camera Error : "));
    errorMsg += ev->errorMsg();
    KMessageBox::error(0, errorMsg);
}

void CameraFolderTree::eventCameraBusy(QCustomEvent *event)
{
    Gphoto_Event_Busy*
        ev(static_cast<Gphoto_Event_Busy *>(event));

    if (ev->busy()) {
        if (!mBusy) {
            emit signal_cameraBusy(true);
            mBusy = true;
        }
    }
    else {
        if (mBusy) {
            emit signal_cameraBusy(false);
            mBusy = false;
        }
    }
}


void CameraFolderTree::slot_contextMenu(KListView* view,
                                        QListViewItem* item,
                                        const QPoint& pos)
{
    if (!view || !item) return;

    CameraFolderItem* camFolderItem =
        static_cast<CameraFolderItem *>(item);


    if (camFolderItem->isRoot()) {

        QPopupMenu popMenu(this);

        if (!mGpController->isCameraInitialised())
            popMenu.insertItem( i18n("Connect"),
                                 10);
        else
            popMenu.insertItem( i18n("Disconnect"),
                                 11);
        int id = popMenu.exec(pos);

        switch(id) {
        case 10: {
            emit signal_cameraFolderClicked();
            camera_connect();
            break;
        }
        case 11: {
            camera_disconnect();
            break;
        }
        }

    }

}

// ----- DnD ---------------------------------------------------

void CameraFolderTree::contentsDragMoveEvent(QDragMoveEvent* event)
{
    if (!QUriDrag::canDecode(event)) {
        event->ignore();
        return;
    }
    event->accept();

    QPoint point(0,event->pos().y());
    QListViewItem *item = itemAt(contentsToViewport(point));
    if (item) {
        if (item != mHighlightedItem) {
            highlightItem(item);
        }
    }
    else
        cleanHighlights();

}

void CameraFolderTree::contentsDragLeaveEvent(QDragLeaveEvent*)
{
    cleanHighlights();
}

void CameraFolderTree::contentsDropEvent(QDropEvent* event)
{
    cleanHighlights();

    if (!QUriDrag::canDecode(event)) {
        event->ignore();
        return;
    }

    QPoint point(0,event->pos().y());
    CameraFolderItem* folderItem =
        static_cast<CameraFolderItem *>(
            itemAt(contentsToViewport(point)));

    if (!folderItem) {
        event->ignore();
        return;
    }

    setSelected(folderItem,true);
    emit signal_cameraFolderClicked();

    QString rejectReason;
    if (!cameraReadyForUpload(rejectReason)) {
        KMessageBox::error(0,rejectReason);
        return;
    }


    QPopupMenu popMenu(this);
    popMenu.insertItem( i18n("Upload"),10);
    popMenu.setMouseTracking(true);
    int id = popMenu.exec(QCursor::pos());
    switch(id) {
    case 10:
        break;
    default:
        return;
    }


    KURL::List srcURLs;
    KURLDrag::decode(event,srcURLs);

    if (!srcURLs.isEmpty()) {
        KURL::List::iterator it;
        for (it = srcURLs.begin(); it != srcURLs.end(); ++it) {
            camera_upload((*it).path());
        }
    }
}

void CameraFolderTree::highlightItem(QListViewItem* item)
{
    if (!item) return;
    cleanHighlights();

    CameraFolderItem *folderItem = static_cast<CameraFolderItem *>(item);
    folderItem->highlight();

    mHighlightedItem = item;
}

void CameraFolderTree::cleanHighlights()
{
    if (!mHighlightedItem) return;

    CameraFolderItem *item = static_cast<CameraFolderItem *>(mHighlightedItem);
    item->unHighlight();

    mHighlightedItem = 0;

}

