/***************************************************************************
 * variant.h
 * This file is part of the KDE project
 * copyright (C)2004-2005 by Sebastian Sauer (mail@dipe.org)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 * You should have received a copy of the GNU Library General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 ***************************************************************************/

#ifndef KROSS_API_VARIANT_H
#define KROSS_API_VARIANT_H

#include <qstring.h>
#include <qvariant.h>
//#include <kdebug.h>

#include "object.h"
#include "value.h"
#include "exception.h"

namespace Kross { namespace Api {

    /**
     * Variant value to represent a QVariant.
     *
     * We handle all primitive types like strings
     * or numerics with Variant cause it spends
     * us more typesafety and handles conversation
     * in a flexible manner.
     */
    class Variant : public Value<Variant, QVariant>
    {
            friend class Value<Variant, QVariant>;
        protected:

            /**
             * Constructor.
             *
             * \param value The initial QVariant-value
             *        this Variant-Object has.
             * \param name The name this Value has.
             */
            Variant(const QVariant& value, const QString& name = "variant");

        public:

            /**
             * Destructor.
             */
            virtual ~Variant();

            /// See \see Kross::Api::Object::getClassName
            virtual const QString getClassName() const;

            /// See \see Kross::Api::Object::getDescription
            virtual const QString getDescription() const;

            /**
             * Try to convert the given \a Object into
             * a QVariant.
             *
             * \throw TypeException If the convert failed.
             * \param object The object to convert.
             * \return The to a QVariant converted object.
             */
            static const QVariant& toVariant(Object* object);

            /**
             * Try to convert the given \a Object into
             * a QString.
             *
             * \throw TypeException If the convert failed.
             * \param object The object to convert.
             * \return The to a QString converted object.
             */
            static const QString toString(Object* object);

            /**
             * Try to convert the given \a Object into
             * a uint.
             *
             * \throw TypeException If the convert failed.
             * \param object The object to convert.
             * \return The to a uint converted object.
             */
            static uint toUInt(Object* object);

            /**
             * Try to convert the given \a Object into
             * a Q_LLONG.
             *
             * \throw TypeException If the convert failed.
             * \param object The object to convert.
             * \return The to a Q_LLONG converted object.
             */
            static Q_LLONG toLLONG(Object* object);

            /**
             * Try to convert the given \a Object into
             * a Q_ULLONG.
             *
             * \throw TypeException If the convert failed.
             * \param object The object to convert.
             * \return The to a Q_ULLONG converted object.
             */
            static Q_ULLONG toULLONG(Object* object);

            /**
             * Try to convert the given \a Object into
             * a boolean value.
             *
             * \throw TypeException If the convert failed.
             * \param object The object to convert.
             * \return The to a bool converted object.
             */
            static bool toBool(Object* object);

            /**
             * Try to convert the given \a Object into
             * a QValueList of QVariant's.
             *
             * \throw TypeException If the convert failed.
             * \param object The object to convert.
             * \return The to a QValueList converted object.
             */
            static QValueList<QVariant> toList(Object* object);
    };

}}

#endif

