// Copyright (c) 2010, Google Inc.
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
//     * Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above
// copyright notice, this list of conditions and the following disclaimer
// in the documentation and/or other materials provided with the
// distribution.
//     * Neither the name of Google Inc. nor the names of its
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE

#include <stdlib.h>
#include <unistd.h>

#include <string>

#include "breakpad_googletest_includes.h"
#include "common/using_std_string.h"
#include "google_breakpad/processor/basic_source_line_resolver.h"
#include "google_breakpad/processor/minidump_processor.h"
#include "google_breakpad/processor/process_state.h"
#ifndef _WIN32
#include "processor/exploitability_linux.h"
#endif  // _WIN32
#include "processor/simple_symbol_supplier.h"

#ifndef _WIN32
namespace google_breakpad {

class ExploitabilityLinuxTest : public ExploitabilityLinux {
 public:
  using ExploitabilityLinux::CalculateAddress;
  using ExploitabilityLinux::DisassembleBytes;
  using ExploitabilityLinux::GetObjdumpInstructionLine;
  using ExploitabilityLinux::TokenizeObjdumpInstruction;
};

class ExploitabilityLinuxTestMinidumpContext : public MinidumpContext {
 public:
  explicit ExploitabilityLinuxTestMinidumpContext(
      const MDRawContextAMD64& context) : MinidumpContext(NULL) {
    valid_ = true;
    SetContextAMD64(new MDRawContextAMD64(context));
    SetContextFlags(MD_CONTEXT_AMD64);
  }
};

}  // namespace google_breakpad
#endif  // _WIN32

namespace {

using google_breakpad::BasicSourceLineResolver;
#ifndef _WIN32
using google_breakpad::ExploitabilityLinuxTest;
using google_breakpad::ExploitabilityLinuxTestMinidumpContext;
#endif  // _WIN32
using google_breakpad::MinidumpProcessor;
using google_breakpad::ProcessState;
using google_breakpad::SimpleSymbolSupplier;

string TestDataDir() {
  return string(getenv("srcdir") ? getenv("srcdir") : ".") +
      "/src/processor/testdata";
}

// Find the given dump file in <srcdir>/src/processor/testdata, process it,
// and get the exploitability rating. Returns EXPLOITABILITY_ERR_PROCESSING
// if the crash dump can't be processed.
google_breakpad::ExploitabilityRating
ExploitabilityFor(const string& filename) {
  SimpleSymbolSupplier supplier(TestDataDir() + "/symbols");
  BasicSourceLineResolver resolver;
  MinidumpProcessor processor(&supplier, &resolver, true);
  processor.set_enable_objdump(true);
  ProcessState state;

  string minidump_file = TestDataDir() + "/" + filename;

  if (processor.Process(minidump_file, &state) !=
      google_breakpad::PROCESS_OK) {
    return google_breakpad::EXPLOITABILITY_ERR_PROCESSING;
  }

  return state.exploitability();
}

TEST(ExploitabilityTest, TestWindowsEngine) {
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_HIGH,
            ExploitabilityFor("ascii_read_av.dmp"));
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_HIGH,
            ExploitabilityFor("ascii_read_av_block_write.dmp"));
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_HIGH,
            ExploitabilityFor("ascii_read_av_clobber_write.dmp"));
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_HIGH,
            ExploitabilityFor("ascii_read_av_conditional.dmp"));
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_HIGH,
            ExploitabilityFor("ascii_read_av_then_jmp.dmp"));
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_HIGH,
            ExploitabilityFor("ascii_read_av_xchg_write.dmp"));
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_HIGH,
            ExploitabilityFor("ascii_write_av.dmp"));
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_HIGH,
            ExploitabilityFor("ascii_write_av_arg_to_call.dmp"));
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_NONE,
            ExploitabilityFor("null_read_av.dmp"));
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_NONE,
            ExploitabilityFor("null_write_av.dmp"));
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_NONE,
            ExploitabilityFor("stack_exhaustion.dmp"));
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_HIGH,
            ExploitabilityFor("exec_av_on_stack.dmp"));
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_MEDIUM,
            ExploitabilityFor("write_av_non_null.dmp"));
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_LOW,
            ExploitabilityFor("read_av_non_null.dmp"));
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_LOW,
            ExploitabilityFor("read_av_clobber_write.dmp"));
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_LOW,
            ExploitabilityFor("read_av_conditional.dmp"));
}

TEST(ExploitabilityTest, TestLinuxEngine) {
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_INTERESTING,
            ExploitabilityFor("linux_null_read_av.dmp"));
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_HIGH,
            ExploitabilityFor("linux_overflow.dmp"));
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_HIGH,
            ExploitabilityFor("linux_stacksmash.dmp"));
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_NONE,
            ExploitabilityFor("linux_divide_by_zero.dmp"));
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_INTERESTING,
            ExploitabilityFor("linux_null_dereference.dmp"));
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_HIGH,
            ExploitabilityFor("linux_jmp_to_0.dmp"));
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_HIGH,
            ExploitabilityFor("linux_outside_module.dmp"));
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_NONE,
            ExploitabilityFor("linux_raise_sigabrt.dmp"));
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_INTERESTING,
            ExploitabilityFor("linux_inside_module_exe_region1.dmp"));
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_INTERESTING,
            ExploitabilityFor("linux_inside_module_exe_region2.dmp"));
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_INTERESTING,
            ExploitabilityFor("linux_stack_pointer_in_stack.dmp"));
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_HIGH,
            ExploitabilityFor("linux_stack_pointer_in_module.dmp"));
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_HIGH,
            ExploitabilityFor("linux_executable_stack.dmp"));
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_HIGH,
            ExploitabilityFor("linux_executable_heap.dmp"));
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_HIGH,
            ExploitabilityFor("linux_jmp_to_module_not_exe_region.dmp"));
#ifndef _WIN32
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_HIGH,
            ExploitabilityFor("linux_write_to_nonwritable_module.dmp"));
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_HIGH,
            ExploitabilityFor("linux_write_to_nonwritable_region_math.dmp"));
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_HIGH,
            ExploitabilityFor("linux_write_to_outside_module.dmp"));
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_HIGH,
            ExploitabilityFor("linux_write_to_outside_module_via_math.dmp"));
  ASSERT_EQ(google_breakpad::EXPLOITABILITY_INTERESTING,
            ExploitabilityFor("linux_write_to_under_4k.dmp"));
#endif  // _WIN32
}

#ifndef _WIN32
TEST(ExploitabilityLinuxUtilsTest, DisassembleBytesTest) {
  ASSERT_FALSE(ExploitabilityLinuxTest::DisassembleBytes("", NULL, 5, NULL));
  uint8_t bytes[6] = {0xc7, 0x0, 0x5, 0x0, 0x0, 0x0};
  char buffer[1024] = {0};
  ASSERT_TRUE(ExploitabilityLinuxTest::DisassembleBytes("i386:x86-64",
                                                        bytes,
                                                        1024,
                                                        buffer));
  std::stringstream objdump_stream;
  objdump_stream.str(string(buffer));
  string line = "";
  while (line.find("<.data>") == string::npos)
    getline(objdump_stream, line);
  getline(objdump_stream, line);
  ASSERT_EQ(line, "   0:\tc7 00 05 00 00 00    \tmov    DWORD PTR [rax],0x5");
}

TEST(ExploitabilityLinuxUtilsTest, GetObjdumpInstructionLine) {
  string disassebly =
      "\n"
      "/tmp/breakpad_mem_region-raw_bytes-tMmMo0:     file format binary\n"
      "// Trying to confuse the parser 0:\n"
      "\n"
      "Disassembly of section .data:\n"
      "\n"
      "0000000000000000 <.data>:\n"
      "   0:\tc7 00 01 00 00 00    \tmov    DWORD PTR [rax],0x1\n"
      "   6:\t5d                   \tpop    rbp\n"
      "   7:\tc3                   \tret    \n"
      "   8:\t55                   \tpush   rbp\n"
      "   9:\t48 89 e5             \tmov    rbp,rsp\n"
      "   c:\t53                   \tpush   rbx\n"
      "   d:\t48                   \trex.W\n"
      "   e:\t81                   \t.byte 0x81\n";
  string line;
  EXPECT_TRUE(ExploitabilityLinuxTest::GetObjdumpInstructionLine(
      disassebly.c_str(), &line));
  EXPECT_EQ("   0:\tc7 00 01 00 00 00    \tmov    DWORD PTR [rax],0x1", line);

  // There is no "0:" after "<.data>:".  Expected to return false.
  disassebly =
      "\n"
      "/tmp/breakpad_mem_region-raw_bytes-tMmMo0:     file format binary\n"
      "// Trying to confuse the parser 0:\n"
      "\n"
      "Disassembly of section .data:\n"
      "\n"
      "   0:\tc7 00 01 00 00 00    \tmov    DWORD PTR [rax],0x1\n"
      "   6:\t5d                   \tpop    rbp\n"
      "   7:\tc3                   \tret    \n"
      "   8:\t55                   \tpush   rbp\n"
      "   9:\t48 89 e5             \tmov    rbp,rsp\n"
      "   d:\t48                   \trex.W\n"
      "0000000000000000 <.data>:\n"
      "   c:\t53                   \tpush   rbx\n";
  EXPECT_FALSE(ExploitabilityLinuxTest::GetObjdumpInstructionLine(
      disassebly.c_str(), &line));
}

TEST(ExploitabilityLinuxUtilsTest, TokenizeObjdumpInstructionTest) {
  ASSERT_FALSE(ExploitabilityLinuxTest::TokenizeObjdumpInstruction("",
                                                                   NULL,
                                                                   NULL,
                                                                   NULL));
  string line = "0: c7 00 05 00 00 00     mov    DWORD PTR [rax],0x5";
  string operation = "";
  string dest = "";
  string src = "";
  ASSERT_TRUE(ExploitabilityLinuxTest::TokenizeObjdumpInstruction(line,
                                                                  &operation,
                                                                  &dest,
                                                                  &src));
  ASSERT_EQ(operation, "mov");
  ASSERT_EQ(dest, "[rax]");
  ASSERT_EQ(src, "0x5");
  line = "0: c3 ret";
  ASSERT_TRUE(ExploitabilityLinuxTest::TokenizeObjdumpInstruction(line,
                                                                  &operation,
                                                                  &dest,
                                                                  &src));
  ASSERT_EQ(operation, "ret");
  ASSERT_EQ(dest, "");
  ASSERT_EQ(src, "");
  line = "0: 5f pop rdi";
  ASSERT_TRUE(ExploitabilityLinuxTest::TokenizeObjdumpInstruction(line,
                                                                  &operation,
                                                                  &dest,
                                                                  &src));
  ASSERT_EQ(operation, "pop");
  ASSERT_EQ(dest, "rdi");
  ASSERT_EQ(src, "");
}

TEST(ExploitabilityLinuxUtilsTest, CalculateAddressTest) {
  MDRawContextAMD64 raw_context;
  raw_context.rdx = 12345;
  ExploitabilityLinuxTestMinidumpContext context(raw_context);
  ASSERT_EQ(context.GetContextAMD64()->rdx, 12345U);
  ASSERT_FALSE(ExploitabilityLinuxTest::CalculateAddress("", context, NULL));
  uint64_t write_address = 0;
  ASSERT_TRUE(ExploitabilityLinuxTest::CalculateAddress("rdx-0x4D2",
                                                        context,
                                                        &write_address));
  ASSERT_EQ(write_address, 11111U);
  ASSERT_TRUE(ExploitabilityLinuxTest::CalculateAddress("rdx+0x4D2",
                                                        context,
                                                        &write_address));
  ASSERT_EQ(write_address, 13579U);
  ASSERT_FALSE(ExploitabilityLinuxTest::CalculateAddress("rdx+rax",
                                                         context,
                                                         &write_address));
  ASSERT_FALSE(ExploitabilityLinuxTest::CalculateAddress("0x3482+0x4D2",
                                                         context,
                                                         &write_address));
}
#endif  // _WIN32

}  // namespace
