/* -*- Mode: c; c-basic-offset: 2 -*-
 *
 * timegm.c - timegm compatibility
 *
 * This file is in the public domain.
 * 
 */

#ifdef HAVE_CONFIG_H
#include <rasqal_config.h>
#endif

#ifdef WIN32
#include <win32_rasqal_config.h>
#endif

#include <stdio.h>
#include <string.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_TIME_H
#include <time.h>
#endif

#include <errno.h>

#include "rasqal.h"
#include "rasqal_internal.h"

/* Taken the following three functions from the kdewin library to avoid
 * a dependency against it */

#if defined(_MSC_VER) && _MSC_VER >= 1400
// use secure functions declared in msvc >= 2005
#define KDEWIN_USE_ENV_S
#endif

static void putenvMsvcrt(const char *name, const char *value)
{
    typedef int (*msvc6putenv)(const char *envstring);
    static msvc6putenv s_msvcrtputenv = 0;
    static int alreadyResolved = 0;
    int i;
    char * a;

    if( !alreadyResolved ) {
#ifdef _MSC_VER
        HANDLE hModule = LoadLibraryA("msvcrt");
#else
        // it doesn't work when msvcr80 isn't loaded - we end up in an error
        // message due to crappy manifest things :(
        // maybe someone has an idea how to fix this.
        //HANDLE hModule = LoadLibraryA("msvcr80");
        HANDLE hModule = NULL;
#endif
        if( hModule )
            s_msvcrtputenv = (msvc6putenv)GetProcAddress(hModule, "_putenv");
        alreadyResolved = 1;
    }
    if( !s_msvcrtputenv )
        return;

    i = strlen(name) + (value ? strlen(value) : 0) + 2;
    a = (char*)malloc(i);
    if (!a) return;

    strcpy(a, name);
    strcat(a, "=");
    if (value)
      strcat(a, value);

    s_msvcrtputenv(a);
    free(a);
}

int setenv(const char *name, const char *value, int overwrite)
{
//FIXME: There is no env in wince, so disable it
#ifndef _WIN32_WCE
#ifndef KDEWIN_USE_ENV_S
    int i, iRet;
    char * a;
#endif

    if (!overwrite && getenv(name)) return 0;

    // make sure to set the env var in all our possible runtime environments
    putenvMsvcrt(name, value);
    //SetEnvironmentVariableA(name, value);     // unsure if we need it...

#ifdef KDEWIN_USE_ENV_S
    return _putenv_s(name, value ? value : "");
#else 
    if (!name) return -1;

    i = strlen(name) + (value ? strlen(value) : 0) + 2;
    a = (char*)malloc(i);
    if (!a) return 1;

    strcpy(a, name);
    strcat(a, "=");
    if (value)
      strcat(a, value);

    iRet = putenv(a);
    free(a);
    return iRet;
#endif
#else
	return -1;
#endif
}

// from kdecore/fakes.c
int unsetenv (const char *name)
{
    if (name == NULL || *name == '\0' || strchr (name, '=') != NULL)
    {
      errno = EINVAL;
      return -1;
    }

    setenv(name, "", 1);
    return 0;
}

time_t
rasqal_timegm(struct tm *tm)
{
  time_t result;
  char *zone;

  zone = getenv("TZ");
  if(zone) {
    /* save it so that setenv() does not destroy shared value */
    size_t zone_len = strlen(zone);
    char *zone_copy = RASQAL_MALLOC(cstring, zone_len + 1);
    if(!zone_copy)
      return -1;

    memcpy(zone_copy, zone, zone_len + 1); /* Copy NUL */
    zone = zone_copy;
  }
    
  setenv("TZ", "UTC", 1);
  tzset();

  result = mktime(tm);

  if(zone)
    setenv("TZ", zone, 1);
  else
    unsetenv("TZ");
  tzset();

  if(zone)
    RASQAL_FREE(cstring, zone);

  return result;
}
