/*
    This file is part of KOrganizer.
    Copyright (c) 2000 Cornelius Schumacher <schumacher@kde.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    As a special exception, permission is given to link this program
    with any edition of Qt, and distribute the resulting executable,
    without including the source code for Qt in the source distribution.
*/

#include <kinstance.h>
#include <klocale.h>
#include <kaboutdata.h>
#include <kiconloader.h>
#include <kaction.h>
#include <kdebug.h>
#include <kprocess.h>
#include <kmessagebox.h>
#include <kfiledialog.h>
#include <kio/netaccess.h>

#include "calendarview.h"

#include "alarmclient.h"
#include "korganizer_part.h"
#include "koviewmanager.h"
#include "kogroupware.h"
#include "koprefs.h"
#include "kodialogmanager.h"
#include "exportwebdialog.h"
#include <libkcal/calendarimap.h>

extern "C"
{
  /**
   * This function is the 'main' function of this part.  It takes
   * the form 'void *init_lib<library name>()  It always returns a
   * new factory object
   */
  void *init_libkorganizer()
  {
    return new KOrganizerFactory;
  }
};

/**
* We need one static instance of the factory for our C 'main'
* function
*/
KInstance *KOrganizerFactory::s_instance = 0L;
KAboutData *KOrganizerFactory::s_about = 0L;

KOrganizerFactory::KOrganizerFactory()
{
}

KOrganizerFactory::~KOrganizerFactory()
{
  delete s_instance;
  s_instance = 0;
  delete s_about;
}

KParts::Part *KOrganizerFactory::createPartObject(QWidget *parentWidget, const char *widgetName,
                                   QObject *parent, const char *name,
                                   const char*,const QStringList& args )
{
  KParts::Part *obj = new KOrganizerPart(parentWidget, widgetName,
                                         parent, name, args );
  return obj;
}

KInstance *KOrganizerFactory::instance()
{
  if ( !s_instance ) {
    s_about = new KAboutData("korganizer", I18N_NOOP("KOrganizer"),"1.99");
    s_instance = new KInstance(s_about);
  }

  kdDebug(5850) << "KOrganizerFactory::instance(): Name: " <<
               s_instance->instanceName() << endl;

  return s_instance;
}

KOrganizerPart::KOrganizerPart(QWidget *parentWidget, const char *widgetName,
                               QObject *parent, const char *name,
                               const QStringList& args ) :
  KParts::ReadOnlyPart(parent, name),
  DCOPObject("KOrganizerSyncIface")
{
  // Setup alarms. MUST be done before instancing CalendarIMAP
  AlarmClient::startAlarmDaemon();
  AlarmClient::startAlarmClient();

  setInstance(KOrganizerFactory::instance());

  // create a canvas to insert our widget
  QWidget *canvas = new QWidget(parentWidget, widgetName);
  canvas->setFocusPolicy(QWidget::ClickFocus);
  setWidget(canvas);

  m_extension = new KOrganizerBrowserExtension(this);

  QVBoxLayout *topLayout = new QVBoxLayout(canvas);

  KGlobal::iconLoader()->addAppDir("korganizer");
  KGlobal::locale()->insertCatalogue("korganizer");

  bool bIMAP = args.contains( "storage=imap" );
  QString initName, initEmail;
  for( QStringList::const_iterator it = args.begin(); it != args.end(); ++it ) {
      if( (*it).left( 5 ) == "name=" )
          initName = (*it).mid( 5 );
      if( (*it).left( 6 ) == "email=" )
          initEmail = (*it).mid( 6 );
  }

  if( !initName.isEmpty() )
    KOPrefs::instance()->setFullName( initName );
  if( !initEmail.isEmpty() )
    KOPrefs::instance()->setEmail( initEmail );

  widget = new CalendarView(canvas,"calendarview",bIMAP);
  topLayout->addWidget(widget);
  widget->show();

  (void)new KAction(i18n("&List"), "list", 0,
                    widget->viewManager(), SLOT(showListView()),
                    actionCollection(), "view_list");
  (void)new KAction(i18n("&Day"), "1day", 0,
                    widget->viewManager(), SLOT(showDayView()),
                    actionCollection(), "view_day");
  (void)new KAction(i18n("W&ork Week"), "5days", 0,
                    widget->viewManager(), SLOT(showWorkWeekView()),
                    actionCollection(), "view_workweek");
  (void)new KAction(i18n("&Week"), "7days", 0,
                    widget->viewManager(), SLOT(showWeekView()),
                    actionCollection(), "view_week");
  (void)new KAction(i18n("&Next X Days"), "nextXdays", 0,
                    widget->viewManager(), SLOT(showNextXView()),
                    actionCollection(), "view_nextx");
  (void)new KAction(i18n("&Month"), "month", 0,
                    widget->viewManager(), SLOT(showMonthView()),
                    actionCollection(), "view_month");
  (void)new KAction(i18n("&To-Do List"), "todo", 0,
                    widget->viewManager(), SLOT(showTodoView()),
                    actionCollection(), "view_todo");
  (void)new KAction(i18n("&Notes"), "notes", 0,
                    widget->viewManager(), SLOT(showNotesView()),
                    actionCollection(), "view_notes");

  setXMLFile( "korganizer_part.rc" );
  kdDebug(5850) << "KOrganizerPart::KOrganizerPart() done" << endl;
}

KOrganizerPart::~KOrganizerPart()
{
  widget->writeSettings();

  closeURL();
}

bool KOrganizerPart::openFile()
{
  widget->openCalendar(m_file);
  widget->show();
  return true;
}


// Groupware slots
void KOrganizerPart::slotSetGroupwareCommunicationEnabled( QObject* kmailMainWin )
{
  CalendarIMAP* calendarImap = dynamic_cast<CalendarIMAP*>( widget->calendar() );
  calendarImap->setTarget( kmailMainWin );
  connect( calendarImap, SIGNAL( updateView( ) ),
	   widget, SLOT( updateView() ) );
  connect( calendarImap, SIGNAL( incidenceConflict( Incidence*, Incidence*, int& ) ),
	   widget, SLOT( slotIncidenceConflict( Incidence*, Incidence*, int& ) ) );

  KOGroupware::create( kmailMainWin, widget, widget->calendar() );
  KOPrefs::instance()->mGroupwareCommunication = true;
}


void KOrganizerPart::slotResourceRequest( const QValueList<QPair<QDateTime, QDateTime> >& busy,
                                          const QCString& resource,
                                          const QString& vCalIn,
                                          bool& vCalInOK,
                                          QString& vCalOut,
                                          bool& vCalOutOK,
                                          bool& isFree,
                                          QDateTime& start, QDateTime& end )
{
  if( KOPrefs::instance()->mGroupwareCommunication )
    KOGroupware::instance()->incomingResourceRequest( busy, resource,
						      vCalIn, vCalInOK,
						      vCalOut, vCalOutOK,
						      isFree,
						      start, end );
  else
    kdDebug(5850) << "KOrganizerPart::slotResourceRequest(): this functionality is only supported in Groupware mode" << endl;
}

void KOrganizerPart::slotEventRequest(const QCString& receiver,
				      const QString& vCalIn,
                                      bool& vCalInOK,
                                      QString& choice,
                                      QString& vCalOut,
                                      bool& vCalOutOK )
{
  choice = "unknown";
  if( KOPrefs::instance()->mGroupwareCommunication ){
    KOGroupware::EventState res =
      KOGroupware::instance()->incomingEventRequest( KOGroupware::Request,
						     receiver,
						     vCalIn, vCalInOK,
						     vCalOut, vCalOutOK );
    switch( res ){
    case KOGroupware::Accepted:
      choice = "accept"; // do *not* change this name: it matches a string in KMail
      break;
    case KOGroupware::ConditionallyAccepted:
      choice = "accept conditionally"; // do *not* change this name: it matches a string in KMail
      break;
    case KOGroupware::Declined:
      choice = "decline"; // do *not* change this name: it matches a string in KMail
      break;
    default:
      choice = "unknown"; // do *not* change this name: it matches a string in KMail
      break;
    }
  }
  else
    kdDebug(5850) << "KOrganizerPart::slotEventRequest(): this functionality is only supported in Groupware mode" << endl;
}


void KOrganizerPart::slotAcceptedEvent( bool tentative,
					const QCString& receiver,
					const QString& vCalIn,
                                        bool& vCalInOK, QString& vCalOut,
                                        bool& vCalOutOK )
{
  if( KOPrefs::instance()->mGroupwareCommunication )
    KOGroupware::instance()->incomingEventRequest( tentative ? KOGroupware::ConditionallyAccepted : KOGroupware::Accepted,
						   receiver,
						   vCalIn, vCalInOK,
						   vCalOut, vCalOutOK );
  else
    kdDebug(5850) << "KOrganizerPart::slotAcceptedEvent(): this functionality is only supported in Groupware mode" << endl;
}


void KOrganizerPart::slotRejectedEvent( const QCString& receiver,
					const QString& vCalIn, bool& vCalInOK,
                                        QString& vCalOut, bool& vCalOutOK )
{
  if( KOPrefs::instance()->mGroupwareCommunication )
    KOGroupware::instance()->incomingEventRequest( KOGroupware::Declined,
						   receiver,
						   vCalIn, vCalInOK,
						   vCalOut, vCalOutOK );
  else
    kdDebug(5850) << "KOGroupware::slotRejectedEvent(): this functionality is only supported in Groupware mode" << endl;
}


void KOrganizerPart::slotIncidenceAnswer( const QCString& sender, const QString& vCalIn,
                                          QString& vCalOut )
{
  if( KOPrefs::instance()->mGroupwareCommunication )
    KOGroupware::instance()->incidenceAnswer( sender, vCalIn, vCalOut );
  else
    kdDebug(5850) << "KOGroupware::slotIncidenceAnswer(): this functionality is only supported in Groupware mode" << endl;
}


void KOrganizerPart::ensureWeAreVisible()
{
  emit signalKOrganizerShow( true );
}


void KOrganizerPart::slotPrint()
{
  ensureWeAreVisible();
  widget->print();
}


void KOrganizerPart::slotPrintPreview()
{
  ensureWeAreVisible();
  widget->printPreview();
}


void KOrganizerPart::slotFileMerge()
{
  ensureWeAreVisible();
  KURL url = KFileDialog::getOpenURL( "", i18n("*.vcs *.ics|Calendar files"), widget );
  if( url.isEmpty() || url.isMalformed() )
    return;

  QString tmpFile;
  if( KIO::NetAccess::download(url, tmpFile) ) {
    bool success = dynamic_cast<CalendarIMAP*>( widget->calendar() )->load( tmpFile );
    KIO::NetAccess::removeTempFile( tmpFile );
    if( success ) {
      //widget->showStatusMessage( i18n("Merged calendar '%1'.").arg( url.prettyURL() ) );
      return;
    }
  }

  KMessageBox::error( widget,
		      i18n("Cannot download calendar from '%1'.").arg( url.prettyURL() ) );
}


void KOrganizerPart::slotFileArchive()
{
  ensureWeAreVisible();
  widget->archiveCalendar();
}


void KOrganizerPart::slotExportICalendar()
{
  ensureWeAreVisible();
  widget->exportICalendar();
}


void KOrganizerPart::slotExportVCalendar()
{
  ensureWeAreVisible();
  widget->exportVCalendar();
}


void KOrganizerPart::slotExportWebCalendar()
{
  ensureWeAreVisible();
  ExportWebDialog *dlg = new ExportWebDialog( widget->calendar(), widget );
  dlg->show();
}

void KOrganizerPart::slotPurgeCompleted()
{
  ensureWeAreVisible();
  widget->purgeCompleted();
}


void KOrganizerPart::slotShowWhatsNextView()
{
  ensureWeAreVisible();
  widget->viewManager()->showWhatsNextView();
}


void KOrganizerPart::slotShowListView()
{
  ensureWeAreVisible();
  widget->viewManager()->showListView();
}


void KOrganizerPart::slotShowDayView()
{
  ensureWeAreVisible();
  widget->viewManager()->showDayView();
}


void KOrganizerPart::slotShowWorkWeekView()
{
  ensureWeAreVisible();
  widget->viewManager()->showWorkWeekView();
}


void KOrganizerPart::slotShowWeekView()
{
  ensureWeAreVisible();
  widget->viewManager()->showWeekView();
}


void KOrganizerPart::slotShowNextXView()
{
  ensureWeAreVisible();
  widget->viewManager()->showNextXView();
}


void KOrganizerPart::slotShowMonthView()
{
  ensureWeAreVisible();
  widget->viewManager()->showMonthView();
}


void KOrganizerPart::slotShowTodoView()
{
  ensureWeAreVisible();
  widget->viewManager()->showTodoView();
}


void KOrganizerPart::slotShowJournalView()
{
  ensureWeAreVisible();
  widget->viewManager()->showJournalView();
}

void KOrganizerPart::slotShowNotesView()
{
  ensureWeAreVisible();
  widget->viewManager()->showNotesView();
}

void KOrganizerPart::slotShowCalendarView()
{
  static bool first = true;

  ensureWeAreVisible();
  if( first ) {
    // On the first visit to the organizer, we set a three day view
    first = false;
    widget->viewManager()->showNextXView();
  } else
    // ... on all else we use what the user was on last
    widget->viewManager()->showAgendaView();
}


void KOrganizerPart::slotUpdate()
{
  ensureWeAreVisible();
  widget->update();
}


void KOrganizerPart::slotGoPrevious()
{
  ensureWeAreVisible();
  widget->goPrevious();
}


void KOrganizerPart::slotGoNext()
{
  ensureWeAreVisible();
  widget->goNext();
}


void KOrganizerPart::slotGoToday()
{
  ensureWeAreVisible();
  widget->goToday();
}


void KOrganizerPart::slotAppointment_new()
{
  ensureWeAreVisible();
  widget->newFloatingEvent();
}


void KOrganizerPart::slotNewTodo()
{
  ensureWeAreVisible();
  widget->newTodo();
}


void KOrganizerPart::slotSubTodo()
{
  ensureWeAreVisible();
  widget->newSubTodo();
}


void KOrganizerPart::slotDeleteIncidence()
{
  ensureWeAreVisible();
  widget->deleteIncidence();
}


void KOrganizerPart::slotEditIncidence()
{
  ensureWeAreVisible();
  widget->editIncidence();
}


void KOrganizerPart::slotTodo_unsub()
{
  ensureWeAreVisible();
  widget->todo_unsub();
}


void KOrganizerPart::slotConfigure()
{
  ensureWeAreVisible();
  widget->edit_options();
}


void KOrganizerPart::slotToggleFilterView( bool visible )
{
  ensureWeAreVisible();
  widget->showFilter( visible );
}

void KOrganizerPart::slotNewNote()
{
  ensureWeAreVisible();
  KOViewManager* viewManager = widget->viewManager();
  viewManager->showNotesView();
  // Now we can rely on there actually being a Notes view.
  viewManager->notesView()->newNote();
}


void KOrganizerPart::slotConfigureDateTime()
{
  ensureWeAreVisible();

  // this is directly taken from KOrganizer::configureDateTime(). We
  // can't call that since we don't have a main window in a part.
  KProcess *proc = new KProcess;
  *proc << "kcmshell" << "language";

  connect(proc,SIGNAL(processExited(KProcess *)),
          SLOT(slotConfigureDateTimeFinished(KProcess *)));

  if (!proc->start()) {
    KMessageBox::sorry( widget,
                        i18n("Couldn't start control module for date and time format."));
  }
}


void KOrganizerPart::slotConfigureDateTimeFinished( KProcess* proc )
{
  delete proc;
}



void KOrganizerPart::slotEditFilters()
{
  ensureWeAreVisible();
  widget->editFilters();
}


void KOrganizerPart::slotShowCategoryEditDialog()
{
  ensureWeAreVisible();
  widget->dialogManager()->showCategoryEditDialog();
}


void KOrganizerPart::slotUpdateView(const QDateTime& start, const QDateTime& end)
{
  widget->updateView(start, end);
}

void KOrganizerPart::slotRefreshAll( const QStringList&,
                                     const QStringList& notes,
                                     const QStringList& )
{
  slotRefreshNotes( notes );
}


void KOrganizerPart::slotRefreshNotes( const QStringList& notes )
{
  if( KOPrefs::instance()->mGroupwareCommunication )
    KOGroupware::instance()->refreshNotes( notes );
  else
    kdDebug(5850) << "KOGroupware::slotRefreshNotes(): this functionality is only supported in Groupware mode" << endl;
}

/*!
  This DCOP interface asks KOrganizer to save its calendar data into
  the specified file. It will typically be invoked by KPilot when the
  sync button is pressed.
  This method is not yet supported in non-Groupware mode.
*/
void KOrganizerPart::pullSyncData( QString filename )
{
  if( KOPrefs::instance()->mGroupwareCommunication )
    KOGroupware::instance()->pullSyncData( filename );
#if 0
  // FIXME: save() is no longer on the Calendar class
  else
    widget->calendar()->save( filename );
#else
  else
    kdDebug(5850) << "KOGroupware::pullSyncData(): this functionality is only supported in Groupware mode" << endl;
#endif
}


/*!
  This DCOP interface asks KOrganizer to discard the calendar data it
  has currently stored and reload it from the specified file. It will
  typically be invoked by KPilot after a sync is completed.
  This method is not yet supported in non-Groupware mode.
*/
void KOrganizerPart::pushSyncData( QString filename )
{
  if( KOPrefs::instance()->mGroupwareCommunication )
    KOGroupware::instance()->pushSyncData( filename );
#if 0
  // FIXME: save() is no longer on the Calendar class
  else
    widget->calendar()->load( filename );
#else
  else
    kdDebug(5850) << "KOGroupware::pushSyncData(): this functionality is only supported in Groupware mode" << endl;
#endif
}


/*!
  This slot is called when the user selects to publish his/her free/busy list.
*/

void KOrganizerPart::slotPublishFreeBusy()
{
  if( KOPrefs::instance()->mGroupwareCommunication )
    KOGroupware::instance()->publishFreeBusy();
  else
    kdDebug(5850) << "KOGroupware::slotPublishFreeBusy(): this functionality is only supported in Groupware mode" << endl;
}


KOrganizerBrowserExtension::KOrganizerBrowserExtension(KOrganizerPart *parent) :
  KParts::BrowserExtension(parent, "KOrganizerBrowserExtension")
{
}

KOrganizerBrowserExtension::~KOrganizerBrowserExtension()
{
}

using namespace KParts;
#include "korganizer_part.moc"




