/*
  Copyright (C) 2008 by Eike Hein <hein@kde.org>

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License as
  published by the Free Software Foundation; either version 2 of
  the License or (at your option) version 3 or any later version
  accepted by the membership of KDE e.V. (or its successor appro-
  ved by the membership of KDE e.V.), which shall act as a proxy 
  defined in Section 14 of version 3 of the license.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program. If not, see http://www.gnu.org/licenses/.
*/


#include "skinlistdelegate.h"
#include "appearancesettings.h"

#include <QModelIndex>
#include <QPainter>


#define MARGIN 3
#define ICON 32


SkinListDelegate::SkinListDelegate(QObject* parent) : QAbstractItemDelegate(parent)
{
}

SkinListDelegate::~SkinListDelegate()
{
}

void SkinListDelegate::paint(QPainter* painter, const QStyleOptionViewItem& option, const QModelIndex &index) const
{
    painter->save();

    setupPaint(painter, option);

    paintIcon(painter, option, index);

    paintText(painter, option, index);

    painter->restore();
}

void SkinListDelegate::setupPaint(QPainter* painter, const QStyleOptionViewItem& option) const
{
    QPalette::ColorGroup cg = option.state & QStyle::State_Enabled ? QPalette::Normal : QPalette::Disabled;
    if (cg == QPalette::Normal && !(option.state & QStyle::State_Active)) cg = QPalette::Inactive;

    if (option.showDecorationSelected && (option.state & QStyle::State_Selected))
    {
        painter->fillRect(option.rect, option.palette.brush(cg, QPalette::Highlight));
        painter->setPen(option.palette.color(cg, QPalette::HighlightedText));
    }
}

void SkinListDelegate::paintIcon(QPainter* painter, const QStyleOptionViewItem& option, const QModelIndex &index) const
{
    QVariant value;

    value = index.data(AppearanceSettings::SkinIcon);

    if (value.isValid() && value.type() == QVariant::Icon)
    {
        int x = option.rect.x() + MARGIN;
        int y = option.rect.y() + (option.rect.height() / 2) - (ICON / 2);

        if (option.direction == Qt::RightToLeft)
            x = option.rect.right() - ICON - MARGIN;

        qvariant_cast<QIcon>(value).paint(painter, x, y, ICON, ICON);
    }
}

void SkinListDelegate::paintText(QPainter* painter, const QStyleOptionViewItem& option, const QModelIndex &index) const
{
    QFont font = option.font;
    int x = option.rect.x() + ICON + (3 * MARGIN);
    int y = option.rect.y();
    int width = option.rect.width() - ICON - (3 * MARGIN);

    QVariant value;

    value = index.data(AppearanceSettings::SkinName);

    if (value.isValid())
    {
        font.setBold(true);
        painter->setFont(font);
        QFontMetrics fontMetrics(font);

        QRect textRect = fontMetrics.boundingRect(value.toString());

        if (option.direction == Qt::RightToLeft)
        {
            if (width < textRect.width())
                x = option.rect.x() + MARGIN;
            else
                x = option.rect.right() - ICON - (3 * MARGIN) - textRect.width();
        }

        y += textRect.height();

        painter->drawText(x, y, fontMetrics.elidedText(value.toString(), option.textElideMode, width));
    }

    value = index.data(AppearanceSettings::SkinAuthor);

    if (value.isValid())
    {
        QString skinAuthor = i18nc("@item:intext", "by %1", value.toString());

        font.setBold(false);
        painter->setFont(font);
        QFontMetrics fontMetrics(font);

        QRect textRect = fontMetrics.boundingRect(skinAuthor);

        if (option.direction == Qt::RightToLeft)
        {
            if (width < textRect.width())
                x = option.rect.x() + MARGIN;
            else
                x = option.rect.right() - ICON - (3 * MARGIN) - textRect.width();
        }

        y += textRect.height();

        painter->drawText(x, y, fontMetrics.elidedText(skinAuthor, option.textElideMode, width));
    }
}

QSize SkinListDelegate::sizeHint(const QStyleOptionViewItem&option, const QModelIndex& index) const
{
    QFont font = option.font;
    QRect name, author;
    int width, height;

    QVariant value;

    value = index.data(AppearanceSettings::SkinName);

    if (value.isValid())
    {
        font.setBold(true);
        QFontMetrics fontMetrics(font);
        name = fontMetrics.boundingRect(value.toString());
    }

    value = index.data(Qt::UserRole + 1);

    if (value.isValid())
    {
        QString skinAuthor = i18nc("@item:intext", "by %1", value.toString());

        font.setBold(false);
        QFontMetrics fontMetrics(font);
        author = fontMetrics.boundingRect(skinAuthor);
    }

    width = qMax(name.width(), author.width());
    QRect textRect(0, 0, width, name.height() + author.height());

    width = ICON + (3 * MARGIN) + textRect.width() + MARGIN;;
    height = qMax(ICON + (2 * MARGIN), textRect.height() + (2 * MARGIN));

    return QSize(width, height);
}
