/*
 * KMLOFax
 *
 * A utility to process facsimiles received with the ELSA
 * MicroLink(tm) Office or MicroLink(tm) ISDN Office modem.
 *
 * Copyright (C) 1999-2001 Oliver Gantz <Oliver.Gantz@epost.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * ------
 * ELSA and MicroLink are trademarks of ELSA AG, Aachen.
 */

#ifndef MLOFILE_H
#define MLOFILE_H

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif 

#include <qfile.h>
#include <qdatetime.h>


#define PAGE_LEN_A4 0	// A4, 297 mm
#define PAGE_LEN_B4 1	// B4, 364 mm
#define PAGE_LEN_UL 2	// unlimited


#define MAX_IMG_BUFF  304
#define MAX_RLE_BUFF 2433
#define MAX_HUF_BUFF 2048



typedef struct {
	uchar code;
	uchar width;
	short len;
} hcode_t;



typedef struct {
	ushort params;
	int lpi;
	int bit_rate;
	int width;
	int length;
	bool comp_2d;
} page_info_t;




class FaxFile: public QFile
{
public:
	FaxFile(const QString& name);
	FaxFile();
	~FaxFile();

	void setName(const QString& name);

	virtual void close();

	void setPageWidth(int width);
	int pageWidth();
	int pageHeight();
	int pageUsedHeight();
	void set2DCompression(bool enable);
	void setMsbFirst(bool enable);

	virtual void rewindPage();

	void readRleLine(short line[]);
	void readImgLine(uchar line[], bool lsb_first);
	int readHuffLine(uchar line[]);

protected:
	void init();
	void initPage();

	virtual int getDataCh();
	bool getDataBits(uchar width);
	void cutDataBits(uchar width);
	bool codeMatches(const uchar code, const uchar width);
	short findMatch(const hcode_t table[]);
	bool skipEol();
	void fillDataLine(uchar line[], short pos, short len, bool lsb_first);
	short getRunLength(const hcode_t table[], uchar max_width);

	bool decode1DDataLine(short run_lengths[], bool probeonly);
	bool decode2DDataLine(short run_lengths[], bool probeonly);
	void decodeDataLine(short run_length[], bool probeonly);

	int storeDataCode(uchar dest[], int pos, const hcode_t& code);
	int encodeDataLine(uchar dest[]);

	int m_width, m_height, m_usedheight;
	int m_bpl;
	bool m_2d;
	bool m_msbfirst;
	short refline[MAX_RLE_BUFF];

	ulong data_buff;
	uchar data_width;
	bool data_left;
};


inline int FaxFile::pageWidth()
{ return m_width; }


inline void FaxFile::set2DCompression(bool enable)
{ m_2d = enable; }


inline void FaxFile::setMsbFirst(bool enable)
{ m_msbfirst = enable; }




class MLOFile : public FaxFile
{
public:
	MLOFile(const QString& name);
	MLOFile();
	~MLOFile();

	bool open(int m);
	virtual void close();

	int readLine(char *line, uint maxlen);

	QString sender();

	int pages();

	bool gotoPage(int page);
	virtual void rewindPage();

	int pageHeight();
	int pageUsedHeight();

	bool fineResolution();

protected:
	void init();

	virtual int getDataCh();

	int m_pages, m_page;
	int m_datapos;

	QString m_sender;

	bool m_fine;
	ushort m_params;
	int m_length;
	int m_bitrate;
};


inline bool MLOFile::fineResolution()
{ return m_fine; }


typedef struct {
	Q_UINT16 tdir_tag;
	Q_UINT16 tdir_type;
	Q_UINT32 tdir_count;
	Q_UINT32 tdir_offset;
} TIFFDirEntry;

typedef enum {
	TIFF_NOTYPE   = 0,
	TIFF_BYTE     = 1,
	TIFF_ASCII    = 2,
	TIFF_SHORT    = 3,
	TIFF_LONG     = 4,
	TIFF_RATIONAL = 5
} TIFFDataType;



class TiffFile: public FaxFile
{
public:
	TiffFile(const QString& name);
	TiffFile();
	~TiffFile();

	bool open(int m);
	virtual void close();

	int pages();

	int pageHeight();
	int pageUsedHeight();

	bool fineResolution();

	void setSender(const QString& sender_);
	QString sender();

	void setTime(const QDateTime& time_);
	void setTime(int year, int month, int day, int hour, int min, int sec);
	QDateTime time();

	void addPage(int page, int pages, int width, int height, bool fine);
	bool finishPage();
	
	bool gotoPage(int page);
	virtual void rewindPage();
	
	void readImgLine(uchar line[], bool lsb_first);
	
	void writeRleLine(short line[]);


protected:
	void init();
	bool readShort(Q_UINT16 *v);
	bool writeShort(Q_UINT16 v);
	bool readLong(Q_UINT32 *v);
	bool writeLong(Q_UINT32 v);
	bool readDirEntry(TIFFDirEntry *entry);
	bool writeDirEntry(int tag, TIFFDataType type, int count, uint offset);
	bool getStrField(TIFFDirEntry *entry, QString *s);
	bool getIntField(TIFFDirEntry *entry, int *i);
	bool getDoubleField(TIFFDirEntry *entry, double *d);

	bool gotoStrip();

	bool m_swabbyteorder;
	int m_pages;
	
	int m_page;
	bool m_fine;
	int m_stripoff;
	int m_rowsperstrip;
	int m_bytesperstrip;
	QString m_sender;
	QDateTime m_time;
	
	int m_strip;
	int m_row;
};


inline int TiffFile::pageHeight()
{ return m_height; }

inline bool TiffFile::fineResolution()
{ return m_fine; }

inline void TiffFile::setSender(const QString& sender_)
{ m_sender = sender_; }

inline void TiffFile::setTime(const QDateTime& time_)
{ m_time = time_; }

inline void TiffFile::setTime(int year, int month, int day, int hour, int min, int sec)
{ m_time.date().setYMD(year, month, day); m_time.time().setHMS(hour, min, sec); }



#endif // MLOFILE_H
