/*
 *  KSeg
 *  Copyright (C) 1999 Ilya Baran
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Send comments and/or bug reports to:
 *                 ibaran@mit.edu
 */


#include <qlayout.h>
#include <qpushbutton.h>
#include "KSegCalculateEditor.H"
#include "G_object.H"
#include "kapp.h"
#include "KSegDocument.H"
#include "KSegWindow.H"
#include <kiconloader.h>
#include <kglobal.h>
#include <kstddirs.h>


class CalculationDialogReferenceFetcher : public ReferenceFetcher
{
public:
  CalculationDialogReferenceFetcher(KSegCalculateEditor *inRef) { ref = inRef; }

  double getValue(int reference_number)
  { return ref->getOutputParents()[reference_number]->getObject()->getNumValue(); }

  KFormula *getFormula(int reference_number)
  { return ((G_valueObject *)(ref->getOutputParents()[reference_number]->getObject()))->getLhs(); }

protected:
  KSegCalculateEditor * ref;
};


KSegCalculateEditor::KSegCalculateEditor(KSegView *view, QString initialString,
					 G_refs initialParents, G_ref *inCurRef)
  : QDialog(view, 0, false), outputParents(initialParents), curRef(inCurRef)
{
  QPushButton *cancel;

  QVBoxLayout *vlayout = new QVBoxLayout(this);

  view->installEventFilter(this);
  view->setMenusEnabled(false);

  //now disable all the other views
  unsigned int i;

  QList<KSegView> allViews = KSegView::getAllViews();

  for(i = 0; i < allViews.count(); ++i) {
    if(allViews.at(i) == view) continue;
    allViews.at(i)->topLevelWidget()->setEnabled(false);
  }

  EditorToolBar *tb = new EditorToolBar(this);
  tb->enableFloating(false);
  tb->enableMoving(false);
  vlayout->addWidget(tb);

  valueDisplay = new QLabel("Value is undefined", this);
  vlayout->addWidget(valueDisplay);
  
  editor = new KFormulaEdit(this, 0, 0, true);
  tb->connectToFormula(editor);
  setFocusProxy(editor);
  editor->getFormula()->setReferenceFetcher(new CalculationDialogReferenceFetcher(this));
  editor->enableSizeHintSignal(true);
  connect(editor, SIGNAL(formulaChanged(const QString &)),
	  this, SLOT(formulaChanged(const QString &)));
  editor->setText(initialString);

  vlayout->addWidget(editor, 1);
  
  QHBoxLayout *hlayout = new QHBoxLayout(vlayout);

  ok = new QPushButton( "OK", this );
  hlayout->addWidget(ok);
  QObject::connect( ok, SIGNAL(clicked()), this, SLOT(OKClicked()) );
  ok->setDefault(true);
  cancel = new QPushButton( "Cancel", this );
  hlayout->addWidget(cancel);
  QObject::connect( cancel, SIGNAL(clicked()), this, SLOT(reject()) );

  formulaChanged(initialString);
}


void KSegCalculateEditor::run()
{
  finished = false;
  show();
  while(!finished) kapp->processOneEvent(); 
}

KSegCalculateEditor::~KSegCalculateEditor()
{
  ((KSegView *)parent())->setMenusEnabled(true);

  unsigned int i;

  QList<KSegView> allViews = KSegView::getAllViews();

  for(i = 0; i < allViews.count(); ++i) {
    allViews.at(i)->topLevelWidget()->setEnabled(true);
  }
}


void KSegCalculateEditor::OKClicked()
{
  outputString = editor->text();
  accept();
}


void KSegCalculateEditor::formulaChanged(const QString & /*newTextDummy*/)
{
  //remove unnecessary parents
  unsigned int i;

  QString newText = editor->text();        
  
  outputString = newText;

  for(i = 0; i < outputParents.size(); ++i) {
    if(!(outputString.contains(QChar(int(REFERENCE_NUM(i)))))) {
      removeParent(i);
      i--;
    }
  }

  //now make sure no reference is too high--if it is, delete it!
  for(i = 0; i < outputString.length(); ++i) {
    if(!IS_REFERENCE(outputString[i].unicode())) continue;

    if(outputString[i].unicode() >= REFERENCE_NUM(outputParents.size())) {
      outputString.remove(i, 1);
    }
  }

  if(outputString != newText) editor->setText(outputString);

  updateValue();
}


void KSegCalculateEditor::removeParent(int i)
{
  unsigned int j;

  outputParents.remove(i);

  for(j = 0; j < outputString.length(); j++) {
    if(!IS_REFERENCE(outputString[j].unicode())) continue;

    if(outputString[j].unicode() > REFERENCE_NUM(i)) {
      outputString[j] = outputString[j].unicode() - 1;
    }
  }

  return;
}


void KSegCalculateEditor::updateValue()
{
  int error;
  double value = 0;
  
  unsigned int i;

  for(i = 0; i < outputParents.count(); ++i) {
    if(fabs(outputParents[i]->getObject()->getNumValue()) >= BIG) {
      value = BIG;
    }
  }
  
  if(value != BIG) {
    value = editor->getFormula()->evaluate(QDict<double>(), &error);
  }

  if(error != NO_ERROR) {
    value = BIG;
    ok->setEnabled(false);
  }
  else ok->setEnabled(true);

  QString t("Value is %1");
  if(value != BIG) t = t.arg(QString::number(value, 'f', 4));
  else t = t.arg("undefined");

  valueDisplay->setText(t);
 
}

bool KSegCalculateEditor::eventFilter(QObject *o, QEvent *e)
{
  if(o != parent()) return false;

  if(e->type() == QEvent::MouseMove) {
    QMouseEvent *m = (QMouseEvent *)e;
    KSegView *v = (KSegView *)parent();
    
    G_refs tmp =
      v->getDocument()->whatAmIOn(m->x() + v->getOffsetX(), m->y() + v->getOffsetY(), false);
    
    unsigned int i;
    for(i = 0; i < tmp.size(); ++i) {
      if(tmp[i] == curRef) continue;
      if(tmp[i]->getType() & G_VALUE && (curRef == 0 || !curRef->isDescend(tmp[i]))) {
	parentWidget()->setCursor(upArrowCursor);

	QString message;

	message = QString("Insert ") + G_ref::getNameFromType(tmp[i]->getType()) + " " +
	  KFormula::toUgly(tmp[i]->getLabel().getText());
	
	((KSegWindow *)(parentWidget()->topLevelWidget()))->statusBarMessage(message);
	return true;
      }
    }
    parentWidget()->unsetCursor();
    ((KSegWindow *)(parentWidget()->topLevelWidget()))->statusBarMessage("Ready");
    return true;
  }
  if(e->type() == QEvent::MouseButtonRelease ||
     e->type() == QEvent::MouseButtonDblClick) {
    return true;
  }
  if(e->type() == QEvent::MouseButtonPress) {
    QMouseEvent *m = (QMouseEvent *)e;
    KSegView *v = (KSegView *)parent();
    
    G_refs tmp =
      v->getDocument()->whatAmIOn(m->x() + v->getOffsetX(), m->y() + v->getOffsetY(), false);
    
    unsigned int i;
    for(i = 0; i < tmp.size(); ++i) {
      if(tmp[i] == curRef) continue;
      if(tmp[i]->getType() & G_VALUE && (curRef == 0 || !curRef->isDescend(tmp[i]))) {
	insertReference(tmp[i]);
	break;
      }
    }

    setActiveWindow();

    return true;
  }

  return false;
}


void KSegCalculateEditor::insertReference(G_ref *r)
{
  //first try to find r in parents:
  int i = outputParents.find(r);

  if(i != -1) {
    editor->insertChar(REFERENCE_NUM(i));
    return;
  }

  if(outputParents.size() + 1 >= MAX_REFERENCES) return; // just in case

  outputParents.append(r);
  editor->insertChar(REFERENCE_NUM(outputParents.size() - 1));  
}


using namespace Box;

EditorToolBar::EditorToolBar(QWidget *parent, const char *name, int _item_size)
  : KToolBar(parent, name, _item_size)
{
  // doesn't matter if we do this twice - KStandardDirs ignores doubled additions
  KGlobal::dirs()->addResourceType("toolbar",
				   KStandardDirs::kde_default("data") + "kformula/pics/");
  insertButton(BarIcon("editcut"), CUT_CHAR, true, "Cut (Ctrl + X)");
  insertButton(BarIcon("editcopy"), COPY_CHAR, true, "Copy (Ctrl + C)");
  insertButton(BarIcon("editpaste"), PASTE_CHAR, true, "Paste (Ctrl + V)");
  
  insertSeparator();
  
  insertButton(BarIcon("undo"), UNDO_CHAR, true, "Undo (Ctrl + Z)");
  insertButton(BarIcon("redo"), REDO_CHAR, true, "Redo (Ctrl + R)");

  insertSeparator();

  insertButton(BarIcon("rsup"), POWER, true, "Power (Ctrl + 6)");
  insertButton(BarIcon("paren"), PAREN, true, "Parentheses (Shift + 9)");
  insertButton(BarIcon("abs"), ABS, true, "Absolute value (Shift + \\)");
  insertButton(BarIcon("frac"), DIVIDE, true, "Fraction (Ctrl + /)");
  insertButton(BarIcon("sqrt"), SQRT, true, "Root (Ctrl + 2)");
}

void EditorToolBar::connectToFormula(KFormulaEdit *formula)
{
  QObject::connect(this, SIGNAL(clicked(int)), formula, SLOT(insertChar(int)));
}

#include "KSegCalculateEditor.moc"
