/*
 *  KSeg
 *  Copyright (C) 1999 Ilya Baran
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Send comments and/or bug reports to:
 *                 ibaran@mit.edu
 */

#include "KSegConstruction.H"
#include "KSegView.H"
#include "G_refs.H"
#include "G_ref.H"
#include "G_object.H"
#include <hash_map>
#include "G_refSearcher.H"

std::vector<KSegConstruction *> KSegConstruction::constructions;

KSegConstruction::KSegConstruction()
{
  constructions.push_back(this);
}

KSegConstruction::~KSegConstruction()
{
  constructions.erase(std::find(constructions.begin(),
				constructions.end(), this));
}


void KSegConstruction::addRef(G_ref *ref)
{
  KSegDocument::addRef(ref);

  //givens will be added into the right place by themselves
  if(ref->getFinal()) final.append(ref);
  if(ref->getInitial()) initial.append(ref);
  if(ref->getType() == G_LOOP) loops.append(ref);
}

void KSegConstruction::delRef(G_ref *ref)
{
  KSegDocument::delRef(ref);

  if(ref->getGiven()) given.removeRef(ref);
  if(ref->getFinal()) final.removeRef(ref);
  if(ref->getInitial()) initial.removeRef(ref);  
  if(ref->getType() == G_LOOP) loops.removeRef(ref);
}

void KSegConstruction::play(G_refs givens, KSegDocument *doc, int iter, bool first)
{
  static hash_map<G_ref *, G_ref *> initialMap; //map of the initial objects
  static NotInConstructionSearcher nicSearcher;
  if(first) {
    initialMap.clear();
    nicSearcher.reInit();
  }

  hash_map<G_ref *, G_ref *> currentMap = initialMap;

  currentMap.resize(allRefs.size());

  int i, j;

  for(i = 0; i < (int)given.count(); i++) {
    currentMap[given[i]] = givens[i];
  }

  QRect win = KSegView::constructionPainter->window();
  if(win.width() > 100 && win.height() > 100) {
    win.setSize(QSize(win.width() - 60, win.height() - 60));
    win.moveBy(30, 30);
  }

  //main loop over all the refs in this construction
  for(i = 0; i < (int)allRefs.count(); i++) {

    //if loop
    if(allRefs[i]->getType() == G_LOOP) {
      if(iter) {
	G_refs newGivens;
	
	for(j = 0; j < int(allRefs[i]->getParents().count()); j++) {
	  newGivens.append(currentMap[allRefs[i]->getParents()[j]]);
	}
	
	play(newGivens, doc, iter - 1, false);
      }
      
      continue;
    }

    //should it be constructed?
    if(allRefs[i]->getGiven()) continue;
    if(allRefs[i]->getFinal() && iter != 0) continue;
    if(allRefs[i]->getInitial() && !first) continue;
    if(nicSearcher.search(allRefs[i]) == false) continue;
    
    //construct it!
    G_refs newParents;

    //set up the parents
    bool construct = true; //will be marked false if any parents are missing
                           // like because they are final

    for(j = 0; j < int(allRefs[i]->getParents().count()); j++) {
      if(currentMap.count(allRefs[i]->getParents()[j]) == 0) {
	construct = false;
	break;
      }
      newParents.append(currentMap[allRefs[i]->getParents()[j]]);      
    }

    if(!construct) continue;

    G_ref *r = new G_ref;

    //creation!
    //check if it's a transform becuase then the type depends on the parent:
    if((allRefs[i]->getType() & G_CURVE) && IS_TRANSFORM(allRefs[i]->getDescendType())) {
      r->create(newParents[0]->getType(), allRefs[i]->getDescendType(), newParents, doc);
    }
    else { //normal creation
      r->create(allRefs[i]->getType(), allRefs[i]->getDescendType(), newParents, doc);
    }

    if(r->getType() == G_CALCULATE) { //we must transfer the formula string
      ((G_calculateObject *)(r->getObject()))->
	setInitialFormulaString(((G_calculateObject *)(allRefs[i]->getObject()))->getFormulaString());
      r->update();
    }

    //set the right draw style and attributes
    allRefs[i]->getDrawstyle()->addReference();
    r->setDrawstyle(allRefs[i]->getDrawstyle());
    
    r->setVisible(allRefs[i]->getVisible());
    r->setLabelVisible(allRefs[i]->getLabelVisible());
    r->getLabel().setText(allRefs[i]->getLabel().getText());
    r->setLabelPos(allRefs[i]->getLabelPos());

    //randomize positions of points and measurements:
    if(r->getType() == G_POINT) {
      if(r->getDescendType() == G_CONSTRAINED_POINT) {
	((G_pointObject *)(r->getObject()))->setP(INTRAND(0, 1000) / 1000.);
	r->update();
      }
      if(r->getDescendType() == G_FREE_POINT) {
	r->getObject()->translate(G_point(INTRAND(win.left(), win.right()),
					  INTRAND(win.top(), win.bottom())));
      }
    }

    if(r->getType() & G_TEXT) {
	r->getObject()->translate(G_point(INTRAND(win.left(), win.right()),
					  INTRAND(win.top(), win.bottom())));
    }

    if(r->isDrawn()) {
      r->getObject()->draw(*(KSegView::constructionPainter));
    }

    //put it in the maps.
    currentMap[allRefs[i]] = r;
    if(allRefs[i]->getInitial()) {
      initialMap[allRefs[i]] = r;
    }
  }

  if(first) { //conserve memory
    initialMap.clear();
    nicSearcher.reInit();
  }
}


bool KSegConstruction::doesMatch(const G_refs& which)
{
  if(which.size() != given.size()) return false;
  
  int i;
  for(i = 0; i < (int)(which.size()); ++i) {
    if((which[i]->getType() & given[i]->whatCanItBe()) == 0) return false;
  }

  return true;
}


bool KSegConstruction::canMakeNormal(G_ref *which)
{
  if(which->getType() == G_LOOP) return false;
  if(which->getNormal() || which->getFinal() || which->getInitial()) return true;
  //otherwise the ref is a given.  Check if there are loops in the document
  if(loops.size() > 0) return false;

  return true;
}


bool KSegConstruction::canMakeGiven(G_ref *which)
{
  if(which->getType() == G_LOOP) return false;
  if(which->getGiven()) return true;

  //if there are loops, return false
  if(loops.size() > 0) return false;

  //now check if this is a constrained point and there is a locus driven by it
  //since if the point is given, it might not be constrained killing the locus
  if(which->getType() == G_POINT && which->getDescendType() == G_CONSTRAINED_POINT) {
    int i;

    for(i = 0; i < int(which->getChildrenConst().size()); ++i) {
      if(which->getChildrenConst()[i]->getType() != G_LOCUS) continue;
      if(which->getChildrenConst()[i]->getDescendType() != G_OBJECT_LOCUS) continue;
      if(which->getChildrenConst()[i]->getParentsConst()[0] != which) continue;
      return false;
    }
  }


  return true;
}


bool KSegConstruction::canMakeFinal(G_ref * /*which*/)
{
  //check it's not ancestor of a given or a loop
  return true;
}


bool KSegConstruction::canMakeInitial(G_ref * /*which*/)
{
  return true;
}


void KSegConstruction::makeNormal(G_ref *which)
{
  if(which->getNormal()) return;

  //insert undo here
  addUndo(new G_undoChangeConstructMode(which));
  
  if(which->getGiven()) given.removeRef(which);
  if(which->getFinal()) final.removeRef(which);
  if(which->getInitial()) initial.removeRef(which);

  which->setNormal();
}


void KSegConstruction::makeGiven(G_ref *which, int where)
{
  if(which->getGiven()) return;

  //insert undo here
  addUndo(new G_undoChangeConstructMode(which));
  
  if(which->getFinal()) final.removeRef(which);
  if(which->getInitial()) initial.removeRef(which);

  which->setGiven();
  if(where == -1) given.append(which);
  else given.insert(where, which);
}


void KSegConstruction::makeFinal(G_ref *which)
{
  if(which->getFinal()) return;

  //insert undo here
  addUndo(new G_undoChangeConstructMode(which));
  
  if(which->getGiven()) given.removeRef(which);
  if(which->getInitial()) initial.removeRef(which);

  which->setFinal();
  final.append(which);
}


void KSegConstruction::makeInitial(G_ref *which)
{
  if(which->getInitial()) return;

  //insert undo here
  addUndo(new G_undoChangeConstructMode(which));
  
  if(which->getGiven()) given.removeRef(which);
  if(which->getFinal()) final.removeRef(which);

  which->setInitial();
  initial.append(which);
}



#include "KSegConstruction.moc"
