/*
 *  KSeg
 *  Copyright (C) 1999 Ilya Baran
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Send comments and/or bug reports to:
 *                 ibaran@mit.edu
 */

#include "G_ref.H"
#include "KSegConstruction.H"
#include "KSegConstructionList.H"


#define INSERT_REF(r) \
{ fstr += G_ref::getNameFromType((r)->getType()) + " "; \
  fstr += QChar(REFERENCE_NUM(curRefNum++)); \
  l.push_back((r)->getLabel().getFormula()); }

#define INSERT_LABEL(r) \
{ fstr += QChar(REFERENCE_NUM(curRefNum++)); \
  l.push_back((r)->getLabel().getFormula()); }

#define END fstr += QString("."); f.parse(fstr); return


void KSegConstructionList::nameObject(G_ref *ref, KFormula &f,
				      ListReferenceFetcher &l, int index)
{
  int i;
  int curRefNum = 0;

  KSegConstruction *doc = (KSegConstruction *)(ref->getDocument());

  l.clear();

  QString fstr;
  fstr.sprintf("%d. ", index + 1);

  //first take care of loops
  if(ref->getType() == G_LOOP) {
    fstr += QString("Loop on ");
    for(i = 0; i < (int)ref->getParents().size(); ++i) {
      if(i > 0) fstr += QString(", ");
      
      INSERT_REF(ref->getParents()[i]);
    }

    END;
  }

  //now take care of givens
  if(ref->getGiven()) {
    i = doc->getGiven().find(ref);
    fstr.sprintf("%d. Given{}({#%d} ", index + 1, i + 1);
    INSERT_REF(ref);

    i = ref->whatCanItBe() - ref->getType(); //all other possibilities
    if(i != 0) {
      fstr += QString(" {}({or ");
      int j;
      for(j = 1; j < i; j = j * 2) {
	if(j & i) {
	  fstr += G_ref::getNameFromType(G_Type(j)) + QString(", ");
	}
      }
      fstr.remove(fstr.length() - 2, 2);
      fstr += QString("}");
    }
    
    END;
  }

  if(ref->getFinal()) fstr += QString("Final ");
  if(ref->getInitial()) fstr += QString("Initial ");

  INSERT_REF(ref);

  fstr += QString(" ");

  //point may be (final), descended from final or (not in construction)
  //having only givens as anc.

  bool implicit_final, not_in_construction, does_not_exist, not_shown;

  implicit_final = !ref->getFinal() && ifSearcher.search(ref);
  
  not_in_construction = !(nicSearcher.search(ref));

  does_not_exist = !(ref->getExists());
  not_shown = !(ref->getVisible());
 
  if(implicit_final || not_in_construction || does_not_exist || not_shown) {
    fstr += QString("{}({");
    
    if(implicit_final) fstr += QString("final, ");
    if(not_in_construction) fstr += QString("not constructed, ");
    if(does_not_exist) fstr += QString("doesn't exist, ");
    if(not_shown) fstr += QString("hidden, ");

    fstr[fstr.length() - 2] = '}';
  }

  //take care of the transforms:
  if(ref->getType() & G_GEOMETRIC && IS_TRANSFORM(ref->getDescendType())) {

    if(ref->getDescendType() == G_TRANSLATED) {
      fstr += QString("is the translation of ");
      INSERT_REF(ref->getParents()[0]);
      fstr += QString(" by the vector from ");
      INSERT_REF(ref->getParents()[1]);
      fstr += QString(" to ");
      INSERT_REF(ref->getParents()[2]);
    }

    if(ref->getDescendType() == G_ROTATED) {
      fstr += QString("is the rotation of ");
      INSERT_REF(ref->getParents()[0]);
      fstr += QString(" about ");
      INSERT_REF(ref->getParents()[1]);
      fstr += QString(" by angle ");
      INSERT_LABEL(ref->getParents()[2]);
      INSERT_LABEL(ref->getParents()[3]);
      INSERT_LABEL(ref->getParents()[4]);
    }

    if(ref->getDescendType() == G_SCALED) {
      fstr += QString("is the image of ");
      INSERT_REF(ref->getParents()[0]);
      fstr += QString(" scaled about ");
      INSERT_REF(ref->getParents()[1]);
      fstr += QString(" by the ratio ");
      INSERT_LABEL(ref->getParents()[2]);
      fstr += QString(":");
      INSERT_LABEL(ref->getParents()[3]);
    }

    if(ref->getDescendType() == G_REFLECTED) {
      fstr += QString("is the reflection of ");
      INSERT_REF(ref->getParents()[0]);
      fstr += QString(" about ");
      INSERT_REF(ref->getParents()[1]);
    }

    END;
  }

  switch(ref->getType()) {
  case G_POINT: //points---------------
    switch(ref->getDescendType()) {
    case G_FREE_POINT:
      fstr += QString("is free");
      break;

    case G_CONSTRAINED_POINT:
      fstr += QString("is on ");
      INSERT_REF(ref->getParents()[0]);
      break;

    case G_INTERSECTION_POINT:
      fstr += QString("is the intersection of ");
      INSERT_REF(ref->getParents()[0]);
      fstr += QString(" and ");
      INSERT_REF(ref->getParents()[1]);
      break;

    case G_INTERSECTION2_POINT:
      fstr += QString("is the 2") + QChar(Box::POWER) + QString("{nd} intersection of ");
      INSERT_REF(ref->getParents()[0]);
      fstr += QString(" and ");
      INSERT_REF(ref->getParents()[1]);
      break;

    case G_MID_POINT:
      fstr += QString("is the midpoint of ");
      INSERT_REF(ref->getParents()[0]);
      break;
    }
    break;

  case G_SEGMENT:
    fstr += QString("has endpoints ");
    INSERT_LABEL(ref->getParents()[0]);
    fstr += QString(" and ");
    INSERT_LABEL(ref->getParents()[1]);
    break;

  case G_RAY:
    if(ref->getDescendType() == G_TWOPOINTS_RAY) {
      fstr += QString("is from ");
      INSERT_REF(ref->getParents()[0]);
      fstr += QString(" through ");
      INSERT_REF(ref->getParents()[1]);
    }
    else if(ref->getDescendType() == G_BISECTOR_RAY) {
      fstr += QString("bisects angle ");
      INSERT_LABEL(ref->getParents()[0]);
      INSERT_LABEL(ref->getParents()[1]);
      INSERT_LABEL(ref->getParents()[2]);
    }
    break;

  case G_LINE:
    switch(ref->getDescendType()) {
    case G_TWOPOINTS_LINE:
      fstr += QString("is between ");
      INSERT_REF(ref->getParents()[0]);
      fstr += QString(" and ");
      INSERT_REF(ref->getParents()[1]);
      break;

    case G_PARALLEL_LINE:
      fstr += QString("is parallel to ");
      INSERT_REF(ref->getParents()[1]);
      fstr += QString(" through ");
      INSERT_REF(ref->getParents()[0]);
      break;

    case G_PERPENDICULAR_LINE:
      fstr += QString("is perpendicular to ");
      INSERT_REF(ref->getParents()[1]);
      fstr += QString(" through ");
      INSERT_REF(ref->getParents()[0]);
      break;
    }
    break;
    
  case G_CIRCLE:
    fstr += QString("is centered at ");
    INSERT_REF(ref->getParents()[0]);
    if(ref->getDescendType() == G_CENTERPOINT_CIRCLE) {
      fstr += QString(" and passes through ");
    }
    else {
      fstr += QString(" and has radius ");
    }
    INSERT_REF(ref->getParents()[1]);
    break;

  case G_ARC:
    fstr += QString("passes through points ");
    INSERT_LABEL(ref->getParents()[0]);
    fstr += QString(", ");
    INSERT_LABEL(ref->getParents()[1]);
    fstr += QString(", and ");
    INSERT_LABEL(ref->getParents()[2]);
    break;

  case G_LOCUS:
    fstr += QString("is traced by ");
    INSERT_REF(ref->getParents()[1]);
    fstr += QString(", driven by ");
    INSERT_REF(ref->getParents()[0]);
    break;

  case G_CALCULATE:
    fstr += QString("involves ");
    for(i = 0; i < (int)ref->getParents().size(); ++i) {
      if(i > 0) fstr += QString(", ");
      
      INSERT_REF(ref->getParents()[i]);
    }
    break;

  case G_MEASURE:
    switch(ref->getDescendType()) {
    case G_DISTANCE_MEASURE:
      fstr += QString("is the distance between ");
      INSERT_REF(ref->getParents()[0]);
      fstr += QString(" and ");
      INSERT_REF(ref->getParents()[1]);      
      break;

    case G_LENGTH_MEASURE:
      fstr += QString("is the length of ");
      INSERT_REF(ref->getParents()[0]);
      break;

    case G_RADIUS_MEASURE:
      fstr += QString("is the radius of ");
      INSERT_REF(ref->getParents()[0]);
      break;

    case G_ANGLE_MEASURE:
      if(ref->getParents()[0]->getType() == G_ARC) { //arc angle
	fstr += QString("is the angle of ");
	INSERT_REF(ref->getParents()[0]);
      }
      else { // three points
	fstr += QString("is the angle between points ");
	INSERT_LABEL(ref->getParents()[0]);
	fstr += QString(", ");
	INSERT_LABEL(ref->getParents()[1]);
	fstr += QString(", and ");
	INSERT_LABEL(ref->getParents()[2]);
      }
      break;

    case G_RATIO_MEASURE:
      fstr += QString("is the ratio of ");
      INSERT_REF(ref->getParents()[0]);
      fstr += QString(" to ");
      INSERT_REF(ref->getParents()[1]);
      break;

    case G_SLOPE_MEASURE:
      fstr += QString("is the slope of ");
      INSERT_REF(ref->getParents()[0]);
      break;

    case G_AREA_MEASURE:
      fstr += QString("is the area of ");
      INSERT_REF(ref->getParents()[0]);
      break;
    }
    break;
  
  default:
    break;
  }

  END;
}
