/***************************************************************************
                          kexdatasource.cpp  -  description
                             -------------------
    begin                : Mon Mar 4 2002
    copyright            : (C) 2002 by Tero Favorin
    email                : tero@favorin.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include <math.h>
#include <qstring.h>
#include <qlist.h>
#include <qmap.h>
#include <qfile.h>
#include <qtextstream.h>
#include <qdom.h>
#include <qbitmap.h>
#include <qpixmapcache.h>
#include <qregexp.h>

#include <kapplication.h>
#include <kstddirs.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <dcopclient.h>
#include "kexdatasource.h"

KExDataSource::KExDataSource(QWidget *w) : QObject(0),DCOPObject("DataSource") {
	parent=w;
	datain=NULL;
	precision=5;
	highPrecision=11;

  KApplication::kApplication()->dcopClient()->setDefaultObject( objId() );

	defaultFlag=QPixmap(21,14);
	defaultFlag.fill(white);
	defaultFlag.setMask(defaultFlag.createHeuristicMask());	// Create transparent Pixmap

	// Currency names
	names.insert("AED",i18n("United Arab Emirates Dirham"));
	names.insert("ANG",i18n("Neth. Antilles Guilders"));
	names.insert("ARP",i18n("Argentine Pesos"));
	names.insert("ATS",i18n("Austrian Schillings"));
	names.insert("AUD",i18n("Australian Dollars"));
	names.insert("BBD",i18n("Barbados Dollars"));
	names.insert("BEF",i18n("Belgian Francs"));
	names.insert("BGL",i18n("Bulgarian Lev"));
	names.insert("BHD",i18n("Bahraini Dinars"));
	names.insert("BMD",i18n("Bermudian Dollars"));
	names.insert("BRL",i18n("Brazilian Reals"));
	names.insert("BSD",i18n("Bahamian Dollars"));
	names.insert("CAD",i18n("Canadian Dollars"));
	names.insert("CHF",i18n("Swiss Francs"));
	names.insert("CNY",i18n("Chinese Renminbi"));
	names.insert("CLP",i18n("Chilean Pesos"));
	names.insert("CZK",i18n("Czech Koruna"));
	names.insert("COP",i18n("Colombian Pesos"));
	names.insert("CYP",i18n("Cypriot Pound"));
	names.insert("DEM",i18n("German Marks"));
	names.insert("DKK",i18n("Danish Kroner"));
	names.insert("DZD",i18n("Algerian Dinar"));
	names.insert("ECS",i18n("Ecuadoran Sucre"));
	names.insert("EGP",i18n("Egyptian Pounds"));
	names.insert("ESP",i18n("Spanish Pesetas"));
	names.insert("EUR",i18n("European Euros"));
	names.insert("FIM",i18n("Finnish Markka"));
	names.insert("FJD",i18n("Fijian Dollars"));
	names.insert("FRF",i18n("French Francs"));
	names.insert("GBP",i18n("British Pounds"));
	names.insert("GHC",i18n("Ghanan New Cedis"));
	names.insert("GRD",i18n("Greek Drachmas"));
	names.insert("HNL",i18n("Honduran Lempiras"));
	names.insert("HKD",i18n("Hong Kong Dollars"));
	names.insert("HUF",i18n("Hungarian Forint"));
	names.insert("IDR",i18n("Indonesian Rupiah"));
	names.insert("IEP",i18n("Irish Punt"));
	names.insert("ILS",i18n("Israeli New Shekels"));
	names.insert("INR",i18n("Indian Rupees"));
	names.insert("ISK",i18n("Icelandic Krona"));
	names.insert("ITL",i18n("Italian Lira"));
	names.insert("JMD",i18n("Jamaican Dollars"));
	names.insert("JOD",i18n("Jordanian Dinar"));
	names.insert("JPY",i18n("Japanese Yen"));
	names.insert("KRW",i18n("South Korean Won"));
	names.insert("KZT",i18n("Kazakhstan Tenge"));
	names.insert("KWD",i18n("Kuwaiti Dinar"));
	names.insert("LBP",i18n("Lebanese Pound"));
	names.insert("LKR",i18n("Sri Lankan Rupees"));
	names.insert("LUF",i18n("Luxembourg Francs"));
	names.insert("MAD",i18n("Moroccan Dirham"));
	names.insert("MMK",i18n("Myanmar Kyat"));
	names.insert("MTL",i18n("Maltese Lira"));
	names.insert("MYR",i18n("Malaysian Ringgit"));
	names.insert("MXN",i18n("Mexican Pesos"));
	names.insert("NLG",i18n("Dutch Guilders"));
	names.insert("NOK",i18n("Norwegian Kroner"));
	names.insert("NZD",i18n("New Zealand Dollars"));
	names.insert("PAB",i18n("Panamaan Balboas"));
	names.insert("PEN",i18n("Peruvian New Soles"));
	names.insert("PHP",i18n("Philippines Pesos"));
	names.insert("PKR",i18n("Pakistani Rupees"));
	names.insert("PLN",i18n("Polish Zloty"));
	names.insert("PTE",i18n("Portuguese Escudo"));
	names.insert("ROL",i18n("Romanian Leu"));
	names.insert("RUR",i18n("Russian Rubles"));
	names.insert("SAR",i18n("Saudi Arabian Riyal"));
	names.insert("SDD",i18n("Sudanese Dinars"));
	names.insert("SEK",i18n("Swedish Krona"));
	names.insert("SGD",i18n("Singapore Dollars"));
	names.insert("SKK",i18n("Slovakian Koruna"));
 	names.insert("THB",i18n("Thai Baht"));
	names.insert("TND",i18n("Tunisian Dinars"));
	names.insert("TRL",i18n("Turkish Lira"));
	names.insert("TTD",i18n("Trinidad and Tobago Dollars"));
	names.insert("TWD",i18n("Taiwanese Dollars"));
	names.insert("UAH",i18n("Ukrainian Hryvna"));
	names.insert("USD",i18n("U.S. Dollars"));
	names.insert("UYU",i18n("Uruguayan Pesos"));
	names.insert("VEB",i18n("Venezuelan Bolivars"));
	names.insert("XAF",i18n("French-African Francs"));
	names.insert("XCD",i18n("East Caribbean Dollars"));
	names.insert("XEU",i18n("European ECUs"));
	names.insert("XPF",i18n("French-Pacific Francs"));
	names.insert("ZAR",i18n("South African Rand"));
	names.insert("ZMK",i18n("Zambian Kwacha"));

	// List of native currencies that have switched to Euro.
	eurolist.append("ATS");
	eurolist.append("BEF");
	eurolist.append("DEM");
	eurolist.append("ESP");
	eurolist.append("FIM");
	eurolist.append("FRF");
	eurolist.append("GRD");
	eurolist.append("IEP");
	eurolist.append("ITL");
	eurolist.append("LUF");
	eurolist.append("NLG");
	eurolist.append("PTE");
	// Euro conversion rates
	data.insert(getKey("ATS","EUR"),KExData(13.7603,QDate(1998,12,31),Conversion));
	data.insert(getKey("BEF","EUR"),KExData(40.3399,QDate(1998,12,31),Conversion));
	data.insert(getKey("DEM","EUR"),KExData(1.95583,QDate(1998,12,31),Conversion));
	data.insert(getKey("ESP","EUR"),KExData(166.386,QDate(1998,12,31),Conversion));
	data.insert(getKey("FIM","EUR"),KExData(5.94573,QDate(1998,12,31),Conversion));
	data.insert(getKey("FRF","EUR"),KExData(6.55957,QDate(1998,12,31),Conversion));
	data.insert(getKey("GRD","EUR"),KExData(340.750,QDate(2001,1,1),Conversion));
	data.insert(getKey("IEP","EUR"),KExData(0.787564,QDate(1998,12,31),Conversion));
	data.insert(getKey("ITL","EUR"),KExData(1936.27,QDate(1998,12,31),Conversion));
	data.insert(getKey("LUF","EUR"),KExData(40.3399,QDate(1998,12,31),Conversion));
	data.insert(getKey("NLG","EUR"),KExData(2.20371,QDate(1998,12,31),Conversion));
	data.insert(getKey("PTE","EUR"),KExData(200.482,QDate(1998,12,31),Conversion));

	addCurrency("EUR");
	readData();	
}
KExDataSource::~KExDataSource(){
	if (datain!=NULL) {
		delete streamin;
		delete datain;
	}         	
}
QString KExDataSource::getKey(QString target, QString base) {
	return target+"/"+base;
}
void KExDataSource::addCurrency(QString code) {
	if (!eurolist.contains(code) && names.contains(code) && !list.contains(code))
		list.append(code);
}
void KExDataSource::addRate(QString target, QString base, Rate rate, QDate date, int confidence) {
	if (!eurolist.contains(target) && !eurolist.contains(base)) {
		KExData d(rate,date,confidence);
		QString key=getKey(target,base);
		if (data.contains(key))
			data.remove(key);
		data.insert(key,d);	
	}
}
double KExDataSource::convert(QString target,QString base, double amount) {
	if (eurolist.contains(target) && eurolist.contains(base)) {
		ExData::ConstIterator it=data.find(getKey(target,"EUR")); // Convert via euro
		if (it!=data.end())
				return convert("EUR",base,round((*it).rate,precision)*amount);
	}
	return getRate(target,base).rate*amount;
}
const KExData KExDataSource::getRate(QString target,QString base) {
	if (target==base)
		return KExData(1,QDate::currentDate(),Reliable);
	ExData::ConstIterator it=data.find(getKey(target,base));
	if (it!=data.end()) // Try to find exact match
		return *it;
	else {
		it=data.find(getKey(base,target)); // Try to find inversed match
		if (it!=data.end()) {
			if ((*it).rate>0)
				return KExData(round(1/(*it).rate,precision),(*it).date,(*it).confidence);		
			else
				return KExData(0,(*it).date,None);
		}
		else {
			KExData d;
			if (eurolist.contains(target) && eurolist.contains(base)) {
				it=data.find(getKey(target,"EUR"));
				d=*it;
				it=data.find(getKey(base,"EUR"));
				d.rate=round(d.rate/(*it).rate,precision);
				if ((*it).date>d.date)
					d.date=(*it).date;
				d.confidence=CrossConversion;
				return d;
			}
			KExData t=toUSD(target);
			KExData b=toUSD(base);
			if (b.rate>0)
				d=KExData(t.rate/b.rate,t.date<b.date ? t.date : b.date,Calculated);
			else
				d=KExData(0,b.date,None);
			return d;
		}
	}
}
KExData KExDataSource::toUSD(QString code) {
	ExData::ConstIterator it;
	KExData d;
	if (code!="USD") {
		if (eurolist.contains(code)) { // Convert national currency units of Euro first to EUR
			it=data.find(getKey(code,"EUR"));
			if (it!=data.end()) {
				d=*it;
				code="EUR";
			} else
				return KExData();
		}
		it=data.find(getKey(code,"USD")); // Get USD rate
		if (it!=data.end()) {
			if (d.rate==0)
				d=*it;
			else {
				Rate r=d.rate;
				d=*it;
				d.rate=r*d.rate;
			}
		} else
			return KExData();
	} else
		d=KExData(1,QDate::currentDate(),Reliable);
	d.rate=round(d.rate,precision);
	return d;
}
const QString KExDataSource::getCurrencyName(QString code) {
	CurrencyNames::ConstIterator it=names.find(code);
	if (it!=names.end())
		return *it;
	else
		return code;
}
const QStringList KExDataSource::getCurrencies(bool oldeuro) {
	typedef QMap<QString,QString> NameMap;
	NameMap namemap;
	QStringList namelist,sortedlist;
  CurrencyNames::ConstIterator it;
	QStringList::ConstIterator it2;
	NameMap::ConstIterator it3;
	QStringList curlist(list);
	if (oldeuro)
		curlist+=eurolist;

  for( it = names.begin(); it != names.end(); ++it ) {
		if (curlist.contains(it.key())>0) {
	  	namemap.insert(it.data(),it.key());
			namelist.append(it.data());
		}
	}
	namelist.sort();
  for (  it2 = namelist.begin(); it2 != namelist.end(); ++it2 ) {
		 it3=namemap.find(*it2);
     sortedlist.append(*it3);
  }	
	return sortedlist;		
}
void KExDataSource::storeData() {
  ExData::ConstIterator it;
	QFile f( locateLocal("appdata", "rates.xml"));
	if ( f.open(IO_WriteOnly) ) {
    QTextStream t( &f );
		t << "<exchangerates>\n";
    QString key,target;
		int slash;
	  for( it = data.begin(); it != data.end(); ++it ) {
			key=it.key();
			slash=key.find('/');
			ASSERT(slash>0);
      target=key.left(slash);
			if (!eurolist.contains(target)) {
				t << "  <rate>\n";
				t << "    <key>"+key+"</key>\n";
				t << "    <rate>"+QString::number((*it).rate)+"</rate>\n";
				t << "    <date>"+QString::number((*it).date.year())+"-"+QString::number((*it).date.month())+"-"+QString::number((*it).date.day())+"</date>\n";
				t << "    <confidence>"+QString::number((*it).confidence)+"</confidence>\n";
				t << "  </rate>\n";
			}
		}
		t << "</exchangerates>\n";
    f.close();
  }	
}

void KExDataSource::readData() {
	QString key,file;
	QDate date;
	Rate rate=0;
	int confidence=None;
	bool ok;

  QDomDocument doc( "rates" );
	file = locate("appdata", "rates.xml");
	if (file.isNull())
		return;
  QFile f( file );
  if ( !f.open( IO_ReadOnly ) )
      return;
  if ( !doc.setContent( &f ) ) {
      f.close();
      return;
  }
  f.close();

	addCurrency("USD");
  QDomElement docElem = doc.documentElement();
  QDomNode n = docElem.firstChild();
  while( !n.isNull() ) {
      QDomElement e = n.toElement();
      if( !e.isNull() ) {
				if (e.tagName()=="rate") {
				  QDomNode n2 = e.firstChild();
				  while( !n2.isNull() ) {
			      QDomElement e2 = n2.toElement();					
				    if( !e2.isNull() ) {
							if (e2.tagName()=="key")
								key=e2.firstChild().toText().data();
							if (e2.tagName()=="rate")
								rate=e2.firstChild().toText().data().toDouble(&ok);
							if (e2.tagName()=="date")
								date=parseDate(e2.firstChild().toText().data(),'-');
							if (e2.tagName()=="confidence")
								confidence=e2.firstChild().toText().data().toInt(&ok);
						}
			      n2 = n2.nextSibling();
					}
					if (date.isValid() && confidence!=None) {
						addCurrency(key.left(key.find('/')));
						data.insert(key,KExData(rate,date,confidence));
					}
				}
      }
      n = n.nextSibling();
  }
}
QDate KExDataSource::parseDate(QString str, QChar sep) {
	bool ok;
	QStringList::ConstIterator it;
	int y,m,d;
	QStringList l=QStringList::split(sep,str,true);
	if (l.count()==3) {
		it=l.at(0);
  	y=(*it).toInt(&ok);
		it=l.at(1);
  	m=(*it).toInt(&ok);
		it=l.at(2);
  	d=(*it).toInt(&ok);
		if (y>0 && m>0 && d>0) {
			return QDate(y,m,d);
		}
	}
	return QDate();
}
void KExDataSource::transferData(KIO::TransferJob *job) {
 if (datain!=NULL) {
		delete streamin;
		delete datain;
 }
 datain=new QByteArray();
 streamin=new QDataStream(*datain,IO_WriteOnly);
 connect( job, SIGNAL( result( KIO::Job * ) ), this, SLOT( slotResult( KIO::Job * ) ) );
 connect( job, SIGNAL( data( KIO::Job *, const QByteArray &  ) ), this, SLOT( slotData( KIO::Job *, const QByteArray &  ) ) );
 connect (job, SIGNAL(infoMessage ( KIO::Job *, const QString & ) ), this, SLOT (slotMessage ( KIO::Job *, const QString & ) ) );
 emit status(Transfering,i18n("Connecting to server"));
}
void KExDataSource::slotResult( KIO::Job *job) {
	bool ok=false;
	if (job->error()) {
		job->showErrorDialog (parent);
		emit status(Error,i18n("Error retrieving currency data"));
	}
	else {
		KIO::TransferJob *tjob=(KIO::TransferJob*)job;
	  if (tjob->isErrorPage ()) {
			KMessageBox::error (parent, i18n("Server returned an error"));
			emit status(Error,i18n("Error retrieving currency data"));
		}	else {
			if (readData(QTextStream( *datain, IO_ReadOnly ))) {
	  		emit status(Ready,i18n("Currency data retrieved successfully"));
        ok=true;
			} else
	  		emit status(Error,i18n("Could not read currency data"));				
		}
	}
	emit dataReady(ok);
}
void KExDataSource::slotMessage ( KIO::Job *, const QString & m) {
	QString msg=m;
	msg=msg.replace(QRegExp("<b>"),"");
	msg=msg.replace(QRegExp("</b>"),"");
	emit status(Transfering,msg);
}
void KExDataSource::slotData ( KIO::Job *, const QByteArray &d) {
	if (d.size()>0)
		*streamin << d;
}
const QPixmap KExDataSource::getFlag(QString code) {
	QPixmap flag;
  if ( !QPixmapCache::find("flag_"+code, flag) ) {
     if (flag.load(locate( "locale",
	 		 QString::fromLatin1( "l10n/%1/flag.png" ) .arg(code.left(2).lower()) ) ))			
	     QPixmapCache::insert("flag_"+code, flag);
		 else {
	     if (flag.load(locate( "icon",
		 		 QString::fromLatin1( "locolor/16x16/apps/kexchange_flag_%1.png" ) .arg(code.left(2).lower()) ) ))			
		     QPixmapCache::insert("flag_"+code, flag);
			 else {
				 flag=defaultFlag;
	       QPixmapCache::insert("flag_"+code, flag);
			 }
		 }
  }
	return flag;
}
QString KExDataSource::getCode(QStringList list, int pos) {
	if (pos>=0) {
		QStringList::ConstIterator it=list.at(pos);
		return *it;
	}
	return QString::null;
}
QString KExDataSource::round(Rate r) {
	int p=precision;
	if (round(r,precision,false)==0 && round(r,highPrecision,false)!=0)
		p=highPrecision;
	return KGlobal::locale()->formatNumber(r,p);
}
/**
	This routine originally by Josef Wolfsteiner from http://www.codeproject.com/cpp/floatutils.asp
*/
Rate KExDataSource::round(Rate doValue, int nPrecision, bool upgrade) {
				Rate result;
        static const Rate doBase = 10.0f;
        Rate doComplete5, doComplete5i;

        doComplete5 = doValue * pow(doBase, (Rate) (nPrecision + 1));

        if(doValue < 0.0f)
                doComplete5 -= 5.0f;
        else
                doComplete5 += 5.0f;

        doComplete5 /= doBase;
        modf(doComplete5, &doComplete5i);

				result=doComplete5i / pow(doBase, (Rate) nPrecision);
				if (result==0 && nPrecision<highPrecision && upgrade)
					result=round(doValue,highPrecision);
        return result;

}

