/*************************************************/
/* methods for class SimW                        */
/*                                               */
/* window displaying logic signal flow           */
/*                                               */
/* Andreas Rostin                                */
/* 15.03.99                                      */
/*************************************************/
#include <ktmainwindow.h>

#include <qpixmap.h> 
#include <qpainter.h>
#include <qpopupmenu.h>
#include <qevent.h>

#include <simw.h>
#include <mainw.h>
#include <netw.h>
#include <xnet.h>
#include <klogic.h>

#include "simw.moc"

/***************************************************/
/* methods of class SimW                           */
/***************************************************/
// main widget content
SimW::SimW(XDeviceNet *_net)
	: QFrame()
{       QString caption = ME;

        caption.append(" - simulation graph");
        setCaption(caption);			// set widgets title
	setBackgroundMode( PaletteBase );	// set widgets background
	resize(SCREEN_SIZE_X, 200);		// set widgets size
	setMaximumWidth(1024);			// set widgets max width

	net = _net;

	ymag = 30;
	ymag_d = ymag + ymag / 2 - 8;
	xmag = 2;

	for (int i = 0; i < 1024; i++) {
		fill0[i] = 0;
		fill1[i] = 1;
	}
	slotcnt = 0;
	slotwidth = 0;

	bar=new KStatusBar(this);
	bar->move(0,180);
	bar->resize(SCREEN_SIZE_X,20);
	activeC = 0;
	MINX = 0;
	c1x = MINX;
	c1x_old = MINX;
	c2x = MINX;
	c2x_old = MINX;

	graphChange();

	simmode = NetWidget::MODE_SIM_MULT;

	csr_pic.load(MainWidget::PATH + "cursor.xpm");
}

SimW::~SimW()
{
}

void SimW::mousePressEvent(QMouseEvent *e)
{	QPainter p;

	p.begin(this);
	eraseCursor(&p);
	if ((e->pos().x() - 10 < c1x) && (e->pos().x() + 10 > c1x)) {
		activeC = 1;
		c1x = e->pos().x();
		if (c1x < MINX) c1x = MINX;
	}
	else if ((e->pos().x() - 10 < c2x) && (e->pos().x() + 10 > c2x)) {
		activeC = 2;
		c2x = e->pos().x();
		if (c2x < MINX) c2x = MINX;
	}
	else activeC = 0;

	if (activeC) {
		eraseCursor(&p);
		drawCursor(&p);
	}
	p.end();
}

void SimW::mouseMoveEvent(QMouseEvent * e)
{	QPainter p;

	if (activeC) {
		p.begin(this);
		eraseCursor(&p);
		if (activeC == 1) c1x = e->pos().x();
		if (c1x < MINX) c1x = MINX;
		if (activeC == 2) c2x = e->pos().x();
		if (c2x < MINX) c2x = MINX;
		draw(&p);
		coord(&p);
		drawCursor(&p);
		p.end();
	}
}

void SimW::mouseReleaseEvent(QMouseEvent *)
{
	activeC = 0;
}

void SimW::resizeEvent( QResizeEvent *)
{
	bar->move(0,height()-20);
	bar->resize(width(),20);
}


void SimW::hide()
{
	emit shidden();
	QWidget::hide();
}

// slot
void SimW::paintEvent(QPaintEvent *)
{	list<XDevice> *l = ldev.First();
	int i;
	QPainter p;
	QPixmap pix(width(), height());

	pix.fill(white);

	if (width() != slotwidth) {
		slotwidth = width();
		if (slotwidth > 1024) slotwidth = 1024;
		for(i=0; i < slotcnt; i++)
			memcpy(slot[i], fill0, slotwidth);
	}

	p.begin(&pix);
	p.setPen(black);
	p.setFont(QFont("helvetica", 8, QFont::Bold));
	i = 0;
	while(l) {
		i++;
		p.drawText(5, ymag + (i * ymag_d) - 6, l->getText());
		l = l->Next();
	}

	bitBlt(&pix, c1x - 4, 2, &csr_pic);
	p.drawLine(c1x, 10, c1x, 1000);
	bitBlt(&pix, c2x - 4, 2, &csr_pic);
	p.drawLine(c2x, 10, c2x, 1000);
	draw(&p);
	coord(&p);
	drawCursor(&p);
	p.end();

	bitBlt(this, 0, 0, &pix);
}

XDeviceNet * SimW::currentNet()
{
	return net;
}

void SimW::changeNet(XDeviceNet *_net)
{
	net = _net;
	graphChange();
}

// slot
// actualize list of devices to display
// calculate new MINX
void SimW::graphChange()
{	int i;
	list<XDevice> *netdev;
	XDevice *dev;
	QString s;

	for(i=0; i < slotcnt; i++)
		memcpy(slot[i], fill0, 1024);

	netdev = net->devPointer()->First();
	slotcnt = 0;
	ldev.Destroy();
	while(netdev) {
		dev = netdev->Get();
		s.sprintf("%s", dev->getText());
		if (dev->drawGraph() && dev->graphEnabled()) {
			if (dev->hasNamedOutput()) {
				list<opStack> *named_output = dev->getNamedORef()->First();
				named_output = named_output->Next();
				while(named_output) {
					s.sprintf("%s(%s)", dev->getText(), named_output->getText());
					ldev.Append(dev, dev->getID(), named_output->getID1())->setText(s);
					if ((int)s.length() * 6 > MINX) MINX = s.length() * 6;
					slotcnt++;
					// prevent slot overflow
					if (slotcnt > MAXY) {
						if (c1x < MINX) c1x = MINX;
						if (c2x < MINX) c2x = MINX;
						repaint(FALSE);
						return;
					}
					named_output = named_output->Next();
				}
			} else {
				ldev.Append(dev, dev->getID())->setText(s);
				if ((int)s.length() * 6 > MINX) MINX = s.length() * 6;
				slotcnt++;
				// prevent slot overflow
				if (slotcnt > MAXY) {
					if (c1x < MINX) c1x = MINX;
					if (c2x < MINX) c2x = MINX;
					repaint(FALSE);
					return;
				}
			}
		}
		netdev = netdev->Next();
	}

	if (c1x < MINX) c1x = MINX;
	if (c2x < MINX) c2x = MINX;
	repaint(FALSE);
}

// slot
void SimW::simStep()
{
	if (!slotwidth) return;

	QPainter(p);
	p.begin(this);
	get();
	draw(&p);
	erase(&p);
	coord(&p);
	drawCursor(&p);
	p.end();
}

void SimW::drawCursor(QPainter *p)
{	QString val;

	if (c1x_old != c1x || c2x_old != c2x) {
		p->setBrush(white);
		p->setPen(white);
		p->drawRect(MINX - 4, 2, width(), 10); 
		bitBlt(this, c1x - 4, 2, &csr_pic);
		bitBlt(this, c2x - 4, 2, &csr_pic);
		val.sprintf("Cursor 1: %d\t\tCursor 2: %d", transval(c1x), transval(c2x));
		bar->message(val);
	}

	p->setPen(black);
	p->drawLine(c1x, 10, c1x, 1000);
	p->drawLine(c2x, 10, c2x, 1000);
	c1x_old = c1x;
	c2x_old = c2x;

}

void SimW::eraseCursor(QPainter *p)
{
	p->setPen(white);
	p->drawLine(c1x,10,c1x,1000);
	p->drawLine(c2x,10,c2x,1000);
}


inline void SimW::get()
{	list<XDevice> *ld = ldev.First();
	int j = 0;
	int startfill_delta, lenfill;

	startfill_delta = slotwidth - xmag;
	if (startfill_delta < 0) startfill_delta = 0;
	lenfill = xmag + 10;

	while(ld) {
		// move existing values to the left (xmag bytes)
		memcpy(oslot[j], slot[j], slotwidth);
		memmove(slot[j], slot[j] + xmag, slotwidth);

		// append new value at the right side (xmag + 10 bytes)
		if (ld->Get()->invertGraph(ld->Get()->output(ld->getID2()), ld->getID2())) {
			memcpy(slot[j] + startfill_delta, fill0, lenfill);
		} else {
			memcpy(slot[j] + startfill_delta, fill1, lenfill);
		}

		// next slot and next device
		j++;
		ld = ld->Next();
	}
}

inline void SimW::draw(QPainter *p)
{	int i, x, y;
	int delta;
	int yval;

	p->setPen(black);
	for (i=0; i < slotcnt; i++) {
		y = (i * ymag_d) + ymag_d;
		delta = 0;
		for (x = MINX; x < slotwidth; x ++) {
			if (slot[i][x] != slot[i][x + 1]) {
				p->drawLine(x, ymag + y, x, y);
				yval = slot[i][x] * ymag + y;
				p->drawLine(x - delta, yval, x, yval);
				delta = 0;
			}
			else delta ++;
		}
		if (slot[i][x]) yval = ymag + y;
		else yval = y;
		p->drawLine(x - delta, yval, x, yval);
	}
}

inline void SimW::erase(QPainter *p)
{	int i, x, y;

	p->setPen(white);
	for (i=0; i < slotcnt; i++) {
		y = (i * ymag_d) + ymag_d;
		for (x = MINX; x < slotwidth; x ++) {
			if ((slot[i][x] != oslot[i][x]) && (slot[i][x + 1] != oslot[i][x])) {
				if (slot[i][x]) p->drawLine(x, ymag + y - 1, x, y);
				else p->drawLine(x, ymag + y, x, y + 1);
			}
		}
	}
}

inline void SimW::coord(QPainter *p)
{	int i;
	QString s;
	int stepn = 0;

	p->setFont(QFont("helvetica", 8, QFont::Bold));

	p->setPen(black);
	// x axis
	p->drawLine(MINX, 30, width(), 30);
	p->drawText(width() - 30, 20, klocale->translate("step"));

	// y axis
	p->drawLine(MINX, 30, MINX, height());

	p->setPen(gray);
	for(i = (slotwidth - xmag); i > MINX; i -= (xmag * 10)) {
		p->drawLine(i, 28, i, height());
	}

	for(i = (slotwidth - xmag); i > MINX; i -= (xmag * 20)) {
		s.setNum(stepn);
		if (stepn > -10)
			p->drawText(i - 5, 26, s);
		else if (stepn > -100)
			p->drawText(i - 8, 26, s);
		else p->drawText(i - 11, 26, s);

		stepn -= 20;
	}
}

// transform screen-coordinate into simulation steps
inline int SimW::transval(int x)
{	int ret;


	// difference of screen coordinate from zero step (must be negative)
	ret = x - (slotwidth - xmag);

	// 1 step: xmag
	ret = ret / xmag;

	return ret;
}

void SimW::setSimMode(int newmode)
{
	simmode = newmode;
}

