/*************************************************/
/* methods for class XWire                       */
/*                                               */
/* visible wire                                  */
/*                                               */
/* Andreas Rostin                                */
/* 15.03.99                                      */
/*************************************************/
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <xwire.h>
#include <device.h>
#include <klogic.h>

/***************************************************/
/* methods of XWire (connection between devices)  */
/***************************************************/
XWire::XWire()
	:Wire()
{	in1_dev = (XDevice *)NULL;		// devices, to which output from out_dev is delivered to
	in1_id = 0;
	ininv[0] = new Device(Device::fINV_INTERNAL);
	ini[0] = 0;
	in2_dev = (XDevice *)NULL;
	in2_id = 0;
	ininv[1] = new Device(Device::fINV_INTERNAL);
	ini[1] = 0;
	out_dev = (XDevice *)NULL;		// device which deliveres input for this wire
	out_id = 0;
	outinv = new XDevice(Device::fINV_INTERNAL, 0, 0);
	outi = 0;
	wire_dev = (XDevice *)NULL;
	wire_id = 0;
	io_w1 = (XWire *)NULL;
	io_w2 = (XWire *)NULL;
	id = uniqueID::getID();
	scanned = 0;
}

XWire::XWire(QPoint p)
	:Wire(p)
{	in1_dev = (XDevice *)NULL;
	in1_id = 0;
	ininv[0] = new Device(Device::fINV_INTERNAL);
	ini[0] = 0;
	in2_dev = (XDevice *)NULL;
	in2_id = 0;
	ininv[1] = new Device(Device::fINV_INTERNAL);
	ini[1] = 0;
	out_dev = (XDevice *)NULL;
	out_id = 0;
	outinv = new XDevice(Device::fINV_INTERNAL, 0, 0);
	outi = 0;
	wire_dev = (XDevice *)NULL;
	wire_id = 0;
	io_w1 = (XWire *)NULL;
	io_w2 = (XWire *)NULL;
	id = uniqueID::getID();
	scanned = 0;
}

XWire::~XWire()
{
	// remove all nodes from the wire
	while(lockLastNode()) {
		removeNode();
	}
	delete ininv[0];
	delete ininv[1];
	delete outinv;
	io_w.Destroy();
}

// get output device
XDevice * XWire::outputDevice()
{
	if (wire_dev) return wire_dev;
	else return out_dev;
}

// get output name
int XWire::outputID()
{
	if (wire_dev) return wire_id;
	else return out_id;
}

// get input device
XDevice * XWire::inputDevice(int no)
{
	if (no == 1) return in1_dev;
	if (no == 2) return in2_dev;
	return (XDevice *)NULL;
}

// get output wire
XWire * XWire::ioWire(int no)
{
	if (no == 1) return io_w1;
	if (no == 2) return io_w2;
	return (XWire *)NULL;
}

// look if a node lies near p
int XWire::contains(QPoint p)
{
	// Wire returns a list!
	if (Wire::contains(p)) return 1;
	else return 0;
}

// erase active node and all affected parts of other wires
void XWire::erase(QPainter *p)
{	list<XWire> *l;

	Wire::erase(p);
	l = io_w.First();
	while(l) {
		if (l->Get()->lockNode(getActiveNode())) 
			l->Get()->Wire::erase(p);
		l = l->Next();
	}
}

// erase the whole wire and all affected wires (used for movement of node-groups)
void XWire::eraseWire(QPainter *p)
{	list<XWire> *l;

	Wire::eraseWire(p);
	l = io_w.First();
	while(l) {
		l->Get()->Wire::eraseWire(p);
		l = l->Next();
	}
}

// draw active node and all affected parts of other wires
void XWire::draw(QPainter *p)
{
	Wire::draw(p);
	list<XWire> *l = io_w.First();
	while(l) {
		if (l->Get()->lockNode(getActiveNode())) 
			l->Get()->Wire::draw(p);
		l = l->Next();
	}
}

// selection of that part of wire contained in rect
int XWire::select(QRect r)
{
	return Wire::select(r, 0);
}

// selection/deselection of a wire
void XWire::select(int f)
{	QRect r;	// not needed

	// call Wire method
	if (f) Wire::select(r, 1);
	else unselect();
}

// return the wire which posesses the node
XWire *XWire::nodeIsForeign(QPoint *pt)
{       list<XWire> *l = io_w.First();
	XWire *w;

        while(l) {
		w = l->Get();
                if (w->nodeIsReference(pt)) return w;
                l = l->Next();
        }
        return (XWire *)NULL;
}

int XWire::getConnectionIndex(XDevice *d)
{
	if (d == in1_dev) return 1;
	if (d == in2_dev) return 2;
	return 0;
}

int XWire::getConnectionID(XDevice *d)
{
	if (d == in1_dev) return in1_id;
	if (d == in2_dev) return in2_id;
	if (d == out_dev) return out_id;
	if (d == wire_dev) return wire_id;
	return 0;
}

// lock connected (if connected) node in other wires
int XWire::createNode(QPoint p)
{	QPoint *pt;
	int ret;

	ret = Wire::createNode(p);
	if (ret) {
		// get reference to the new node
		pt = getActiveNode();

		// lock node in connected wire
		if (io_w1 && io_w1->Wire::isPart(pt))
			io_w1->lockNode(pt);

		if (io_w2 && io_w2->Wire::isPart(pt))
			io_w2->lockNode(pt);
	}
	return ret;
}

void XWire::updateNode(QPoint p)
{	int idx;

	// node contains the old position of the point
        if (0 != (idx = Wire::activeIsInput())) {
		if (idx == 1) in1_dev->eraseConnectionLines();
		if (idx == 2) in2_dev->eraseConnectionLines();
        }
        if (Wire::activeIsOutput() && out_dev) {
                out_dev->eraseConnectionLines();
        }

	// update the node
	Wire::updateNode(p);

        // node contains the new position of the point
        if (0 != (idx = Wire::activeIsInput())) {
		if (idx == 1) in1_dev->drawConnectionLines();
		if (idx == 2) in2_dev->drawConnectionLines();
        }
        if (Wire::activeIsOutput() && out_dev) {
                out_dev->drawConnectionLines();
        }
} 

// device-wire-connections: first diconnect devices, then remove node
// wire-wire-connections  : make a disconnect from the wire which holds a copy
XWire * XWire::removeNode()
{	XWire *w = (XWire *)NULL;
	QPoint *pt = getActiveNode();
	int idx;

	//---- wire connections ----
	//  remove the copy in the foreign node
	//  then remove the node itself

	// this posesses the active node
	if (activeIsReference() && (NULL != (w = ioWire(getActiveConnIndex())))) {
		w->lockNode(pt);
		w->disconnectWire(this);	// w: foreign wire
		return w;
	}
	// this contains only a copy of node
	if (NULL != (w = nodeIsForeign(pt))) {
		w->lockNode(pt);
		this->disconnectWire(w);	// this: foreign wire
		return w;
	}

	//---- device connections ----
	// remove point-reference of device-connections in this wire
	if (0 != (idx = activeIsInput())) {
		if (idx == 1) disconnectInput(in1_dev);
		if (idx == 2) disconnectInput(in2_dev);
	}
	if (activeIsOutput()) 
		disconnectOutput();
	// remove point
	Wire::removeNode();
	return (XWire *)NULL;
}

void XWire::releaseNode()
{	int idx;

	if (0 != (idx = activeIsInput())) {
		if (idx == 1) in1_dev->drawConnectionLines();
		if (idx == 2) in2_dev->drawConnectionLines();
	}
	if (activeIsOutput() && out_dev) {
		out_dev->drawConnectionLines();
	}
	Wire::releaseNode();
}

// returns error if connected already, 1 if succesful
// pay attention to the inverters on device-level!
// if invert is set, invertation is forced
int XWire::connectInput(XDevice *dev, int invert, int connection_id)
{	int ret;

	// remember the point, connected to
	ret = Wire::connectInput();
	if ((WOK1 != ret) && (WOK2 != ret)) return ret;

	// remember the device, connected to
	if (WOK1 == ret) {
		in1_dev = dev;
		in1_id = connection_id;
		if (invert == 1) ini[0] = 1;
		else ini[0] = 0;
		// add reference to this wire in device
		in1_dev->addInput(this, 1);
	}
	else {
		in2_dev = dev;
		in2_id = connection_id;
		if (invert == 1) ini[1] = 1;
		else ini[1] = 0;
		// add reference to this wire in device
		in2_dev->addInput(this, 2);
	}

	// logical connection
	if ((dev == in1_dev) && in1_dev && out_dev)
		setLogicInput(in1_dev, out_dev);
	if ((dev == in1_dev) && in1_dev && wire_dev)
		setLogicInput(in1_dev, wire_dev);
	if ((dev == in2_dev) && in2_dev && wire_dev)
		setLogicInput(in2_dev, wire_dev);
	if ((dev == in1_dev) && in1_dev && !wire_dev && !out_dev)
		setLogicInput(in1_dev, (Device *)NULL);
	if ((dev == in2_dev) && in2_dev && !wire_dev)
		setLogicInput(in2_dev, (Device *)NULL);

	return WOK;
}

// if called from device: attention!! reference will be destroyed in device!!
// pay attention to the inverters on device-level!
void XWire::disconnectInput(XDevice *dev)
{	int idx = 0;

	if (in1_dev == dev) idx = 1;
	if (in2_dev == dev) idx = 2;
	if (!idx) return;

	// if not two devices are connected anymore, logical disconnection
	if ((dev == in1_dev) && in1_dev && out_dev)
		resetLogicInput(in1_dev, out_dev);
	if ((dev == in1_dev) && in1_dev && wire_dev)
		resetLogicInput(in1_dev, wire_dev);
	if ((dev == in2_dev) && in2_dev && wire_dev)
		resetLogicInput(in2_dev, wire_dev);
	if ((dev == in1_dev) && in1_dev && !wire_dev && !out_dev)
		resetLogicInput(in1_dev, (Device *)NULL);
	if ((dev == in2_dev) && in2_dev && !wire_dev)
		resetLogicInput(in2_dev, (Device *)NULL);

	// remove reference to this wire in device
	dev->removeInput(this, idx);

	// remove point-reverence, connected to
	Wire::disconnectInput(idx);

	// remove references to object, connected to
	if (idx == 1) {
		if (in2_dev) {
			in1_dev = in2_dev;
			in1_id = in2_id;
			in2_dev = (XDevice *)NULL;
			in2_id = 0;
		}
		else {
			in1_dev = (XDevice *)NULL;
			in1_id = 0;
		}
	}
	else {
		in2_dev = (XDevice *)NULL;
		in2_id = 0;
	}
}

// connect to output of a device (logical, physical, references)
// pay attention to the inverters on device-level!
// if invert is 1, invertation is forced
// if invert is 0, the default is used
// if invert is -1, non-invertation is forced
int XWire::connectOutput(XDevice *dev, int invert, int connection_id)
{	int ret;
	XDevice *_outdev;

	// if there is already an output, connection will fail
	if (wire_dev) return WFAIL;
	if (in1_dev && in2_dev) return WFAIL;

	// reference to the device as output giving device
	out_dev = dev;
	out_id = connection_id;

	// remember point (active node), the device will be connected to
	if (WOK != (ret = Wire::connectOutput())) return ret;

	// force invertation?
	if (invert == 1) outi = 1;
	else if (invert == -1) outi = 0;
	// set inverter by default?
	else outi = out_dev->isInverted();

	// add reference to this wire in device
	out_dev->addOutput(this);

	// add output-device in all connected wires, build log.connections
	scanned = 1;
	resetScan();
	_outdev = getOutputDev(&connection_id);
	resetScan();
	setOutputDev(_outdev, connection_id);
	if (out_dev) {
		// make logic connections
		setLogicOutput();
	}

	return WOK;
}

// disconnect output of a device
// pay attention to the inverters on device-level!
void XWire::disconnectOutput()
{
	// remove reference to this wire in device
	if (out_dev)
		out_dev->removeOutput(this);

	// remove point-reference (not longer output-point)
	Wire::disconnectOutput();

	// remove device in all connected wires, remove log.connections
	if (out_dev) {
		resetLogicOutput();
		scanned = 1;
		resetScan();
		removeOutputDev();
		out_dev = (XDevice *)NULL;
		out_id = 0;
	}
}

void XWire::invertInput(XDevice *d, bool set)
{	int idx = 3;

	// nothing to do?
	if (in1_dev == d) {
		if ((ini[0]) && (set == TRUE)) return;
		if ((!ini[0]) && (set == FALSE)) return;
		idx = 0;
	}
	if (in2_dev == d) {
		if ((ini[1]) && (set == TRUE)) return;
		if ((!ini[1]) && (set == FALSE)) return;
		idx = 1;
	}
	if (idx == 3) {
		warning("wrong call to XWire::invertLogic");
		return;
	}

	// redirect logical connections
	if ((in1_dev == d) && out_dev)
		resetLogicInput(in1_dev, out_dev);
	if ((in1_dev == d) && wire_dev)
		resetLogicInput(in1_dev, wire_dev);
	if ((in2_dev == d) && wire_dev)
		resetLogicInput(in2_dev, wire_dev);
	if ((in1_dev == d) && !wire_dev && !out_dev)
		resetLogicInput(in1_dev, (Device *)NULL);
	if ((in2_dev == d) && !wire_dev)
		resetLogicInput(in2_dev, (Device *)NULL);

	if (set == TRUE) ini[idx] = 1;
	else ini[idx] = 0;

	if ((in1_dev == d) && out_dev)
		setLogicInput(in1_dev, out_dev);
	if ((in1_dev == d) && wire_dev)
		setLogicInput(in1_dev, wire_dev);
	if ((in2_dev == d) && wire_dev)
		setLogicInput(in2_dev, wire_dev);
	if ((in1_dev == d) && !wire_dev && !out_dev)
		setLogicInput(in1_dev, (Device *)NULL);
	if ((in2_dev == d) && !wire_dev)
		setLogicInput(in2_dev, (Device *)NULL);

}

int XWire::inputIsInverted(XDevice *d)
{
	if (in1_dev == d) return ini[0];
	else if (in2_dev == d) return ini[1];
	else return -1;
}

void XWire::invertOutput(bool set)
{	XDevice *_outdev;
	int dev_id;

	// nothing to do?
	if (outi && (set == TRUE)) return;
	if (!outi && (set == FALSE)) return;

	// redirect logical connections
	if (out_dev)
		resetLogicOutput();

	// remove logical connections
	scanned = 1;
	resetScan();
	removeOutputDev();

	if (set == TRUE) outi = 1;
	else outi = 0;

	if (out_dev)
		setLogicOutput();

	// rescan wire-connections for new output-device
	// determine output-device
	resetScan();
	_outdev = getOutputDev(&dev_id);
	resetScan();
	if (_outdev) setOutputDev(_outdev, dev_id);
}

int XWire::outputIsInverted()
{
	return outi;
}

void XWire::setLogicInput(Device *devi, Device *devo)
{	Device *in_dev;
	int idx = 3;
	int dev_id_i;
	int dev_id_o;

	if (devi == in1_dev) {
		idx = 0;
		in_dev = in1_dev;
		dev_id_i = in1_id;
	}
	else {
		idx = 1;
		in_dev = in2_dev;
		dev_id_i = in2_id;
	}
	if (devo == out_dev) dev_id_o = out_id;
	else dev_id_o = wire_id;

	if (devo) {
		if (ini[idx]) {
			if (outi) {
				in_dev->connectDevices(ininv[idx], dev_id_i, 0);
				ininv[idx]->connectDevices(outinv, 0, dev_id_o);
			}
			else {
				in_dev->connectDevices(ininv[idx], dev_id_i, 0);
				ininv[idx]->connectDevices(devo, 0, dev_id_o);
			}
		}
		else {
			if (outi) {
				in_dev->connectDevices(outinv, dev_id_i, 0);
			}
			else {
				in_dev->connectDevices(devo, dev_id_i, dev_id_o);
			}
		}
	}
	else {
		if (ini[idx]) {
			in_dev->connectDevices(ininv[idx], dev_id_i, 0);
		}
	}
}

void XWire::resetLogicInput(Device *devi, Device *devo)
{	Device *in_dev;
	int idx = 3;
	int dev_id_i;
	int dev_id_o;

	if (devi == in1_dev) {
		idx = 0;
		in_dev = in1_dev;
		dev_id_i = in1_id;
	}
	else {
		idx = 1;
		in_dev = in2_dev;
		dev_id_i = in2_id;
	}
	if (devo == out_dev) dev_id_o = out_id;
	else dev_id_o = wire_id;

	if (devo) {
		if (ini[idx]) {
			if (outi) {
				in_dev->disconnectDevices(ininv[idx], dev_id_i, 0);
				ininv[idx]->disconnectDevices(outinv, 0, 0);
			}
			else {
				in_dev->disconnectDevices(ininv[idx], dev_id_i, 0);
				ininv[idx]->disconnectDevices(devo, 0, dev_id_o);
			}
		}
		else {
			if (outi) in_dev->disconnectDevices(outinv, dev_id_i, 0);
			else in_dev->disconnectDevices(devo, dev_id_i, dev_id_o);
		}
	}
	else {
		if (ini[idx]) in_dev->disconnectDevices(ininv[idx], dev_id_i, 0);
	}
}

// new output-device
void XWire::setLogicOutput()
{	int dev_id_i;

	if (out_dev) {
		Device *di;
		if (ini[0] && in1_dev) {
			di = ininv[0];
			dev_id_i = 0;
		}
		else {
			di = in1_dev;
			dev_id_i = in1_id;
		}

		if (di) {
			if (outi) {
				di->connectDevices(outinv, dev_id_i, 0);
				outinv->connectDevices(out_dev, 0, out_id);
			}
			else di->connectDevices(out_dev, dev_id_i, out_id);
		}
		else {
			if (outi)
				outinv->connectDevices(out_dev, 0, out_id);
		}
		return;
	}
	if (wire_dev) {
		if (in1_dev && ini[0]) {
			ininv[0]->connectDevices(wire_dev, 0, wire_id);
		}
		else if (in1_dev && !ini[0]) {
			in1_dev->connectDevices(wire_dev, in1_id, wire_id);
	}
		if (in2_dev && ini[1]) {
			ininv[1]->connectDevices(wire_dev, 0, wire_id);
		}
		else if (in2_dev && !ini[1]) {
			in2_dev->connectDevices(wire_dev, in2_id, wire_id);
		}
		return;
	}
}

void XWire::resetLogicOutput()
{	int dev_id_i;

	if (out_dev) {
		Device *di;
		if (ini[0] && in1_dev) {
			di = ininv[0];
			dev_id_i = 0;
		}
		else {
			di = in1_dev;
			dev_id_i = in1_id;
		}

		if (di) {
			if (outi) {
				di->disconnectDevices(outinv, dev_id_i, 0);
				outinv->disconnectDevices(out_dev, 0, out_id);
			}
			else di->disconnectDevices(out_dev, dev_id_i, out_id);
		}
		else {
			if (outi)
				outinv->disconnectDevices(out_dev, 0, out_id);
		}
		return;
	}
	if (wire_dev) {
		if (in1_dev && ini[0]) {
			ininv[0]->disconnectDevices(wire_dev, 0, wire_id);
		}
		else if (in1_dev && !ini[0]) {
			in1_dev->disconnectDevices(wire_dev, in1_id, wire_id);
		}
		if (in2_dev && ini[1]) {
			ininv[1]->disconnectDevices(wire_dev, 0, wire_id);
		}
		else if (in2_dev && !ini[1]) {
			in2_dev->disconnectDevices(wire_dev, in2_id, wire_id);
		}
		return;
	}
}

// check if name is an input
// xdevice is usually a net device
int XWire::isNamedInput(XDevice *dev, int dev_id)
{
	if (in1_dev == dev && in1_id == dev_id) return 1;
	if (in2_dev == dev && in2_id == dev_id) return 1;
	return 0;
}

// check if name is an output
// xdevice is usually a net device
int XWire::isNamedOutput(XDevice *dev, int dev_id)
{
	if ((out_dev == dev || wire_dev == dev) && out_id == dev_id) return 1;
	return 0;
}

// checks, if actice-node of wire w is connected to this wire
// w is the active wire
// this is the foreign wire, the wire to check and perhaps in future the connected wire
int XWire::checkConnection(XWire *w)
{
	// active node of w must be the first or last node of w
	if (!w->activeIsEnd()) return WOK;

	// dont connect, if active node is an input/output-node in w (already connected)
	if ((w->activeIsInput()) || (w->activeIsOutput())) return WOK;

	// don't connect if active node is connected
	// (direct wire-loop not allowed)
	if (!io_w.Get(w) && (io_w1 != w) && (io_w2 != w)) {
		// check wire-connection, lock node
		// on success: insert/lock foreign node from w
		QPoint *pt = w->getActiveNode();
		if (NULL == Wire::checkConnection(pt)) return WOK;

		// make connection
		return connectWire(w);
	}
	return WOK;
}

// w is the owner of the node
// this now has a pointer to the foreign node of w
int XWire::connectWire(XWire *w)
{	XDevice *_outdev;
	int ret;
	int dev_id;

	// remember the connected wire(!)
	io_w.Append(w);

	// no of output-devices must be 0 or 1
	// w doesn't know anything about this wire yet
	scanned = 1;
	resetScan();
	if (1 < countOutputDev()) {
		disconnectWire(w);
		return WSHORT;
	}

	// remember this wire in the other (active) wire
	if (WOK != (ret = w->addWireReference(this))) return ret;

	// determine output-device
	resetScan();
	_outdev = getOutputDev(&dev_id);
	resetScan();
	if (_outdev) setOutputDev(_outdev, dev_id);
	resetScan();

	return WCONN;
}

// called by wire which contains a copy of the node
// node must be locked
int XWire::addWireReference(XWire *new_conn)
{	
	io_w.Append(new_conn);

        // add reference to the connected wire
        if (io_w1 == NULL) {
		Wire::connectWire(1);
		io_w1 = new_conn;
	}
        else if (io_w2 == NULL) {
		Wire::connectWire(2);
		io_w2 = new_conn;
	}
	else return WFATAL;
	return WOK;
}

// remove wire-wire connection
// owner/copy information is not relevant
void XWire::disconnectWire(XWire *w)
{	XDevice *_outdev;
	int dev_id;

	// remove logical connections (in both wires)
	scanned = 1;
	resetScan();
	removeOutputDev();
	resetScan();

	// remove connection references in both wires
	removeWireReference(w);
	w->removeWireReference(this);

	// rescan logical connections in this wire
	_outdev = getOutputDev(&dev_id);
	resetScan();
	if (_outdev) setOutputDev(_outdev, dev_id);

	// rescan logical connections in connected wire
	_outdev = w->getOutputDev(&dev_id);
	w->resetScan();
	if (_outdev) w->setOutputDev(_outdev, dev_id);

	// remove foreign node
	Wire::cutNode();

	// remove/destroy node
	w->Wire::removeNode();
}

// node must be locked in the wire
// owner/copy information is not relevant
void XWire::removeWireReference(XWire *old_conn)
{
	// remove wire reference
	io_w.Destroy(old_conn);

        if (io_w1 == old_conn) {
        	// remove node reference
		Wire::disconnectWire(1);
		// remove wire reference
		io_w1 = (XWire *)NULL;
	}
        if (io_w2 == old_conn) {
        	// remove node reference
		Wire::disconnectWire(2);
		// remove wire reference
		io_w2 = (XWire *)NULL;
	}
}

// determine no of output-devices
// be sure you have called resetScan() before
int XWire::countOutputDev()
{	int ret = 0;
	XDevice *d1 = (XDevice *)NULL, *d2 = (XDevice *)NULL;
	int output_id1 = 0, output_id2 = 0;

	if (!scanned) {
		scanned = 1;
		if (out_dev) {
			output_id1 = out_id;
			if (outi) d1 = outinv;
			else d1 = out_dev;
			ret++;
		}
		if (wire_dev) {
			output_id1 = wire_id;
			d1 = wire_dev;
			ret++;
		}
		list<XWire> *l = io_w.First();
		while(l) {
			d2 = l->Get()->getOutputDev(&output_id2);
			if (d2 && (d2 != d1)) ret++;
			else if (d1 && d1 == d2 && output_id1 && output_id1 != output_id2) ret++;
			if (!d1) {
				d1 = d2;
				output_id1 = output_id2;
			}
			l = l->Next();
		}
	}
	return ret;
}

// determine output-device
// be sure you have called resetScan() before
XDevice * XWire::getOutputDev(int *dev_id)
{	XDevice *d;

	if (!scanned) {
		scanned = 1;
		if (out_dev) {
			*dev_id = out_id;
			if (outi) return outinv;
			else return out_dev;
		}
		list<XWire> *l = io_w.First();
		while(l) {
			d = l->Get()->getOutputDev(dev_id);
			if (d) return d;
			l = l->Next();
		}
	}
	return (XDevice *)NULL;
}

// set output-device in this and all connected wires, build logical conn.
// be sure you have called resetScan() before
void XWire::setOutputDev(XDevice *dev, int dev_id)
{
	if (!scanned) {
		scanned = 1;
		list<XWire> *l = io_w.First();
		if (!out_dev && (wire_dev != dev) && (dev != NULL)) {
			wire_dev = dev;
			wire_id = dev_id;
			outi = 0;
			setLogicOutput();
		}
		while(l) {
			l->Get()->setOutputDev(dev, dev_id);
			l = l->Next();
		}
	}
}

// remove output-device in this and all connected wires
// be sure you have called resetScan() before
void XWire::removeOutputDev()
{
	if (!scanned) {
		scanned = 1;
		if (!out_dev) resetLogicOutput();
		wire_dev = (XDevice *)NULL;
		wire_id = 0;
		if (!out_dev) outi = 0;
		list<XWire> *l = io_w.First();
		while(l) {
			l->Get()->removeOutputDev();
			l = l->Next();
		}
	}
}

// needed to prevent loops
void XWire::resetScan()
{
	if (scanned) {
		scanned = 0;
		list<XWire> *l = io_w.First();
		while(l) {
			l->Get()->resetScan();
			l = l->Next();
		}
	}
}

int XWire::wasScanned()
{
	return scanned;
}

int XWire::setColor(int force)
{
	if (out_dev) {
		if (outi) {
			if (!force && !outinv->outputChanged()) return 0;
			if (outinv->output(out_id)) Wire::setColor(Qt::red);
			else Wire::setColor(Qt::blue);
			return 1;
		}
		else {
			if (!force && !out_dev->outputChanged()) return 0;
			if (out_dev->output(out_id)) Wire::setColor(Qt::red);
			else Wire::setColor(Qt::blue);
			return 1;
		}
	}
	else if (wire_dev) {
		if (outi) {
			if (!force && !outinv->outputChanged()) return 0;
			if (outinv->output(wire_id)) Wire::setColor(Qt::red);
			else Wire::setColor(Qt::blue);
			return 1;
		}
		else {
			if (!force && !wire_dev->outputChanged()) return 0;
			if (wire_dev->output(wire_id)) Wire::setColor(Qt::red);
			else Wire::setColor(Qt::blue);
			return 1;
		}
	}
	else Wire::setColor(Qt::black);
	return 1;
}

int XWire::getID()
{
	return id;
}
