#include <stdlib.h>

#include <qfile.h>
#include <qwidget.h>

#include <kmessagebox.h>
#include <klocale.h>

#include <kputil.h>

#include "kpmatrix.h"

const unsigned int growbytes = 1024;

void
KPMatrix::init (double *_matrix, unsigned int _nc, unsigned int _nr,
		    bool deep)
{
  nc=_nc;
  nr=_nr;

  bdeep = deep;

  amin = new double [nc];
  amax = new double [nc];
  for (unsigned int i=0;i<nc;i++)
    {
      amin [i] = 1e300;
      amax [i] = -1e300;
    }

  //We want to grow by roughly growbytes bytes whenever
  // set() oversteps the bounds of the matrix (for deep matrices)
  if (bdeep)
    growby = ::kpmax (growbytes/nc, 100);

  if (bdeep)
    {
      if (_matrix)
	{ 
	  unsigned int index;
	  data = new double [nr*nc];
	  if (_matrix)
	    for (unsigned int i=0, ni=0; i<nr; i++, ni+=nc)
	      for (unsigned int j=0; j<nc; j++)
		{
		  index = ni+j;
		  data [index] = _matrix [index];
		  minmax (j, index);
		}
	  memnr=nr=_nr;
	}
      else
	{
	  data = new double [nc*growby];  //growby rows
	  memnr = growby;
	  nr=0;
	}
    }
  else //not deep
    data = _matrix;


}

KPMatrix::KPMatrix (unsigned int _nc)
{
  init (0, _nc, 0, TRUE);
}

KPMatrix::KPMatrix (double *_matrix, unsigned int _nc, unsigned int _nr,
		    bool deep)
{
  init (_matrix, _nc, _nr, deep);
}

void
KPMatrix::set (unsigned int row, unsigned int col, double value)
{
  //  printf ("%d %d %d %d  %d %f\n",	  row, col, nr, nc, memnr, value);
  if (row<memnr && col<nc)
    {
      int index = row*nc + col;
      data [index] = value;
      minmax (col, index);
      if (row>=nr)
	nr = row+1;
      return;
    }
  
  //What if row > memnr+growby?

  unsigned int size=memnr*nc, i;
  unsigned int newsize = (memnr+growby)*nc;
  double *datanew = new double [newsize];
  for (i=0; i<size; i++) // copy
    datanew [i] = data [i];
  for (;i<newsize;i++)       // zero the extra space
    datanew [i] =0;

  memnr += growby;
  nr = row+1;

  delete [] data;
  data = datanew;

  data [row*nc +col] = value;

}

KPMatrix::~KPMatrix()
{
  if (bdeep)
    delete data;
  delete amin;
  delete amax;
}


KPMatrix *
KPMatrix::load (const QString &filename, QWidget *parent)
{

  QString qs;
  QFile file (filename);
  file.open (IO_ReadOnly);
  unsigned int size = file.size();

  int e,f,g;
  int savepos;

  qs.sprintf ("Loading %s", (const char *)filename);

  if (size==0)
    {
      KMessageBox::
        sorry ( parent, i18n("Cannot load matrix.  No data."),
                i18n("Matrix Read Error") );  
      return 0L;
    }


  //Skip initial comments
  do
    {
      savepos = file.at();
#if QT_VERSION<200
      char *tmp = new char [512];
      e=file.readLine (tmp, 512);
      qs=tmp;
      delete tmp;
#else
      e=file.readLine (qs, 512);
#endif

      if (e<0) //or e<1?
	{
	  
	  KMessageBox::
	    sorry ( parent, i18n("Cannot load matrix.  No data."),
		    i18n("Matrix Read Error") );        
	  return 0L;
	}
    } while (qs[0]=='#' || qs[0]=='%' || qs[0]=='\n');



  int nc=0;
  //How many columns?

  //column separators are ' ', ',', ';', ':'
  for (unsigned int i=0;i<qs.length();i++)
    if (qs[i]==',' || qs[i]==';' || qs[i]==':')
      qs[i]=' ';
  QString qs2=qs.simplifyWhiteSpace();
  qs=qs2;

  f=0;
  while ( (f=qs.find (' ', f))!=-1)
    {
      nc++;
      f++;
    }

  nc++;

  KPMatrix *m;
  /*  if (nc==1)
    m = new KPMatrix (2);
    else*/

  m = new KPMatrix (nc);

  file.at(savepos);

  int ir=0, ic=0;
  int ncm1=nc-1;
  int cnt=0;

  //if QT_VERSION<200
  char *tmp;
  tmp = new char [512];
  while (file.readLine (tmp, 512)>0)
    {
      qs=tmp;
      
      //while (file.readLine (qs, 512)>0)
      //    {
      //      cout << (const char*)qs << endl;
      
      //commas can be column separators
      for (unsigned int i=0;i<qs.length();i++)
	if (qs[i]==',')
	  qs[i]=' ';
      qs2=qs.simplifyWhiteSpace(); 
      qs=qs2;
      //      cout << qs << endl;
      g=f=0;
      ic=0;
      //Skip comment lines  (later: process commands)
      if (qs[0]=='#' || qs[0]=='%' || qs[0]=='\n')
	continue;
      while (ic<nc)
	{
	  double d;
	  
	  f=qs.find (' ', f);
	  if (f==-1 && ic!=ncm1)
	    {
	      KMessageBox::
                sorry ( parent, i18n("Cannot load matrix.  It is not in\n"
				     " MxN (rectangular) format."),
                        i18n("Matrix Format Error") ); 
	      delete m;
	      return 0L;
	    }
	  
	  if (f==-1)
	    f=qs.length();
	  d=atof (qs.mid (g,f-g));
	  if (nc==1)
	    {
	      m->set (ir, 0, cnt);
	      m->set (ir, 1, d);
	    }
	  else
	    m->set (ir, ic, d);
	  
	  cnt++;
	  ic++;
	  f++;
	  g=f;
	}
      ir++;
    }
  delete tmp;
  
  return m;
}
