/***************************************************************************
                          ksetiwatch.cpp  -  description
                             -------------------
    begin                : Fre Okt 22 17:39:47 CEST 1999
    copyright            : (C) 1999 by Gordon Machel
    email                : gmachel@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include <qlist.h>
#include <qlabel.h>
#include <qmessagebox.h>
#include <qfileinfo.h>
#include <qtextstream.h>

#include <klocale.h>
#include <kstatusbar.h>
#include <kmessagebox.h>

#include <unistd.h>

#include "ksetiwatch.h"
#include "setiloc.h"
#include "skymap.h"
#include "csvdatabase.h"


// skymap-related variables
SkyMap	*SKYMAP = 0;
bool		SKYMAP_LOADED = false;
bool		SKYMAP_VISIBLE = false;
bool		SHOW_TEXT = true;
bool		FILL_MARKER = true;
int			MARKER_SIZE = 2;

extern QList<SetiLoc> SetiList;

extern int		RefreshInterval;
extern int		StatusInterval;
extern QTimer *RefreshTimer;
extern bool		HMS;
extern bool		NEWPEAK_RECORD;
extern bool		NEWPEAK_CURRENTWU;
extern WUScore Record;
extern bool		MINIMIZETOPANEL;
extern bool   DRAWGRID;

struct csvHeaderField	{
											const char *text;
											int type;
											};

csvHeaderField csv_userinfo[] = 	{
		  														{"date", 1},
  																{"id", 0},
  																{"key", 0},
  																{"email_addr", 1},
  																{"user_name", 1},
  																{"url", 1},
  																{"country", 1},
  																{"postal_code", 0},
		  														{"show_name", 1},
		  														{"show_email", 1},
  																{"venue", 0},
  																{"register_time", 1},
  																{"last_wu_time", 1},
  																{"last_result_time", 1},
  																{"nwus", 0},
  																{"nresults", 0},
		  														{"total_cpu", 0},
  																{"params_index", 0}
  																};
#define CSV_USERINFO_COLUMNS 18

csvHeaderField csv_datainfo[] = 	{
				  												{"task", 1},
  																{"version", 0},
 					 												{"name", 1},
				  												{"data_type", 1},
  																{"data_class", 0},
  																{"splitter_version", 1},
  																{"start_ra", 0},
				  												{"start_dec", 0},
  																{"end_ra", 0},
  																{"end_dec", 0},
  																{"angle_range", 0},
				  												{"time_recorded", 1},
  																{"subband_center", 0},
  																{"subband_base", 0},
  																{"subband_sample_rate", 0},
				  												{"fft_len", 0},
  																{"ifft_len", 0},
  																{"subband_number", 0},
  																{"receiver", 1},
				  												{"nsamples", 0},
  																{"tape_version", 0}
  																};
#define CSV_DATAINFO_COLUMNS 21

csvHeaderField csv_state[] =	{
		  												{"ncfft", 0},
  														{"cr", 0},
  														{"fl", 0},
  														{"cpu", 0},
  														{"prog", 0},
  														{"bs_power", 0},
  														{"bs_score", 0},
		  												{"bs_bin", 0},
  														{"bs_fft_ind", 0},
  														{"bs_chirp_rate", 0},
  														{"bs_fft_len", 0},
  														{"bg_score", 0},
  														{"bg_power", 0},
  														{"bg_chisq", 0},
  														{"bg_bin", 0},
  														{"bg_fft_ind", 0},
  														{"bg_chirp_rate", 0},
  														{"bg_fft_len", 0},
  														{"bp_score", 0},
  														{"bp_power", 0},
  														{"bp_mean", 0},
  														{"bp_period", 0},
  														{"bp_freq_bin", 0},
  														{"bp_time_bin", 0},
  														{"bp_chirp_rate", 0},
  														{"bp_fft_len", 0},
  														{"bt_score", 0},
  														{"bt_power", 0},
  														{"bt_mean", 0},
  														{"bt_period", 0},
  														{"bt_bperiod", 0},
  														{"bt_tpotind0_0", 0},  														
  														{"bt_tpotind0_1", 0},  														
  														{"bt_tpotind1_0", 0},  														
  														{"bt_tpotind1_1", 0},  														
  														{"bt_tpotind2_0", 0},  														
  														{"bt_tpotind2_1", 0},
  														{"bt_freq_bin", 0},
  														{"bt_time_bin", 0},
  														{"bt_chirp_rate", 0},
  														{"bt_scale", 0},
  														{"bt_fft_len", 0}  														
	          		              };
#define CSV_STATE_COLUMNS 42

Ksetiwatch::Ksetiwatch(const char* name) : KMainWindow(0, name)
{
	QString tlabel[6] = {
											i18n("Analysis"),
											i18n("Data Info"),
											i18n("User Info"),
											i18n("Completed Work Units"),
											i18n("Options"),
											i18n("About")
											};
		
	initWUScore(&Record);
	readConfig();
	
  // load the skymap
  loader = new KIconLoader();
  if(loader)
  	{
		StarMap = loader->loadIcon("starmap", KIcon::User, 0, KIcon::DefaultState, 0, true);
		SKYMAP_LOADED = !StarMap.isNull();
		
		QPixmap iconpic;
		iconpic = loader->loadIcon("ksetiwatch", KIcon::Desktop);
		setIcon(iconpic);
	  }
	
  RefreshTimer = new QTimer(this);
  RefreshTimer->start(RefreshInterval, FALSE);

  // status bar must be initialized before setting the central widget !
	statusBar()->insertItem("",0, 1);
	statusBar()->setItemAlignment(0, AlignLeft|AlignVCenter);
  	
	ktb = new KTabCtl(this);
	ana = new Analysis(ktb);
	ana->setMinimumSize(640, 130);
	ktb->addTab(ana, tlabel[0]);
	di = new DataInfo(&StarMap, ktb);
	di->setMinimumSize(640, 130);
	ktb->addTab(di, tlabel[1]);
	ui = new UserInfo(ktb);
	ui->setMinimumSize(640, 130);
	ktb->addTab(ui, tlabel[2]);
	cwu = new CompletedWUList(&StarMap, ktb);
	cwu->setMinimumSize(640, 130);
	ktb->addTab(cwu, tlabel[3]);
	opt = new Options(ktb);
	opt->setMinimumSize(640, 130);
	ktb->addTab(opt, tlabel[4]);
	abt = new About(ktb);
	abt->setMinimumSize(640, 130);
	ktb->addTab(abt, tlabel[5]);
	setCentralWidget(ktb);

  // connect the refresh timer to the Analysis list slot
  handleTabSelection(SetiLoc::AnalysisList);
						
	completed_timer = new QTimer(this);
  connect( completed_timer, SIGNAL(timeout()), SLOT(checkWUStatus()) );
	// an interval of 2 sec should be enough
	// obviously not; reduced to 500 ms
	completed_timer->start(StatusInterval, FALSE);
		
  connect(ktb, SIGNAL(tabSelected(int)), this, SLOT(handleTabSelection(int)));
	connect(ana, SIGNAL(listModified()), cwu, SLOT(refreshTreeList()));
	connect(ana, SIGNAL(listModified()), di, SLOT(refreshList()));
	connect(ana, SIGNAL(listModified()), ui, SLOT(refreshList()));

	startClients();
					
	ksdock = new DockWidget(this);
  connect(ana, SIGNAL(selectionChanged(SetiLoc*)), ksdock, SLOT(updateDockWidget(SetiLoc*)));
	connect(opt, SIGNAL(dockStatusChanged()), this, SLOT(toggleIcon()));
  // if ksetiwatch was closed in minimized state start it minimized again
  if(MINIMIZETOPANEL) ksdock->dock();
	}
	
Ksetiwatch::~Ksetiwatch()
	{
  }

void Ksetiwatch::closeEvent(QCloseEvent *e)
	{
	saveConfig();
	// clear the list, otherwise there may be a segfault on exit
	SetiList.clear();
	KMainWindow::closeEvent( e );
	}

bool Ksetiwatch::event(QEvent *e)
{
switch( (int)e->type() )
	{
	case QEvent::Hide:
		{
		if(MINIMIZETOPANEL)
			{
			SetiLoc* loc = getLocation(SetiLoc::AnalysisList, ana->selectedItem());
			ksdock->dock(loc);
			ksdock->setActiveWindow();
			}
		startmin = true;
		break;
		}
	case QEvent::Show:
		{
		startmin = false;
		break;
		}
	}
return KMainWindow::event( e );
}

void Ksetiwatch::saveConfig()
	{
	QString entry;
	
	config = kapp->config();

	config->setGroup("General Options");
	QRect r(pos().x(), pos().y(), size().width(), size().height());
	config->writeEntry("WindowSize", r);
	config->writeEntry("RefreshInterval", RefreshInterval);
	config->writeEntry("TimeFormat", HMS);
	config->writeEntry("NewPeak_Record", NEWPEAK_RECORD);
	config->writeEntry("NewPeak_CurrentWU", NEWPEAK_CURRENTWU);
	config->writeEntry("MinimizeToPanel", MINIMIZETOPANEL);
	config->writeEntry("DrawGrid", DRAWGRID);
	config->writeEntry("StartMinimized", startmin);
	config->writeEntry("MarkerSize", MARKER_SIZE);
	config->writeEntry("FillMarker", FILL_MARKER);
	config->writeEntry("ShowText", SHOW_TEXT);
		
	config->setGroup("Seti Locations");
  int count = SetiList.count();
  config->writeEntry("count", count);

  SetiLoc *loc;
  int i = 1;
	for( loc=SetiList.first(); loc != 0; loc=SetiList.next() )
		{
		entry.sprintf("Location_%d",i);
		config->writeEntry(entry, loc->location());
		entry.sprintf("Directory_%d",i);
		config->writeEntry(entry, loc->directory());
		entry.sprintf("Color_%d",i);
		config->writeEntry(entry, loc->color() );
		entry.sprintf("Result_Timestamp_%d",i);
		config->writeEntry(entry, loc->timestamp());
		entry.sprintf("CommandLine_Arguments_%d",i);
		config->writeEntry(entry, loc->cmdLineArgs());
		entry.sprintf("Startup_%d",i);
		config->writeEntry(entry, loc->startup());
		entry.sprintf("LogWU_%d",i);
		config->writeEntry(entry, loc->logwu());
		i++;
		}		

  config->setGroup("Record Scores");
  // this is to avoid rounding errors
  entry.sprintf("%f", Record.spike);
  config->writeEntry("Spike", entry);
  config->writeEntry("Spike_ChirpRate", Record.spike_chirprate);
  config->writeEntry("Spike_WUName", Record.spike_wu_name);
  config->writeEntry("Gaussian", Record.gaussian);
  config->writeEntry("Gaussian_Power", Record.gaussian_power);
  config->writeEntry("Gaussian_Chisq", Record.gaussian_chisq);
  config->writeEntry("Gaussian_ChirpRate", Record.gaussian_chirprate);
  config->writeEntry("Gaussian_WUName", Record.gaussian_wu_name);
  config->writeEntry("Pulse", Record.pulse);
  config->writeEntry("Pulse_Power", Record.pulse_power);
  config->writeEntry("Pulse_Period", Record.pulse_period);
  config->writeEntry("Pulse_ChirpRate", Record.pulse_chirprate);
  config->writeEntry("Pulse_WUName", Record.pulse_wu_name);
  config->writeEntry("Triplet", Record.triplet);
  config->writeEntry("Triplet_Power", Record.triplet_power);
  config->writeEntry("Triplet_Period", Record.triplet_period);
  config->writeEntry("Triplet_ChirpRate", Record.triplet_chirprate);
  config->writeEntry("Triplet_WUName", Record.triplet_wu_name);

  // write the config to disk immediately and not at shutdown
  config->sync();
  }

void Ksetiwatch::readConfig()
	{
	QString d, l, c, r;
	
	config = kapp->config();

	config->setGroup("General Options");
	QRect sdefault(0, 0, 640, 165);
  QRect s = config->readRectEntry("WindowSize", &sdefault);
  this->setGeometry(s);
  RefreshInterval   = config->readNumEntry("RefreshInterval", 15000);
  HMS               = config->readBoolEntry("TimeFormat", true);
  NEWPEAK_RECORD    = config->readBoolEntry("NewPeak_Record", false);
  NEWPEAK_CURRENTWU = config->readBoolEntry("NewPeak_CurrentWU", false);
  MINIMIZETOPANEL   = config->readBoolEntry("MinimizeToPanel", false);
  DRAWGRID          = config->readBoolEntry("DrawGrid", true);
  startmin          = config->readBoolEntry("StartMinimized", false);
  MARKER_SIZE       = config->readNumEntry("MarkerSize", 2);
  FILL_MARKER       = config->readBoolEntry("FillMarker", true);
  SHOW_TEXT         = config->readBoolEntry("ShowText", true);

	config->setGroup("Seti Locations");
	int count = config->readNumEntry("count");
	
	if(count > 0)
		{
		for(int i=1;i<=count;i++)
			{
			d.sprintf("Directory_%d", i);
			l.sprintf("Location_%d", i);
			c.sprintf("Color_%d", i);
			r.sprintf("Result_Timestamp_%d", i);
			SetiLoc *loc = new SetiLoc(config->readEntry(d), config->readEntry(l), config->readColorEntry(c) );
			// there seems to be a bug in the KConfig class; group has to be set once more here
			config->setGroup("Seti Locations");
			loc->setTimestamp(config->readEntry(r, ""));
			d.sprintf("CommandLine_Arguments_%d", i);
			loc->setCmdLineArgs(config->readEntry(d, ""));
			d.sprintf("Startup_%d", i);
			loc->setStartup(config->readBoolEntry(d, false));
			d.sprintf("LogWU_%d", i);
			loc->setLogwu(config->readBoolEntry(d, true));
			SetiList.append(loc);			
			}
		}

  config->setGroup("Record Scores");
	Record.spike              = config->readDoubleNumEntry("Spike", 0.0);
	Record.spike_chirprate    = config->readDoubleNumEntry("Spike_ChirpRate", 0.0);		
	Record.spike_wu_name      = config->readEntry("Spike_WUName", "unknown");
	Record.gaussian           = config->readDoubleNumEntry("Gaussian", 0.0);
	Record.gaussian_power     = config->readDoubleNumEntry("Gaussian_Power", 0.0);
	Record.gaussian_chisq     = config->readDoubleNumEntry("Gaussian_Chisq", 0.0);
	Record.gaussian_chirprate = config->readDoubleNumEntry("Gaussian_ChirpRate", 0.0);
	Record.gaussian_wu_name   = config->readEntry("Gaussian_WUName", "unknown");
	Record.pulse              = config->readDoubleNumEntry("Pulse", 0.0);
	Record.pulse_power        = config->readDoubleNumEntry("Pulse_Power", 0.0);
	Record.pulse_period       = config->readDoubleNumEntry("Pulse_Chisq", 0.0);
	Record.pulse_chirprate    = config->readDoubleNumEntry("Pulse_ChirpRate", 0.0);
	Record.pulse_wu_name      = config->readEntry("Pulse_WUName", "unknown");
	Record.triplet            = config->readDoubleNumEntry("Triplet", 0.0);
	Record.triplet_power      = config->readDoubleNumEntry("Triplet_Power", 0.0);
	Record.triplet_period     = config->readDoubleNumEntry("Triplet_Chisq", 0.0);
	Record.triplet_chirprate  = config->readDoubleNumEntry("Triplet_ChirpRate", 0.0);
	Record.triplet_wu_name    = config->readEntry("Triplet_WUName", "unknown");
	}

void Ksetiwatch::handleTabSelection(int id)
	{
	disconnect(RefreshTimer, SIGNAL(timeout()), ana, SLOT(refreshList()));
	disconnect(RefreshTimer, SIGNAL(timeout()), di, SLOT(refreshList()));
	disconnect(RefreshTimer, SIGNAL(timeout()), ui, SLOT(refreshList()));
	switch(id)
		{
		case SetiLoc::AnalysisList:
			{
		  connect( RefreshTimer, SIGNAL(timeout()), ana, SLOT(refreshList()) );
			statusBar()->changeItem(i18n("Press the right mouse button to edit the list."), 0);
			ana->refreshList();
		  break;
		  }
		case SetiLoc::DataInfoList:
			{
		  connect( RefreshTimer, SIGNAL(timeout()), di, SLOT(refreshList()) );
			statusBar()->changeItem(i18n("Ready."), 0);
			di->refreshList();
		  break;
		  }
		case SetiLoc::UserInfoList:
			{
		  connect( RefreshTimer, SIGNAL(timeout()), ui, SLOT(refreshList()) );
			statusBar()->changeItem(i18n("Ready."), 0);
			ui->refreshList();
		  break;
		  }
		case 3: // Completed Work Units
			{
			statusBar()->changeItem(i18n("Right click for options."), 0);
			break;
			}
		default:
			{
			statusBar()->changeItem(i18n("Ready."), 0);
			break;
			}
		}
	}
				
/** Checks for a completed work unit */
void Ksetiwatch::checkWUStatus()
{
SetiLoc *loc;
char *ext("");

QString state_file;
QString data_file;
QString result_file;
QString resultheader_file;
QString wtemp_file;

completed_timer->stop();

for( loc=SetiList.first(); loc != 0; loc=SetiList.next() )
	{
	state_file = loc->directory() + "/state";
	data_file = loc->directory() + "/work_unit";
	result_file = loc->directory() + "/result";
	resultheader_file = loc->directory() + "/result_header";
	wtemp_file = loc->directory() + "/wtemp";

	switch(loc->checkVersion())
		{
		case 1: // version 1.0x
			{
			ext = ".txt";
			break;
			}
		default: // version 2.x and higher
			{
			ext = ".sah";
			break;
			}
		}
  state_file += ext;
  data_file += ext;
  result_file += ext;
  resultheader_file += ext;
  wtemp_file += ext;

  // depending on the settings of the combobox add items to or delete items from the list
  ana->updateAnalysisList(loc);

  // check and display the status
  loc->checkStatus(state_file, data_file, result_file, resultheader_file, wtemp_file);
  	  	
	if(loc->logwu() == true)
		{
		if(QFile::exists(data_file) == false)
			{
			// work unit file doesn't exist; this is an indication of a completed work unit
			if(QFile::exists(result_file) == true && QFile::exists(resultheader_file) == false)
				{
				// result file exists, while result_header is gone; we're getting closer
				// in the following, we're using the timestamp of the result file to
				// prevent multiple logging
				QFileInfo fi(result_file);
				QDateTime dt = QDateTime(fi.lastModified());
				QString res_dt = dt.toString();
				if(res_dt != loc->timestamp())
					{
					loc->setTimestamp(res_dt);
					logWorkUnit(loc);
					// save the config to make sure that the timestamp is written to disk
					saveConfig();
					cwu->refreshList(loc);
					}
				}
			}
		}
	}
completed_timer->start(StatusInterval, FALSE);
}

/** Writes all important work unit results to a file named SETILog.csv */
void Ksetiwatch::logWorkUnit(SetiLoc *loc)
{
QString buffer, entry;
char *ext("");
int i;
int error(0);
int warn(0);
int version;
QString setilog_file(loc->directory() + "/SETILog.csv");
QString user_file(loc->directory() + "/user_info");
QString result_file(loc->directory() + "/result");
QString state_file(loc->directory() + "/state");

	version = loc->checkVersion();
	switch(version)
		{
		case 1: // version 1.0x
			{
			ext = ".txt";
			break;
			}
		default: // version 2.x and higher
			{
			ext = ".sah";
			break;
			}
		}
  state_file += ext;
  user_file += ext;
  result_file += ext;

QFile f_sel(setilog_file);
QFile f_usr(user_file);
QFile f_res(result_file);
QFile f_stt(state_file);

// check if SETILog.csv needs to be upgraded
CSVDataBase csv((const char *)setilog_file);
bool upgrade = false;
if(f_sel.exists())
	{
	if(csv.open(IO_ReadOnly))
		{
		int cols = csv.columnCount();
		if(cols < CSV_USERINFO_COLUMNS+CSV_DATAINFO_COLUMNS+CSV_STATE_COLUMNS)
			{
			// rename SETILog.csv to SETILog.old
			if(0 != rename((const char*)setilog_file, (const char*)(loc->directory()+"/SETILog.old")))
				{
				error = 3;
				debug("Renaming SETILog.csv failed.");
				}
			else
				{
				upgrade = true;
				}
			}
		csv.close();
  	}
  }

// create new SETILog.csv if necessary
if(f_sel.exists() == false && error == 0)
	{
	if(f_sel.open(IO_ReadWrite))
		{
		// write the definition line
	  for(i=0;i<CSV_USERINFO_COLUMNS;i++)
	  	{
	  	f_sel.putch('"');
	  	f_sel.writeBlock(csv_userinfo[i].text, strlen(csv_userinfo[i].text));
	  	f_sel.putch('"');	f_sel.putch(',');
	  	}
	  for(i=0;i<CSV_DATAINFO_COLUMNS;i++)
	  	{
	  	f_sel.putch('"');
	  	f_sel.writeBlock(csv_datainfo[i].text, strlen(csv_datainfo[i].text));
	  	f_sel.putch('"');	f_sel.putch(',');
	  	}
	  for(i=0;i<CSV_STATE_COLUMNS;i++)
	  	{
	  	f_sel.putch('"');
	  	f_sel.writeBlock(csv_state[i].text, strlen(csv_state[i].text));
			f_sel.putch('"');
	  	if(i < CSV_STATE_COLUMNS-1) f_sel.putch(',');
	  	}
	  // if the upgrade flag is set copy the data from SETILog.old to the new file
	  if(upgrade)
	  	{
			QFile oldf_sel(loc->directory()+"SETILog.old");
			if(oldf_sel.open(IO_ReadWrite))
				{
				QTextStream tout(&oldf_sel);
				QTextStream tin(&f_sel);
				QString s;
				i = 0;
				while(!oldf_sel.atEnd())
					{
					s = tout.readLine();
					// discard the first line
					if(i > 0) tin << endl << s;
					i++;
					}
				oldf_sel.close();
				}
			else
				{
				error = 4;
				debug("Could not create new SETILog.csv.");
				}
			}
	  f_sel.close();
	  }
	}	

QFileInfo finfo((const char*)result_file);
QDateTime dt = finfo.lastModified();
if(dt.isValid())
	{
	QDate d = dt.date();
	buffer.sprintf("\"%d.%02d.%02d ", d.year(), d.month(), d.day() );
	QTime t = dt.time();
	buffer += t.toString() + "\",";
	}
else
	{
	QDate d = QDate::currentDate();
	buffer.sprintf("\"%d.%02d.%02d ", d.year(), d.month(), d.day() );
	QTime t = QTime::currentTime();
	buffer += t.toString() + "\",";
	}

// open the 'user info' file for reading the necessary information
if(f_usr.exists() && error == 0)
	{
	for(i=1;i<CSV_USERINFO_COLUMNS;i++)
		{
  	if(i == 4) // instead of "user_name" take "name" to retrieve the users's name
  		{
			buffer += constructEntry((const char*)user_file, "name", csv_userinfo[i].type);
  		}
  	else
  		{
			buffer += constructEntry((const char*)user_file, csv_userinfo[i].text, csv_userinfo[i].type);
			}
		}
  }
else
	error = 1;

// open the 'result' file for reading the necessary information
if(f_res.exists() && error == 0)
	{
	for(i=0;i<CSV_DATAINFO_COLUMNS;i++)
		{
		if(i == 1) // version
			{
			entry.sprintf("%d,", version);
			buffer += entry;
			}
		else
			buffer += constructEntry((const char*)result_file, csv_datainfo[i].text, csv_datainfo[i].type);
		}
	}
else if(error == 0)
	error = 2;

// open the 'state' file for reading the necessary information
// if we cannot find the state file retrieve stored values in SetiLoc
bool statefile_exists(f_stt.exists());
if(error == 0)
	{
	for(i=0;i<CSV_STATE_COLUMNS;i++)
		{
		if(statefile_exists)
			{
			buffer += constructEntry((const char*)state_file, csv_state[i].text, csv_state[i].type);
			}
		else
			{
			QString s;
			WUScore* wu = loc->topScore();
			switch(i)
				{
				case 0: // ncfft
					s = "9024";
					break;
				case 1: // cr
					s = "-9.996793e+00";
					break;
				case 2: // fl
					s = "32768";
					break;
				case 3: // cpu
					s.setNum((double)loc->CPUTime());
					break;
				case 4: // prog
					s = "1.0";
					break;
				case 5: // bs_power
					s.setNum(wu->spike);
					break;
				case 6: // bs_score
					s.setNum(wu->spike_score);
					break;
				case 9:
					s.setNum(wu->spike_chirprate);
					break;				
				case 7: // don't care about unneeded parameters
				case 8:
				case 10:
					s = "0";
					break;
				case 11:
					s.setNum(wu->gaussian);
					break;
				case 12: // bg_power
					s.setNum(wu->gaussian_power);
					break;
				case 13: // bg_chisq
					s.setNum(wu->gaussian_chisq);
					break;
				case 16: // bg_chirp_rate
					s.setNum(wu->gaussian_chirprate);
					break;
				case 14: // don't care about unneeded parameters
				case 15:
				case 17:
					s = "0";
					break;
				case 18: // bp_score
					s.setNum(wu->pulse);
					break;
				case 19: // bp_power
					s.setNum(wu->pulse_power);
					break;
				case 21: // bp_period
					s.setNum(wu->pulse_period);
					break;
				case 24: // bp_chirprate
					s.setNum(wu->pulse_chirprate);
					break;
				case 20:
				case 22:
				case 23:
				case 25:
					s = "0";
					break;
				case 26: // bt_score
					s.setNum(wu->triplet);
					break;
				case 27: // bt_power
					s.setNum(wu->triplet_power);
					break;
				case 29: // bt_period
					s.setNum(wu->triplet_period);
					break;
				case 39: // bt_chirprate
					s.setNum(wu->triplet_chirprate);
					break;				
				case 28: // don't care about unneeded parameters
				case 30:
				case 31:
				case 32:
				case 33:
				case 34:
				case 35:
				case 36:
				case 37:
				case 38:
				case 40:
				case 41:
					s = "0";
					break;
				default:
					s = "";
					break;
				}
			buffer += s;
			buffer += ',';
			warn = 1;
			}
		}
	buffer = buffer.left(buffer.length()-1);
	}

if(error == 0)
	{			
  if(f_sel.open(IO_Append | IO_WriteOnly))
  	{
  	f_sel.putch('\n');
  	f_sel.writeBlock((const char *)buffer, strlen(buffer));
  	
  	f_sel.close();
  	
  	if(warn)
  		{
  		QString buf;
  		buf.sprintf(i18n("Ksetiwatch has logged the work unit for \"%s\" without\n"
						  		"existing 'state.sah' file. Some values might therefore be wrong\n"
						  		"or out of date. Consider adding the '-stop_after_process' \n"
						  		"command line option when starting the SETI@home client."),
						  		(const char*)loc->location());
 			QMessageBox* mb = new QMessageBox(i18n("Warning"),
 													(const char*)buf, QMessageBox::Information,
  												QMessageBox::Ok|QMessageBox::Default,
  												0, 0, 0, 0, false);
 			mb->show();
  		}
  	}
  }
else
	{
	if(f_sel.isOpen()) f_sel.close();
	if(f_usr.isOpen()) f_usr.close();
	if(f_res.isOpen()) f_res.close();
	if(f_stt.isOpen()) f_stt.close();
	buffer.sprintf(i18n("Missing file(s). Logging of work unit failed.\nError code: %d"), error);
	KMessageBox::error(this, (const char*)buffer);
	}
}

/** constructs a string from readEntry  */
QString Ksetiwatch::constructEntry(const char* fn, const char *id, int type)
{
QString entry = SetiLoc::readEntry(fn, id);
if(type == 0)
	{
	entry.append(",");
	}
if(type == 1)
	{
	if(!entry.isEmpty())
		{
		entry.insert(0, '\"');
		entry.append("\",");
		}
	else
		{
		entry.append(",");
		}
	}

return entry;
}

/** Starts the SETI@home clients if the startup switch is on. */
void Ksetiwatch::startClients()
{
SetiLoc *loc;

for( loc=SetiList.first(); loc != 0; loc=SetiList.next() )
	{
	if(loc->startup() == true) loc->startClient();
	}
}

/** Returns the corresponding SetiLoc entry from
		a QListView item in the Analysis list. */
SetiLoc* Ksetiwatch::getLocation(int list, QListViewItem* it)
{
SetiLoc *loc = 0;

if(it)
	{
	// identify corresponding SetiLoc item
	for(loc=SetiList.first(); loc != 0; loc=SetiList.next() )
		{
		if(loc->item(list) == it) break;
		}
	}
return( loc );
}

/** initializes the record score with zeros and empty strings */
void Ksetiwatch::initWUScore(WUScore* score)
{
score->spike           = 0.0;
score->spike_score     = 0.0;
score->spike_bin       = 0;
score->spike_fft_index = 0;
score->spike_fft_len   = 0;
score->spike_chirprate = 0.0;
score->spike_wu_name   = "";

score->gaussian           = 0.0;
score->gaussian_power     = 0.0;
score->gaussian_bin       = 0;
score->gaussian_chisq     = 0.0;
score->gaussian_true_mean = 0.0;
score->gaussian_sigma     = 0.0;
score->gaussian_fft_index = 0;
score->gaussian_fft_len   = 0;
score->gaussian_chirprate = 0.0;
score->gaussian_wu_name   = "";
for(int i=0;i<64;i++) score->gaussian_data[i] = 0.0;

score->pulse           = 0.0;
score->pulse_power     = 0.0;
score->pulse_mean      = 0.0;
score->pulse_period    = 0.0;
score->pulse_freq_bin  = 0;
score->pulse_time_bin  = 0;
score->pulse_fft_len   = 0;
score->pulse_chirprate = 0.0;
score->pulse_wu_name   = "";

score->triplet            = 0.0;
score->triplet_power      = 0.0;
score->triplet_mean       = 0.0;
score->triplet_period     = 0.0;
score->triplet_bperiod    = 0.0;
score->triplet_tpotind0_0 = 0;
score->triplet_tpotind0_1 = 0;
score->triplet_tpotind1_0 = 0;
score->triplet_tpotind1_1 = 0;
score->triplet_tpotind2_0 = 0;
score->triplet_tpotind2_1 = 0;
score->triplet_freq_bin   = 0;
score->triplet_time_bin   = 0.0;
score->triplet_scale      = 0.0;
score->triplet_fft_len    = 0;
score->triplet_chirprate  = 0.0;
score->triplet_wu_name    = "";
}

/** Toggles dock icon on or off when the user makes changes in the options. */
void Ksetiwatch::toggleIcon()
{
if(MINIMIZETOPANEL)
	ksdock->dock();
else
	ksdock->undock();
}
