/***************************************************************************
                          setiloc.cpp  -  description
                             -------------------
    begin                : Wed Oct 27 1999
    copyright            : (C) 1999 by Gordon Machel
    email                : gmachel@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include <kapp.h>
#include <klocale.h>

#include <qtextstream.h>
#include <qfileinfo.h>
#include <qmessagebox.h>
#include <qbitmap.h>
#include <qdir.h>

#include <sys/types.h>
#include <signal.h>
#include <unistd.h>
#include <math.h>

#include "setiloc.h"
#include "analysis.h"
#include "datainfo.h"
#include "userinfo.h"
#include "ksetiwatch.h"

extern bool HMS;
extern int  RefreshInterval;
extern int  StatusInterval;
extern bool		NEWPEAK_RECORD;
extern bool		NEWPEAK_CURRENTWU;
extern WUScore Record;

SetiLoc::SetiLoc(const QString& dir, const QString& loc, QColor col)
	{
	slocation = loc;
	sdirectory = dir;
	scolor = col;
	status = Stopped;
	for(int i=0;i<3;i++)
	  {
  	graphvsbl[i] = false;
  	graphwidget[i] = 0;
  	}
	sresult_timestamp = "";
	if(RefreshInterval < 90000)
		statuscheckcount = 90000/StatusInterval;
	else
		statuscheckcount = RefreshInterval/StatusInterval;	 			
	chirprate = -1000.0;
	progress = -1.0;
	progress_old = -1.0;
	cputime = 0.0;
	totaltime = 0.0;
	wu_completed = 0;
	progressrate = 0.0;
	
	// this ensures that all data are read when checking for the first time
	max.spike    = -1.0;
	max.gaussian = -1.0;
	max.pulse    = -1.0;
	max.triplet  = -1.0;
	
	for(int i=0; i<3; i++) listitem[i] = 0;
	
	statusIcons = new KIconLoader();
	status_icon[0] = statusIcons->loadIcon("mini-setistopped", KIcon::User);
	status_icon[1] = statusIcons->loadIcon("mini-setirunning", KIcon::User);	
	status_icon[2] = statusIcons->loadIcon("mini-setifinished", KIcon::User);
	status_icon[3] = statusIcons->loadIcon("mini-setiloading", KIcon::User);
	
	client_pid = (pid_t)0;	
	}

SetiLoc::~SetiLoc()
	{
	}

// adds an item to the list
QListViewItem* SetiLoc::addItem(int list, QListView* lv)
{
QListViewItem* it;

if(list == AnalysisList)
	{
	it = new AnalysisListViewItem(lv);
	}
else if(list == DataInfoList)
	{
	it = new DataInfoListViewItem(lv);
	}
else if(list == UserInfoList)
	{
	it = new UserInfoListViewItem(lv);
	}
else
	{
	it = new QListViewItem(lv);
	}

listitem[list] = it;
return(it);
}

/** Removes the QListView item from the list. */
void SetiLoc::removeItem(int list)
{
if(listitem[list]) delete listitem[list];
listitem[list] = 0;
}

// removes all ListView items
void SetiLoc::removeAllItems()
{
for(int i=0;i<3;i++) removeItem(i);
}

bool SetiLoc::refresh(int list)
	{
	QString buf;
	QString title;
	QString val;
	QString state_file = sdirectory + "/state";
	QString data_file = sdirectory + "/work_unit";
	QString user_file = sdirectory + "/user_info";
	int h, m, s, k, l;
	int version;
	WUScore wusg, record_old, max_old;
	double tmp;

	version = checkVersion();
	switch(version)
		{
		case 1: // version 1.0x
			{
			val = ".txt";
			break;
			}
		default: // version 2.x and higher
			{
			val = ".sah";
			break;
			}
		}
  state_file += val;
  data_file += val;
  user_file += val;

	bool new_record = false;	
	record_old = Record;
	max_old = max;
	
	wusg = readStateData( state_file );
	if(wusg.spike > max.spike || max.spike == -1.0)
		{
		max.spike = wusg.spike;
		max.spike_bin = wusg.spike_bin;
		max.spike_chirprate = wusg.spike_chirprate;
		max.spike_fft_index = wusg.spike_fft_index;
		max.spike_fft_len = wusg.spike_fft_len;
		max.spike_score = wusg.spike_score;		
		}
	if(wusg.gaussian > max.gaussian || max.gaussian == -1.0)	
		{
		max.gaussian = wusg.gaussian;
		max.gaussian_bin = wusg.gaussian_bin;
		max.gaussian_power = wusg.gaussian_power;
		max.gaussian_chisq = wusg.gaussian_chisq;
		max.gaussian_chirprate = wusg.gaussian_chirprate;
		max.gaussian_sigma = wusg.gaussian_sigma;
		max.gaussian_true_mean = wusg.gaussian_true_mean;
		max.gaussian_fft_index = wusg.gaussian_fft_index;
		max.gaussian_fft_len = wusg.gaussian_fft_len;
		readGaussianData( state_file );
		emit newGaussian(this);
		}
	if(wusg.pulse > max.pulse || max.pulse == -1.0)
		{
		max.pulse           = wusg.pulse;
		max.pulse_power     = wusg.pulse_power;
		max.pulse_mean      = wusg.pulse_mean;
		max.pulse_period    = wusg.pulse_period;
		max.pulse_freq_bin  = wusg.pulse_freq_bin;
		max.pulse_time_bin  = wusg.pulse_time_bin;
		max.pulse_fft_len   = wusg.pulse_fft_len;
		max.pulse_chirprate = wusg.pulse_chirprate;
		readPulseData( state_file );
		emit newPulse(this);
		}

	if(wusg.triplet > max.triplet || max.triplet == -1.0)
		{
		max.triplet            = wusg.triplet;
		max.triplet_power      = wusg.triplet_power;
		max.triplet_mean       = wusg.triplet_mean;
		max.triplet_period     = wusg.triplet_period;
		max.triplet_bperiod    = wusg.triplet_bperiod;
		max.triplet_tpotind0_0 = wusg.triplet_tpotind0_0;
		max.triplet_tpotind0_1 = wusg.triplet_tpotind0_1;
		max.triplet_tpotind1_0 = wusg.triplet_tpotind1_0;
		max.triplet_tpotind1_1 = wusg.triplet_tpotind1_1;
		max.triplet_tpotind2_0 = wusg.triplet_tpotind2_0;
		max.triplet_tpotind2_1 = wusg.triplet_tpotind2_1;
		max.triplet_freq_bin   = wusg.triplet_freq_bin;
		max.triplet_time_bin   = wusg.triplet_time_bin;
		max.triplet_scale      = wusg.triplet_scale;
		max.triplet_fft_len    = wusg.triplet_fft_len;
		max.triplet_chirprate  = wusg.triplet_chirprate;
		readTripletData( state_file );
		emit newTriplet(this);
		}
		
	if(wusg.spike > Record.spike)
		{
		Record.spike = wusg.spike;
		Record.spike_bin = wusg.spike_bin;
		Record.spike_chirprate = wusg.spike_chirprate;
		Record.spike_fft_index = wusg.spike_fft_index;
		Record.spike_fft_len = wusg.spike_fft_len;
		Record.spike_score = wusg.spike_score;
		Record.spike_wu_name = readWUName( data_file );
		new_record = true;
		}
	if(wusg.gaussian > Record.gaussian)
		{
		Record.gaussian = wusg.gaussian;
		Record.gaussian_bin = wusg.gaussian_bin;
		Record.gaussian_power = wusg.gaussian_power;
		Record.gaussian_chisq = wusg.gaussian_chisq;
		Record.gaussian_sigma = wusg.gaussian_sigma;
		Record.gaussian_true_mean = wusg.gaussian_true_mean;
		Record.gaussian_chirprate = wusg.gaussian_chirprate;
		Record.gaussian_fft_index = wusg.gaussian_fft_index;
		Record.gaussian_fft_len = wusg.gaussian_fft_len;
		Record.gaussian_wu_name = readWUName( data_file );
		for(int i=0;i<64;i++) Record.gaussian_data[i] = max.gaussian_data[i];
		new_record = true;
		emit newGaussian(this);
		}
	if(wusg.pulse > Record.pulse)
		{
		Record.pulse           = wusg.pulse;
		Record.pulse_power     = wusg.pulse_power;
		Record.pulse_mean      = wusg.pulse_mean;
		Record.pulse_period    = wusg.pulse_period;
		Record.pulse_freq_bin  = wusg.pulse_freq_bin;
		Record.pulse_time_bin  = wusg.pulse_time_bin;
		Record.pulse_fft_len   = wusg.pulse_fft_len;
		Record.pulse_chirprate = wusg.pulse_chirprate;
		Record.pulse_wu_name   = readWUName( data_file);
		for(int i=0;i<512;i++) Record.pulse_data[i] = max.pulse_data[i];
		new_record = true;
		emit newPulse(this);
		}
	if(wusg.triplet > Record.triplet)
		{
		Record.triplet            = wusg.triplet;
		Record.triplet_power      = wusg.triplet_power;
		Record.triplet_mean       = wusg.triplet_mean;
		Record.triplet_period     = wusg.triplet_period;
		Record.triplet_bperiod    = wusg.triplet_bperiod;
		Record.triplet_tpotind0_0 = wusg.triplet_tpotind0_0;
		Record.triplet_tpotind0_1 = wusg.triplet_tpotind0_1;
		Record.triplet_tpotind1_0 = wusg.triplet_tpotind1_0;
		Record.triplet_tpotind1_1 = wusg.triplet_tpotind1_1;
		Record.triplet_tpotind2_0 = wusg.triplet_tpotind2_0;
		Record.triplet_tpotind2_1 = wusg.triplet_tpotind2_1;
		Record.triplet_freq_bin   = wusg.triplet_freq_bin;
		Record.triplet_time_bin   = wusg.triplet_time_bin;
		Record.triplet_scale      = wusg.triplet_scale;
		Record.triplet_fft_len    = wusg.triplet_fft_len;
		Record.triplet_chirprate  = wusg.triplet_chirprate;
		Record.triplet_wu_name    = readWUName( data_file);
		for(int i=0;i<512;i++) Record.triplet_data[i] = max.triplet_data[i];
		new_record = true;
		emit newTriplet(this);
		}
	
	if(NEWPEAK_RECORD)
		{
		if(wusg.spike > record_old.spike)
			{
			QString msgbuf;
			msgbuf.sprintf(i18n("You have found a new record peak:\n"
									"peak = %.3lf at %.4lf Hz/s chirp rate,\n"
									"found in work unit %s.\n\n"
									"Previous peak:\n"
									"peak = %.3lf at %.4lf Hz/s chirp rate,\n"
									"found in work unit %s."),
									Record.spike, Record.spike_chirprate,
									(const char*)Record.spike_wu_name,
									record_old.spike, record_old.spike_chirprate,
									(const char*)record_old.spike_wu_name);
 			title.sprintf(i18n("New Record Spike for %s"), (const char*)slocation);			
 			QMessageBox* mb = new QMessageBox(title, msgbuf, QMessageBox::Information,
  												QMessageBox::Ok|QMessageBox::Default, 0, 0, 0, 0, false);
  		mb->show();
			}
		if(wusg.gaussian > record_old.gaussian)
			{
			QString msgbuf;
			msgbuf.sprintf(i18n("You have found a new record gaussian:\n"
									"score = %.4lf, power = %.2lf, fit = %.2lf at %.4lf Hz/s chirp rate,\n"
									"found in work unit %s.\n\n"
									"Previous gaussian:\n"
									"score = %.4lf, power = %.2lf, fit = %.2lf at %.4lf Hz/s chirp rate,\n"
									"found in work unit %s.\n"),
									Record.gaussian, Record.gaussian_power, Record.gaussian_chisq,
									Record.gaussian_chirprate, (const char*)Record.gaussian_wu_name,
									record_old.gaussian, record_old.gaussian_power,
									record_old.gaussian_chisq, record_old.gaussian_chirprate,
									(const char*)record_old.gaussian_wu_name);
 			title.sprintf(i18n("New Record Gaussian for %s"), (const char*)slocation);			
 			QMessageBox* mb = new QMessageBox(title, msgbuf, QMessageBox::Information,
  												QMessageBox::Ok|QMessageBox::Default, 0, 0, 0, 0, false);
 			mb->show();
			}
		if(wusg.pulse > record_old.pulse)
			{
			QString msgbuf;
			msgbuf.sprintf(i18n("You have found a new record pulse:\n"
									"score = %.4lf, power = %.2lf, period = %.4lf at %.4lf Hz/s chirp rate,\n"
									"found in work unit %s.\n\n"
									"Previous pulse:\n"
									"score = %.4lf, power = %.2lf, period = %.4lf at %.4lf Hz/s chirp rate,\n"
									"found in work unit %s.\n"),
									Record.pulse, Record.pulse_power, Record.pulse_period,
									Record.pulse_chirprate, (const char*)Record.pulse_wu_name,
									record_old.pulse, record_old.pulse_power,
									record_old.pulse_period, record_old.pulse_chirprate,
									(const char*)record_old.pulse_wu_name);
 			title.sprintf(i18n("New Record Pulse for %s"), (const char*)slocation);			
 			QMessageBox* mb = new QMessageBox(title, msgbuf, QMessageBox::Information,
  												QMessageBox::Ok|QMessageBox::Default, 0, 0, 0, 0, false);
 			mb->show();
			}
		if(wusg.triplet > record_old.triplet)
			{
			QString msgbuf;
			msgbuf.sprintf(i18n("You have found a new record triplet:\n"
									"score = %.4lf, period = %.4lf at %.4lf Hz/s chirp rate,\n"
									"found in work unit %s.\n\n"
									"Previous triplet:\n"
									"score = %.4lf, period = %.4lf at %.4lf Hz/s chirp rate,\n"
									"found in work unit %s.\n"),
									Record.triplet, Record.triplet_period,
									Record.triplet_chirprate, (const char*)Record.triplet_wu_name,
									record_old.triplet, record_old.triplet_period,
									record_old.triplet_chirprate, (const char*)record_old.triplet_wu_name);
 			title.sprintf(i18n("New Record Pulse for %s"), (const char*)slocation);			
 			QMessageBox* mb = new QMessageBox(title, msgbuf, QMessageBox::Information,
  												QMessageBox::Ok|QMessageBox::Default, 0, 0, 0, 0, false);
 			mb->show();
			}

		if(NEWPEAK_CURRENTWU && max_old.spike != -1.0 && max_old.gaussian != -1.0 &&
		    max_old.pulse != -1 && max_old.triplet != -1)
			{			
  		if(wusg.spike > max_old.spike && wusg.spike < record_old.spike)
  			{
  			QString msgbuf;
  			msgbuf.sprintf(i18n("You have found a new high peak in the current WU:\n"
  									"peak = %.3lf at %.4lf Hz/s chirp rate.\n\n"
  									"Previous peak:\n"
  									"peak = %.3lf at %.4lf Hz/s chirp rate."),
  									wusg.spike, wusg.spike_chirprate,
  									max_old.spike, max_old.spike_chirprate);
  			title.sprintf(i18n("New High Spike for %s"), (const char*)slocation);
  			QMessageBox* mb = new QMessageBox(title, msgbuf, QMessageBox::Information,
  												QMessageBox::Ok|QMessageBox::Default, 0, 0, 0, 0, false);
  			mb->show();
  			}
  		if(wusg.gaussian > max_old.gaussian && wusg.gaussian < record_old.gaussian)
  			{
  			QString msgbuf;
  			msgbuf.sprintf(i18n("You have found a new high gaussian in the current WU:\n"
  									"score = %.4lf, power = %.2lf, fit = %.2lf at %.4lf Hz/s chirp rate.\n\n"
  									"Previous gaussian:\n"
  									"score = %.4lf, power = %.2lf, fit = %.2lf at %.4lf Hz/s chirp rate."),
  									wusg.gaussian, wusg.gaussian_power, wusg.gaussian_chisq, wusg.gaussian_chirprate,
  									max_old.gaussian, max_old.gaussian_power,
  									max_old.gaussian_chisq, max_old.gaussian_chirprate);
  			title.sprintf(i18n("New High Gaussian for %s"), (const char*)slocation);
  			QMessageBox* mb = new QMessageBox(title, msgbuf, QMessageBox::Information,
  												QMessageBox::Ok|QMessageBox::Default, 0, 0, 0, 0, false);
  			mb->show();
  			}
  		if(wusg.pulse > max_old.pulse && wusg.pulse < record_old.pulse)
  			{
  			QString msgbuf;
  			msgbuf.sprintf(i18n("You have found a new high pulse in the current WU:\n"
  									"score = %.4lf, power = %.2lf, period = %.4lf at %.4lf Hz/s chirp rate.\n\n"
  									"Previous pulse:\n"
  									"score = %.4lf, power = %.2lf, period = %.4lf at %.4lf Hz/s chirp rate."),
  									wusg.pulse, wusg.pulse_power, wusg.pulse_period, wusg.pulse_chirprate,
  									max_old.pulse, max_old.pulse_power,
  									max_old.pulse_period, max_old.pulse_chirprate);
  			title.sprintf(i18n("New High Pulse for %s"), (const char*)slocation);
  			QMessageBox* mb = new QMessageBox(title, msgbuf, QMessageBox::Information,
  												QMessageBox::Ok|QMessageBox::Default, 0, 0, 0, 0, false);
  			mb->show();
  			}
  		if(wusg.triplet > max_old.triplet && wusg.triplet < record_old.triplet)
  			{
  			QString msgbuf;
  			msgbuf.sprintf(i18n("You have found a new high triplet in the current WU:\n"
  									"score = %.4lf, period = %.4lf at %.4lf Hz/s chirp rate.\n\n"
  									"Previous triplet:\n"
  									"score = %.4lf, period = %.4lf at %.4lf Hz/s chirp rate."),
  									wusg.triplet, wusg.triplet_period, wusg.triplet_chirprate,
  									max_old.triplet, max_old.triplet_period, max_old.triplet_chirprate);
  			title.sprintf(i18n("New High Triplet for %s"), (const char*)slocation);
  			QMessageBox* mb = new QMessageBox(title, msgbuf, QMessageBox::Information,
  												QMessageBox::Ok|QMessageBox::Default, 0, 0, 0, 0, false);
  			mb->show();
  			}
			}
		}
	
	// this is set to true if statuschanged() needs to be emitted
	bool stchange(false);	
	switch(list)
		{
		case AnalysisList:
			{
    	// exit if client is loading
    	if(status == Loading) break;
    	
    	// set location text
    	if(listitem[list]) listitem[list]->setText(0, slocation);
    	
    	if (QFile::exists(state_file) && listitem[list])
    		{
    		if(status != Finished)
    			{
      		// set the actual chirp rate
  				val = readEntry(state_file, "cr" );
      		chirprate = val.toDouble();
      		val.sprintf("%.4f",chirprate);
      		listitem[list]->setText(2, val);
      		
      		// set the progress in %
      		val = readEntry(state_file, "prog" );
      		tmp = 100.0*val.toDouble();
   		  	/* Check if processing of current WU has been reset (e.g. by switching
        	   to an new version of the S@h client). */
        	/* check against -0.000001 to avoid the effect of rounding errors */
          if((tmp-progress) < -0.000001)
            {
            // initialize max score
            Ksetiwatch::initWUScore(&max);
            stchange = true;
            }
      		if((tmp-progress) > 0.000001) stchange = true;
      		progress = tmp;
      		val.sprintf("%.3f",progress);
      		listitem[list]->setText(3, val);
      		}

    		// status is now handled separately
				    		
    		// set the elapsed CPU time		
    		val = readEntry(state_file, "cpu" );
    		cputime = val.toDouble();
    		val = convertTime(cputime, HMS);
    		listitem[list]->setText(4, val);

    		// calculate and set the progress rate
    		progressrate = (cputime > 0.0) ? (3600.0/cputime)*progress : 0.0;
    		val.sprintf("%.5f",progressrate);
    		listitem[list]->setText(6, val);
    		
    		/* Calculate and set the MFlops. According to the Seti@home home page
    		   one work unit requires 2 Teraflops. */
    		/* 3.0 clients require more: ~2.65 TeraFlops (depending on WU)
    		   A more precise calculation needs to be done! */
    		double tf;
    		if(version == 3)
    			tf = 2.65e12;
    		else
    			tf = 2.0e12;
    		mflops = tf*progressrate/(100.0e6*3600.0);
    		val.sprintf("%.2f",mflops);
    		listitem[list]->setText(7, val);
    		
    		if(status != Finished)
    			{
      		// calculate and set the time required to finish this work unit
      		if(progressrate > 0.0)
      			{
      			double tl = timeleft;
  	    		timeleft = 3600.0*(100.0 - progress)/progressrate;
    	      val = convertTime(timeleft, HMS);
    	      if(tl != timeleft) stchange = true;
    	      }
    	    else
    	    	{
    	    	val = "";
    	    	timeleft = 0.0;
    	    	stchange = true;
    	    	}
      		listitem[list]->setText(5, val);
      		}

    		// set the highest peak so far
    		if(max.spike > 1.0e6)
    			val.sprintf("%.8e", max.spike);
    		else    				
	    		val.sprintf("%.3f",max.spike);
    		listitem[list]->setText(8, val);

    		// display the strongest gaussian according to its score value
    		if(max.gaussian > 0.0)
	    		val.sprintf(" scr=%.4f, pwr=%.2f, fit=%.2f", max.gaussian,
  	  								max.gaussian_power, max.gaussian_chisq);
  	  	else
  	  		val = "";
    		listitem[list]->setText(9, val);    	
    		
    		// display the strongest pulse according to its score value
    		if(max.pulse > 0.0)
	    		val.sprintf(" scr=%.4f, pwr=%.2f, prd=%.4f", max.pulse,
  	  								max.pulse_power, max.pulse_period);
  	  	else
  	  		val = "";
    		listitem[list]->setText(10, val);    	
    		
    		// display the strongest triplet according to its score value
    		if(max.triplet > 0.0)
	    		val.sprintf(" scr=%.4f, prd=%.4f", max.triplet, max.triplet_period);
  	  	else
  	  		val = "";
    		listitem[list]->setText(11, val);    	
    		}
    	break;
    	}
    case DataInfoList:
    	{
    	if(listitem[list] == 0) break;					//exit if listitem doesn't exist
    	listitem[list]->setText(0, slocation);
    	if (QFile::exists(data_file))
    		{
	    	val = readEntry(data_file, "start_ra" );
	    	sra = val.toDouble();
	    	h = static_cast<int>( sra );
	    	m = static_cast<int>( (sra - h)*60 );
	    	s = static_cast<int>( (sra - h)*3600 - m*60 );
	    	from.sprintf("%d hr %d min %d sec RA ", h, m, s);
	    	val = readEntry(data_file, "start_dec" );
	    	sdec = val.toDouble();
	    	if(sdec >= 0.0) from += '+'; else from += '-';
	    	double absdec = fabs(sdec);
	    	h = static_cast<int>( absdec );
	    	m = static_cast<int>( (absdec - h)*60 );
	    	s = static_cast<int>( (absdec - h)*3600 - m*60 );
	    	buf.sprintf(" %d deg %d' %d'' Dec", h, m, s);
	    	from += buf;
	    	listitem[list]->setText(1, from);

	    	wu_name = readEntry(data_file, "name" );
	    	
	    	val = readEntry(data_file, "time_recorded" );
	    	k = val.find('(');
	    	l = val.find(')');
	    	if(k > -1 && l > -1 && l > k)
		    	recorded = val.mid(k+1, l-k-1);
		    else
		    	recorded = "";
	    	listitem[list]->setText(2, recorded);
	    	
	    	source = readEntry(data_file, "receiver" );
	    	if(source == "ao1420")
	    		listitem[list]->setText(3, i18n("Arecibo Telescope "));
	    	
    		val = readEntry(data_file, "subband_base" );
    		base_frequency = val.toDouble()/1.0e9;
    		val.sprintf("  %.9f GHz", base_frequency);
    		listitem[list]->setText(4, val);    	
    	  }
    	else
    		{
    		for(int i=1;i<5;i++)
    			{
    			listitem[list]->setText(i, 0);
    			}
    		}
    	break;
    	}
    case UserInfoList:
    	{
    	if(listitem[list] == 0) break;					//exit if listitem doesn't exist
    	listitem[list]->setText(0, slocation);
    	if (QFile::exists(user_file) )
    		{
	    	val = readEntry(user_file, "name" );
    		listitem[list]->setText(1, val);
	    	
	    	val = readEntry(user_file, "nresults" );
    		wu_completed = val.toInt();
    		val.sprintf("%d", wu_completed);
    		listitem[list]->setText(2, val);

	    	val = readEntry(user_file, "total_cpu" );
    		total_cpu = val.toDouble();
    		val = convertTime(total_cpu, HMS);
    		listitem[list]->setText(3, val);
    		
    		average_cpu = (wu_completed > 0) ? total_cpu/wu_completed : 0.0;
    		val = convertTime(average_cpu, HMS);
    		listitem[list]->setText(4, val);

	    	val = readEntry(user_file, "last_result_time" );
	    	k = val.find('(');
	    	l = val.find(')');
	    	if(k > -1 && l > -1 && l > k)
		    	last_result = val.mid(k+1, l-k-1);
		    else
		    	last_result = "";
	    	listitem[list]->setText(5, "  " + last_result);
	    	
	    	val = readEntry(user_file, "register_time" );
	    	k = val.find('(');
	    	l = val.find(')');
	    	if(k > -1 && l > -1 && l > k)
		    	register_time = val.mid(k+1, l-k-1);
		    else
		    	register_time = "";
	    	listitem[list]->setText(6, "  " + register_time);
	    	}
    	else
    		{
    		for(int i=1;i<7;i++)
    			{
    			listitem[list]->setText(i, 0);
    			}
    		wu_completed = 0;
    		total_cpu = 0.0;
    		}
	    break;
			}
    }

  if(stchange) emit statusChanged();
  return( new_record );
  }

int SetiLoc::checkStatus(const QString& state_f, const QString& data_f, const QString& result_f,
			  								 const QString&  resultheader_f, const QString& wtemp_f)
	{
	// define this for efficiency reasons
  bool workunitfileExists(QFile::exists(data_f));

  int version = checkVersion();
	// check the status of the SETI@home client
	int oldstatus = status;
	if((QFile::exists(wtemp_f) && version >= 2) || (workunitfileExists == false && version == 1))
		{
		if(QFile::exists(resultheader_f) == false && workunitfileExists == false)
			{
			if(/*QFile::exists(state_f) &&*/ QFile::exists(result_f))
				{
				status = Finished;
				}
			else
				{
				status = Loading;
				}
			}
		}
	else // wtemp.sah doesn't exist
		{
    if(status == Loading)
    	{
 			progress_old = progress;
 			// set max score to 0
 			Ksetiwatch::initWUScore(&max);
 			// refresh Data and User Info list
 			status = Running;
 			refresh(AnalysisList);
 			refresh(DataInfoList);
 			refresh(UserInfoList);
 			// reset the check count
			if(RefreshInterval < 90000)
	 			statuscheckcount = 90000/StatusInterval;
	 		else
	 			statuscheckcount = RefreshInterval/StatusInterval;
	 		emit newWorkUnit(this);	 			
 			}

 		if(status == Running && client_pid != 0)
 			{	
  	  if(client.isRunning() == false && isClientRunning(client_pid) == false)
  	  	{
  	  	status = Stopped;
  	  	client_pid = (pid_t)0;
  	  	// status would stay Running otherwise
  	  	refresh(AnalysisList);
  	  	progress_old = -1.0;
  	  	}
  	  }
 		if(status == Stopped && status == oldstatus)
 			{	
  	  if(client.isRunning())
  	  	{
  	  	client_pid = (pid_t)client.getPid();
  	  	status = Running;
  	  	}
  	  if(isClientRunning() == true)
  	  	{
  	  	client_pid = (pid_t)getClientPid();
  	  	status = Running;
  	  	}
  	  }
  	  	
    /* Evaluate the status of this location (either running or stopped)
       by comparing the current progress value to that of 60 sec ago
       (progress_old). If RefreshInterval is greater than 60 sec status will
       be checked every time this function is called. */
 		if(client_pid == 0 && (--statuscheckcount <= 0 || status == Stopped))
 			{
 			// this is to avoid a stopped process as being indicated as running
 			// during the first check
 			if(progress_old == -1.0)
 				{
 				progress_old = progress;
 				}
 			// now compare
 			if(progress > progress_old)
 				{
 				status = Running;
 				progress_old = progress;
 				}
 			else if(progress == progress_old)
 				{
 				status = Stopped;
 				client_pid = (pid_t)0;
 				}
 			// reset the check count
			if(RefreshInterval < 90000)
	 			statuscheckcount = 90000/StatusInterval;
	 		else
	 			statuscheckcount = RefreshInterval/StatusInterval;	 			
 			}		
		}		
	
	if(status != oldstatus && listitem[AnalysisList])
		{
		showStatusIcon();
		emit statusChanged();
		
		switch(status)
			{
			case Finished:
				{
				listitem[AnalysisList]->setText(2, 0);
				listitem[AnalysisList]->setText(3, "100.000");
				listitem[AnalysisList]->setText(5, 0);
				break;
				}
			case Loading:
				{
    		for(int i=2;i<12;i++) listitem[AnalysisList]->setText(i, 0);
				break;
				}
			}
  	}

  if(status == Loading && listitem[AnalysisList])
  	{
  	QFileInfo fi(wtemp_f);
  	// approximate file size of WUs is 360000
  	int loaded = (int)( 100*((double)fi.size()/360000.0) );
  	if(loaded > 100) loaded = 100;
  	
  	char buf[16];
  	sprintf(buf, "%d %%", loaded);
  	listitem[AnalysisList]->setText(3, (const char*)buf);  	
  	}

  return(status);
	}

QString SetiLoc::convertTime(double time, bool hms)
	{
	int d, h, m, s;
	char t[64];
	
  d = static_cast< int >( time ) / 86400;
  h = static_cast< int >( time ) % 86400 / 3600;
  m = static_cast< int >( time ) % 3600 / 60;
  s = static_cast< int >( time ) % 60;
  // display with days or not
  if( hms || (hms == FALSE && d == 0))
  	{
  	h += d*24;
  	sprintf(t, "%d:%02d:%02d", h, m, s);
  	}
  else
  	sprintf(t, "%dd:%02d:%02d:%02d", d, h, m, s);
  return( QString(t) );
  }

QString SetiLoc::readEntry(const QString& fn, const QString& e)
	{
	QString l,s;
	QString v("");
	int eq(-1);
	bool found(false);
	QFile sah(fn);
	
	if(sah.open(IO_ReadOnly))
		{
		QTextStream t(&sah);
	 	while( !t.eof() )
 			{
 			s = t.readLine();
			// the next line is to prevent hangs when monitoring NFS directories
			// (suggested by Frode Tenneboe)
			if (s.isEmpty()) break;
	 		eq = s.find('=');
	 		if(eq != -1)
	 			{
				l = s.left(eq);
				if(l == e) {found = true; break;}
	 			}
	 		}
	 	if(found) v = s.right( s.length()-eq-1 );
	 	sah.close();
	 	}
	
	return( v );
	}
						
WUScore SetiLoc::readStateData(const QString& sf) const
{	
	WUScore sg;
	QString val;
		
 	if (QFile::exists(sf))
 		{
		// read the highest peak so far    				
		val = readEntry( sf, "bs_power" );
		sg.spike = val.toDouble();
		
		// read the binary value of the highest peak
		val = readEntry(sf, "bs_bin");
		sg.spike_bin = val.toInt();

		// read the corresponding chirp rate of the highest peak    				
		val = readEntry( sf, "bs_chirp_rate" );
		sg.spike_chirprate = val.toDouble();

		// read the fft index of the highest peak
		val = readEntry(sf, "bs_fft_ind");
		sg.spike_fft_index = val.toInt();
		
		// read the fft length of the highest peak
		val = readEntry(sf, "bs_fft_len");
		sg.spike_fft_len = val.toInt();
		
		// read the score of the highest peak
		val = readEntry(sf, "bs_score");
		sg.spike_score = val.toDouble();
		
		// read the strongest gaussian according to its score value
		val  = readEntry( sf, "bg_score" );
		sg.gaussian = val.toDouble();

		// read the binary value of the highest gaussian
		val = readEntry(sf, "bg_bin");
		sg.gaussian_bin = val.toInt();

		// read the gaussian power
		val  = readEntry( sf, "bg_power" );
		sg.gaussian_power = val.toDouble();

		// read the gaussian chi square
		val  = readEntry( sf, "bg_chisq" );
		sg.gaussian_chisq = val.toDouble();

		// read the fft length of the highest gaussian
		val = readEntry(sf, "bg_fft_len");
		sg.gaussian_fft_len = val.toInt();
						
		// read the corresponding chirp rate of the highest gaussian    				
		val = readEntry( sf, "bg_chirp_rate" );
		sg.gaussian_chirprate = val.toDouble();

		// read the sigma value of the top Gaussian
		val = readEntry( sf, "bg_sigma" );
		sg.gaussian_sigma = val.toDouble();

		// read the mean value of the top Gaussian
		val = readEntry( sf, "bg_true_mean" );
		sg.gaussian_true_mean = val.toDouble();

		// read the fft index of the top Gaussian
		val = readEntry( sf, "bg_fft_ind" );
		sg.gaussian_fft_index = val.toInt();
		
		// read the score of the top pulse (maybe later: and update
		// the rest if it is higher than the previous score)
		val = readEntry( sf, "bp_score" );
		//if( val.toDouble() > sg.pulse)
			{
			sg.pulse = val.toDouble();
			
			// read the power of the top pulse
			val = readEntry( sf, "bp_power" );
			sg.pulse_power = val.toDouble();
			
			// read the mean value of the top pulse
			val = readEntry( sf, "bp_mean" );
			sg.pulse_mean = val.toDouble();

			// read the period of the top pulse
			val = readEntry( sf, "bp_period" );
			sg.pulse_period = val.toDouble();
			
			// read the chirp rate of the top pulse
			val = readEntry( sf, "bp_chirp_rate" );
			sg.pulse_chirprate = val.toDouble();
			
			// TODO: freq_bin, time_bin, fft_len
			// just fill with zeros for the time being
			sg.pulse_freq_bin = 0;
			sg.pulse_time_bin = 0;
			sg.pulse_fft_len  = 0;
			}

		// read the score of the top triplet (maybe later: and update
		// the rest if it is higher than the previous score)
		val = readEntry( sf, "bt_score" );
		//if( val.toDouble() > sg.triplet)
			{
			sg.triplet = val.toDouble();
			
			// read the power of the top triplet
			val = readEntry( sf, "bt_power" );
			sg.triplet_power = val.toDouble();
			
			// read the mean value of the top pulse
			val = readEntry( sf, "bt_mean" );
			sg.triplet_mean = val.toDouble();

			// read the period of the top pulse
			val = readEntry( sf, "bt_period" );
			sg.triplet_period = val.toDouble();
			
			// read the chirp rate of the top pulse
			val = readEntry( sf, "bt_chirp_rate" );
			sg.triplet_chirprate = val.toDouble();
			
			// read the index values
			val = readEntry( sf, "bt_tpotind0_0" );
			sg.triplet_tpotind0_0 = val.toInt();
			val = readEntry( sf, "bt_tpotind0_1" );
			sg.triplet_tpotind0_1 = val.toInt();
			val = readEntry( sf, "bt_tpotind1_0" );
			sg.triplet_tpotind1_0 = val.toInt();
			val = readEntry( sf, "bt_tpotind1_1" );
			sg.triplet_tpotind1_1 = val.toInt();
			val = readEntry( sf, "bt_tpotind2_0" );
			sg.triplet_tpotind2_0 = val.toInt();
			val = readEntry( sf, "bt_tpotind2_1" );
			sg.triplet_tpotind2_1 = val.toInt();
			
			// TODO: bperiod, tpotindx_x, freq_bin, time_bin, scale, fft_len
			// just fill with zeros for the time being
			sg.triplet_bperiod    = 0.0;
			sg.triplet_freq_bin   = 0;
			sg.triplet_time_bin   = 0.0;
			sg.triplet_scale      = 0.0;
			sg.triplet_fft_len    = 0;			
		  }
		}
 	else
 		{
 		// file couldn't be opened or doesn't exist
 		// fill structure with zeros
 		Ksetiwatch::initWUScore(&sg);
 		}
	return( sg );
}


QString SetiLoc::readWUName(const QString& wuf) const
{	
	QString val("");
	
 	if (QFile::exists(wuf) )
 		{
		// read work unit name    				
		val = readEntry(wuf, "name" );
		}
	
	return(val);
}

void SetiLoc::setTimestamp(const QString& ts)
{
sresult_timestamp = ts;
}

QString SetiLoc::timestamp() const
{
return sresult_timestamp;
}

/**
 * Check the version of the running SETI@home client
 *
 */
int SetiLoc::checkVersion()
{
int version(0);
QString version_file(this->directory() + "/version.sah");

if(QFile::exists((const char*)version_file))
	{
	QString entry = readEntry((const char*)version_file, "major_version");
	version = entry.toInt();
	}
else if(QFile::exists((const char*)(this->directory() + "/version.txt")))
	version = 1;

return(version);
}

/** Displays the status icon in the Analysis list. */
void SetiLoc::showStatusIcon()
{
int cw(28);

if(listitem[AnalysisList])
	{
  // center the pixmap in the column (why can't Qt handle this?)
  cw = listitem[AnalysisList]->listView()->columnWidth(1);
  if(cw < 28) cw = 28;

  // For some reason unknown to me, I have to subtract 5 to prevent the status column
  // width from increasing steadily when switching between the combobox items.
  cw -= 5;

  QPixmap stpix(cw, 16);
  QPainter p(&stpix);
  p.fillRect(0, 0, cw, 16, white);
  p.drawPixmap((cw-status_icon[status].width())/2, 0, status_icon[status]);
  p.end();
  		
  stpix.setMask( stpix.createHeuristicMask() );
  listitem[AnalysisList]->setPixmap(1, stpix);
  }
}

/** Starts the SETI@home client. */
bool SetiLoc::startClient()
{
bool ret;

client.clearArguments();
client << (const char*)(sdirectory + "/setiathome");
if(scmdlineargs.isEmpty() == false)
	{
	int p1(0), p2;
	do
		{
		p2 = scmdlineargs.find(' ', p1);
		if(p2 >= 0)
			{
			client << (const char*)scmdlineargs.mid(p1, p2-p1);
			p1 = p2+1;
			}
		} while(p2 != -1);
	client << (const char*)scmdlineargs.right(scmdlineargs.length()-p1);
	}
// we have to change to the setiathome directory to properly start the client
QString curdir = QDir::currentDirPath();
QDir::setCurrent((const char*)sdirectory);
client.start(KProcess::DontCare, KProcess::NoCommunication);
QDir::setCurrent((const char*)curdir);
// wait a tenth of a sec
usleep(100000);
ret = isClientRunning(client.getPid());
if(ret == true)
	{
	client_pid = (pid_t)client.getPid();
	// the next three lines are required to set this location in the
	// Running state (and show it immediately); a bit tricky.
	refresh(AnalysisList);
	setState(Running);
	showStatusIcon();
	emit statusChanged();
	}

return(ret);
}

/** Stops the SETI@home client. */
int SetiLoc::stopClient()
{
int ret(-1);

if(client_pid > 0)
	{
	ret = ::kill(client_pid, SIGTERM);
	if(ret == 0)
		{
		client_pid = (pid_t)0;
		// the next three lines are required to set this location in the
		// Stopped state (and show it immediately); a bit tricky.
		refresh(AnalysisList);
		setState(Stopped);
		showStatusIcon();
		emit statusChanged();
		}
	}

return(ret);
}

bool SetiLoc::readGaussianData(const QString& sf)
{
QFile statefile(sf);
QString s, l;
int eq(-1);

bool status = false;
if(statefile.open(IO_ReadOnly))
	{
	QTextStream t(&statefile);
 	while( !t.eof() )
		{
		s = t.readLine();
		if (s.isEmpty()) break;
 		eq = s.find('=');
 		if(eq != -1)
 			{
			l = s.left(eq);
			if(strcmp((const char*)l, "bg_pot 0") == 0) {status = true; break;}
 			}
 		}
 	if(status)
 		sscanf((const char*)s, "bg_pot 0=%lf", &max.gaussian_data[0]);
 	else
 		{
 		statefile.close();
 		return false;
 		}

  char id[32];
  status = true;
  for(int i=1;i<=63;i++)
  	{
  	s = t.readLine();
  	if(s.isEmpty()) {status = false; break;}
  	sprintf(id, "bg_pot %d=%%lf", i);
  	sscanf((const char*)s, id, &max.gaussian_data[i]);
  	}
  statefile.close();
  }

return status;
}

bool SetiLoc::readPulseData(const QString& sf)
{
bool status = false;
QString buf;
char byte[2];        // stores a byte value

buf = readEntry(sf, "bp_pot");
if(!buf.isEmpty())
  {
  QTextStream ts(&buf, IO_ReadOnly);
  int val;
  int i = 0;
  while(!ts.atEnd() && i < 512)
    {
    ts >> byte[0] >> byte[1];
    sscanf(byte, "%x", &val);
    max.pulse_data[i] = (unsigned short int)val;
    i++;
    }
  status = true;
  }

return status;
}

bool SetiLoc::readTripletData(const QString& sf)
{
bool status = false;
QString buf;
char byte[2];        // stores a byte value

buf = readEntry(sf, "bt_pot");
if(!buf.isEmpty())
  {
  QTextStream ts(&buf, IO_ReadOnly);
  int val;
  int i = 0;
  while(!ts.atEnd() && i < 512)
    {
    ts >> byte[0] >> byte[1];
    sscanf(byte, "%x", &val);
    max.triplet_data[i] = (unsigned short int)val;
    i++;
    }
  status = true;
  }

return status;
}

/** Retrieves the pid. */
int SetiLoc::getClientPid()
{
int pid(-1);
QFile pidfile((const char*)(sdirectory + "/pid.sah"));

if(pidfile.open(IO_ReadOnly))
	{
	QTextStream t(&pidfile);
	t >> pid;
	pidfile.close();
	}
return(pid);
}

/** checks whether the S@h client is running */
bool SetiLoc::isClientRunning(pid_t pid)
{
int err(-1);

// read from pid.sah if pid is zero
if(pid == 0) pid = getClientPid();
// don't send a signal but only check for error;
// if zero, process is alive
if(pid != -1) err = ::kill(pid, 0);
if(err == 0)
	return(true);
else
	return(false);
}
