/***************************************************************************
                          kfilecoderdoc.cpp  -  description
                             -------------------                                         
    begin                : ven avr 23 18:58:17 CEST 1999
                                           
    copyright            : (C) 1999 by Franois Dupoux                         
    email                : fdupoux@free.fr                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/

#include <qmessagebox.h>
#include <qprogressdialog.h>
#include <qcstring.h>
#include <qfileinfo.h>
#include <qdir.h>

#include <kmessagebox.h>
#include <kpassdlg.h>
#include <kurl.h>
#include <kio/netaccess.h>
#include <kmimetype.h>
#include <kdirsize.h>

#include <sys/vfs.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <time.h>
#include <grp.h>
#include <pwd.h>

#include "PasswordTest.h"
#include "kfilecoder.h"
#include "kfilecoderdoc.h"
#include "filelib.h"
#include "algo_pc1.h"
#include "algo_idea.h"
#include "kalgodlg.h"
#include "misc.h"
#include "random.h"

// nPassLen, szName, szDescription, (*addFileFunction), (*extractFilesFunction)
KAlgoInfo g_algo[] =
{
	{"PC1", "hard 128 bits encryption", addFileAlgoPc1, extractFilesAlgoPc1},
	{"IDEA", "International Data Encryption Algorithm (128 bits)", addFileAlgoIdea, extractFilesAlgoIdea}
	//{"XOR", "eXclusive OR, very low security", addFileAlgoXor, extractFilesAlgoXor},
	//{0, 0, 0, 0}
};

char *g_szEndFileHeader = "END-OF-FILE-HEADER";
char *g_szBeginFileHeader = "BEGIN-OF-FILE-HEADER";

// ===========================================================================================================================
KFileCoderDoc::KFileCoderDoc(QObject *parent, const char *)
{
	m_mainWnd = (QWidget *) parent;
	m_bArchiveOpened = false;
	eraseMemberData(true);
}
	
// ===========================================================================================================================
void KFileCoderDoc::eraseMemberData(bool bErasePassword)
{
	m_dwAlgorithmUsed = 0; // default algorith
	m_dwFlags = 0;	
	m_dwNbFiles = 0;
	m_bIsWorking = false;
	
	memset(m_szArchivePath, 0, MAXPATHLEN);
	
	// password tester
	memset(m_cPasswordTestMixer, 0, ENCODING_LEN);
	memset(m_cHashTestPass, 0, ENCODING_LEN);
		
	
	// delete password in the memory, to increase the security
	if (bErasePassword)
	{
		// password of the user, with a 128 bits len
		memset(m_cPassword, 0, ENCODING_LEN);
		memset(m_cMixerCodingKey, 0, ENCODING_LEN);	

		// mixer number coding all others mixers
		memset(m_cMixerCodingMixer, 0, ENCODING_LEN);
	}
}

// ===========================================================================================================================
KFileCoderDoc::~KFileCoderDoc()
{
	eraseMemberData(true);
}

// ===========================================================================================================================
char *KFileCoderDoc::getArchivePath()
{	return m_szArchivePath;
}

// ===========================================================================================================================
DWORD KFileCoderDoc::getNbFiles()
{	return m_dwNbFiles;
}

// ===========================================================================================================================
DWORD KFileCoderDoc::getAlgoUsed()
{	return m_dwAlgorithmUsed;
}

// ===========================================================================================================================
void KFileCoderDoc::setListView (QListView *lv)
{	m_List = lv;	
}

// ===========================================================================================================================
int KFileCoderDoc::createArchive(const char *szFilename, const BYTE *cPassword, DWORD dwAlgorithmUsed) // Create a new archive and write header: put m_cPassword, m_szArchivePAth, m_fdArchive
{
	QString strMess;
	int nRes;
	
	// Archive is already opened
	if (m_bArchiveOpened == true)
	{	 KMessageBox::error( m_mainWnd, i18n("An archive is already opened. You must close it before."));
		return ERROR_FAILED;
	}

 	// The file already exists: overwrite ?
	if (access(szFilename, F_OK) == 0)
	{	strMess.sprintf(i18n("The archive <b>%s</b> already exists. Do you want to overwrite it ? All data will be erased."), szFilename);
		if (KMessageBox::questionYesNo(m_mainWnd, strMess) == KMessageBox::No) // No clicked
			return ERROR_FAILED;
	}
	
	m_List -> clear();

	// Creating the file
	m_fArchive = fopen(szFilename, "w+b");
	if (!m_fArchive)
	{	strMess.sprintf(i18n("Unable to open the file <b>%s</b> for Reading and Writing."), szFilename);	
		KMessageBox::error( m_mainWnd, strMess);
		return ERROR_FAILED;
	}

	// initialize the PC2 algorithm
	pc2Init();
	
	// Put flags and datas
	m_bArchiveOpened = true;
	strcpy(m_szArchivePath, szFilename);
	memcpy(m_cPassword, cPassword, ENCODING_LEN);

	// Set default flags for this version of KFileCoder:
	m_dwFlags = 0;
	
	// calculate the mixer number, which will be used to mix the password and to check password is good
	generateRandomNumber(m_cPasswordTestMixer);
	
	// calculate the password test hash table
	getPasswordTestHashTable(m_cHashTestPass, cPassword, m_cPasswordTestMixer);
	
	// calculate the mixer number m_cMixerCodingMixer, which will be used to encode all (random numbers
	// used to make all files encoded with different keys)
	// m_cMixerCodingKey is the key used to encode all mixer
	generateRandomNumber(m_cMixerCodingMixer);
	calculatePC1Hash(m_cMixerCodingKey, m_cMixerCodingMixer, ENCODING_LEN, m_cPassword);
	
	// Write the header
	m_dwAlgorithmUsed = dwAlgorithmUsed;
	nRes = writeArchiveHeader();
	if (nRes == -1)
	{	closeArchive();
		return ERROR_FAILED; // error
	}

	// Set caption of the window
	m_mainWnd -> setCaption(m_szArchivePath);

	return ERROR_SUCCESS; // Success	
}


// ===========================================================================================================================
int KFileCoderDoc::openArchive(const char *szFilename, const BYTE *cPassword) // Open the archive and read header: put m_cPassword, m_szArchivePAth, m_fdArchive
{	
	QString strMess;
	BYTE cHashTestTable[ENCODING_LEN+1];
	int nRes;

	// Archive is already opened
	if (m_bArchiveOpened == true)
	{	KMessageBox::error(m_mainWnd, i18n("An archive is already opened. You must close it before."));
		return ERROR_FAILED;
	}

	// Opening the file
	m_fArchive = fopen(szFilename, "r+b"); // read + write
	if (!m_fArchive)
	{	strMess.sprintf(i18n("Unable to open the file <b>%s</b> for Reading and Writing."), szFilename);	
		KMessageBox::error( m_mainWnd, strMess);
		return ERROR_FAILED;
	}
	
	// initialize the PC2 algorithm
	pc2Init();
		
	// Put flags and datas
	m_bArchiveOpened = true;
	strcpy(m_szArchivePath, szFilename);

	// copy password: must be done before readArchiveHeader()
	if (cPassword) // if the archive is really opened (not after a delete files)
	{	memcpy(m_cPassword, cPassword, ENCODING_LEN);
	}
	
	// Read the header
	nRes = readArchiveHeader();
	if (nRes == -1)
	{	closeArchive();
		m_bArchiveOpened = false;
		return ERROR_FAILED; // error
	}
	
	// check the password: create the table from the entered password
	getPasswordTestHashTable(cHashTestTable, m_cPassword, m_cPasswordTestMixer);
		
	// compare the archive written hash with the new one (just calculated)
	if (memcmp(cHashTestTable, m_cHashTestPass, ENCODING_LEN) != 0) // if differents
	{	KMessageBox::error(m_mainWnd, i18n("Incorrect password."));
		closeArchive();
		return ERROR_FAILED;
	}		
	
	// Redraw the List View
	listView_Fill();

	// Set caption of the window
	m_mainWnd -> setCaption(m_szArchivePath);

	return ERROR_SUCCESS; // Success	
}

// ===========================================================================================================================
int KFileCoderDoc::closeArchive(bool bClearList /*=true*/, bool bErasePassword/*=true*/) // Close the current Archive
{
	// No rchive opened
	if (m_bArchiveOpened == false)
	{ return ERROR_FAILED;
	}

	fclose(m_fArchive);
	m_bArchiveOpened = false;
	m_fArchive = 0;
	
	eraseMemberData(false);

	if (bClearList)
		m_List -> clear();

	// Update menu & toolbar commands
	((KFileCoderApp *) m_mainWnd) -> updateCommandsAccess();

	// Set caption of the window
	m_mainWnd -> setCaption("");

	return ERROR_SUCCESS;
}

// ===========================================================================================================================
int KFileCoderDoc::writeArchiveHeader() // When creating new archive
{
	KArchiveHeader head;
	int nRes;

	// Archive is not opened
	if (m_bArchiveOpened != true)
	{	KMessageBox::error(m_mainWnd, i18n("An archive must be opened to write header."));
		return ERROR_FAILED;
	}

	// Go to the begin of the file
	fseek(m_fArchive, 0, SEEK_SET);

	// initialize header
	memset(&head, 0, sizeof(KArchiveHeader));
	
	// Prepare datas of header which will be written to the archive
	sprintf (head.szPgm, "KFileCod01");
	head.dwAlgorithmUsed = m_dwAlgorithmUsed;
	head.dwFlags = m_dwFlags;

	// copy password test data
	memcpy(head.cHashTestPass, m_cHashTestPass, ENCODING_LEN);
	memcpy(head.cMixerCodingMixer, m_cMixerCodingMixer, ENCODING_LEN);
	
	// encode the password test mixer in the header
	pc1Encode(head.cEncodedPasswordTestMixer, m_cPasswordTestMixer, ENCODING_LEN, m_cMixerCodingKey);
	
	// Write the header
	nRes = fwrite(&head, sizeof(KArchiveHeader), 1, m_fArchive);
	fflush(m_fArchive);
	if (nRes != 1)
	{	KMessageBox::error(m_mainWnd, i18n("Can't write header of Archive."));
		return ERROR_FAILED;
	}

	return ERROR_SUCCESS;
}

// ===========================================================================================================================
int KFileCoderDoc::readArchiveHeader()// When opening an archive		
{	
	KArchiveHeader head;
	int nRes;
	QString strMess;

	// Archive is net opened
	if (m_bArchiveOpened != true)
	{	KMessageBox::error( m_mainWnd, i18n("An archive must be opened to read header."));
		return ERROR_FAILED;
	}

	// Go to the begin of the file
	fseek(m_fArchive, 0, SEEK_SET);

	errno = 0;
	nRes = fread(&head, sizeof(KArchiveHeader), 1, m_fArchive);
	if (nRes != 1)
	{	strMess.sprintf(i18n("Can't read header of Archive: Error %s"), strerror(errno));
		KMessageBox::error( m_mainWnd, strMess);
		return ERROR_FAILED;
	}

	// Compare header with "KFileCoder document"
	head.szPgm[11] = 0;
	
	if (strcmp(head.szPgm, "KFileCoder") == 0) // This is an old archive format (before 0.4)
	{	KMessageBox::error(m_mainWnd, i18n("This archive was created with version 0.1 or 0.2 or 0.3 of KFileCoder. Use an older version to read this archive."));
		return ERROR_FAILED;
	}	

	if (strcmp(head.szPgm, "KFileCod01") != 0) // The opened archive is not a KFileCoder archive
	{	KMessageBox::error(m_mainWnd, i18n("This file is not a KFileCoder archive."));
		return ERROR_FAILED;
	}	

	// Get flags
	m_dwFlags = head.dwFlags;

	// Read the number of the algorithm used to encode files
	m_dwAlgorithmUsed = head.dwAlgorithmUsed;
	if (head.dwAlgorithmUsed > ALGO_SUPPORTED_NB)
	{	KMessageBox::error (m_mainWnd, i18n("The algorith which was used to encode the archive is not supported in this version of KFileCoder. Please, download a new one on: http://kfilecoder.sourceforge.net/."));
		return ERROR_FAILED;
	}

	// copy password test hash table
	memcpy(m_cHashTestPass, head.cHashTestPass, ENCODING_LEN);
	memcpy(m_cMixerCodingMixer, head.cMixerCodingMixer, ENCODING_LEN);
	
	// m_cMixerCodingKey is the key used to encode all mixer
	calculatePC1Hash(m_cMixerCodingKey, head.cMixerCodingMixer, ENCODING_LEN, m_cPassword);
		
	// encode the password test mixer in the header
	pc1Decode(m_cPasswordTestMixer, head.cEncodedPasswordTestMixer, ENCODING_LEN, m_cMixerCodingKey);
	
	return ERROR_SUCCESS;
}

// ===========================================================================================================================
int KFileCoderDoc::addDirectory(QString strDir, DWORD dwCompress, QWORD qwDirSize, QProgressDialog *dlgProgress)
{
	QDir dir;
	QString strMess;
	QString strFilename;
	DWORD i;
	int nFiles = 0;
	QFileInfo fi;
	bool bProgressCreated;
	int nRes;
		
	//printf("==============> Adding dir %s\n", strDir.data());
	
	// remove the final '/' if exists
	if (strDir.right(1) == "/")
		strDir.truncate(strDir.length()-1);
	
	dir.setPath(strDir);
	dir.setFilter(QDir::Files | QDir::Readable | QDir::Hidden | QDir::NoSymLinks);

	bProgressCreated = (dlgProgress == NULL);
	
	// Draw the progress dialog
	if (dlgProgress == NULL)
	{	dlgProgress = new QProgressDialog (i18n("Adding files to archive.."),0, (DWORD) qwDirSize, m_mainWnd,"",true);
		dlgProgress->setCaption("KFileCoder");
		dlgProgress->show();
		dlgProgress->setProgress(0L);
	}
	
	// 0. -*-*-*-*-*-*-*-*-*- Check it's a valid directory -*-*-*-*-*-*-*-*-*-
	if (!dir.isReadable() || !dir.exists())
	{	
		strMess.sprintf (i18n("Can't access to directory <b>%s</b>"), strDir.data());
		nRes = KMessageBox::warningContinueCancel(m_mainWnd, strMess, QString::null, i18n("Continue"));		
		
		if (bProgressCreated)
			delete dlgProgress;
		
		if (nRes == KMessageBox::Continue)
			return ERROR_SUCCESS;
		else
			return ERROR_STOP;
	}
	
	// 1. -*-*-*-*-*-*-*-*-*- First, list all files -*-*-*-*-*-*-*-*-*-

	for (i=0; i < dir.count(); i++)
	{
		strFilename = formatFullPath(strDir.data(), dir[i].data());
		fi.setFile(strFilename);
		nRes = addFile(strFilename, dwCompress);
		if (nRes == ERROR_STOP)
		{	if (bProgressCreated)
				delete dlgProgress;

			return ERROR_STOP;	
		}
		nFiles++;
		dlgProgress->setProgress(dlgProgress->progress()+fi.size()); // Progress Dialog
	}

	// 2. -*-*-*-*-*-*-*-*-*- Second, list all dir -*-*-*-*-*-*-*-*-*-

	dir.setFilter(QDir::Dirs | QDir::Readable | QDir::Executable | QDir::Hidden | QDir::NoSymLinks);
	dir.setPath(strDir);

	for (i=0; i < dir.count(); i++)
	{
		if (strcmp(dir[i], ".") != 0 && strcmp(dir[i], "..") != 0)
		{
			strFilename = formatFullPath(strDir.data(), dir[i].data());
			nRes = addDirectory(strFilename, dwCompress, 0L, dlgProgress);
			if (nRes == ERROR_STOP)
			{	if (bProgressCreated)
					delete dlgProgress;

				return ERROR_STOP;	
			}
		}
	}
	
	if (bProgressCreated)
		delete dlgProgress;
		
	return nFiles;
}

// ===========================================================================================================================
int KFileCoderDoc::addFile(KURL urlFile, DWORD dwCompress)
{	
	qDebug ("============> Enter in addFile(%s)\n", urlFile.path().data());
	
	int nRes;
	bool bRes;	
	QString strMess;
	QString strFilename;
	FILE *fFileToAdd;
	char szTempPath[MAXPATHLEN];
	char szFilename[MAXPATHLEN];
	char szFileName[MAXPATHLEN];
	char szFilePath[MAXPATHLEN];
	BYTE cHashControl[ENCODING_LEN+1];
	QWORD qwOriginalFileSize;
	QWORD qwCompressedSize;
	QWORD qwEncodedFileSize;
	QWORD qwDiskFreeSpace;
	BYTE cFileKey[ENCODING_LEN+1];
	BYTE cFileMixer[ENCODING_LEN+1];
	BYTE cEncodedFileMixer[ENCODING_LEN+1];
	QFileInfo fi;
	
	// Download file if need (if url is "http://...")
	if (!(KIO::NetAccess::download(urlFile, strFilename))) // Open the Archive
		return ERROR_FAILED;

	// Check it's not a directory
	fi.setFile(strFilename);
	if (fi.isDir())
	{	QWORD qwDirSize;
		KURL urlDir(strFilename);
		qwDirSize = KDirSize::dirSize(urlDir);
		return addDirectory(strFilename, dwCompress, qwDirSize, NULL);
	}

	// Check archive an is opened
	if (m_bArchiveOpened != true)
	{	KMessageBox::error (m_mainWnd, i18n("An archive must be opened to add a file."));
		return ERROR_FAILED;
	}

	// Check the file to add is not the current archive (add a file into itself)
	if (urlFile.path() == QString(m_szArchivePath))
	{	KMessageBox::error (m_mainWnd, i18n("Cannot add <b>%s</b>. You can't add the archive file into itself !"), m_szArchivePath);
		return ERROR_FAILED;
	}
	
	// Calculate FileName and FilePath from szFilename
	sprintf(szFilename, "%s", strFilename.data());
	getFileName(szFileName, szFilename);
	getFilePath(szFilePath, szFilename);

	// Check the file is not already in the archive
	bRes = doesFileExistsInArchive(szFileName, szFilePath);
	if (bRes == true)
	{	strMess.sprintf(i18n("The file <b>%s</b> is already present in the archive. KFileCoder can't add two files with the same name."), formatFullPath(szFilePath, szFileName).data());
		nRes = KMessageBox::warningContinueCancel(m_mainWnd, strMess, QString::null, i18n("Continue"));
		if (nRes == KMessageBox::Continue)
			return ERROR_SUCCESS;
		else
			return ERROR_STOP;
	}

	// Test access rights
	if (access(szFilename, F_OK | R_OK) == -1)
	{	strMess.sprintf(i18n("Access denied to file <b>%s</b>"), szFilename);
		nRes = KMessageBox::warningContinueCancel(m_mainWnd, strMess, QString::null, i18n("Continue"));
		if (nRes == KMessageBox::Continue)
			return ERROR_SUCCESS;
		else
			return ERROR_STOP;
	}
	
	strMess.sprintf(i18n("Adding %s..."), szFilename);
	((KFileCoderApp *)m_mainWnd)->slotStatusMsg(strMess);

	// update user interface
  g_app -> processEvents();
		
	// check file is not empty
	fi.setFile(szFilename);
	qwOriginalFileSize = fi.size();

	// check gzip if present on the system	
	if ((dwCompress == COMPRESS_BZIP2) && (checkCompress(szFilename, "bzip2") != 0))
	{	strMess.sprintf(i18n("bzip2 can't be run. Then, gzip will be used instead of bzip2 to compress the file <b>%s</b>"), szFilename);
		KMessageBox::information (m_mainWnd, strMess, QString::null, i18n("Don't show me this message again"));
		dwCompress = COMPRESS_GZIP;
	}
	
	// check bzip2 if present on the system	
	if ((dwCompress == COMPRESS_GZIP) && (checkCompress(szFilename, "gzip") != 0))
	{	strMess.sprintf(i18n("gzip can't be run. Then, the compression won't be used to encode the file <b>%s</b>"), szFilename);
		KMessageBox::information (m_mainWnd, strMess, QString::null, i18n("Don't show me this message again"));
		dwCompress = COMPRESS_NONE;
	}

	// --------- Open file to add ------------------------
	
	if (dwCompress == COMPRESS_NONE)
	{
		fFileToAdd = fopen(szFilename, "rb");
		errno = 0;
		if (!fFileToAdd)
		{	strMess.sprintf(i18n("Can't open file <b>%s</b> for reading.<br>: Error %s"), szFilename, strerror(errno));
			KMessageBox::error (m_mainWnd, strMess);
			return ERROR_FAILED;
		}
	}
	else if (dwCompress == COMPRESS_GZIP)
	{
		sprintf(szTempPath, "gzip -c \"%s\"", szFilename);
		fFileToAdd = popen(szTempPath, "r");
		errno = 0;
		if (!fFileToAdd)
		{	strMess.sprintf(i18n("Can't open file <b>%s</b> for reading.<br>: Error %s"), szFilename, strerror(errno));
			KMessageBox::error (m_mainWnd, strMess);
			return ERROR_FAILED;
		}
	}
	else if (dwCompress == COMPRESS_BZIP2)
	{
		sprintf(szTempPath, "bzip2 -c \"%s\"", szFilename);
		fFileToAdd = popen(szTempPath, "r");
		errno = 0;
		if (!fFileToAdd)
		{	strMess.sprintf(i18n("Can't open file <b>%s</b> for reading.<br>: Error %s"), szFilename, strerror(errno));
			KMessageBox::error (m_mainWnd, strMess);
			return ERROR_FAILED;
		}
	}
	else // invalid compression mode
	{	return ERROR_FAILED;
	}


	// Check there is enought free disk space
	nRes = getDiskFreeSpaceForFile(&qwDiskFreeSpace, m_szArchivePath);
	if (nRes != -1 && qwDiskFreeSpace  < qwOriginalFileSize)
	{	strMess.sprintf(i18n("There is not enought disk free space, this operation need.\nAvaible space: %s\nNeed space: %s"), formatSize(qwDiskFreeSpace).data(), formatSize(qwOriginalFileSize).data());
		KMessageBox::error (m_mainWnd, strMess);
		return ERROR_FAILED;
	}
		
	// create a 128 bits key
	generateRandomNumber(cFileMixer);
	
	// encode the cFileMixer which will be written in the archive
	pc1Encode(cEncodedFileMixer, cFileMixer, ENCODING_LEN, m_cMixerCodingKey);
	
	// Hash random 128 bits number with the main 128 bits password
	calculatePC1Hash(cFileKey, cFileMixer, ENCODING_LEN, m_cPassword);

	// go to the end of the archive before adding files
	nRes = fseek(m_fArchive, 0L, SEEK_END);
	if (nRes != 0)
	{	KMessageBox::error (m_mainWnd, i18n("Can't go to the end of the archive"));
		return ERROR_FAILED;
	}
	
	// skip bytes to write the header
	DWORD dwFileHeaderSize = strlen(g_szBeginFileHeader) + sizeof(DWORD) + ENCODING_LEN + sizeof(UINT) + strlen(szFileName) +sizeof(UINT) + strlen(szFilePath) + sizeof(QWORD) + sizeof(QWORD) + sizeof(QWORD)
	+ sizeof(QWORD) + ENCODING_LEN + sizeof(DWORD) + sizeof(QWORD) + sizeof(QWORD) + sizeof(QWORD) + ENCODING_LEN + FILEHEADER_RESERVED_SIZE + strlen(g_szEndFileHeader);
	
	
	long lFileHeaderPosition = ftell(m_fArchive);
	
	// skip the header. Do not use fseek, because end of file
	BYTE cZero = 0;
	nRes = fwrite(&cZero, 1, dwFileHeaderSize, m_fArchive);
	if ((DWORD)nRes != dwFileHeaderSize)
	{	KMessageBox::error (m_mainWnd, i18n("Can't write pre header to the archive"));
		return ERROR_FAILED;
	}
	
	// =+=+=+=+=+=+=+=+=+=+ Add encoding algorithm function =+=+=+=+=+=+=+=+=+=+

	nRes = 0;
	memset(cHashControl, 0, ENCODING_LEN);
	nRes = g_algo[m_dwAlgorithmUsed].addFileFunction(m_mainWnd, szFilename, cFileKey, fFileToAdd, m_fArchive, cHashControl, &qwCompressedSize, &qwEncodedFileSize);
	
	// =+=+=+=+=+=+=+=+=+=+ Add encoding algorithm function =+=+=+=+=+=+=+=+=+=+

	// Close file added to archive
	if (dwCompress == COMPRESS_NONE)
		fclose(fFileToAdd);
	else
		pclose(fFileToAdd);
			
	if (nRes == -1)	
		return ERROR_FAILED;

	// --------- Write header of file to add ------------------------	
	
	// go to the begin of the file header
	fseek(m_fArchive, lFileHeaderPosition, SEEK_SET);
	
	nRes = writeFileHeader(m_fArchive, szFilename, szFileName, szFilePath, cEncodedFileMixer, qwCompressedSize, qwEncodedFileSize, dwCompress, cHashControl);
	if (nRes == -1)	
	{	strMess.sprintf(i18n("Can't write file <b>%s</b> to archive: Error %s"), szFilename, strerror(errno));
		KMessageBox::error (m_mainWnd, strMess);
		fclose(fFileToAdd);
		return ERROR_FAILED;
	}
	
	// go after file encoded data
	nRes = fseek(m_fArchive, 0L, SEEK_END);
	if (nRes != 0)
	{	KMessageBox::error (m_mainWnd, i18n("Can't go to the end of the archive, after encoded data"));
		return ERROR_FAILED;
	}

	// Add file to the ListView
	struct stat fStat;
	memset(&fStat, 0, sizeof(fStat));
	stat(szFilename, &fStat);

	listView_AddItem(szFileName, szFilePath, qwOriginalFileSize, qwEncodedFileSize, fStat.st_mode, fStat.st_uid, fStat.st_gid, dwCompress);

	m_dwNbFiles++;

	qDebug ("==========> Exit of addFile(%s)\n", urlFile.path().data());
	return ERROR_SUCCESS;
}

// ===========================================================================================================================
int KFileCoderDoc::extractFiles(const char *szDestPath, bool bOnlySelFiles)
{	char szFileName[MAXPATHLEN];
	char szFilePath[MAXPATHLEN];
	char szFullpath[MAXPATHLEN];
	char szPath[MAXPATHLEN];
	char szTempPath[MAXPATHLEN];
	QString strMess;
	QFileInfo fi;
	QFileInfo fiArchive;
	int nRes;
	bool bOverwriteYesToAll = false;
	bool bOverwriteNoToAll = false;
	QWORD qwDiskFreeSpace;
	FILE *fFileToDecode;
	bool bSkipFile;
	BYTE cFileMixer[ENCODING_LEN+1];
	BYTE cEncodedFileMixer[ENCODING_LEN+1];
	BYTE cFileKey[ENCODING_LEN+1];
	BYTE cHashControl[ENCODING_LEN+1];
	BYTE cHashOriginalControl[ENCODING_LEN+1];
	QWORD qwArchiveSize;
	DWORD dwCompress;

	QWORD qwOriginalFileSize;
	QWORD qwEncodedFileSize;
	QWORD qwCompressedFileSize;
	mode_t nPermissions;
	uid_t st_uid;
	gid_t st_gid;
	time_t st_mtime;
	
	// Check archive an is opened
	if (m_bArchiveOpened != true)
	{	KMessageBox::error(m_mainWnd, i18n("An archive must be opened to extract all its files."));
		return ERROR_FAILED;
	}
	
	// Go to the begin of the archive, after Header
  fseek(m_fArchive, sizeof(KArchiveHeader), SEEK_SET);
	fiArchive.setFile(m_szArchivePath);
	qwArchiveSize = (QWORD) fiArchive.size();
 	
	// Draw the progress dialog
	fi.setFile(m_szArchivePath);
	QProgressDialog dlgProgress(i18n("Extracting files of archive.."),0, fi.size(), m_mainWnd,"",true);
  dlgProgress.setCaption("KFileCoder");
	dlgProgress.show();
	dlgProgress.setProgress(ftell(m_fArchive));
	
	while (((QWORD)ftell(m_fArchive)) < qwArchiveSize) // while not the end of the archive
	{	
   	// ----- Read "Header" of the file -----
		errno = 0;
		
		nRes = readFileHeader(m_fArchive, szFileName, szFilePath, &qwOriginalFileSize, &qwCompressedFileSize, &qwEncodedFileSize,
		&nPermissions, &st_uid, &st_gid, &st_mtime, cEncodedFileMixer, &dwCompress, cHashOriginalControl);
		
		if (nRes == -1)
  	{	strMess.sprintf(i18n("Can't read files from archive: Error %s"), strerror(errno));
			KMessageBox::error (m_mainWnd, strMess);
			return ERROR_FAILED;
		}		
			
		// Calculate szFullpath (path and name)
		if (szFilePath[0] == '/')
			sprintf (szFullpath, "%s%s/%s", szDestPath, szFilePath, szFileName);
		else
			sprintf (szFullpath, "%s/%s/%s", szDestPath, szFilePath, szFileName);
		
		// Calculate szPath (path only) to create directory
		sprintf (szPath, "%s", formatFullPath(szDestPath, szFilePath).data());

  	// Check there is enought free disk space
		nRes = getDiskFreeSpaceForFile(&qwDiskFreeSpace, szFullpath);
		if (nRes != -1 && qwDiskFreeSpace  < qwOriginalFileSize)
		{	KMessageBox::error(m_mainWnd, i18n("There is not enought disk free space, this operation need."));
   		return ERROR_FAILED;
		}

		// update user interface
  	g_app -> processEvents();
			
		// -------------- The file already exists: overwrite ? --------------------
		bSkipFile = false;
		if (access(szFullpath, F_OK) == 0)
		{	nRes = 0;	

      // Ask question "Owerwrite ?" only if YesToAll or NoToAll have not been selected before
			if (bOverwriteYesToAll == false && bOverwriteNoToAll == false)
			{	strMess.sprintf(i18n("The file <b>%s</b> already exists. Do you want to overwrite it ? All data will be erased."), szFullpath);
				
				nRes = KMessageBox::questionYesNo(m_mainWnd, strMess); // Do not support "Yes/No for all files"
			
				/* KDE 1.x version
				KMsgBox msg(m_mainWnd, "KFileCoder", strMess, KMessageBox::QUESTION, i18n("Yes"), i18n("No"), i18n("Yes to all"), i18n("No to all"));
	    			nRes = dlg.exec();
				if (nRes == 3) // Yes to all
					bOverwriteYesToAll = true;
				if (nRes == 4) // No to all
					bOverwriteNoToAll = true;*/
			}

			if (bOverwriteNoToAll == true || nRes == KMessageBox::No) // We must not owerwrite ==> Skip file in archive
				bSkipFile = true;
		}

		// =+=+=+=+=+=+=+=+=+=+ Extracting and decoding with algorithm function =+=+=+=+=+=+=+=+=+=+
		if ((bSkipFile == false) && (bOnlySelFiles == false || isFileSelected(szFileName, szFilePath) == true)) // If we must extract file
		{
			strMess.sprintf(i18n("Extracting %s..."), szFullpath);
			((KFileCoderApp *)m_mainWnd)->slotStatusMsg(strMess);
	
			// update user interface
  		g_app -> processEvents();

			// Make all dirs of path: create directory if they doesn'y exists
			createAllDirsOfPath(szPath);	

			// Creating the file
			if (dwCompress == COMPRESS_NONE)
			{
  			fFileToDecode = fopen(szFullpath, "w+b");
				if (!fFileToDecode)
				{	strMess.sprintf(i18n("Unable to open the file <b>%s</b> for Writing."), szFullpath);	
					KMessageBox::error (m_mainWnd, strMess);
					return ERROR_FAILED;
				}	
			}
			else if (dwCompress == COMPRESS_GZIP)
			{
				sprintf(szTempPath, "gzip -d > \"%s\"", szFullpath);
  			fFileToDecode = popen(szTempPath, "w");
				if (!fFileToDecode)
				{	strMess.sprintf(i18n("Unable to open the file <b>%s</b> for Writing."), szFullpath);	
					KMessageBox::error (m_mainWnd, strMess);
					return ERROR_FAILED;
				}	
			}
			else if (dwCompress == COMPRESS_BZIP2)
			{
				sprintf(szTempPath, "bzip2 -d > \"%s\"", szFullpath);
  			fFileToDecode = popen(szTempPath, "w");
				if (!fFileToDecode)
				{	strMess.sprintf(i18n("Unable to open the file <b>%s</b> for Writing."), szFullpath);	
					KMessageBox::error (m_mainWnd, strMess);
					return ERROR_FAILED;
				}	
			}
			else
			{	return ERROR_FAILED;
			}

			// decode the cFileMixer which was read from archive
			pc1Decode(cFileMixer, cEncodedFileMixer, ENCODING_LEN, m_cMixerCodingKey);

			// Hash random 128 bits number with the main 128 bits password
			calculatePC1Hash(cFileKey, cFileMixer, ENCODING_LEN, m_cPassword);

			// Extract All Files: calling the function which Extract all Files witch the current Algorithm
			nRes = 0;
			memset(cHashControl, 0, ENCODING_LEN);
			nRes = g_algo[m_dwAlgorithmUsed].extractFilesFunction(m_mainWnd, szFullpath, cFileKey, fFileToDecode, m_fArchive, cHashControl, qwCompressedFileSize, qwEncodedFileSize);

			// close file to decode
			if (dwCompress == COMPRESS_NONE)
				fclose(fFileToDecode);
			else
				pclose(fFileToDecode);
			
			if (nRes == -1)
				return ERROR_FAILED;			

			// set access rights and owners
			chmod(szFullpath, nPermissions);
			chown(szFullpath, st_uid, st_gid);	 // only root can change the owner		
			
			// =+=+=+=+=+=+=+=+=+=+ Extracting and decoding with algorithm function =+=+=+=+=+=+=+=+=+=+

			// Compare real hash control and good hash control
  		if (memcmp(cHashOriginalControl, cHashControl, ENCODING_LEN) != 0) // "hash control" are differents
  		{	strMess.sprintf(i18n("The file <b>%s</b> was extracted, but its 128 bits hash control table is not the same as original. There was a error in datas, or password is bad."), szFullpath);
				nRes = KMessageBox::warningContinueCancel(m_mainWnd, strMess, QString::null, i18n("Continue"));
				if (nRes == KMessageBox::Cancel)
					return ERROR_SUCCESS;
			}

		}
		else // If we mustn't extract file ==> Skip it
		{	fseek(m_fArchive, qwEncodedFileSize, SEEK_CUR); // skip file data
		}

		// Progress Dialog
		dlgProgress.setProgress(ftell(m_fArchive));
	}

	return ERROR_SUCCESS;
}

// ===========================================================================================================================
long int KFileCoderDoc::testFiles(QString *strErrorFiles)
{	char szFileName[MAXPATHLEN];
	char szFilePath[MAXPATHLEN];
	char szFullpath[MAXPATHLEN];
	QString strMess;
	QFileInfo fi;
	QFileInfo fiArchive;
	int nRes;
	BYTE cFileMixer[ENCODING_LEN+1];
	BYTE cEncodedFileMixer[ENCODING_LEN+1];
	BYTE cFileKey[ENCODING_LEN+1];
	BYTE cHashControl[ENCODING_LEN+1];
	BYTE cHashOriginalControl[ENCODING_LEN+1];
	QWORD qwArchiveSize;
	long int nErrorFiles;
	DWORD dwCompress;

	QWORD qwEncodedFileSize;
	QWORD qwCompressedFileSize;
	QWORD qwOriginalFileSize;
	mode_t nPermissions;
	uid_t st_uid;
	gid_t st_gid;
	time_t st_mtime;
	
	nErrorFiles = 0L;
	
	// Check archive an is opened
	if (m_bArchiveOpened != true)
	{	KMessageBox::error(m_mainWnd, i18n("An archive must be opened to check all its files."));
		return ERROR_FAILED;
	}
	
	// Go to the begin of the archive, after Header
  fseek(m_fArchive, sizeof(KArchiveHeader), SEEK_SET);
	fiArchive.setFile(m_szArchivePath);
	qwArchiveSize = (QWORD) fiArchive.size();
 	
	// Draw the progress dialog
	fi.setFile(m_szArchivePath);
	QProgressDialog dlgProgress(i18n("Checking files of archive.."),0, fi.size(), m_mainWnd,"",true);
  dlgProgress.setCaption("KFileCoder");
	dlgProgress.show();
	dlgProgress.setProgress(ftell(m_fArchive));
	
	while (((QWORD)ftell(m_fArchive)) < qwArchiveSize) // while not the end of the archive
	{	
   	// ----- Read "Header" of the file -----
		errno = 0;
		
		nRes = readFileHeader(m_fArchive, szFileName, szFilePath, &qwOriginalFileSize, &qwCompressedFileSize, &qwEncodedFileSize,
		&nPermissions, &st_uid, &st_gid, &st_mtime, cEncodedFileMixer, &dwCompress, cHashOriginalControl);
		if (nRes == -1)
  	{	strMess.sprintf(i18n("Can't read file <b>%s</b> from archive: Error %s"), szFullpath, strerror(errno));
			KMessageBox::error (m_mainWnd, strMess);
			return ERROR_FAILED;
		}
		   	
		strMess.sprintf(i18n("Checking %s..."), szFullpath);
		((KFileCoderApp *)m_mainWnd)->slotStatusMsg(strMess);
	
		// update user interface
  	g_app -> processEvents();
			
		// =+=+=+=+=+=+=+=+=+=+ check file =+=+=+=+=+=+=+=+=+=+
		{
			// decode the cFileMixer which was read from archive
			pc1Decode(cFileMixer, cEncodedFileMixer, ENCODING_LEN, m_cMixerCodingKey);

			// Hash random 128 bits number with the main 128 bits password
			calculatePC1Hash(cFileKey, cFileMixer, ENCODING_LEN, m_cPassword);

			// Extract All Files: calling the function which Extract all Files witch the current Algorithm
			nRes = 0;
			memset(cHashControl, 0, ENCODING_LEN);
			nRes = g_algo[m_dwAlgorithmUsed].extractFilesFunction(m_mainWnd, NULL, cFileKey, NULL, m_fArchive, cHashControl, qwCompressedFileSize, qwEncodedFileSize);
						
			if (nRes == -1)
				return ERROR_FAILED;			

			// =+=+=+=+=+=+=+=+=+=+ Extracting and decoding with algorithm function =+=+=+=+=+=+=+=+=+=+

			// Compare real hash control and good hash control
  		if (memcmp(cHashOriginalControl, cHashControl, ENCODING_LEN) != 0) // "hash control" are differents
  		{	nErrorFiles++;
  			strMess.sprintf("<li><b>Name:</b>%s; <b>Path:</b>%s; <b>Size:</b> %s</li>", szFileName, szFilePath, formatSize(qwOriginalFileSize).data());
  			(*strErrorFiles) += strMess;
  		}
		}

		// Progress Dialog
		dlgProgress.setProgress(ftell(m_fArchive));
	}

	return nErrorFiles;
}

// ===========================================================================================================================
int KFileCoderDoc::deleteSelectedFilesInArchive()			
{
	// Copy the archive to a new one: copy only files which are not selected, using function isFileSelected();
	// Then move the current archive, and recreate a new one, from the old which is moved, by copying selected files

	//printf("========================> Enter in deleteSelectedFilesInArchive()\n");
	
	char szOldArchive[MAXPATHLEN];
	char szNewArchive[MAXPATHLEN];
	char szFileName[MAXPATHLEN];
	char szFilePath[MAXPATHLEN];
	KArchiveHeader head;
	QWORD qwFileHeadSize;
	QWORD qwTotalBytesToCopy;
	QWORD qwArchiveSize;
	QWORD qwEncodedFileSize;
	int nRes;
	QWORD qwDiskFreeSpace;
	QString strMess;
	QWORD qwCount;
	int cCurChar;
	QFileInfo fi;
	QWORD qwPos1, qwPos2;

	// Check archive an is opened
	if (m_bArchiveOpened != true)
	{	KMessageBox::error(m_mainWnd, i18n("An archive must be opened to delete files."));
		return ERROR_FAILED;
	}

	// If no file selected, cancel
	if (getNbSelectedFiles() == 0)
	{	KMessageBox::error (m_mainWnd, i18n("There are no selected files."));
		return ERROR_FAILED;
	}	

	// Initializations
	sprintf (szOldArchive, "%s", m_szArchivePath);
	sprintf (szNewArchive, "%s-new", m_szArchivePath); 	
	fi.setFile(m_szArchivePath);
	qwArchiveSize = fi.size();
	
  // Check there is enought free disk space
	nRes = getDiskFreeSpaceForFile(&qwDiskFreeSpace, szNewArchive);
	if (nRes != -1 && qwDiskFreeSpace < qwArchiveSize)
	{	KMessageBox::error (m_mainWnd, i18n("There is not enought disk free space, this operation need."));
   	return ERROR_FAILED;
	}
	
	((KFileCoderApp *)m_mainWnd)->slotStatusMsg(i18n("Deleting files from archive..."));

	// Close the archive
	closeArchive(false, false);

	// Open the files
	FILE *fOld, *fNew;
	
	errno = 0;
	fOld = fopen(szOldArchive, "rb");
	if (!fOld)
	{	KMessageBox::error (m_mainWnd, i18n("Can't delete files from archive. Error %s"), strerror(errno));
		return ERROR_FAILED;
	}

	errno = 0;
  fNew = fopen(szNewArchive, "w+b");
	if (!fNew)
	{	KMessageBox::error (m_mainWnd, i18n("Can't delete files from archive. Error %s"), strerror(errno));
		return ERROR_FAILED;
	}

 	// +=+=+=+=+=+=+=+= BEGIN: Copy the header and the files from OldArchive to NewArchive +=+=+=+=+=+=+=+=

	// 1. Copy the header
	errno = 0;
	nRes = fread(&head, sizeof(KArchiveHeader), 1, fOld);	
	if (nRes != 1)	
  {	KMessageBox::error (m_mainWnd, i18n("Can't read old archive. Error %s"), strerror(errno));
		fclose(fOld);
		fclose(fNew);
		return ERROR_FAILED;
	}

	errno = 0;
	nRes = fwrite(&head, sizeof(KArchiveHeader), 1, fNew);	
	if (nRes != 1)	
  {	KMessageBox::error(  m_mainWnd, i18n("Can't write new archive. Error %s"), strerror(errno));
		fclose(fOld);
		fclose(fNew);
		return ERROR_FAILED;
	}
	
	// progress dialog
	QProgressDialog dlgProgress(i18n("Deleting files from archive..."),0, (int) qwArchiveSize, m_mainWnd, "", true);
	dlgProgress.setCaption("KFileCoder");
	dlgProgress.show();
	dlgProgress.setAutoClose(true);
	dlgProgress.setProgress(0L);
	
	// 2. Copy the files if they are selected
	while (((QWORD)ftell(fOld)) < qwArchiveSize) // while !feof
	{	
		// update user interface
  	g_app -> processEvents();
			
		qwPos1 = (QWORD) ftell(fOld); // position in archive before file header

		errno = 0;
		nRes = readFileHeader(fOld, szFileName, szFilePath, NULL, NULL, &qwEncodedFileSize, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
		if (nRes == -1)
  	{	strMess.sprintf(i18n("Can't read file <b>%s</b> from archive: Error %s"), formatFullPath(szFilePath, szFileName).data(), strerror(errno));
			KMessageBox::error (m_mainWnd, strMess);
			return ERROR_FAILED;
		}
		
		qwPos2 = (QWORD) ftell(fOld); // position in archive after file header
		
		if (isFileSelected(szFileName, szFilePath) == false) // If file is not selected ==> Do not delete ==> Copy it to new archive
		{	
			// come back in the archive, before the file header to copy it
			fseek(fOld, qwPos1, SEEK_SET);
			
			// number of bytes to copy
			qwFileHeadSize = qwPos2 - qwPos1;	
			qwTotalBytesToCopy = qwFileHeadSize + qwEncodedFileSize; // <File Header> + <File encoded contents> + <File CRC> 	
			//printf("============= Copy %s/%s (%llu bytes)\n", szFilePath, szFileName, qwEncodedFileSize);			

      // ******* Copy (qwFileSize + ENCODING_LEN) Bytes: file contents + crc *******
			for (qwCount=0LL; qwCount < qwTotalBytesToCopy; qwCount++)
			{
				errno = 0;
				cCurChar = fgetc(fOld);
				if (cCurChar == EOF)
  			{	strMess.sprintf(i18n("Can't read old archive: Error %s"), strerror(errno));
  				KMessageBox::error (m_mainWnd, strMess);
					fclose(fOld);
					fclose(fNew);
					return ERROR_FAILED;
				}
				
				errno = 0;
				nRes = fputc(cCurChar, fNew);
  			if (nRes == EOF)
  			{	strMess.sprintf(i18n("Can't write new archive: Error %s"), strerror(errno));
  				KMessageBox::error (m_mainWnd, strMess);
					fclose(fOld);
					fclose(fNew);
					return ERROR_FAILED;
				}					
			}
			
			dlgProgress.setProgress((int)ftell(fOld));
		}
		else // File not selected ==> Skip it
		{
			//printf("============= Delete %s/%s\n", szFilePath, szFileName);			
  		fseek(fOld, qwEncodedFileSize, SEEK_CUR); // Skip file contents
		}

	}
 	
 	// +=+=+=+=+=+=+=+= END: Copy the header and the files from OldArchive to NewArchive +=+=+=+=+=+=+=+=

	// Close files
	fclose(fOld);
	fclose(fNew);

	// If operation succedded: Delete Old Archive
	deleteFile(szOldArchive);

	// If operation succedded: Rename the new archive to szOldArchive
	nRes = rename(szNewArchive, szOldArchive);
	if (nRes == -1)
	{	strMess.sprintf(i18n("Can't delete files from archive. Error %s"), m_szArchivePath, strerror(errno));	
		KMessageBox::error (m_mainWnd, strMess);
		return ERROR_FAILED;
	}
	
	// Open the new archive
	openArchive(szOldArchive, 0); // 0 because the password is already mixed and written in m_cPassword
	
	//printf("========================> Exit of deleteSelectedFilesInArchive()\n");

	return ERROR_SUCCESS;
}

// ===========================================================================================================================
int KFileCoderDoc::deleteFile(char *szFilename)
{
	int nRes;
	char szNewFilename[MAXPATHLEN+1];
	FILE *fStream;
	QWORD qwFileSize;
	QWORD qwFilePos;
	
	//printf("===> Deleting %s\n", szFilename);
	
	memset(szNewFilename, 'a', MAXPATHLEN);
	szNewFilename[MAXPATHLEN-1] = 0; // end of string in ASCII

	// Delete contents of the old file
	if ((fStream = fopen(szFilename,"a+b")) == NULL)
		return ERROR_FAILED;
	
	// get file size
	fseek(fStream, 0, SEEK_END);
	qwFileSize = (QWORD) ftell(fStream);
	
	// go to the begin of the file
	fseek(fStream, 0, SEEK_SET);
		
	for (qwFilePos=0; qwFilePos < qwFileSize; qwFilePos++);
	{	
		if (qwFilePos % 2 == 0)
			fputc(0x55, fStream);
		else
			fputc(0xAA, fStream);
	}
	
	fflush(fStream);
	fclose(fStream);
	
	truncate(szFilename, 0); // empty file contents

	// rename and delete old file
	nRes = rename(szFilename, szNewFilename); // to delete the filename of the archive
	if (nRes != -1) // do not delete another existing user file
	{	unlink(szNewFilename);
	}

	return ERROR_SUCCESS;
}

// ===========================================================================================================================
int KFileCoderDoc::listView_Fill()
{
	//printf ("====================> Enter in listView_Fill()\n");
	
	char szFileName[MAXPATHLEN];
	char szFilePath[MAXPATHLEN];
	QWORD qwOriginalFileSize;
	QWORD qwEncodedFileSize;
	mode_t nPermissions;
	uid_t userOwner;
	gid_t groupOwner;
	time_t st_mtime;
	QString strMess;
	int nRes;
	QFileInfo fiArchive;
	QWORD qwArchiveSize;
	DWORD dwCompress;
	
	// Check archive an is opened
	if (m_bArchiveOpened != true)
		return ERROR_FAILED;
	
	// Empty the list
	m_List -> clear();
	m_dwNbFiles = 0;
	fiArchive.setFile(m_szArchivePath);
	qwArchiveSize = fiArchive.size();
	
	// Skip header
	nRes = fseek(m_fArchive, sizeof(KArchiveHeader), SEEK_SET);
		
	while (((QWORD)ftell(m_fArchive)) < qwArchiveSize) // while not the end of the archive
	{	
   	// ----- Read "Header" of the file -----	
		nRes = readFileHeader(m_fArchive, szFileName, szFilePath, &qwOriginalFileSize, NULL, &qwEncodedFileSize,
		&nPermissions, &userOwner, &groupOwner, &st_mtime, NULL, &dwCompress, NULL);
		if (nRes == -1)
  	{	strMess.sprintf(i18n("Can't read files headers from archive."), formatFullPath(szFilePath, szFileName).data());
			KMessageBox::error (m_mainWnd, strMess);
			return ERROR_FAILED;
		}
		
		// ----- Skip the file contents and control hash table -----
  	fseek(m_fArchive, (long) qwEncodedFileSize, SEEK_CUR); // skip contents
		
		// Add Item to list View
		listView_AddItem(szFileName, szFilePath, qwOriginalFileSize, qwEncodedFileSize, nPermissions, userOwner, groupOwner, dwCompress);
  	g_app -> processEvents();
		m_dwNbFiles++;
	}
	
	//printf ("====================> Exit of listView_Fill()\n");
	return ERROR_SUCCESS;
}

// ===========================================================================================================================
int KFileCoderDoc::listView_AddItem(const char *szName, const char *szPath, QWORD qwOriginalFileSize, QWORD qwEncodedFileSize, mode_t nPermissions, uid_t userOwner, gid_t groupOwner, DWORD dwCompress)
{
	QString strFullPath;
	QString strTemp;
	QWORD qwRatio;

	// Prepare text to add
	strFullPath = formatFullPath(szPath, szName);
	
	// Add item to list
	QListViewItem *lvi;

	// insert the new item after the last one
	lvi = new QListViewItem(m_List);

	lvi -> setText(0, szName);
	lvi -> setText(1, formatSize(qwOriginalFileSize).data()); // Convert QWORD into a string in the good format (Bytes or KBytes or MBytes)
	lvi -> setText(2, formatSize(qwEncodedFileSize).data()); // Convert QWORD into a string in the good format (Bytes or KBytes or MBytes)
	lvi -> setText(3, szPath);
	
	// permissions
	lvi -> setText(4, parsePermissions(nPermissions));
	
	// owners
	strTemp.sprintf("%s / %s", getUserNameFromUID(userOwner).data(), getGroupNameFromGID(groupOwner).data());
	lvi -> setText(5, strTemp);
	
	// compression
	if (dwCompress == COMPRESS_NONE)
		strTemp.sprintf(i18n("none"));
	else if (dwCompress == COMPRESS_GZIP)
		strTemp.sprintf("gzip");
	else if (dwCompress == COMPRESS_BZIP2)
		strTemp.sprintf("bzip2");
	else
		strTemp.sprintf(i18n("unknown"));
	lvi -> setText(6, strTemp);
	
	// ratio (compressed perf)
	if ((qwOriginalFileSize > 0) && (qwEncodedFileSize > 0))
	{	qwRatio = (100 * qwEncodedFileSize) / qwOriginalFileSize;
		strTemp.sprintf("%d %%", (int)qwRatio);
		lvi -> setText(7, strTemp);
	}
	
	// icon
	lvi -> setPixmap(0, KMimeType::pixmapForURL(strFullPath, 0, KIcon::Small)); // should add true after Small
	
	// Update menu & toolbar commands
	((KFileCoderApp *) m_mainWnd) -> updateCommandsAccess();
	
	return ERROR_SUCCESS;
}

// ===========================================================================================================================
int KFileCoderDoc::writeFileString(FILE *qfOut, const char *szString, BYTE *cKey)
{
	BYTE cEncodedString[MAXPATHLEN+1];
	int nRes;
	UINT nLen;

	memset(cEncodedString, 0, MAXPATHLEN);
	nLen = strlen(szString);

	// encode the string, then hackers won't see the names of the files encoded in the archive
	pc1Encode(cEncodedString, (BYTE *) szString, nLen, cKey);

	// Write size of the string
	nRes = fwrite(&nLen, sizeof(UINT), 1, qfOut);	
	if (nRes != 1)	
  {	KMessageBox::error (m_mainWnd, i18n("Can't write to file."));
		return ERROR_FAILED;
	}
	
	// Write the string
	nRes = fwrite(cEncodedString, nLen, 1, qfOut);	
	if (nRes != 1)	
  {	KMessageBox::error (m_mainWnd, i18n("Can't write to file."));
		return ERROR_FAILED;
	}	

	return ERROR_SUCCESS;
}

// ===========================================================================================================================
int KFileCoderDoc::readFileString(FILE *qfIn, char *szString, BYTE *cKey)
{
	BYTE cEncodedString[MAXPATHLEN+1];
	QString strMess;
	int nRes;
	UINT nLen;
	
	// init
	memset(szString, 0, MAXPATHLEN);
	
	// Read size of the string
	errno = 0;
	nRes = fread(&nLen, sizeof(UINT), 1, qfIn);	
	
	if (nRes != 1)	
  {	strMess.sprintf(i18n("Can't read from file: Error %s"), strerror(errno));
	 	KMessageBox::error (m_mainWnd, strMess);
		return ERROR_FAILED;
	}
	
	// Read the string
	errno = 0;
	nRes = fread(cEncodedString, nLen, 1, qfIn);	
	if (nRes != 1)	
  {	strMess.sprintf(i18n("Can't read from file: Error %s"), strerror(errno));
	 	KMessageBox::error (m_mainWnd, strMess);
		return ERROR_FAILED;
	}
	
	// encode the string, then hackers won't see the names of the files encoded in the archive
	pc1Decode((BYTE *) szString, cEncodedString, nLen, cKey);

	szString[nLen] = 0; // End of the string

	return ERROR_SUCCESS;
}

// ===========================================================================================================================
bool KFileCoderDoc::isFileSelected(const char *szFileName, const char *szFilePath)
{
	QListViewItem *lviCurItem;
	QListViewItem *lviFirst;
	
	lviCurItem = lviFirst = m_List -> firstChild();
	if (lviCurItem == NULL)
	{	return false;
	}

	do
	{	if ( (QString(szFileName) == lviCurItem -> text(0)) && (QString(szFilePath) == lviCurItem -> text(3)) )
		{	return (lviCurItem -> isSelected());
		}

    lviCurItem = lviCurItem -> nextSibling();
	} while(lviCurItem && lviCurItem != lviFirst);

	return false; // Error: item not in the list
}

// ===========================================================================================================================
int KFileCoderDoc::getNbSelectedFiles()
{
	int nSel = 0;
	QListViewItem *lviCurItem;
	QListViewItem *lviFirst;
	
	lviCurItem = lviFirst = m_List -> firstChild();
	if (lviCurItem == NULL)
		return false;

	do
	{	if (lviCurItem -> isSelected())
			nSel++;

    lviCurItem = lviCurItem -> nextSibling();
	} while(lviCurItem && lviCurItem != lviFirst);

	return nSel; // Error: item not in the list
}

// ===========================================================================================================================
bool KFileCoderDoc::doesFileExistsInArchive(const char *szFileName, const char *szFilePath)
{
	QListViewItem *lviCurItem;
	QListViewItem *lviFirst;
	
	lviCurItem = lviFirst = m_List -> firstChild();
	if (lviCurItem == NULL)
		return false;

	do
	{	if ( strcmp(szFileName, lviCurItem -> text(0)) == 0 && strcmp(szFilePath, lviCurItem -> text(3)) == 0 )
		{	return true;
		}

    lviCurItem = lviCurItem -> nextSibling();
	} while(lviCurItem && lviCurItem != lviFirst);

	return false;
}

// ===========================================================================================================================
int KFileCoderDoc::writeFileHeader(FILE *fArchive, char *szFullPath, char *szFileName, char *szFilePath, BYTE *cEncodedFileMixer,
QWORD qwCompressedFileSize, QWORD qwEncodedFileSize, DWORD dwCompress, BYTE *cHashControl)
{
	struct stat fStat;
	char cReserved[FILEHEADER_RESERVED_SIZE+1];
	BYTE cEncodedFilenameMixer[ENCODING_LEN+1];
	BYTE cFilenameMixer[ENCODING_LEN];
	BYTE cFilenameKey[ENCODING_LEN];
	QWORD qwUID, qwGID, qwPermissions;
	QWORD qwTime; // 64 bist value to allow future extensions of the current 32 bits time
	int nRes;
	DWORD dwFilenamesEncodingAlgo;
			
	// get infos about file
	memset(&fStat, 0, sizeof(fStat));
	memset(cReserved, 0, FILEHEADER_RESERVED_SIZE);
	stat(szFullPath, &fStat);
	
	qwPermissions = (QWORD) fStat.st_mode;
	qwUID = (QWORD) fStat.st_uid;
	qwGID = (QWORD) fStat.st_gid;
	qwTime = (QWORD) fStat.st_mtime;
	QWORD qwOriginalFileSize = (QWORD) fStat.st_size;
	
	// --------- write magic string ---------
	nRes = fwrite(g_szBeginFileHeader, strlen(g_szBeginFileHeader), 1, fArchive);	
	if (nRes != 1)	
		return ERROR_FAILED;
		
	// --------- encode file name and patch with a new key ---------
	dwFilenamesEncodingAlgo = 0; // PC1 standard
	nRes = fwrite(&dwFilenamesEncodingAlgo, sizeof(DWORD), 1, fArchive);	
	if (nRes != 1)	
		return ERROR_FAILED;
	
	// calculate the mixer number, which will be used to mix the random number
	generateRandomNumber(cFilenameMixer);
	
	// decode the cFilenameMixer used to calculate the key used to encode filenames/paths
	pc1Encode(cEncodedFilenameMixer, cFilenameMixer, ENCODING_LEN, m_cMixerCodingKey);
	
	nRes = fwrite(cEncodedFilenameMixer, ENCODING_LEN, 1, fArchive);	
	if (nRes != 1)	
		return ERROR_FAILED;
	
	calculatePC1Hash(cFilenameKey, cFilenameMixer, ENCODING_LEN, m_cPassword);
	
	// --------- write file name / path ---------
	nRes = writeFileString(fArchive, szFileName, cFilenameKey); // filename (ex: "image.png")
	if (nRes == -1)
		return ERROR_FAILED;
	
	nRes = writeFileString(fArchive, szFilePath, cFilenameKey); // filepath (ex:"/home/user")
	if (nRes == -1)
		return ERROR_FAILED;

	// write file permissions
	nRes = fwrite(&qwPermissions, sizeof(QWORD), 1, fArchive);	
	if (nRes != 1)	
		return ERROR_FAILED;

	// write owner user
	nRes = fwrite(&qwUID, sizeof(QWORD), 1, fArchive);	
	if (nRes != 1)	
		return ERROR_FAILED;
	
	// write owner group
	nRes = fwrite(&qwGID, sizeof(QWORD), 1, fArchive);	
	if (nRes != 1)	
		return ERROR_FAILED;
		
	// write 64 bits last-modif-date/time number
	nRes = fwrite(&qwTime, sizeof(QWORD), 1, fArchive);	
	if (nRes != 1)	
		return ERROR_FAILED;
	
	// write number used to mix password of this file
	nRes = fwrite(cEncodedFileMixer, ENCODING_LEN, 1, fArchive);	
	if (nRes != 1)	
		return ERROR_FAILED;
	
	// write compression mode used
	nRes = fwrite(&dwCompress, sizeof(DWORD), 1, fArchive);	
	if (nRes != 1)	
		return ERROR_FAILED;
	
	// write file original size
	nRes = fwrite(&qwOriginalFileSize, sizeof(QWORD), 1, fArchive);	
	if (nRes != 1)	
		return ERROR_FAILED;		
	
	// write file compressed size
	nRes = fwrite(&qwCompressedFileSize, sizeof(QWORD), 1, fArchive);	
	if (nRes != 1)	
		return ERROR_FAILED;
	
	// write file encoded/compressed size
	nRes = fwrite(&qwEncodedFileSize, sizeof(QWORD), 1, fArchive);	
	if (nRes != 1)	
		return ERROR_FAILED;
	
	// write hash control
	nRes = fwrite(cHashControl, ENCODING_LEN, 1, fArchive);	
	if (nRes != 1)	
		return ERROR_FAILED;
	
	// write reserved bytes (for future extensions)
	nRes = fwrite(cReserved, FILEHEADER_RESERVED_SIZE, 1, fArchive);	
	if (nRes != 1)	
		return ERROR_FAILED;

	// write magic string
	nRes = fwrite(g_szEndFileHeader, strlen(g_szEndFileHeader), 1, fArchive);	
	if (nRes != 1)	
		return ERROR_FAILED;
	
	return ERROR_SUCCESS;
}

// ===========================================================================================================================
int KFileCoderDoc::readFileHeader(FILE *fArchive, char *szFileName, char *szFilePath, QWORD *qwOriginalFileSize, QWORD *qwCompressedFileSize, QWORD *qwEncodedFileSize,
mode_t *nPermissions, uid_t *st_uid, gid_t *st_gid, time_t *st_mtime, BYTE *cEncodedFileMixer, DWORD *dwCompress, BYTE *cHashControl)
{
	int nRes;
	char szTemp[512];
	BYTE cEncodedFilenameMixer[ENCODING_LEN+1];
	BYTE cFilenameMixer[ENCODING_LEN+1];
	BYTE cFilenameKey[ENCODING_LEN+1];
	QWORD qwTime; // 64 bist value to allow future extensions of the current 32 bits time
	QWORD qwUID, qwGID, qwPermissions;
	DWORD dwFilenamesEncodingAlgo;
	
	// --------- read magic string ---------
	nRes = fread(szTemp, strlen(g_szBeginFileHeader), 1, fArchive);	
	if (nRes != 1)	
		return ERROR_FAILED;
	
	// test begin flag
	if (memcmp(szTemp, g_szBeginFileHeader, strlen(g_szBeginFileHeader)) != 0)
	{	printf ("Unable to read begin of file header flag\n");
		return ERROR_FAILED;	
	}
	
	// --------- encode file name and patch with a new key ---------
	nRes = fread(&dwFilenamesEncodingAlgo, sizeof(DWORD), 1, fArchive);	
	if (nRes != 1)	
		return ERROR_FAILED;
	
	// calculate the mixer number, which will be used to mix the random number
	nRes = fread(cEncodedFilenameMixer, ENCODING_LEN, 1, fArchive);	
	if (nRes != 1)	
		return ERROR_FAILED;
	
	// decode the cFilenameMixer used to calculate the key used to encode filenames/paths
	pc1Decode(cFilenameMixer, cEncodedFilenameMixer, ENCODING_LEN, m_cMixerCodingKey);
	
	calculatePC1Hash(cFilenameKey, cFilenameMixer, ENCODING_LEN, m_cPassword);
			
	// read file name
	nRes = readFileString(fArchive, szFileName, cFilenameKey);
	if (nRes == -1)
		return ERROR_FAILED;

	// read file path
	nRes = readFileString(fArchive, szFilePath, cFilenameKey);
	if (nRes == -1)
		return ERROR_FAILED;
	
	// read file permissions
	if (nPermissions)
	{	nRes = fread(&qwPermissions, sizeof(QWORD), 1, fArchive);	
		*nPermissions = (mode_t) qwPermissions;
		if (nRes != 1)	
			return ERROR_FAILED;
	}
	else
	{	// move in the file, as if we had read
		fseek(fArchive, sizeof(QWORD), SEEK_CUR);
	}

	// read owner user
	if (st_uid)
	{	nRes = fread(&qwUID, sizeof(QWORD), 1, fArchive);	
		*st_uid = (uid_t) qwUID;
		if (nRes != 1)	
			return ERROR_FAILED;
	}
	else
	{	// move in the file, as if we had read
		fseek(fArchive, sizeof(QWORD), SEEK_CUR);
	}
	
	// read owner group
	if (st_gid)
	{	nRes = fread(&qwGID, sizeof(QWORD), 1, fArchive);	
		*st_gid = (gid_t) qwGID;
		if (nRes != 1)	
			return ERROR_FAILED;
	}
	else
	{	// move in the file, as if we had read
		fseek(fArchive, sizeof(QWORD), SEEK_CUR);
	}
		
	// read last-modif-date/time (64 bits to allow the future 64 bits value)
	if (st_mtime)
	{	nRes = fread(&qwTime, sizeof(QWORD), 1, fArchive);	
		*st_mtime = (time_t) qwTime;
		if (nRes != 1)	
			return ERROR_FAILED;
	}
	else
	{	// move in the file, as if we had read
		fseek(fArchive, sizeof(QWORD), SEEK_CUR);
	}
	
	// read number used to mix password of this file
	if (cEncodedFileMixer)
	{
		nRes = fread(cEncodedFileMixer, ENCODING_LEN, 1, fArchive);	
		if (nRes != 1)	
			return ERROR_FAILED;
	}
	else
	{
		fseek(fArchive, ENCODING_LEN, SEEK_CUR);	
	}	

	// read compression level used
	if (dwCompress)
	{
		nRes = fread(dwCompress, sizeof(DWORD), 1, fArchive);	
		if (nRes != 1)	
			return ERROR_FAILED;
	}
	else
	{
		fseek(fArchive, sizeof(DWORD), SEEK_CUR);	
	}	
	
	
	// read file original size
	if (qwOriginalFileSize)
	{	nRes = fread(qwOriginalFileSize, sizeof(QWORD), 1, fArchive);	
		if (nRes != 1)	
			return ERROR_FAILED;
	}
	else
	{	// move in the file, as if we had read
		fseek(fArchive, sizeof(QWORD), SEEK_CUR);
	}
	
	// read file original size
	if (qwCompressedFileSize)
	{	nRes = fread(qwCompressedFileSize, sizeof(QWORD), 1, fArchive);	
		if (nRes != 1)	
			return ERROR_FAILED;
	}
	else
	{	// move in the file, as if we had read
		fseek(fArchive, sizeof(QWORD), SEEK_CUR);
	}
	
	// read file original size
	if (qwEncodedFileSize)
	{	nRes = fread(qwEncodedFileSize, sizeof(QWORD), 1, fArchive);	
		if (nRes != 1)	
			return ERROR_FAILED;
	}
	else
	{	// move in the file, as if we had read
		fseek(fArchive, sizeof(QWORD), SEEK_CUR);
	}
	
	// read encoded size
	if (cHashControl)
	{	nRes = fread(cHashControl, ENCODING_LEN, 1, fArchive);
		if (nRes != 1)
			return ERROR_FAILED;
	}
	else
	{	// move in the file, as if we had read
		fseek(fArchive, ENCODING_LEN, SEEK_CUR);
	}
	
	// read reserved bytes (for future extensions)
	nRes = fseek(fArchive, FILEHEADER_RESERVED_SIZE, SEEK_CUR);	
	if (nRes != 0)	
		return ERROR_FAILED;	
	
	// read magic string
	nRes = fread(szTemp, strlen(g_szEndFileHeader), 1, fArchive);	
	if (nRes != 1)	
		return ERROR_FAILED;
	
	// test end flag
	if (memcmp(szTemp, g_szEndFileHeader, strlen(g_szEndFileHeader)) != 0)
	{	printf ("File %s %s. Unable to read end of file header flag\n", szFileName, szFilePath);
		return ERROR_FAILED;	
	}
	
	return ERROR_SUCCESS;
}

// ===========================================================================================================================
QString KFileCoderDoc::getUserNameFromUID(uid_t uid)
{
	struct passwd *pwStuct;
	QString strResult;
	
	while ((pwStuct = getpwent()))
	{
		// if this is the searched uid
		if (pwStuct -> pw_uid == uid)
		{	strResult.sprintf("%s (%d)", pwStuct -> pw_name, uid);
			endpwent();
			return strResult;	
		}
	}
	
	strResult.setNum(uid);
	endpwent();
	return strResult;
}

// ===========================================================================================================================
QString KFileCoderDoc::getGroupNameFromGID(gid_t gid)
{
	struct group *groupStuct;
	QString strResult;

	while ((groupStuct = getgrent()))
	{
		// if this is the searched gid
		if (groupStuct -> gr_gid == gid)
		{	strResult.sprintf("%s (%d)", groupStuct -> gr_name, gid);
			endgrent();
			return strResult;	
		}
	}

	strResult.setNum(gid);
	endgrent();
	return strResult;
}
