/***************************************************************************
                          kobjectimpl.h  -  description
                             -------------------
    begin                : Mon Oct 16 2000
    copyright            : (C) 2000 by Sergio Moretti
    email                : sermore@libero.it
    revision             : $Revision: 1.3 $
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef KOBJECTIMPL_H
#define KOBJECTIMPL_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qdom.h>
#include <qstring.h>
#include <qdragobject.h>
#include <qobject.h>
#include "km.h"
//#include "kobject.h"

#ifdef DMALLOC
#include <dmalloc.h>
#endif


using namespace KM;

class KContainerImpl;
class KObject;

/** 
 *@short Abstract base class for all objects implementation
 *@author Sergio Moretti
 *@version $Id$
 * State of object:
 * @see id :     object identifier unique in document
 * @type   :     object type
 * @dom    :     dom element associated to object
 * 
 */
class KObjectImpl : public QObject {

  Q_OBJECT

  friend class KFactoryImpl;
  friend class KObject;

  /** document identifier in XML prolog */
  static const char DOCID[];
  /** list of mimetypes onto which this object can be converted */ 
  static const char MIMETYPE[][50];
  /** xml document identifier string */
  static const char DOCTYPE[];

public:
  virtual ~KObjectImpl();

  // Public State

  /** identity of object */
  int id() const { return _id; }
  /** get type of object */
  int type() const { return _type; }
  /** DOM Element associated to object */
  QDomElement dom() const { return _dom; }
  /** type string */
  QString typeStr() const;
  /** document name */
  virtual const char * docId() const { return DOCID; }
  /** dom document type */
  virtual const char * docType() const { return DOCTYPE; }
  /** mime type */
  virtual const char * mimeType(int i) const { return MIMETYPE[i]; }

  // Public Operations
  /** return true if object don't have a parent */
  bool isRoot() const { return container() == 0; }
  /** return true if the modification flag has been setted
   * Modification Flags:
   * MOD_STATE  : state of object
   * MOD_CONF   : configuration of object
   * MOD_VIEW   : object need display update
   * MOD_ERR    : fatal error is occurred on object
   */
  bool isModified(ModType type) const;
  /** set or reset the modification flag, this change can be propagated up 
   * in the object tree. A change in configuration or state in a
   * child can set state modification in parent.
   * 
   * PRP_NONE   : don't propagate change
   * PRP_THIS   : only this object modification is propagated up to root
   * PRP_ALL    : all objects in hierarchy will signal the change
   */
  void setModified(ModType type, PropMode prop, bool mod = true);
  /** run this procedure periodically */
  virtual void runPeriodically();

  // Object Creation and Initialization

  /** load state and identity from QDomElement */
  virtual void load(KContainerImpl *container, const QDomElement &e);
  /** update DOM to current state */
  virtual void update();

protected:
  /** constructor for cloning */
  KObjectImpl(int type);
  /** abstract clone operation */
  virtual KObjectImpl * clone() const = 0;

  /**copy state but not identity from QDomElement, deepCopy is necessary for 
   *KContainer copy */
  virtual void loadData();
  virtual void updateConf();
  virtual void updateState();

  // Protected State

  /** parent container */
  KContainerImpl * container() const { return _container; }

  // Protected Operation on Object

  /** create a DOM from object */
  virtual QDomDocument toDom() const;

  // Global Configuration Management

  /** global instance reference */
  const KObjectImpl * global() const { return _global; } // == 0 ? this : _global; }
  /** get/set use of global configuration */
  bool isGlobal() const { return _cfg_global; }
  void setGlobal(bool g) { _cfg_global = g; }

private:
  /** parent container */
  KContainerImpl *_container;
  /** object identifier */
  int _id;
  /** state modification flag */
  int _modified;
  /** global instance reference */
  KObjectImpl *_global;
  /** use of global configuration */
  bool _cfg_global;
  /** DOM element */
  QDomElement _dom;
  /** type of object */
  int _type;
};

#endif // KOBJECTIMPL_H

/* Local Variables: */
/* mode: c++ */
/* End: */
