/***************************************************************************
                          ktransferimpl.cpp  -  description
                             -------------------
    begin                : Tue Oct 17 2000
    copyright            : (C) 2000 by Sergio Moretti
    email                : sermore@libero.it
    revision             : $Revision: 1.5 $
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <kdebug.h>
#include <qdir.h>
#include <klocale.h>
#include <qfileinfo.h>
#include <qtextstream.h>
#include <kapp.h>
#include "utils.h"
//#include "kcontainerimpl.h"
#include "ktransfer.h"
#include "ktmanagerimpl.h"
#include "ktransferimpl.h"


const char KTransferImpl::DOCID[] = "Transfer";
const char KTransferImpl::MIMETYPE[][50] = { "application/" DOC "-transfer",
					     "text/x-" DOC "-transfer" };
const char KTransferImpl::DOCTYPE[] = "<!DOCTYPE Transfer >";


KTransferImpl::KTransferImpl(int type) :  KObjectImpl(type) {
  _timeStart.setHMS(0, 0, 0);
  _retry = 0;
  _lastBytesRead = 0;
  _mediumBandwidth = 0;
  _bandwidth = 0;
  _timeEst.setHMS(0, 0, 0);
  _lastReadNotify.setHMS(0, 0, 0);
  _cmdStatus = CMD_RUNNABLE;
  _rsmStatus = RSM_UNKNOWN;
  _killed = false;
  connect(&_wait, SIGNAL(timeout()), this, SLOT(slotTimeout()));
}

KTransferImpl::~KTransferImpl(){
  //kdDebug(D_INI) << name() << ": Transfer destroy" << endl;
}

/** load transfer from dom */
void KTransferImpl::loadData() {
  kdDebug(D_INI) << name() << ": transfer loadData" << endl;
  KObjectImpl::loadData();
  // state
  _remote = dom().attribute("URL", "");
  _local.setPath(dom().attribute("File", "")); //FIXME !! is correct??
  _status = TRN_READY;
  _cmdStatus = CMD_RUNNABLE;
  _rsmStatus = 
    static_cast<ResumeStatus>(dom().attribute("ResumeStatus", "0").toInt());
  _resumed = dom().attribute("Resumed", "0").toInt();
  _len = dom().attribute("Len", "0").toInt();
  _partial = dom().attribute("Partial", "0").toInt();
  _priority = dom().attribute("Priority", "0").toInt();
  // config
  _cfg_checkResume = dom().attribute("CheckResume", "1").toInt();
  _cfg_maxResume = dom().attribute("MaxResume", "10").toInt();
  _cfg_autoResume = dom().attribute("AutoResume", "0").toInt();
  _cfg_waitResume = dom().attribute("WaitResume", "20").toInt();
  _cfg_maxRetry = dom().attribute("MaxRetry", "3").toInt();
  _cfg_waitRetry = dom().attribute("WaitRetry", "2").toInt();
  _cfg_bandSampleTime = dom().attribute("BandSampleTime", "2000").toInt();
  _cfg_readNotifyTime = dom().attribute("ReadNotifyTime", "1000").toInt();
  _cfg_priorityPolicy = 
    dom().attribute("PriorityPolicy", QString().setNum(PRI_FIFO)).toInt();
  _cfg_logProcess = dom().attribute("LogProcess", "0").toInt();
}

void KTransferImpl::updateState() {
  KObjectImpl::updateState();
  dom().setAttribute("URL", _remote.url());
  dom().setAttribute("File", _local.path());
  dom().setAttribute("ResumeStatus", _rsmStatus);
  dom().setAttribute("Resumed", _resumed);
  dom().setAttribute("Len", _len);
  dom().setAttribute("Partial", _partial);
  dom().setAttribute("Priority", _priority);
}

void KTransferImpl::updateConf() {
  KObjectImpl::updateConf();
  dom().setAttribute("CheckResume", _cfg_checkResume);
  dom().setAttribute("MaxResume", _cfg_maxResume);
  dom().setAttribute("AutoResume", _cfg_autoResume);
  dom().setAttribute("WaitResume", _cfg_waitResume);
  dom().setAttribute("MaxRetry", _cfg_maxRetry);
  dom().setAttribute("WaitRetry", _cfg_waitRetry);
  dom().setAttribute("BandSampleTime", _cfg_bandSampleTime);
  dom().setAttribute("ReadNotifyTime", _cfg_readNotifyTime);
  dom().setAttribute("PriorityPolicy", _cfg_priorityPolicy);
  dom().setAttribute("LogProcess", _cfg_logProcess);
}

KTManagerImpl * KTransferImpl::manager() const { 
  return dynamic_cast<KTManagerImpl*>(container()); 
}

const KTransferImpl * KTransferImpl::global() const { 
  return dynamic_cast<const KTransferImpl*>(KObjectImpl::global()); 
}

QString KTransferImpl::statusStr() const {
  switch (status()) {
  case TRN_READY:
  case TRN_FINISH:
    return cmdStatusStr();
  case TRN_START:
    return i18n("Start");
  case TRN_CONNECT:
    return i18n("Connect");
  case TRN_DOWNLOAD:
    return i18n("Download");
  }
  kdFatal() << name() << ": unknown status " << status() << endl;
  return QString::null;
}

QString KTransferImpl::cmdStatusStr() const {
  switch (cmdStatus()) {
  case CMD_RUNNABLE:
    return i18n("Ready");
  case CMD_EXITED:
    return i18n("Wait");
  case CMD_KILLED:
    return i18n("Killed");
  case CMD_FATALERROR:
    return i18n("Error");
  case CMD_RUNNING:
    return i18n("Running");
  }
  kdFatal() << name() << ": unknown state " << status() << endl;
  return QString::null;
}

QString KTransferImpl::filStatusStr() const {
  if (partial() == 0)
    return i18n("Empty");
  else if (partial() < len())
    return i18n("Partial");
  else if (partial() == len())
    return i18n("Complete");
  kdError() << name() << ": partial > len " << statusStr() << endl;
  return QString::null;
};

QString KTransferImpl::rsmStatusStr() const {
  switch (rsmStatus()) {
  case RSM_UNKNOWN:
    return i18n("???");
  case RSM_CHECKING:
    return i18n("***");
  case RSM_CHECKFAIL:
    return i18n("---");
  case RSM_YES:
    return i18n("Yes");
  case RSM_NO:
    return i18n("No");
  }
  kdFatal() << name() << ": unknown state " << status() << endl;
  return QString::null;
};

/** compare priority, return 1, 0, -1 */
int KTransferImpl::cmpPriority(const KTransferImpl *t) const {
  int p1 = priority() + manager()->getPriority();
  int p2 = t->priority() + manager()->getPriority();
  if ( p1 != p2 )
    return p1 > p2 ? 1 : -1;
  switch (getPriorityPolicy()) {
  case PRI_FIFO:
    return ( id() > t->id() ? -1 : 1);
  case PRI_LIFO:
    return ( id() < t->id() ? -1 : 1);
  case PRI_SHORT:
    return (len() == t->len() ? 
	    (id() > t->id() ? -1 : 1) : (len() > t->len() ? -1 : 1));
  case PRI_LONG:
    return (len() == t->len() ? 
	    (id() > t->id() ? -1 : 1) : (len() < t->len() ? -1 : 1));
  case PRI_USER:
    return 0;
  }
  kdFatal(D_RUN) << name() << ": priority policy unknown " << getPriorityPolicy()<< endl;
  return 0;
}

KURL KTransferImpl::getTempFile(const KURL &, const KURL &lcl) const {
  KURL tmp;
  tmp.setPath(QFileInfo(manager()->getWorkingDir(), 
			lcl.fileName()).absFilePath());
  return tmp;
}

KURL KTransferImpl::getTmpFile() const { 
  return getTempFile(remote(), local()); 
}

bool KTransferImpl::getCheckResume() const {
  return isGlobal() ? global()->getCheckResume() : _cfg_checkResume;
}
	
int KTransferImpl::getMaxResume() const {
  return isGlobal() ? global()->getMaxResume() : _cfg_maxResume;
}
	
bool KTransferImpl::getAutoResume() const {
  return isGlobal() ? global()->getAutoResume() : _cfg_autoResume;
}

int KTransferImpl::getWaitResume() const {
  return isGlobal() ? global()->getWaitResume() : _cfg_waitResume;
}

int KTransferImpl::getMaxRetry() const {
  return isGlobal() ? global()->getMaxRetry() : _cfg_maxRetry;
}

int KTransferImpl::getWaitRetry() const {
  return isGlobal() ? global()->getWaitRetry() : _cfg_waitRetry;
}

int KTransferImpl::getBandSampleTime() const {
  return isGlobal() ? global()->getBandSampleTime() : _cfg_bandSampleTime;
}

int KTransferImpl::getReadNotifyTime() const {
  return isGlobal() ? global()->getReadNotifyTime() : _cfg_readNotifyTime;
}

int KTransferImpl::getPriorityPolicy() const {
  return isGlobal() ? global()->getPriorityPolicy() : _cfg_priorityPolicy;
}

bool KTransferImpl::getLogProcess() const {
  return isGlobal() ? global()->getLogProcess() : _cfg_logProcess;
}

bool KTransferImpl::isComplete() const {
  return (partial() == len() && len() > 0);
}

void KTransferImpl::setLen(int l) {
  kdDebug(D_RUN) << name() << " set len = " << l << endl;
  _len = l;
}

void KTransferImpl::setPartial(int p) {
  kdDebug(D_RUN) << name() << " set partial = " << p << endl;
  _partial = p;
}

/** check if process is running */
bool KTransferImpl::isRunning() const {
  return (cmdStatus() == CMD_RUNNING);
}

/** check if the process can be started */
bool KTransferImpl::isRunnable(bool automatic) const {
  if (isComplete() || isRunning())
    return false;
  return (cmdStatus() == CMD_RUNNABLE 
	  && ((automatic && status() == TRN_READY) // autodownload
	      || (status() == TRN_FINISH // autoresume
		  && getAutoResume() 
		  && isResumable()
		  && (!getMaxResume() || resumed() < getMaxResume())))
	  );
}

void KTransferImpl::raiseFatalError() {
  if (getLogProcess()) {
    QTextStream log(&_fatalErrorLog, IO_WriteOnly | IO_Append);
    log << "\n-------------------------------------------\n";
    _logFile.open(IO_ReadOnly);
    QTextStream flog(&_logFile);
    //_logFile.at(_logFile.size() - 10000 > 0 ? _logFile.size() - 10000 : 0);
    while (!flog.atEnd()) {
      QString line = flog.readLine();
      log << line << endl;
    }
    _logFile.close();
  }
  setModified(MOD_ERR, PRP_THIS);
}

bool KTransferImpl::start() {
  kdDebug(D_RUN) << name() << ":" << statusStr() << ":start " << endl;
  if (isRunning()) {
    kdError(D_RUN) << name() << ": start a running process" << endl;
    return false;
  }
  if (rsmStatus() == RSM_UNKNOWN && getCheckResume()) {
    checkResumeCapability();
    setModified(MOD_STATE, PRP_ALL);
    if (cmdStatus() != CMD_EXITED)
      return false;
  }
  _fatalError = false;
  _fatalErrorLog = QString::null;
  _retry = 0;
  _lastBytesRead = 0;
  _mediumBandwidth = 0;
  _bandwidth = 0L;
  _timeEst.setHMS(0, 0, 0);
  _lastReadNotify.setHMS(0, 0, 0);
  _killed = false;
  if (getLogProcess()) {
    QString logname = QFileInfo(manager()->getWorkingDir(), 
				local().fileName() + ".log").absFilePath();
    _logFile.setName(logname);
    _logFile.open(IO_WriteOnly | IO_Append);
  }
  bool retval;
  if (!(retval = processStart())) {
    _logFile.close();
  }
  return retval;
}

bool KTransferImpl::kill(bool setKilled) {
  kdDebug(D_RUN) << name() << ":" << statusStr() << ":kill " << endl;
  kdError(!isRunning(), D_RUN) << name() << ": kill process not running" << endl;
  _killed = setKilled;
  return processKill();
  //FIXME attendi che processo termini??
}

void KTransferImpl::opProcessStart() {
  kdDebug(D_RUN) << name() << ":" << statusStr() << "process start " << endl;
  _resumed++;
  setCmdStatus(CMD_RUNNING);
  setStatus(TRN_START);
  container()->itemActivate(true);
  setModified(MOD_STATE, PRP_ALL);
}

void KTransferImpl::opRunStart() {
  kdDebug(D_RUN) << name() << ":" << statusStr() << ":run start " << endl;
  //FIXME mettici _retry++
  _retry++;
}

void KTransferImpl::opRunEnd() {
  kdDebug(D_RUN) << name() << ":" << statusStr() << ":run end " << endl;
}

void KTransferImpl::opConnect() {
  kdDebug(D_RUN) << name() << ":" << statusStr() << ":connect " << endl;
  setStatus(TRN_CONNECT);
  setModified(MOD_STATE, PRP_ALL);
}

void KTransferImpl::opDownload() {
  kdDebug(D_RUN) << name() << ":" << statusStr() << ":download p=" << partial() << ",l=" << len() << endl;
  _timeStart.start();
  _lastReadTime.start();
  _lastBytesRead = partial();
  _lastBytesStart = partial();
  setStatus(TRN_DOWNLOAD);
  setModified(MOD_STATE, PRP_ALL);
}

void KTransferImpl::opDataRead(int datalen) {
  kdDebug(D_RUN) << name() << ":" << statusStr() << ":read data " << datalen << endl;
  _partial += datalen;
  calcBandwidth();
  int time = _lastReadNotify.elapsed();
  if (time > getReadNotifyTime()) {
    _lastReadNotify.start();
    setModified(MOD_READ, PRP_THIS);
  }
}

void KTransferImpl::opError(const QString &err) {
  kdDebug(D_RUN) << name() << ":" << statusStr() << ": error " << err << endl;
  //send message
  //setModified(MOD_STATE, PRP_ALL);
}

void KTransferImpl::opFatalError(const QString &err) {
  kdDebug(D_RUN) << name() << ":" << statusStr() << ": fatal error " << err << endl;
  _fatalError = true;
  _fatalErrorLog = err;
  if (isRunning())
    kill();
}

void KTransferImpl::opProcessEnd(int exitStatus) {
  kdDebug(D_RUN) << name() << ": process end " << exitStatus << endl;
  _logFile.close();
  if (cmdStatus() == CMD_RUNNING) {
    //kdWarning(D_RUN) << name() << ": process exit in CMD_RUNNING state" << endl;
    setCmdStatus(CMD_EXITED);
  }
  if (cmdStatus() != CMD_EXITED && cmdStatus() != CMD_FATALERROR) {
    opFatalError(QString("%1 : exit state not correct").arg(statusStr()));
    setCmdStatus(CMD_FATALERROR);
  }
  if (cmdStatus() != CMD_FATALERROR) {
    if (!isComplete())
      if (_killed)
	setCmdStatus(CMD_KILLED);
      else
	_wait.start(getWaitResume()*1000, true);
    else
      setCmdStatus(CMD_RUNNABLE);
    if (isComplete()) {
      KURL tmpfile = getTmpFile();
      if (tmpfile != local()) {
	if (QDir::current().exists(local().path()))
	  saveFile(local().path());
	//FIXME recupero errore
	bool check = QDir::current().rename(tmpfile.path(), local().path());
	kdError(check == false, D_RUN) << name() << ": move tmpfile " << tmpfile.path() << " -> " << local().path() << endl;
      }
    }
  }
  _timeStart.setHMS(0,0,0);
  if (getLogProcess())
    _logFile.close();
  setStatus(TRN_FINISH);
  container()->itemActivate(false);
  setModified(MOD_STATE, PRP_ALL);
  if (_fatalError)
    raiseFatalError();
}

void KTransferImpl::opProcessOutput(const QString &data) {
  if (getLogProcess())
    _logFile.writeBlock(data, data.length());
  emit sigProcessOutput(data);
}

bool KTransferImpl::processCheckResume() {
  setRsmStatus(RSM_CHECKING);
  int oldPartial = partial();
  QString tmpfile = getTmpFile().path();
  QString tmpfileSave = tmpfile + ".tmp";
  if (QDir::current().exists(tmpfile))
    QDir::current().rename(tmpfile, tmpfileSave);
  QFile f(tmpfile);
  f.open(IO_WriteOnly);
  //FIXME!! gestione lunghezza, se > len ?
  char buffer[20];
  f.writeBlock(buffer, 20);
  f.close();
  setPartial(20);
  bool retval = false;
  if (!start()) {
    setRsmStatus(RSM_CHECKFAIL);
    goto exit;
  }
  //QTime timeout;
  //timeout.start();
  while (!isRunning()) // && timeout.elapsed() < 20000)
    kapp->processEvents();
  if (!isRunning()) {
    if (rsmStatus() == RSM_CHECKING)
      setRsmStatus(RSM_CHECKFAIL);
    goto exit;
  }
  //timeout.start();
  while (isRunning() && rsmStatus() == RSM_CHECKING)
    //&& timeout.elapsed() < 40000)
    kapp->processEvents();
  kdDebug(D_RUN) << name() << ": check = " << rsmStatusStr() << endl;
  if (isRunning())
    kill(false);
  while (isRunning()) // && timeout.elapsed() < 40000)
    kapp->processEvents();
 exit:
  kdFatal(isRunning(), D_RUN) << name() << ":" << statusStr() << ": unable to stop process " << endl;
  switch (rsmStatus()) {
  case RSM_UNKNOWN:
    kdFatal(D_RUN) << name() << ":" << statusStr() << ": resume unknown after check" << endl;
    retval = false;
    break;
  case RSM_CHECKING:
    kdFatal(cmdStatus() != CMD_KILLED, D_RUN) << name() << ": " << statusStr() << ": resume check exit but not killed " << endl;
    setRsmStatus(RSM_UNKNOWN);
    retval = false;
    break;
  case RSM_CHECKFAIL:
    retval = false;
    break;
  case RSM_YES:
  case RSM_NO:
    retval = true;
    break;
  default:
    kdFatal(D_RUN) << name() << ": unknown resume state " << rsmStatus() << endl;
  }
  setPartial(oldPartial);
  QDir::current().remove(tmpfile);
  if (QDir::current().exists(tmpfileSave))
    QDir::current().rename(tmpfileSave, tmpfile);
  setModified(MOD_STATE, PRP_ALL);
  return retval;  
}

void KTransferImpl::calcBandwidth() {
  int time = _lastReadTime.elapsed();
  // FIXME!! gestisci meglio intervallo di campionamento
  if (time > getBandSampleTime()) {
    // calculate instantaneous bandwidth
    _bandwidth = (int)((1000.0 * (partial() - _lastBytesRead)) / time);
    _lastReadTime.start();
    _lastBytesRead = partial();
  }
  time = _timeStart.elapsed();
  if (time > 0)
    _mediumBandwidth = (int)((1000.0 * (partial() - _lastBytesStart)) / time);
  //debug("READ %l %l", _bandwidth, _mediumBandwidth);//(const char*)_parsebuf);
}

QTime KTransferImpl::estTime() {
  if (mediumBandwidth())
    return QTime().addSecs((len() - partial())/ mediumBandwidth());
  return QTime();
}

/** check for resume capabilities */
void KTransferImpl::checkResumeCapability() {
  processCheckResume();
}

//////////////  SLOTS

void KTransferImpl::slotTimeout() {
  if (status() == TRN_FINISH && cmdStatus() == CMD_EXITED) {
    setCmdStatus(CMD_RUNNABLE);
    setModified(MOD_STATE, PRP_ALL);
  }
}

#include "ktransferimpl.moc"
