/**
 * changesview.cpp
 *
 * Copyright (C)  2004  Zack Rusin <zack@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307  USA
 */
#include "changesview.h"

#include "entries.h"
#include "entrytoken.h"
#include "grouptoken.h"
#include "configmanager.h"
#include "kernel.h"

#include <kfiledialog.h>
#include <kmessagebox.h>
#include <klistview.h>
#include <ksqueezedtextlabel.h>
#include <kpushbutton.h>
#include <klocale.h>
#include <kdebug.h>

using namespace KConfigEditor;

ChangesView::ChangesView( QWidget *parent )
    : ChangesViewUI( parent )
{
    connect( m_removeButton, SIGNAL(clicked()),
             SLOT(removeEntry()) );
    connect( m_backupButton, SIGNAL(clicked()),
             SLOT(generateBackupScript()) );
    connect( m_propagationButton, SIGNAL(clicked()),
             SLOT(generatePropagationScript()) );
}

void ChangesView::fillChanges( QDict<EntryToken>  &changes, QDict<EntryToken>  &pendingChanges )
{
    m_entries = changes;
    m_pendingEntries = pendingChanges;

    m_changesView->clear();

    QDictIterator<EntryToken> itr( m_entries );
    EntryToken *token = 0;
    while ( (token=itr.current()) ) {
        bool saved = !pendingChanges.find( token->url().url() );
        new QListViewItem( m_changesView, token->url().url(),
                           token->valueAsString(), token->initialValueAsString(),
                           saved ? i18n( "Yes" ): i18n( "No" ) );
        ++itr;
    }
}

void ChangesView::removeEntry()
{
    QListViewItem *item = m_changesView->selectedItem();

    if ( !item ) {
        KMessageBox::information( this, i18n( "You have to select an item which you want to remove first." ),
                                  i18n( "No Item Selected" ) );
    } else {
        KURL url( item->text( 0 ) );
        EntryToken *token = 0;
        if ( (token=m_entries[url.url()]) ) {
            QStringList lst;
            token->restoreToken( token->url(), lst );
            if ( !m_pendingEntries[ token->url().url() ] ) {
                kdDebug()<<"has to save it again"<<endl;
                kernel->manager()->saveToken( token );
            }
            m_statusLabel->setText( i18n( "Token %1 has been restored" ).arg( item->text( 0 ) ) );
        } else {
            m_statusLabel->setText( i18n( "Token %1 doesn't exist!" ).arg( item->text( 0 ) ) );
        }
    }
    delete item;
}

void ChangesView::generateBackupScript()
{
    QString contents = kernel->manager()->backupScript();

    writeFile( contents );
}

void ChangesView::generatePropagationScript()
{
    QString contents = kernel->manager()->propagationScript();
    writeFile( contents );
}

void ChangesView::writeFile( const QString &contents )
{
    QString fileName = KFileDialog::getSaveFileName();

    if ( !fileName.isEmpty() ) {
        if ( !fileName.endsWith( ".js" ) )
            fileName += ".js";

        if ( QFile::exists( fileName ) ) {
            if ( KMessageBox::questionYesNo( this, i18n( "File %1 exists. "
                                                         "Do you want to overwrite it?" ).arg( fileName ),
                                             i18n( "File Exists" ) ) == KMessageBox::No ) {
                return;
            }
        }

        QFile file( fileName );
        if ( file.open( IO_WriteOnly ) ) {
            QTextStream stream( &file );
            stream << contents << "\n";
            file.close();
        }
    }

}

#include "changesview.moc"
