/**
 * configmanager.cpp
 *
 * Copyright (C)  2004  Zack Rusin <zack@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307  USA
 */
#include "config.h"
#include "configmanager.h"

#include "configurationview.h"
#include "changesdialog.h"
#include "changesview.h"

#include "subsystems/kde/kdesubsystem.h"
#include "subsystems/kde/kdescriptgenerator.h"
#if HAVE_GCONF
#include "subsystems/gnome/gnomesubsystem.h"
#include "subsystems/gnome/gnomescriptgenerator.h"
#endif
#include "interfaces/subsystem.h"
#include "interfaces/scriptgenerator.h"
#include "grouptoken.h"
#include "entrytoken.h"

#include <kmessagebox.h>
#include <kapplication.h>
#include <klocale.h>
#include <kurl.h>

using namespace KConfigEditor;

ConfigManager::ConfigManager( QObject *parent )
    : QObject( parent ), m_waitingParsers( 0 )
{
    initSubsystems();
    initGenerators();
}

void ConfigManager::initSubsystems()
{
    /*
    Subsystem *sub = new KDESubsystem( this );
    m_subsystems.insert( "kde", sub );
    connect( sub, SIGNAL(parsingFinished(Subsystem*)),
             SLOT(slotParsingFinished(Subsystem*)) );
    */
#if HAVE_GCONF
    Subsystem *sub = new GNOMESubsystem( this );
    m_subsystems.insert( "gnome", sub );
    connect( sub, SIGNAL(parsingFinished(Subsystem*)),
             SLOT(slotParsingFinished(Subsystem*)) );
#endif
}

void ConfigManager::initGenerators()
{
    m_generators.insert( "kde", new KDEScriptGenerator() );
#if HAVE_GCONF
    m_generators.insert( "gnome", new GNOMEScriptGenerator() );
#endif
}

void ConfigManager::parse( ConfigurationView *view )
{
    m_waitingParsers = 0;
    QDictIterator<Subsystem> itr( m_subsystems );
    while ( itr.current() ) {
        ++m_waitingParsers;
        itr.current()->parse( view );
        ++itr;
    }
}

void ConfigManager::slotParsingFinished( Subsystem * )
{
    --m_waitingParsers;
    if ( !m_waitingParsers )
        emit parsingFinished();
}

void ConfigManager::save()
{
    QDictIterator<Subsystem> itr( m_subsystems );
    bool saved = false;
    while ( itr.current() ) {
        bool subSaved = itr.current()->save();
        if ( !saved )
            saved = subSaved;
        ++itr;
    }
    if ( saved ) {
        KMessageBox::information( kapp->mainWidget(), i18n("Configuration has been saved." ),
                                  i18n("Configuration Saved") );
    } else {
        KMessageBox::information( kapp->mainWidget(), i18n("No changes to save." ),
                                  i18n("Nothing to Save") );
    }
}

void ConfigManager::showChangesView()
{
    QDict<EntryToken> pendingEntries;
    QDict<EntryToken> sessionChanges;

    QDictIterator<Subsystem> itr( m_subsystems );
    Subsystem *subsystem;
    while ( (subsystem=itr.current()) ) {
        QDict<GroupToken> groups( subsystem->topLevelGroups() );
        QDictIterator<GroupToken> gitr( groups );
        while ( gitr.current() ) {
            QDict<EntryToken> &schanges = gitr.current()->sessionModifiedEntries();
            QDictIterator<EntryToken> sitr( schanges );
            while ( sitr.current() ) {
                sessionChanges.insert( sitr.current()->url().url(), sitr.current() );
                ++sitr;
            }

            if ( gitr.current()->hasPendingChanges() ) {
                QDict<EntryToken> &entries = gitr.current()->modifiedEntries();
                QDictIterator<EntryToken> eitr( entries );
                while ( eitr.current() ) {
                    pendingEntries.insert( eitr.current()->url().url(), eitr.current() );
                    ++eitr;
                }
            }
            ++gitr;
        }
        ++itr;
    }

    if ( sessionChanges.isEmpty() ) {
        KMessageBox::information( kapp->mainWidget(), i18n( "There are no changes to display." ),
                                  i18n( "No Changes" ) );
    } else {
        ChangesDialog *dialog = new ChangesDialog( kapp->mainWidget() );
        dialog->view()->fillChanges( sessionChanges, pendingEntries );
        dialog->show();
    }
}

static QString createScriptHeader()
{
    QString str(
        "#!/usr/bin/env kjscmd\n"
        "# The script has been generated by KConfigEditor\n"
        "# You need KJSEmbed from KDE 3.3 to run this script!\n" );
    return str;
}

QString ConfigManager::backupScript()
{
    bool initialized = false;
    QString scriptText;

    QDictIterator<ScriptGenerator> itr( m_generators );
    while ( itr.current() ) {
        QString contents;
        Subsystem *system = m_subsystems[ itr.current()->protocol() ];
        if ( system )
            contents = itr.current()->backupScript( system->topLevelGroups() ) + "\n";
        else {
            kdWarning()<<"There's no subsystem for \""
                       << itr.current()->protocol() <<"\" generator."<<endl;
        }
        if ( !contents.isEmpty() ) {
            if ( !initialized ) {
                scriptText = createScriptHeader();
                scriptText += "# The script restores configuration to its previous state\n";
                initialized = true;
            }
            scriptText += contents;
        }

        ++itr;
    }
    return scriptText;
}

QString ConfigManager::propagationScript()
{
    bool initialized = false;
    QString scriptText;

    QDictIterator<ScriptGenerator> itr( m_generators );
    while ( itr.current() ) {
        Subsystem *system = m_subsystems[ itr.current()->protocol() ];
        QString contents;
        if ( system )
            contents = itr.current()->propagationScript( system->topLevelGroups() ) + "\n";
        else {
            kdWarning()<<"There's no subsystem for \""
                       << itr.current()->protocol() <<"\" generator."<<endl;
        }
        if ( !contents.isEmpty() ) {
            if ( !initialized ) {
                scriptText = createScriptHeader();
                scriptText += "# The script changes configuration to its current state\n";
                initialized = true;
            }
            scriptText += contents;
        }
        ++itr;
    }
    return scriptText;
}

QString ConfigManager::fullBackupScript()
{
    QString scriptText = createScriptHeader();
    scriptText += "# The script restores full configuration\n";

    QDictIterator<ScriptGenerator> itr( m_generators );
    while ( itr.current() ) {
        Subsystem *system = m_subsystems[ itr.current()->protocol() ];
        if ( system )
            scriptText += itr.current()->fullBackupScript( system->topLevelGroups() ) + "\n";
        else {
            kdWarning()<<"There's no subsystem for \""
                       << itr.current()->protocol() <<"\" generator."<<endl;
        }
        ++itr;
    }
    return scriptText;
}

void ConfigManager::saveToken( EntryToken *token )
{
    QString protocol = token->url().protocol();
    Subsystem *subsystem = m_subsystems[ protocol ];
    Q_ASSERT( subsystem );

    if ( subsystem )
        subsystem->saveToken( token );
}

#include "configmanager.moc"
