/**
 * entrytoken.h
 *
 * Copyright (C)  2004  Zack Rusin <zack@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307  USA
 */
#ifndef ENTRYTOKEN_H
#define ENTRYTOKEN_H

#include "token.h"
#include "grouptoken.h"
#include "treeitems.h"

#include <stdlib.h>

#include <kurl.h>
#include <kdebug.h>

#include <qstringlist.h>
#include <qvaluelist.h>
#include <qvariant.h>
#include <qstring.h>


namespace KConfigEditor {
    class ConfigurationView;

    /**
     * EntryToken is the class representing the base config entries.
     * In incapsulates all features of KConfig and KCfg entries.
     * EntryToken's are formed by merging informations from KConfig
     * and KCFG files.
     */
    class EntryToken : public Token
    {
    public:
        struct Choice {
            QString name;
            QString label;
            QString whatsThis;
        };

        virtual ~EntryToken() { }

        /**
         * Returns the token as the QVariant
         */
        virtual QVariant variant() const =0;
        /**
         * Sets the token from the QVariant
         * @return true if successful
         */
        virtual void setVariant( const QVariant & ) =0;

        /**
         * Returns the initial value of this token.
         */
        virtual QVariant initialValue() const =0;

        /**
         * Returns the initial value of this token as string
         */
        virtual QString initialValueAsString() const =0;

        /**
         * Does the best it can to convert the currently held type
         * to a string representation.
         */
        virtual QString  valueAsString() const =0;

        virtual void setValueFromString( const QString & ) = 0;

        /**
         * Returns the name of the type as a string.
         */
        virtual QString typeName() const = 0;

        /**
         * The type of the value held by the token as a QVariant::Type
         */
        virtual Type    type() const =0;

        bool contains( const QString &str ) const
        {
            return ( ( name().find( str, 0, false ) != -1 ) ||
                     ( valueAsString().find( str, 0, false ) != -1 ) ||
                     ( label().find( str, 0, false ) != -1 )||
                     ( whatsThis().find( str, 0, false ) != -1 )
                );
        }

        GroupToken* parent() const { return m_parent; }

        /**
         * Short description of an entry.
         */
        QString label() const { return m_label; }
        void setLabel( const QString &label ) {
            m_label = label;
        }

        /**
         * Represents a more detailed description of an entry.
         */
        QString whatsThis() const { return m_whatsThis; }
        void setWhatsThis( const QString &whats ) {
            m_whatsThis = whats;
        }

        /**
         * Default value as set in the KCfg file
         */
        QString defaultValue() const { return m_default; }
        void setDefaultValue( const QString &defaultVal ) {
            m_default = defaultVal;
        }

        /**
         * For entries with a predefined set of available entries
         * (enums) represents the set of available options.
         */
        QValueList<Choice> choices() const { return m_choices; }
        void setChoices( const QValueList<Choice> &choices ) {
            m_choices = choices;
        }

        /**
         * Indentifies hidden config entries (set in KCfg files)
         */
        void setHidden( bool hidden ) {
            m_hidden = hidden;
        }
        bool hidden() const { return m_hidden; }

        double min() const { return m_min; }
        void setMin( double min ) {
            m_min = min;
        }

        double max() const { return m_max; }
        void setMax( double max ) {
            m_max = max;
        }

        QStringList values() const { return m_values; }
        void setValues( const QStringList &values ) {
            m_values = values;
        }

        /**
         * Is the token taken from an enviroment variable or script.
         */
        void setFromEnvironment( bool fromEnv ) {
            m_fromEnv = fromEnv;
        }
        bool fromEnvironment() const { return m_fromEnv; }

        bool hasPendingChanges() const { return m_modified; }
        void prunePendingChanges() { m_modified = false; }


        void attach( ConfigurationView * ) {
            //m_view = new EntryItem( static_cast<GroupItem*>( m_parent->view() ), this );
        }

    protected:
        friend class GroupToken;
        EntryToken( GroupToken *parent, const QString &name,
                    bool hidden=false )
            : Token( parent->url() ), m_parent( parent ),
              m_max( 0 ), m_min( 0 ),
              m_hidden( hidden ), m_fromEnv( false )
        {
            if ( name.startsWith( "/" ) )
                m_url.setPath( name );
            else
                m_url.addPath( name );
        }


        void copyFromToken( EntryToken *token )
        {
            m_parent = token->m_parent;
            m_label  = token->m_label;
            m_whatsThis = token->m_whatsThis;
            m_choices = token->m_choices;
            m_default = token->m_default;
            m_hidden = token->m_hidden;
            // token->m_immutable;
            m_fromEnv = token->m_fromEnv;
            setValueFromString( token->valueAsString() );
        }
    protected:
        GroupToken*  m_parent;

    private:
        QString      m_label;
        QString      m_whatsThis;
        QValueList<Choice> m_choices;
        QString      m_default;
        double       m_max;
        double       m_min;
        QStringList  m_values;
        bool         m_hidden;
        bool         m_fromEnv;
    };

    /**
     * Hack to not have to write a lot of redundant code in entries.h and not
     * still be able to hold lists of EntryToken's (if entrytoken would be a template
     * we wouldn't be able to do that)
     */
    template <typename T>
    class EntriesBase : public EntryToken
    {
    public:
        T value() const { return m_value; }
        void setValue( const T &val ) {
            m_value = val;
            finishSetting();
        }

        virtual void restoreToken( const KURL &item, QStringList & ) {
            if ( item == url() ) {
                kdDebug()<<"restoring token at "<<item.prettyURL()<<endl;
                m_value = m_initialValue;
                finishSetting();
            }
        }

    protected:
        EntriesBase( GroupToken *parent, const QString &name,
                     bool hidden=false )
            : EntryToken( parent, name, hidden ), m_initialized( false )
        {
        }

        void finishSetting() {
            if ( !m_initialized ) {
                m_initialValue = m_value;
                m_initialized = true;
            } else {
                if ( m_value != m_initialValue ) {
                    m_parent->entryModified( m_parent, this );
                    m_modified = true;
                } else {
                    m_parent->entryRestored( m_parent, this );
                    m_modified = false;
                }
            }
        }


    protected:
        bool m_initialized;
        T m_value;
        T m_initialValue;
    };
}

#endif
