/*
 * KHeiseReg
 *
 * A utility to search for articles within the Heise register.
 *
 * Copyright (C) 2002 Oliver Gantz <Oliver.Gantz@ePost.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <qlayout.h>
#include <qsize.h>
#include <qgroupbox.h>
#include <qiconset.h>

#include <klocale.h>
#include <kapplication.h>
#include <kpushbutton.h>
#include <kstdguiitem.h>
#include <kiconloader.h>

#include "global.h"
#include "searchdlg.h"



SearchDlg::SearchDlg(QWidget *parent, const char *name): KDialog(parent, name)
{
	QBoxLayout *vbox, *hbox, *svbox, *shbox;
	QGridLayout *grid;
	QLabel *label;
	QGroupBox *gbox;
	KPushButton *bhelp, *bdefault, *bsearch, *bclose;
	QSize size;

	config = kapp->config();

	vbox = new QVBoxLayout(this, marginHint());

	grid = new QGridLayout(6, 3, spacingHint());
	vbox->addLayout(grid);

	label = new QLabel(i18n("&Bywords:"), this);
	label->setFixedSize(label->sizeHint());
	grid->addWidget(label, 0, 0, AlignVCenter | AlignRight);

	bywords = new QLineEdit(this);
	label->setBuddy(bywords);
	bywords->setMinimumSize(bywords->sizeHint());
	grid->addMultiCellWidget(bywords, 0, 0, 1, 5);

	label = new QLabel(i18n("&Authors:"), this);
	label->setFixedSize(label->sizeHint());
	grid->addWidget(label, 1, 0, AlignVCenter | AlignRight);

	authors = new QLineEdit(this);
	label->setBuddy(authors);
	authors->setMinimumSize(authors->sizeHint());
	grid->addMultiCellWidget(authors, 1, 1, 1, 5);

	label = new QLabel(i18n("&Editors:"), this);
	label->setFixedSize(label->sizeHint());
	grid->addWidget(label, 2, 0, AlignVCenter | AlignRight);

	editors = new QLineEdit(this);
	label->setBuddy(editors);
	editors->setMinimumSize(editors->sizeHint());
	grid->addWidget(editors, 2, 1);

	label = new QLabel(i18n("E&ditions:"), this);
	label->setFixedSize(label->sizeHint());
	grid->addWidget(label, 2, 2, AlignVCenter | AlignRight);

	editions = new QLineEdit(this);
	label->setBuddy(editions);
	editions->setMinimumSize(editions->sizeHint());
	grid->addWidget(editions, 2, 3);

	label = new QLabel(i18n("&Years:"), this);
	label->setFixedSize(label->sizeHint());
	grid->addWidget(label, 2, 4, AlignVCenter | AlignRight);

	years = new QLineEdit(this);
	label->setBuddy(years);
	years->setMinimumSize(years->sizeHint());
	grid->addWidget(years, 2, 5);

	grid->setColStretch(1, 1);
	grid->setColStretch(3, 1);
	grid->setColStretch(5, 1);

	hbox = new QHBoxLayout(spacingHint());
	vbox->addLayout(hbox);

	gbox = new QGroupBox(i18n("Magazines"), this);

	svbox = new QVBoxLayout(gbox, spacingHint());
	svbox->addSpacing(fontMetrics().height());

	shbox = new QHBoxLayout(spacingHint());
	svbox->addLayout(shbox);
	ctmag = new QCheckBox(i18n("c'&t Magazine"), gbox);
	ctmag->setFixedSize(ctmag->sizeHint());
	shbox->addWidget(ctmag);
	shbox->addStretch();

	shbox = new QHBoxLayout(spacingHint());
	svbox->addLayout(shbox);
	ixmag = new QCheckBox(i18n("&iX Magazine"), gbox);
	ixmag->setFixedSize(ixmag->sizeHint());
	shbox->addWidget(ixmag);
	shbox->addStretch();

	svbox->addStretch();

	svbox->activate();

	hbox->addWidget(gbox);

	gbox = new QGroupBox(i18n("Options"), this);

	svbox = new QVBoxLayout(gbox, spacingHint());
	svbox->addSpacing(fontMetrics().height());

	shbox = new QHBoxLayout(spacingHint());
	svbox->addLayout(shbox);
	case_sensitive = new QCheckBox(i18n("Ca&se sensitive"), gbox);
	case_sensitive->setFixedSize(case_sensitive->sizeHint());
	shbox->addWidget(case_sensitive);
	shbox->addStretch();

	shbox = new QHBoxLayout(spacingHint());
	svbox->addLayout(shbox);
	fuzzy = new QCheckBox(i18n("&Fuzzy"), gbox);
	fuzzy->setFixedSize(fuzzy->sizeHint());
	shbox->addWidget(fuzzy);
	shbox->addStretch();

	shbox = new QHBoxLayout(spacingHint());
	svbox->addLayout(shbox);
	shbox->addSpacing(19);
	min_threshold = new QLabel(QString::fromLatin1("0%"), gbox);
	min_threshold->setFixedSize(min_threshold->sizeHint());
	min_threshold->setEnabled(false);
	shbox->addWidget(min_threshold);
	threshold = new QSlider(0, 100, 10, 0, QSlider::Horizontal, gbox);
	threshold->setTickmarks(QSlider::Below);
	threshold->setTickInterval(10);
	threshold->setMinimumSize(threshold->sizeHint());
	threshold->setEnabled(false);
	shbox->addWidget(threshold);
	max_threshold = new QLabel(QString::fromLatin1("100%"), gbox);
	max_threshold->setFixedSize(max_threshold->sizeHint());
	max_threshold->setEnabled(false);
	shbox->addWidget(max_threshold);

	shbox = new QHBoxLayout(spacingHint());
	svbox->addLayout(shbox);
	cumulative = new QCheckBox(i18n("C&umulative"), gbox);
	cumulative->setFixedSize(cumulative->sizeHint());
	shbox->addWidget(cumulative);
	shbox->addStretch();

	svbox->addStretch();
	svbox->activate();

	hbox->addWidget(gbox);

	vbox->addStretch();

	hbox = new QHBoxLayout(spacingHint());
	vbox->addLayout(hbox);

	bhelp = new KPushButton(KStdGuiItem::help(), this);
	bdefault = new KPushButton(KStdGuiItem::defaults(), this);
	bdefault->setFixedWidth(bdefault->sizeHint().width());
	bsearch = new KPushButton(SmallIconSet(QString::fromLatin1("find")), i18n("&Search"), this);
	bsearch->setDefault(true);
	bclose = new KPushButton(KStdGuiItem::close(), this);
	bclose->setFixedWidth(bclose->sizeHint().width());

	size = bhelp->sizeHint().expandedTo(bsearch->sizeHint());
	bdefault->setFixedHeight(size.height());
	bhelp->setFixedSize(size);
	bsearch->setFixedSize(size);
	bclose->setFixedHeight(size.height());

	hbox->addWidget(bhelp);
	hbox->addWidget(bdefault);
	hbox->addStretch();
	hbox->addWidget(bsearch);
	hbox->addWidget(bclose);

	vbox->activate();

	setCaption(kapp->makeStdCaption(i18n("Search")));

	connect(fuzzy, SIGNAL(toggled(bool)), this, SLOT(slotFuzzyToggled(bool)));
	connect(bhelp, SIGNAL(clicked()), this, SLOT(slotHelp()));
	connect(bdefault, SIGNAL(clicked()), this, SLOT(slotDefault()));
	connect(bsearch, SIGNAL(clicked()), this, SLOT(slotSearch()));
	connect(bclose, SIGNAL(clicked()), this, SLOT(close()));

	slotDefault();
}


void SearchDlg::slotFuzzyToggled(bool on)
{
	min_threshold->setEnabled(on);
	threshold->setEnabled(on);
	max_threshold->setEnabled(on);
}


void SearchDlg::slotHelp()
{
	kapp->invokeHelp(QString::fromLatin1("search-help"));
}


void SearchDlg::slotDefault()
{
	config->setGroup(QString::fromLatin1(ID_PREF_GROUP_SEARCH));
	bywords->setText(config->readEntry(ID_PREF_SEA_BYWORDS, QString::fromLatin1(PREF_SEA_BYWORDS)));
	authors->setText(config->readEntry(ID_PREF_SEA_AUTHORS, QString::fromLatin1(PREF_SEA_AUTHORS)));
	editors->setText(config->readEntry(ID_PREF_SEA_EDITORS, QString::fromLatin1(PREF_SEA_EDITORS)));
	editions->setText(config->readEntry(ID_PREF_SEA_EDITIONS, QString::fromLatin1(PREF_SEA_EDITIONS)));
	years->setText(config->readEntry(ID_PREF_SEA_YEARS, QString::fromLatin1(PREF_SEA_YEARS)));
	ctmag->setChecked(config->readBoolEntry(ID_PREF_SEA_CTMAG, PREF_SEA_CTMAG));
	ixmag->setChecked(config->readBoolEntry(ID_PREF_SEA_IXMAG, PREF_SEA_IXMAG));
	case_sensitive->setChecked(config->readBoolEntry(ID_PREF_SEA_CASE, PREF_SEA_CASE));
	fuzzy->setChecked(config->readBoolEntry(ID_PREF_SEA_FUZZY, PREF_SEA_FUZZY));
	cumulative->setChecked(config->readBoolEntry(ID_PREF_SEA_CUMUL, PREF_SEA_CUMUL));
	threshold->setValue(config->readNumEntry(ID_PREF_SEA_THRESHOLD, PREF_SEA_THRESHOLD));
}


void SearchDlg::slotSearch()
{
	unsigned char mag = REG_MAGAZINE_NONE;

	mask.setBywords(bywords->text());
	mask.setAuthors(authors->text());
	mask.setEditors(editors->text());
	mask.setEditions(editions->text());
	mask.setYears(years->text());
	if (ctmag->isChecked())
		mag |= REG_MAGAZINE_CT;
	if (ixmag->isChecked())
		mag |= REG_MAGAZINE_IX;
	mask.setMagazines(mag);
	mask.setCaseSensitive(case_sensitive->isChecked());
	mask.setFuzzy(fuzzy->isChecked());
	mask.setCumulative(cumulative->isChecked());
	mask.setThreshold(threshold->value());

	emit search(mask);
}
