/***************************************************************************
                          kbudget.cpp  -  description
                             -------------------
    begin                : Sat Aug 10 16:03:35 CST 2002
    copyright            : (C) 2002 by Richard Garand
    email                : richard@garandnet.net
    $Id: kbudget.cpp,v 1.15 2002/09/06 23:39:58 richard Exp $
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// include files for QT
#include <qdir.h>
#include <qprinter.h>
#include <qpainter.h>

// include files for KDE
#include <kiconloader.h>
#include <kmessagebox.h>
#include <kfiledialog.h>
#include <kmenubar.h>
#include <kstatusbar.h>
#include <klocale.h>
#include <kconfig.h>
#include <kstdaction.h>
#include <kiconloader.h>

// application specific includes
#include "kbudget.h"
#include "kbudgetview.h"
#include "kbudgetdoc.h"
#include "knewaccount.h"
#include "knewtransaction.h"

#define ID_STATUS_MSG 1

KBudgetApp::KBudgetApp(QWidget* , const char* name):KMainWindow(0, name)
{
  config=kapp->config();

  ///////////////////////////////////////////////////////////////////
  // call inits to invoke all other construction parts
  initStatusBar();
  initActions();
  initDocument();
  initView();
	
  readOptions();
  if ( firstTime )
    KMessageBox::information(0, "KBudget is still in early development. It already has some working features, but other things will be missing, broken, or inconsistent.", "Warning");
  
  ///////////////////////////////////////////////////////////////////
  // disable actions at startup
//  fileSave->setEnabled(false);
//  fileSaveAs->setEnabled(false);
  filePrint->setEnabled(false);
  editCut->setEnabled(false);
  editCopy->setEnabled(false);
  editPaste->setEnabled(false);
  viewYear->setEnabled(false);
  viewSpecial->setEnabled(false);
  accountDelete->setEnabled(false);
  accountsRename->setEnabled(false);
}

KBudgetApp::~KBudgetApp()
{

}

void KBudgetApp::initActions()
{
  fileNewWindow = new KAction(i18n("New &Window"), 0, 0, this, SLOT(slotFileNewWindow()), actionCollection(),"file_new_window");
  fileNew = KStdAction::openNew(this, SLOT(slotFileNew()), actionCollection());
  fileOpen = KStdAction::open(this, SLOT(slotFileOpen()), actionCollection());
  fileOpenRecent = KStdAction::openRecent(this, SLOT(slotFileOpenRecent(const KURL&)), actionCollection());
  fileSave = KStdAction::save(this, SLOT(slotFileSave()), actionCollection());
  fileSaveAs = KStdAction::saveAs(this, SLOT(slotFileSaveAs()), actionCollection());
  fileClose = KStdAction::close(this, SLOT(slotFileClose()), actionCollection());
  filePrint = KStdAction::print(this, SLOT(slotFilePrint()), actionCollection());
  fileQuit = KStdAction::quit(this, SLOT(slotFileQuit()), actionCollection());
  editCut = KStdAction::cut(this, SLOT(slotEditCut()), actionCollection());
  editCopy = KStdAction::copy(this, SLOT(slotEditCopy()), actionCollection());
  editPaste = KStdAction::paste(this, SLOT(slotEditPaste()), actionCollection());
  viewYear = new KAction(i18n("This &Year"), 0, 0, this, SLOT(slotViewYear()), actionCollection(), "view_year");
  viewMonth = new KAction(i18n("This &Month"), 0, 0, this, SLOT(slotViewMonth()), actionCollection(), "view_month");
  viewSpecial = new KAction(i18n("Select &Period"), 0, 0, this, SLOT(slotViewSpecial()), actionCollection(), "select_period");
  accountsNew = new KAction(i18n("&New Account"), "folder_new", 0, this, SLOT(slotAccountsNew()), actionCollection(), "accounts_new");
  accountsRename = new KAction(i18n("&Rename account"), "edit", 0, 0, 0, actionCollection(), "accounts_rename");
  accountDelete = new KAction(i18n("&Delete Account"), "editdelete", 0, this, SLOT(slotDeleteAccount()), actionCollection(), "accounts_delete");
  transactionsNew = new KAction(i18n("&New Transaction"), "mail_forward", 0, this, SLOT(slotTransactionsNew()), actionCollection(), "transactions_new");
  viewToolBar = KStdAction::showToolbar(this, SLOT(slotViewToolBar()), actionCollection());
  viewStatusBar = KStdAction::showStatusbar(this, SLOT(slotViewStatusBar()), actionCollection());

  fileNewWindow->setStatusText(i18n("Opens a new application window"));
  fileNew->setStatusText(i18n("Creates a new document"));
  fileOpen->setStatusText(i18n("Opens an existing document"));
  fileOpenRecent->setStatusText(i18n("Opens a recently used file"));
  fileSave->setStatusText(i18n("Saves the actual document"));
  fileSaveAs->setStatusText(i18n("Saves the actual document as..."));
  fileClose->setStatusText(i18n("Closes the actual document"));
  filePrint ->setStatusText(i18n("Prints out the actual document"));
  fileQuit->setStatusText(i18n("Quits the application"));
  editCut->setStatusText(i18n("Cuts the selected section and puts it to the clipboard"));
  editCopy->setStatusText(i18n("Copies the selected section to the clipboard"));
  editPaste->setStatusText(i18n("Pastes the clipboard contents to actual position"));
  viewYear->setStatusText(i18n("Opens the summary for the current year"));
  viewMonth->setStatusText(i18n("Opens the summary for the current month"));
  viewSpecial->setStatusText(i18n("Opens the summary for a selected period"));
  accountsNew->setStatusText(i18n("Creates a new acccount"));
  accountsRename->setStatusText(i18n("Renames the currently selected account"));
  transactionsNew->setStatusText(i18n("Creates a new transaction"));
  viewToolBar->setStatusText(i18n("Enables/disables the toolbar"));
  viewStatusBar->setStatusText(i18n("Enables/disables the statusbar"));
  accountDelete->setStatusText(i18n("Deletes the currently selected account"));

  // use the absolute path to your kbudgetui.rc file for testing purpose in createGUI();
  createGUI();

}


void KBudgetApp::initStatusBar()
{
  ///////////////////////////////////////////////////////////////////
  // STATUSBAR
  // TODO: add your own items you need for displaying current application status.
  statusBar()->insertItem(i18n("Ready."), ID_STATUS_MSG);
}

void KBudgetApp::initDocument()
{
  doc = new KBudgetDoc(this);
  connect(doc, SIGNAL(saveableStatus(bool)), SLOT(slotEnableSave(bool)));
  doc->newDocument();
}

void KBudgetApp::initView()
{ 
  ////////////////////////////////////////////////////////////////////
  // create the main widget here that is managed by KTMainWindow's view-region and
  // connect the widget to your document to display document contents.

  view = new KBudgetView(this);
  connect(view, SIGNAL(accountSelectionStatus(bool)), SLOT(slotEnableAccountFunctions(bool)));
  connect(accountsRename, SIGNAL(activated()), view, SLOT(slotRenameAccount()));
  doc->addView(view);
  setCentralWidget(view);	
  setCaption(doc->URL().fileName(),false);

}

void KBudgetApp::openDocumentFile(const KURL& url)
{
  slotStatusMsg(i18n("Opening file..."));

  doc->openDocument(url);
  fileOpenRecent->addURL(url);
  slotStatusMsg(i18n("Ready."));
}


KBudgetDoc *KBudgetApp::getDocument() const
{
  return doc;
}

void KBudgetApp::saveOptions()
{	
  config->setGroup("General Options");
  config->writeEntry("Geometry", size());
  config->writeEntry("Show Toolbar", viewToolBar->isChecked());
  config->writeEntry("Show Statusbar",viewStatusBar->isChecked());
  config->writeEntry("ToolBarPos", (int) toolBar("mainToolBar")->barPos());
  fileOpenRecent->saveEntries(config,"Recent Files");
}


void KBudgetApp::readOptions()
{
	
  config->setGroup("General Options");

  // bar status settings
  bool bViewToolbar = config->readBoolEntry("Show Toolbar", true);
  viewToolBar->setChecked(bViewToolbar);
  slotViewToolBar();

  bool bViewStatusbar = config->readBoolEntry("Show Statusbar", true);
  viewStatusBar->setChecked(bViewStatusbar);
  slotViewStatusBar();


  // bar position settings
  KToolBar::BarPosition toolBarPos;
  toolBarPos=(KToolBar::BarPosition) config->readNumEntry("ToolBarPos", KToolBar::Top);
  toolBar("mainToolBar")->setBarPos(toolBarPos);
	
  // initialize the recent file list
  fileOpenRecent->loadEntries(config,"Recent Files");

  QSize size=config->readSizeEntry("Geometry");
  if(!size.isEmpty())
  {
    resize(size);
  }

  firstTime = config->readBoolEntry("First time", true);
  config->writeEntry("First time", false);
}

void KBudgetApp::saveProperties(KConfig *_cfg)
{
  if(doc->URL().fileName()!=i18n("Untitled") && !doc->isModified())
  {
    // saving to tempfile not necessary

  }
  else
  {
    KURL url=doc->URL();	
    _cfg->writeEntry("filename", url.url());
    _cfg->writeEntry("modified", doc->isModified());
    QString tempname = kapp->tempSaveName(url.url());
    QString tempurl= KURL::encode_string(tempname);
    KURL _url(tempurl);
    doc->saveDocument(_url);
  }
}


void KBudgetApp::readProperties(KConfig* _cfg)
{
  QString filename = _cfg->readEntry("filename", "");
  KURL url(filename);
  bool modified = _cfg->readBoolEntry("modified", false);
  if(modified)
  {
    bool canRecover;
    QString tempname = kapp->checkRecoverFile(filename, canRecover);
    KURL _url(tempname);
  	
    if(canRecover)
    {
      doc->openDocument(_url);
      doc->setModified();
      setCaption(_url.fileName(),true);
      QFile::remove(tempname);
    }
  }
  else
  {
    if(!filename.isEmpty())
    {
      doc->openDocument(url);
      setCaption(url.fileName(),false);
    }
  }
}		

bool KBudgetApp::queryClose()
{
  return doc->saveModified();
}

bool KBudgetApp::queryExit()
{
  saveOptions();
  return true;
}

/////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATION
/////////////////////////////////////////////////////////////////////

void KBudgetApp::slotFileNewWindow()
{
  slotStatusMsg(i18n("Opening a new application window..."));
	
  KBudgetApp *new_window= new KBudgetApp();
  new_window->show();

  slotStatusMsg(i18n("Ready."));
}

void KBudgetApp::slotFileNew()
{
  slotStatusMsg(i18n("Creating new document..."));

  if(!doc->saveModified())
  {
     // here saving wasn't successful

  }
  else
  {	
    doc->newDocument();		
    setCaption(doc->URL().fileName(), false);
  }

  slotStatusMsg(i18n("Ready."));
}

void KBudgetApp::slotFileOpen()
{
  slotStatusMsg(i18n("Opening file..."));
	
  if(!doc->saveModified())
  {
     // here saving wasn't successful

  }
  else
  {	
    KURL url=KFileDialog::getOpenURL(QString::null,
        i18n("*.kbudget|KBudget files\n*|All files"), this, i18n("Open File..."));
    if(!url.isEmpty())
    {
      if ( doc->openDocument(url) ) {
        setCaption(url.fileName(), false);
        fileOpenRecent->addURL( url );
      } else {
        KMessageBox mb;
        mb.sorry(this, "Error: couldn't open file", "Error");
      }
    }
  }
  slotStatusMsg(i18n("Ready."));
}

void KBudgetApp::slotFileOpenRecent(const KURL& url)
{
  slotStatusMsg(i18n("Opening file..."));
	
  if(!doc->saveModified())
  {
     // here saving wasn't successful
  }
  else
  {
    if ( doc->openDocument(url) )
      setCaption(url.fileName(), false);
    else {
      KMessageBox mb;
      mb.sorry(this, "Error: couldn't open file", "Error");
    }
  }
                             void slotEnableDelete(bool enabled);
  slotStatusMsg(i18n("Ready."));
}

void KBudgetApp::slotFileSave()
{
  slotStatusMsg(i18n("Saving file..."));

  if ( doc->URL().filename() == i18n("Untitled") )
    slotFileSaveAs();
  else
    doc->saveDocument(doc->URL());

  slotStatusMsg(i18n("Ready."));
}

void KBudgetApp::slotFileSaveAs()
{
  slotStatusMsg(i18n("Saving file with a new filename..."));

  KURL url=KFileDialog::getSaveURL(QDir::currentDirPath(),
        i18n("*.kbudget|KBudget files\n*|All files"), this, i18n("Save as..."));
  if(!url.isEmpty())
  {
    if ( url.fileName().find('.') == -1 )
      url.setPath(url.path().append(".kbudget"));
    doc->saveDocument(url);
    fileOpenRecent->addURL(url);
    setCaption(url.fileName(),doc->isModified());
  }

  slotStatusMsg(i18n("Ready."));
}

void KBudgetApp::slotFileClose()
{
  slotStatusMsg(i18n("Closing file..."));
	
  close();

  slotStatusMsg(i18n("Ready."));
}

void KBudgetApp::slotFilePrint()
{
  slotStatusMsg(i18n("Printing..."));

  QPrinter printer;
  if (printer.setup(this))
  {
    view->print(&printer);
  }

  slotStatusMsg(i18n("Ready."));
}

void KBudgetApp::slotFileQuit()
{
  slotStatusMsg(i18n("Exiting..."));
  saveOptions();
  // close the first window, the list makes the next one the first again.
  // This ensures that queryClose() is called on each window to ask for closing
  KMainWindow* w;
  if(memberList)
  {
    for ( unsigned int i= 0; i< memberList->count(); i++ )
    {
      // only close the window if the closeEvent is accepted. If the user presses Cancel on the saveModified() dialog,
      // the window and the application stay open.
      if ( memberList->at(i)->close() )
	break;
    }
  }
}

void KBudgetApp::slotEditCut()
{
  slotStatusMsg(i18n("Cutting selection..."));

  slotStatusMsg(i18n("Ready."));
}

void KBudgetApp::slotEditCopy()
{
  slotStatusMsg(i18n("Copying selection to clipboard..."));

  slotStatusMsg(i18n("Ready."));
}

void KBudgetApp::slotEditPaste()
{
  slotStatusMsg(i18n("Inserting clipboard contents..."));

  slotStatusMsg(i18n("Ready."));
}

void KBudgetApp::slotViewYear()
{
  slotStatusMsg(i18n("Peparing data..."));
  
  slotStatusMsg(i18n("Ready."));
}

void KBudgetApp::slotViewMonth()
{
  slotStatusMsg(i18n("Peparing data..."));

  QDate date = QDate::currentDate();
  view->selectMonth(date.year(), date.month() - 1);

  slotStatusMsg(i18n("Ready."));
}

void KBudgetApp::slotViewSpecial()
{
  slotStatusMsg(i18n("Peparing data..."));

  slotStatusMsg(i18n("Ready."));
}

void KBudgetApp::slotAccountsNew()
{
  slotStatusMsg(i18n("Creating a new account..."));
  (new KNewAccount(this))->show();
  slotStatusMsg(i18n("Ready."));
}

void KBudgetApp::slotTransactionsNew()
{
  slotStatusMsg(i18n("Creating a new transaction..."));
  (new KNewTransaction(this))->show();
  slotStatusMsg(i18n("Ready."));
}

void KBudgetApp::slotViewToolBar()
{
  slotStatusMsg(i18n("Toggling toolbar..."));
  ///////////////////////////////////////////////////////////////////
  // turn Toolbar on or off
  if(!viewToolBar->isChecked())
  {
    toolBar("mainToolBar")->hide();
  }
  else
  {
    toolBar("mainToolBar")->show();
  }		

  slotStatusMsg(i18n("Ready."));
}

void KBudgetApp::slotViewStatusBar()
{
  slotStatusMsg(i18n("Toggle the statusbar..."));
  ///////////////////////////////////////////////////////////////////
  //turn Statusbar on or off
  if(!viewStatusBar->isChecked())
  {
    statusBar()->hide();
  }
  else
  {
    statusBar()->show();
  }

  slotStatusMsg(i18n("Ready."));
}


void KBudgetApp::slotStatusMsg(const QString &text)
{
  ///////////////////////////////////////////////////////////////////
  // change status message permanently
  statusBar()->clear();
  statusBar()->changeItem(text, ID_STATUS_MSG);
}

void KBudgetApp::slotEnableSave(bool enabled)
{
  fileSave->setEnabled(enabled);
}

void KBudgetApp::slotDeleteAccount()
{
  doc->deleteAccount(view->currentAccount());
}

void KBudgetApp::slotEnableAccountFunctions(bool enabled)
{
  accountDelete->setEnabled(enabled);
  accountsRename->setEnabled(enabled);
}
