/***************************************************************************
                          kbundoaction.cpp  -  description
                             -------------------
    begin                : Tue Sep 10 2002
    copyright            : (C) 2002 by Richard Garand
    email                : richard@garandnet.net
    $Id: kbundoaction.cpp,v 1.1 2002/09/13 18:43:08 richard Exp $
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qdatetime.h>

#include "kbundoaction.h"
#include "kbudgetdoc.h"
#include "account.h"
#include "transaction.h"

KBUndoAction::KBUndoAction(KBudgetDoc* doc)
  : m_doc(doc)
{
}

KBUndoAction::~KBUndoAction()
{
}

KBUndoGroup::KBUndoGroup(KBudgetDoc* doc)
  : KBUndoAction(doc)
{
}

KBUndoGroup::~KBUndoGroup()
{
}

void KBUndoGroup::undo()
{
  for ( list.first(); list.current(); list.next() )
    list.current()->undo();
}

void KBUndoGroup::redo()
{
  for ( list.first(); list.current(); list.next() )
    list.current()->redo();
}

void KBUndoGroup::add(KBUndoAction* act)
{
  list.append(act);
}

KBAccountUndo::KBAccountUndo(KBudgetDoc* doc, actions action, int id)
  : KBUndoAction(doc), m_action(action), m_id(id), savedAcct(0)
{
  if ( action == UA_DEL ) {
    // save a copy of the account
    allocatePtr();
    *savedAcct = doc->getAccount(id);
  } else if ( action == UA_RENAME )
    savedName = doc->getAccount(id).name();
  else if ( action == UA_STBALANCE )
    savedValue = doc->getAccount(id).startingBalance();
}

KBAccountUndo::~KBAccountUndo()
{
  if ( savedAcct )
    delete savedAcct;
}

void KBAccountUndo::undo()
{
  switch ( m_action ) {
  case UA_ADD:
    storeAccount();
    m_doc->deleteAccount(m_id);
    break;
  case UA_DEL:
    recreateAccount();
    break;
  case UA_RENAME: {
    QString tempstr = savedName;
    savedName = m_doc->getAccount(m_id).name();
    m_doc->getAccount(m_id).setName(tempstr);
    // TODO: remove this
    m_doc->signalAcctChanged(m_id);
    break;
  } case UA_STBALANCE: {
    float tempValue = savedValue;
    savedValue = m_doc->getAccount(m_id).startingBalance();
    m_doc->getAccount(m_id).setStartingBalance(tempValue);
    // TODO: remove this
    m_doc->signalAcctChanged(m_id);
    break;
  } default:
    break;
  }
}

void KBAccountUndo::redo()
{
  switch ( m_action ) {
  case UA_ADD:
    recreateAccount();
    break;
  case UA_DEL:
    storeAccount();
    m_doc->deleteAccount(m_id);
    break;
  case UA_RENAME: {
    QString tempstr = savedName;
    savedName = m_doc->getAccount(m_id).name();
    m_doc->getAccount(m_id).setName(tempstr);
    // TODO: remove this
    m_doc->signalAcctChanged(m_id);
    break;
  } case UA_STBALANCE: {
    float tempValue = savedValue;
    savedValue = m_doc->getAccount(m_id).startingBalance();
    m_doc->getAccount(m_id).setStartingBalance(tempValue);
    // TODO: remove this
    m_doc->signalAcctChanged(m_id);
    break;
  } default:
    break;
  }
}

void KBAccountUndo::allocatePtr()
{
  if ( savedAcct )
    delete savedAcct;
  savedAcct = new Account;
}

void KBAccountUndo::storeAccount()
{
  allocatePtr();
  *savedAcct = m_doc->getAccount(m_id);
}

void KBAccountUndo::recreateAccount()
{
  if ( !savedAcct )
    return;
  m_doc->insertAccount(m_id, savedAcct);
}

KBTransactionUndo::KBTransactionUndo(KBudgetDoc* doc, actions action, int id)
  : KBUndoAction(doc), m_action(action), m_id(id), saved(0)
{
  if ( action == UT_DEL ) {
    // save a copy of the transaction
    allocatePtr();
    *saved = doc->getTransaction(id);
  } else if ( action == UT_DESCRIPTION )
    savedDesc = doc->getTransaction(id).description();
  else if ( action == UT_MODIFIED ) {
    savedDesc = doc->getTransaction(id).description();
    savedValue = doc->getTransaction(id).value();
    savedDate = doc->getTransaction(id).date();
  }
}

KBTransactionUndo::~KBTransactionUndo()
{
  if ( saved )
    delete saved;
}

void KBTransactionUndo::undo()
{
  switch ( m_action ) {
  case UT_ADD:
    store();
    m_doc->deleteTransaction(m_id);
    break;
  case UT_DEL:
    recreate();
    break;
  case UT_DESCRIPTION: {
    QString tempstr = savedDesc;
    savedDesc = m_doc->getTransaction(m_id).description();
    m_doc->getTransaction(m_id).setDescription(tempstr);
    // TODO: remove this
    m_doc->signalAcctChanged(m_doc->getTransaction(m_id).from());
    m_doc->signalAcctChanged(m_doc->getTransaction(m_id).to());
    break;
  } case UT_MODIFIED: {
    float tempVal = savedValue;
    QDate tempDate = savedDate;
    QString tempstr = savedDesc;
    Transaction& tr = m_doc->getTransaction(m_id);

    savedDesc = tr.description();
    savedValue = tr.value();
    savedDate = tr.date();

    tr.setData(tempVal, tempDate, tempstr);
    
    // TODO: remove this
    m_doc->signalAcctChanged(m_doc->getTransaction(m_id).from());
    m_doc->signalAcctChanged(m_doc->getTransaction(m_id).to());
    break;
  } default:
    break;
  }
}

void KBTransactionUndo::redo()
{
  switch ( m_action ) {
  case UT_ADD:
    recreate();
    break;
  case UT_DEL:
    store();
    m_doc->deleteTransaction(m_id);
    break;
  case UT_DESCRIPTION: {
    QString tempstr = savedDesc;
    savedDesc = m_doc->getTransaction(m_id).description();
    m_doc->getTransaction(m_id).setDescription(tempstr);
    // TODO: remove this
    m_doc->signalAcctChanged(m_doc->getTransaction(m_id).from());
    m_doc->signalAcctChanged(m_doc->getTransaction(m_id).to());
    break;
  } case UT_MODIFIED: {
    float tempVal = savedValue;
    QDate tempDate = savedDate;
    QString tempstr = savedDesc;
    Transaction& tr = m_doc->getTransaction(m_id);

    savedDesc = tr.description();
    savedValue = tr.value();
    savedDate = tr.date();

    tr.setData(tempVal, tempDate, tempstr);

    // TODO: remove this
    m_doc->signalAcctChanged(m_doc->getTransaction(m_id).from());
    m_doc->signalAcctChanged(m_doc->getTransaction(m_id).to());
    break;
  } default:
    break;
  }
}

void KBTransactionUndo::allocatePtr()
{
  if ( saved )
    delete saved;
  saved = new Transaction;
}

void KBTransactionUndo::store()
{
  allocatePtr();
  *saved = m_doc->getTransaction(m_id);
}

void KBTransactionUndo::recreate()
{
  if ( !saved )
    return;
  m_doc->insertTransaction(m_id, saved);
  m_doc->getAccount(saved->from()).addTransaction(m_id);
  m_doc->getAccount(saved->to()).addTransaction(m_id);
}
