/* This file is part of the KDE project
   Copyright (C) 2004 Jaroslaw Staniek <js@iidea.pl>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#include <kexidb/utils.h>
#include <kexidb/cursor.h>

#include <qmap.h>

#include <klocale.h>

using namespace KexiDB;

//! Cache
QMap< uint, TypeGroupList > tlist;
QMap< uint, QStringList > nlist;
QMap< uint, QStringList > slist;

static void initList()
{
	if (!tlist.isEmpty())
		return;

	for (uint t=0; t<=KexiDB::Field::LastType; t++) {
		const uint tg = KexiDB::Field::typeGroup( t );
		TypeGroupList list;
		QStringList name_list, str_list;
		if (tlist.find( tg )!=tlist.end()) {
			list = tlist[ tg ];
			name_list = nlist[ tg ];
			str_list = slist[ tg ];
		}
		list+= t;
		name_list += KexiDB::Field::typeName( t );
		str_list += KexiDB::Field::typeString( t );
		tlist[ tg ] = list;
		nlist[ tg ] = name_list;
		slist[ tg ] = str_list;
	}
}

const TypeGroupList KexiDB::typesForGroup(KexiDB::Field::TypeGroup typeGroup)
{
	initList();
	return tlist[ typeGroup ];
}

QStringList KexiDB::typeNamesForGroup(KexiDB::Field::TypeGroup typeGroup)
{
	initList();
	return nlist[ typeGroup ];
}

QStringList KexiDB::typeStringsForGroup(KexiDB::Field::TypeGroup typeGroup)
{
	initList();
	return slist[ typeGroup ];
}

QMap< uint, Field::Type > def_tlist;

static void initDefList()
{
	if (!def_tlist.isEmpty())
		return;
	def_tlist[ Field::InvalidGroup ] = Field::InvalidType;
	def_tlist[ Field::TextGroup ] = Field::Text;
	def_tlist[ Field::IntegerGroup ] = Field::Integer;
	def_tlist[ Field::FloatGroup ] = Field::Float;
	def_tlist[ Field::BooleanGroup ] = Field::Boolean;
	def_tlist[ Field::DateTimeGroup ] = Field::Date;
	def_tlist[ Field::BLOBGroup ] = Field::BLOB;
}

KexiDB::Field::Type KexiDB::defaultTypeForGroup(KexiDB::Field::TypeGroup typeGroup)
{
	initDefList();
	return (typeGroup <= Field::LastTypeGroup) ?  def_tlist[ typeGroup ] : Field::InvalidType;
}

void KexiDB::getHTMLErrorMesage(Object* obj, QString& msg, QString &details)
{
	Connection *conn = 0;
	if (!obj || !obj->error()) {
		if (dynamic_cast<Cursor*>(obj)) {
			conn = dynamic_cast<Cursor*>(obj)->connection();
			obj = conn;
		}
		else {
			return;
		}
	}
	if (dynamic_cast<Connection*>(obj)) {
		conn = dynamic_cast<Connection*>(obj);
	}
	if (!obj || !obj->error())
		return;
	//lower level message is added th the details, if there is alread message specified
	if (msg.isEmpty())
		msg = "<p>" + obj->errorMsg();
	else
		details += "<p>" + obj->errorMsg();

	if (!obj->serverErrorMsg().isEmpty())
		details += "<p><b><nobr>" +i18n("Message from server:") + "</nobr></b><br>" + obj->serverErrorMsg();
	if (conn && !conn->recentSQLString().isEmpty())
		details += "<p><b><nobr>" +i18n("SQL statement:") + "</nobr></b><br>" + conn->recentSQLString();
	int serverResult;
	QString serverResultName;
	if (obj->serverResult()!=0) {
		serverResult = obj->serverResult();
		serverResultName = obj->serverResultName();
	}
	else {
		serverResult = obj->previousServerResult();
		serverResultName = obj->previousServerResultName();
	}
	if (!serverResultName.isEmpty())
		details += (QString("<p><b><nobr>")+i18n("Server result name:")+"</nobr></b><br>"+serverResultName);
	if (!details.isEmpty()) {
		details += (QString("<p><b><nobr>")+i18n("Server result number:")+"</nobr></b><br>"+QString::number(serverResult));
	}
}

void KexiDB::getHTMLErrorMesage(Object* obj, QString& msg)
{
	getHTMLErrorMesage(obj, msg, msg);
}

void KexiDB::getHTMLErrorMesage(Object* obj, ResultInfo *result)
{
	getHTMLErrorMesage(obj, result->msg, result->desc);
}

int KexiDB::idForObjectName( Connection &conn, const QString& objName, int objType )
{
	RowData data;
	if (!conn.querySingleRecord(QString("select o_id from kexi__objects where o_name='%1' and o_type=%2")
		.arg(objName.lower()).arg(objType), data))
		return 0;
	bool ok;
	int id = data[0].toInt(&ok);
	return ok ? id : 0;
}

