/***************************************************************************
 *   Copyright (C) 2006 by Thomas Kadauke                                  *
 *   tkadauke@gmx.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,      *
 *   Boston, MA 02110-1301, USA.                                           *
 ***************************************************************************/

// KDE includes
#include <kdebug.h>

// WorKflow includes
#include "typemanager.h"
#include "datatype.h"
#include "conversion.h"
#include "pathconversion.h"

using namespace WorKflow;

TypeManager* TypeManager::s_self = 0;

TypeManager* TypeManager::self()
{
  if (!s_self) {
    s_self = new TypeManager();
  }
  return s_self;
}

void TypeManager::registerType(Datatype* type)
{
  m_typeMap[type->id()] = type;
  m_cacheDirty = true;
}

void TypeManager::unregisterType(Datatype* type)
{
  m_typeMap.erase(type->id());
  m_cacheDirty = true;
}

void TypeManager::registerConversion(Conversion* conversion)
{
  m_atomarConversions.append(conversion);
  m_cacheDirty = true;
}

void TypeManager::unregisterConversion(Conversion* conversion)
{
  m_atomarConversions.erase(m_atomarConversions.find(conversion));
  m_cacheDirty = true;
}

Datatype* TypeManager::find(const QString& id)
{
  TypeMap::ConstIterator it = m_typeMap.find(id);
  if (it != m_typeMap.end()) {
    return it.data();
  } else {
    kdWarning() << k_funcinfo << "unknown type id: " << id << endl;
    return 0;
  }
}

Conversion* TypeManager::findConversion(Datatype* source, Datatype* dest)
{
  if (m_cacheDirty)
    rebuildConversionCache();

  Internal::ConversionKey c(source, dest);
  ConversionMap::ConstIterator it = m_conversionMap.find(c);
  if (it != m_conversionMap.end())
    return it.data();
  else {
    return findPath(source, dest);
  }
}

TypeManager::TypeList TypeManager::types() const
{
  TypeList result;
  for (TypeMap::ConstIterator i = m_typeMap.begin(); i != m_typeMap.end(); ++i) {
    result.append(i.data());
  }
  return result;
}

TypeManager::TypeManager()
  : m_cacheDirty(true)
{

}

TypeManager::~TypeManager()
{

}

void TypeManager::rebuildConversionCache()
{
  m_conversionMap.clear();
  for (ConversionList::ConstIterator i = m_upcastConversions.begin(); i != m_upcastConversions.end(); ++i)
    delete *i;

  for (ConversionList::ConstIterator i = m_conversionPaths.begin(); i != m_conversionPaths.end(); ++i)
    delete *i;

  // build all upcast conversions
  for (TypeMap::ConstIterator i = m_typeMap.begin(); i != m_typeMap.end(); ++i) {
    Datatype* tp = i.data();
    Datatype* base = tp->baseType();

    if (base) {
      Conversion* conv = new Conversion(0);
      conv->setSourceTypeId(tp->id());
      conv->setDestTypeId(base->id());
      m_upcastConversions.append(conv);
      addCacheEntry(conv);
    }
  }

  // add atomar conversions to map
  for (ConversionList::ConstIterator i = m_atomarConversions.begin(); i != m_atomarConversions.end(); ++i)
    addCacheEntry(*i);

  m_cacheDirty = false;
}

Conversion* TypeManager::findPath(Datatype* source, Datatype* dest)
{
  kdDebug() << k_funcinfo << endl;

  QValueList<Datatype*> queue;
  QMap<Datatype*, bool> visited;
  QMap<Datatype*, Conversion*> path;

  bool found = false;

  queue.append(source);
  while (!queue.isEmpty()) {
    Datatype* tp = queue.front();
    queue.pop_front();

    ConversionList edges = adj(tp);
    for (ConversionList::ConstIterator i = edges.begin(); i != edges.end(); ++i) {
      Datatype* d = find((*i)->destTypeId());
      if (!visited[d]) {
        visited[d] = true;
        path[d] = *i;
        if (d == dest) {
          found = true;
          break;
        }
        queue.append(d);
      }
    }

    if (found)
      break;
  }

  if (found) {
    kdDebug() << "found conversion path (reverse order):" << endl;
    PathConversion* conv = new PathConversion;
    conv->setSourceTypeId(source->id());
    conv->setDestTypeId(dest->id());

    ConversionList conversions;

    // rebuild path (reverse order)
    Datatype* node = dest;
    while (node != source) {
      kdDebug() << node->id() << endl;
      conversions.push_front(path[node]);
//       conv->addConversion(path[node]);
      node = find(path[node]->sourceTypeId());
    }
    kdDebug() << node->id() << endl;

    // correct order
    for (ConversionList::ConstIterator i = conversions.begin(); i != conversions.end(); ++i)
      conv->addConversion(*i);

    // add to cache
    addCacheEntry(conv);

    return conv;
  } else {
    return 0;
  }
}

TypeManager::ConversionList TypeManager::adj(Datatype* source)
{
  ConversionList result;

  for (ConversionList::ConstIterator i = m_upcastConversions.begin(); i != m_upcastConversions.end(); ++i)
    if ((*i)->sourceTypeId() == source->id())
      result.append(*i);

  for (ConversionList::ConstIterator i = m_atomarConversions.begin(); i != m_atomarConversions.end(); ++i)
    if ((*i)->sourceTypeId() == source->id())
      result.append(*i);

  return result;
}

void TypeManager::addCacheEntry(Conversion* conversion)
{
  Datatype* source = find(conversion->sourceTypeId());
  Datatype* dest = find(conversion->destTypeId());

  if (!source || !dest) {
    kdWarning() << "Conversion " << conversion->id() << " has unknown source or dest type!" << endl;
    return;
  }

  Internal::ConversionKey c(source, dest);
  m_conversionMap[c] = conversion;
}
