/***************************************************************************
 *   Copyright (C) 2006 by Thomas Kadauke                                  *
 *   tkadauke@gmx.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,      *
 *   Boston, MA 02110-1301, USA.                                           *
 ***************************************************************************/

// KDE includes
#include <kdebug.h>

// Qt includes
#include <qdom.h>

// WorKflow includes
#include "value.h"
#include "datatype.h"
#include "conversion.h"
#include "typemanager.h"

using namespace WorKflow;

Value::Value()
  : m_type(0)
{

}

Value Value::singleString(const QString& typeId, const QString& value)
{
  Value result(typeId);
  result.addString(value);
  return result;
}

Value Value::singleNumber(const QString& typeId, double value)
{
  Value result(typeId);
  result.addNumber(value);
  return result;
}

Value Value::singleBoolean(const QString& typeId, bool value)
{
  Value result(typeId);
  result.addBoolean(value);
  return result;
}

Value Value::stringList(const QString& typeId, const QStringList& list)
{
  Value result(typeId);
  result.addStringList(list);
  return result;
}

Value Value::dcopRefList(const QString& typeId, const QValueList<DCOPRef>& list)
{
  Value result(typeId);
  for (QValueList<DCOPRef>::ConstIterator i = list.begin(); i != list.end(); ++i)
    result.addDcopRef(*i);
  return result;
}

Value::Value(const QString& typeId)
{
  m_type = TypeManager::self()->find(typeId);
}

bool Value::isEmpty() const
{
  return m_data.isEmpty();
}

void Value::clear()
{
  m_data.clear();
}

QString Value::nextString()
{
  if (isEmpty())
    return QString();

  QString value = m_data.first();
  m_data.pop_front();
  return value;
}

double Value::nextNumber()
{
  if (isEmpty())
    return 0.0;

  double value = m_data.first().toDouble();
  m_data.pop_front();
  return value;
}

bool Value::nextBoolean()
{
  if (isEmpty())
    return false;

  bool value = m_data.first().toInt();
  m_data.pop_front();
  return value;
}

DCOPRef Value::nextDcopRef()
{
  if (isEmpty())
    return DCOPRef();

  return stringToDcopRef(nextString());
}

QStringList Value::asStringList() const
{
  return m_data;
}

void Value::addString(const QString& value)
{
  m_data.append(value);
}

void Value::addNumber(double value)
{
  m_data.append(QString::number(value));
}

void Value::addBoolean(bool value)
{
  m_data.append(QString::number(static_cast<int>(value)));
}

void Value::addDcopRef(const DCOPRef& ref)
{
  m_data.append(QString(ref.app()) + '\n' + QString(ref.obj()));
}

void Value::addStringList(const QStringList& list)
{
  m_data += list;
}

Datatype* Value::type() const
{
  return m_type;
}

Value Value::convertTo(Datatype* dest)
{
  if (dest == type()) {
    return *this;
  }

  Conversion* conv = TypeManager::self()->findConversion(type(), dest);
  if (!conv) {
    kdWarning() << k_funcinfo << "Can't convert between " << type()->id() << " and " << dest->id() << endl;
    return *this;
  }

  return conv->convert(*this);
}

bool Value::operator==(const Value& val) const
{
  return m_type == val.m_type && m_data == val.m_data;
}

bool Value::operator!=(const Value& val) const
{
  return !operator==(val);
}

Value Value::enumValue(const QString& typeId, const QString& key)
{
  return Value::singleString(typeId, key);
}

Value Value::enumValue(const QString& typeId, int index)
{
  Value result(typeId);
  const QStringList& keys = result.type()->enumKeys();
  result.addString(*(keys.at(index)));
  return result;
}

QString Value::enumKey() const
{
  return m_data.first();
}

int Value::enumIndex() const
{
  const QStringList& keys = type()->enumKeys();
  return keys.findIndex(enumKey());
}

QString Value::enumName() const
{
  const QStringList& names = type()->enumNames();
  return *(names.at(enumIndex()));
}

void Value::readXML(const QDomElement& e)
{
  m_data.clear();

  QDomNode n = e.firstChild();
  while (!n.isNull()) {
    QDomElement e = n.toElement();
    if (!e.isNull()) {
      if (e.tagName() == "item") {
        QString item = e.text();
        if (!item.isNull())
          m_data.append(item);
      }
    }

    n = n.nextSibling();
  }
}

void Value::writeXML(QDomDocument& doc, QDomElement& e)
{
  QDomElement valueTag = doc.createElement("value");

  for (QStringList::ConstIterator i = m_data.begin(); i != m_data.end(); ++i) {
    QDomElement item = doc.createElement("item");
    item.appendChild(doc.createTextNode(*i));
    valueTag.appendChild(item);
  }

  e.appendChild(valueTag);
}

QValueList<DCOPRef> Value::asDcopRefList() const
{
  QValueList<DCOPRef> result;
  for (QStringList::ConstIterator i = m_data.begin(); i != m_data.end(); ++i) {
    result.append(stringToDcopRef(*i));
  }
  return result;
}

DCOPRef Value::stringToDcopRef(const QString& ref) const
{
  QStringList parts = QStringList::split('\n', ref);
  return DCOPRef((*parts.at(0)).utf8(), (*parts.at(1)).utf8());
}
