/***************************************************************************
 *   Copyright (C) 2006 by Thomas Kadauke                                  *
 *   tkadauke@gmx.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,      *
 *   Boston, MA 02110-1301, USA.                                           *
 ***************************************************************************/

// KDE includes
#include <kdebug.h>

// Qt includes
#include <qregexp.h>

// WorKflow includes
#include "lineedit.h"
#include "parameter.h"

using namespace WorKflow;

class LineEdit::Private
{
public:
  JoinType joinType;
  QString joinString;
  QRegExp splitExpression;
};

LineEdit::LineEdit(QWidget* parent, const char* name)
  : KLineEdit(parent, name), WidgetInterface()
{
  d = new LineEdit::Private;
  d->joinType = SingleItem;

  connect(this, SIGNAL(textChanged(const QString&)), this, SLOT(slotChanged()));
}

LineEdit::LineEdit(const QString& string, QWidget* parent, const char* name)
  : KLineEdit(string, parent, name)
{
  d = new LineEdit::Private;
}

LineEdit::~LineEdit()
{
  delete d;
}

void LineEdit::setValue(const Value& value)
{
  QString txt;
  if (d->joinType == SingleItem) {
    txt = value.asStringList().first();
  } else {
    txt = value.asStringList().join(d->joinString);
  }

  if (txt != text())
    setText(txt);
}

Value LineEdit::value()
{
  if (d->joinType == SingleItem) {
    return Value::singleString(parameter()->typeId(), text());
  } else {
    return Value::stringList(parameter()->typeId(), QStringList::split(d->splitExpression, text()));
  }
}

LineEdit::JoinType LineEdit::joinType() const
{
  return d->joinType;
}

void LineEdit::setJoinType(JoinType type)
{
  d->joinType = type;
}

QString LineEdit::joinString() const
{
  return d->joinString;
}

void LineEdit::setJoinString(const QString& str)
{
  d->joinString = str;
}

QString LineEdit::splitExpression() const
{
  return d->splitExpression.pattern();
}

void LineEdit::setSplitExpression(const QString& expression)
{
  d->splitExpression = QRegExp(expression);
}

void LineEdit::slotChanged()
{
  if (!parameter())
    return;

  emit changed(value());
}

void LineEdit::setParameterId(const QString& paramId)
{
  WidgetInterface::setParameterId(paramId);
}

QString LineEdit::parameterId() const
{
  return WidgetInterface::parameterId();
}

#include "lineedit.moc"
