/*
  Copyright (c) 2012-2020 Laurent Montel <montel@kde.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "selectthunderbirdfilterfileswidget.h"
#include "ui_selectthunderbirdfilterfileswidget.h"
#include "mailimporter/filterthunderbird.h"
#include <QAbstractButton>
#include <QListWidgetItem>

#include <KLocalizedString>
using namespace MailCommon;

SelectThunderbirdFilterFilesWidget::SelectThunderbirdFilterFilesWidget(const QString &defaultSettingPath, QWidget *parent)
    : QWidget(parent)
    , ui(new Ui::SelectThunderbirdFilterFilesWidget)
{
    ui->setupUi(this);
    connect(ui->buttonGroup, QOverload<QAbstractButton *>::of(&QButtonGroup::buttonClicked), this, &SelectThunderbirdFilterFilesWidget::slotButtonClicked);
#if QT_VERSION < QT_VERSION_CHECK(5, 15, 0)
    connect(ui->profiles, QOverload<int>::of(&KComboBox::currentIndexChanged), this, &SelectThunderbirdFilterFilesWidget::slotProfileChanged);
#else
    connect(ui->profiles, QOverload<int, const QString &>::of(&KComboBox::currentIndexChanged), this, &SelectThunderbirdFilterFilesWidget::slotProfileChanged);
#endif

    ui->listFiles->setSelectionMode(QAbstractItemView::MultiSelection);
    QString defaultProfile;
    QMap<QString, QString> listProfile = MailImporter::FilterThunderbird::listProfile(defaultProfile, defaultSettingPath);
    QMap<QString, QString>::const_iterator i = listProfile.constBegin();
    while (i != listProfile.constEnd()) {
        QString name = i.key();
        if (i.value() == defaultProfile) {
            name += i18n(" (default)");
        }
        ui->profiles->addItem(name, i.value());
        ++i;
    }
    ui->fileUrl->setEnabled(true);
    ui->profiles->setEnabled(false);
    ui->listFiles->setEnabled(false);
    slotProfileChanged(0);
    connect(ui->fileUrl, &KUrlRequester::textChanged, this, &SelectThunderbirdFilterFilesWidget::slotUrlChanged);
    connect(ui->listFiles, &QListWidget::itemSelectionChanged, this, &SelectThunderbirdFilterFilesWidget::slotItemSelectionChanged);
}

SelectThunderbirdFilterFilesWidget::~SelectThunderbirdFilterFilesWidget()
{
    delete ui;
}

void SelectThunderbirdFilterFilesWidget::slotItemSelectionChanged()
{
    Q_EMIT enableOkButton(!ui->listFiles->selectedItems().isEmpty());
}

void SelectThunderbirdFilterFilesWidget::slotUrlChanged(const QString &path)
{
    Q_EMIT enableOkButton(!path.isEmpty());
}

void SelectThunderbirdFilterFilesWidget::slotButtonClicked(QAbstractButton *button)
{
    if (button == ui->selectFile) {
        ui->fileUrl->setEnabled(true);
        ui->profiles->setEnabled(false);
        ui->listFiles->setEnabled(false);
        Q_EMIT enableOkButton(!ui->fileUrl->text().isEmpty());
    } else {
        ui->fileUrl->setEnabled(false);
        ui->profiles->setEnabled(true);
        ui->listFiles->setEnabled(true);
        slotItemSelectionChanged();
    }
}

void SelectThunderbirdFilterFilesWidget::slotProfileChanged(int index)
{
    if (index >= ui->profiles->count()) {
        return;
    }

    QStringList listFilterFiles;
    const QString path(MailImporter::FilterThunderbird::defaultSettingsPath() + ui->profiles->itemData(index).toString());
    QDir dir(path);
    const QStringList subDir = dir.entryList(QDir::AllDirs | QDir::NoDotAndDotDot, QDir::Name);
    for (const QString &mailPath : subDir) {
        const QString subMailPath(path + QLatin1Char('/') + mailPath);
        QDir dirMail(subMailPath);
        const QStringList subDirMail = dirMail.entryList(QDir::AllDirs | QDir::NoDotAndDotDot, QDir::Name);
        for (const QString &file : subDirMail) {
            const QString filterFile(subMailPath + QLatin1Char('/') + file + QLatin1String("/msgFilterRules.dat"));
            if (QFile(filterFile).exists()) {
                listFilterFiles << filterFile;
            }
        }
    }
    ui->listFiles->clear();
    ui->listFiles->addItems(listFilterFiles);
}

QStringList SelectThunderbirdFilterFilesWidget::selectedFiles() const
{
    QStringList listFiles;
    if (ui->selectFile->isChecked()) {
        listFiles << ui->fileUrl->url().path();
    } else {
        const QList<QListWidgetItem *> list = ui->listFiles->selectedItems();
        listFiles.reserve(list.count());
        for (QListWidgetItem *item : list) {
            listFiles << item->text();
        }
    }
    return listFiles;
}

void SelectThunderbirdFilterFilesWidget::setStartDir(const QUrl &url)
{
    ui->fileUrl->setFilter(QStringLiteral("*.dat"));
    ui->fileUrl->setStartDir(url);
}
