/*
    This file is part of KOrganizer.
    Copyright (c) 2002 Klarlvdalens Datakonsult AB <info@klaralvdalens-datakonsult.se>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    As a special exception, permission is given to link this program
    with any edition of Qt, and distribute the resulting executable,
    without including the source code for Qt in the source distribution.
*/

/* This comes from KMLineEdit in KMail. This code reduplication is
   unfortunate, but necessary in order to avoid compile-time
   dependencies between kdenetwork and kdepim, unless the whole scheme
   of addressbook-based completion is moved into kdelibs. Copyright of
   the initial version (C) 1998 Markus Wuebben.
*/

#include <config.h>

#include "kolineedit.h"
#include "koeditordetails.h"

#include <kcompletion.h>
#include <kcompletionbox.h>
#include <kstdaccel.h>
#include <kdebug.h>
#include <kurl.h>
#include <kurldrag.h>
#include <kconfig.h>
#include <qregexp.h>
#include <kabc/addressbook.h>
#include <kabc/stdaddressbook.h>
#include <kabc/distributionlist.h>

#include <ctype.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <pwd.h>


KCompletion* KOLineEdit::s_completion = 0;
bool KOLineEdit::s_addressesDirty = false;

KOLineEdit::KOLineEdit(KOEditorDetails* editor,
                       bool useCompletion,
                       QWidget *parent, const char *name)
    : KOLineEditInherited(parent,name)
{
  mEditor = editor;
  m_useCompletion = useCompletion;
  mMailEdit = 0;

  init();
}

KOLineEdit::KOLineEdit(bool useCompletion,
                       QWidget *parent, const char *name)
    : KOLineEditInherited(parent,name)
{
  mEditor = 0;
  m_useCompletion = useCompletion;
  mMailEdit = 0;

  init();
}

KOLineEdit::KOLineEdit(QWidget *parent, const char *name)
    : KOLineEditInherited(parent,name)
{
  mEditor = 0;
  m_useCompletion = true;
  mMailEdit = 0;

  init();
}

void KOLineEdit::init()
{

  if ( !s_completion ) {
      s_completion = new KCompletion();
      s_completion->setOrder( KCompletion::Sorted );
      s_completion->setIgnoreCase( true );
  }

  installEventFilter(this);

  if ( m_useCompletion )
  {
      setCompletionObject( s_completion, false ); // we handle it ourself
      connect( this, SIGNAL( completion(const QString&)),
               this, SLOT(slotCompletion() ));

      KCompletionBox *box = completionBox();
      connect( box, SIGNAL( highlighted( const QString& )),
               this, SLOT( slotPopupCompletion( const QString& ) ));
      connect( completionBox(), SIGNAL( userCancelled( const QString& )),
               SLOT( setText( const QString& )));


      // Whenever a new KOLineEdit is created (== a new composer is created),
      // we set a dirty flag to reload the addresses upon the first completion.
      // The address completions are shared between all KOLineEdits.
      // Is there a signal that tells us about addressbook updates?
      s_addressesDirty = true;
  }  
}

//-----------------------------------------------------------------------------
KOLineEdit::~KOLineEdit()
{
  removeEventFilter(this);
}


//-----------------------------------------------------------------------------
void KOLineEdit::setEmailLineEdit( QLineEdit* mailEdit )
{
  mMailEdit = mailEdit;
}


//-----------------------------------------------------------------------------
void KOLineEdit::setFont( const QFont& font )
{
    KOLineEditInherited::setFont( font );
    if ( m_useCompletion )
        completionBox()->setFont( font );
}

//-----------------------------------------------------------------------------
bool KOLineEdit::eventFilter(QObject *o, QEvent *e)
{
#ifdef KeyPress
#undef KeyPress
#endif

  if (e->type() == QEvent::KeyPress)
  {
    QKeyEvent* k = (QKeyEvent*)e;

    if (KStdAccel::shortcut(KStdAccel::SubstringCompletion).contains(KKey(k)))
    {
      doCompletion(true);
      return TRUE;
    }
    // ---sven's Return is same Tab and arrow key navigation start ---
    if ((k->key() == Key_Enter || k->key() == Key_Return) &&
        !completionBox()->isVisible())
    {
      return TRUE;
    }
    if (k->state()==ControlButton && k->key() == Key_Right)
    {
      if ((int)text().length() == cursorPosition()) // at End?
      {
        doCompletion(true);
        return TRUE;
      }
      return FALSE;
    }
    if (k->state()==ControlButton && k->key() == Key_V)
    {
      paste();
      return TRUE;
    }
    if (k->key() == Key_Up)
    {
      return TRUE;
    }
    if (k->key() == Key_Down)
    {
      return TRUE;
    }
    // ---sven's Return is same Tab and arrow key navigation end ---

  }
  return KOLineEditInherited::eventFilter(o, e);
}

void KOLineEdit::mouseReleaseEvent( QMouseEvent * e )
{
   if (m_useCompletion && (e->button() == MidButton))
   {
      KOLineEditInherited::mouseReleaseEvent(e);
      return;
   }
   KOLineEditInherited::mouseReleaseEvent(e);
}

void KOLineEdit::insert(const QString &t)
{
  KOLineEditInherited::insert(t);
  return;
}

void KOLineEdit::paste()
{
    KOLineEditInherited::paste();
}

//-----------------------------------------------------------------------------
void KOLineEdit::cursorAtEnd()
{
    setCursorPosition( text().length() );
}


void KOLineEdit::undo()
{
    QKeyEvent k(QEvent::KeyPress, 90, 26, 16 ); // Ctrl-Z
    keyPressEvent( &k );
}

//-----------------------------------------------------------------------------
void KOLineEdit::doCompletion(bool ctrlT)
{
    if ( !m_useCompletion )
        return;

    QString s(text());
    QString prevAddr;
    int n = s.findRev(',');
    if (n>= 0)
    {
        prevAddr = s.left(n+1) + ' ';
        s = s.mid(n+1,255).stripWhiteSpace();
    }

    KCompletionBox *box = completionBox();

    if ( s.isEmpty() )
    {
        box->hide();
        return;
    }

    KGlobalSettings::Completion  mode = completionMode();

    if ( s_addressesDirty )
        loadAddresses();

    QString match;
    int curPos = cursorPosition();
    if ( mode != KGlobalSettings::CompletionNone )
    {
        match = s_completion->makeCompletion( s );
        if (match.isNull() && mode == KGlobalSettings::CompletionPopup)
          match = s_completion->makeCompletion( "\"" + s );
    }

    // kdDebug(5006) << "** completion for: " << s << " : " << match << endl;

    if ( ctrlT )
    {
        QStringList addresses = s_completion->items();
        QStringList::Iterator it = addresses.begin();
        QStringList completions;
        for (; it != addresses.end(); ++it)
        {
            if ((*it).find(s,0,false) >= 0)
                completions.append( *it );
        }

        if (completions.count() > 1) {
            m_previousAddresses = prevAddr;
            box->setItems( completions );
            box->setCancelledText( text() );
            box->popup();
        }
        else if (completions.count() == 1)
            setText(prevAddr + completions.first());
        else
            box->hide();

        cursorAtEnd();
        return;
    }

    switch ( mode )
    {
        case KGlobalSettings::CompletionPopup:
        {
            if ( !match.isNull() )
            {
                m_previousAddresses = prevAddr;
                box->setItems( s_completion->allMatches( s ));
                box->insertItems( s_completion->allMatches( "\"" + s ));
                box->setCancelledText( text() );
                box->popup();
            }
            else
                box->hide();

            break;
        }

        case KGlobalSettings::CompletionShell:
        {
            if ( !match.isNull() && match != s )
            {
                setText( prevAddr + match );
                cursorAtEnd();
            }
            break;
        }

        case KGlobalSettings::CompletionMan: // Short-Auto in fact
        case KGlobalSettings::CompletionAuto:
        {
            if ( !match.isNull() && match != s )
            {
                QString adds = prevAddr + match;
                validateAndSet( adds, curPos, curPos, adds.length() );
            }
            break;
        }

        default: // fall through
        case KGlobalSettings::CompletionNone:
            break;
    }
}

//-----------------------------------------------------------------------------
void KOLineEdit::slotPopupCompletion( const QString& completion )
{
    setText( m_previousAddresses + completion );
    cursorAtEnd();
}

//-----------------------------------------------------------------------------
void KOLineEdit::loadAddresses()
{
    s_completion->clear();
    s_addressesDirty = false;

    QStringList addresses;
    KOabcBridge::addresses(&addresses);
    QStringList::Iterator it2 = addresses.begin();
    for (; it2 != addresses.end(); ++it2) {
    	s_completion->addItem( *it2 );
    }
}


//-----------------------------------------------------------------------------
// From KMMessage
QStringList KOLineEdit::splitEmailAddrList(const QString& aStr)
{
  // Features:
  // - always ignores quoted characters
  // - ignores everything (including parentheses and commas)
  //   inside quoted strings
  // - supports nested comments
  // - ignores everything (including double quotes and commas)
  //   inside comments

  QStringList list;

  if (aStr.isEmpty())
    return list;

  QString addr;
  uint addrstart = 0;
  int commentlevel = 0;
  bool insidequote = false;

  for (uint index=0; index<aStr.length(); index++) {
    // the following conversion to latin1 is o.k. because
    // we can safely ignore all non-latin1 characters
    switch (aStr[index].latin1()) {
    case '"' : // start or end of quoted string
      if (commentlevel == 0)
        insidequote = !insidequote;
      break;
    case '(' : // start of comment
      if (!insidequote)
        commentlevel++;
      break;
    case ')' : // end of comment
      if (!insidequote) {
        if (commentlevel > 0)
          commentlevel--;
        else {
          kdDebug(5006) << "Error in address splitting: Unmatched ')'"
                        << endl;
          return list;
        }
      }
      break;
    case '\\' : // quoted character
      index++; // ignore the quoted character
      break;
    case ',' :
      if (!insidequote && (commentlevel == 0)) {
        addr = aStr.mid(addrstart, index-addrstart);
        if (!addr.isEmpty())
          list += addr.simplifyWhiteSpace();
        addrstart = index+1;
      }
      break;
    }
  }
  // append the last address to the list
  if (!insidequote && (commentlevel == 0)) {
    addr = aStr.mid(addrstart, aStr.length()-addrstart);
    if (!addr.isEmpty())
      list += addr.simplifyWhiteSpace();
  }
  else
    kdDebug(5006) << "Error in address splitting: "
                  << "Unexpected end of address list"
                  << endl;

  return list;
}

//-----------------------------------------------------------------------------
// From KMMessage
QString KOLineEdit::stripEmailAddr(const QString& aStr)
{
  QStringList list = splitEmailAddrList(aStr);
  QString result, totalResult, partA, partB;
  int i, j, len;
  for (QStringList::Iterator it = list.begin(); it != list.end(); ++it)
  {
    char endCh = '>';

    i = (*it).find('<');
    if (i<0)
    {
      i = (*it).find('(');
      endCh = ')';
    }
    if (i<0) result = *it;
    else {
      partA = (*it).left(i).stripWhiteSpace();
      j = (*it).find(endCh,i+1);
      if (j<0) result = *it;
      else {
        partB = (*it).mid(i+1, j-i-1).stripWhiteSpace();

        if (partA.find('@') >= 0 && !partB.isEmpty()) result = partB;
        else if (!partA.isEmpty()) result = partA;
        else result = (*it);

        len = result.length();
        if (result[0]=='"' && result[len-1]=='"')
          result = result.mid(1, result.length()-2);
        else if (result[0]=='<' && result[len-1]=='>')
          result = result.mid(1, result.length()-2);
        else if (result[0]=='(' && result[len-1]==')')
          result = result.mid(1, result.length()-2);
      }
    }
    if (!totalResult.isEmpty()) totalResult += ", ";
    totalResult += result;
  }
  return totalResult;
}

//-----------------------------------------------------------------------------
// This is from KMMessage.
QCString KOLineEdit::getEmailAddr(const QString& aStr)
{
  int a, i, j, len, found = 0;
  QChar c;
  // Find the '@' in the email address:
  a = aStr.find('@');
  if (a<0) return aStr.latin1();
  // Loop backwards until we find '<', '(', ' ', or beginning of string.
  for (i = a - 1; i >= 0; i--) {
    c = aStr[i];
    if (c == '<' || c == '(' || c == ' ') found = 1;
    if (found) break;
  }
  // Reset found for next loop.
  found = 0;
  // Loop forwards until we find '>', ')', ' ', or end of string.
  for (j = a + 1; j < (int)aStr.length(); j++) {
    c = aStr[j];
    if (c == '>' || c == ')' || c == ' ') found = 1;
    if (found) break;
  }
  // Calculate the length and return the result.
  len = j - (i + 1);
  return aStr.mid(i+1,len).latin1();
}


/*!
   This overloaded version splits the text into a name and an email
   address, puts the name into this line edit and the email address
   into mLineEdit.
*/
void KOLineEdit::setText( const QString& text )
{
  QString name = stripEmailAddr( text );
  QString email = getEmailAddr( text );

  KOLineEditInherited::setText( name );
  if( mMailEdit )
    mMailEdit->setText( email );
}


void KOabcBridge::addresses(QStringList* result) // includes lists
{
  KABC::AddressBook *addressBook = KABC::StdAddressBook::self();
  KABC::AddressBook::Iterator it;
  for( it = addressBook->begin(); it != addressBook->end(); ++it ) {
    QStringList emails = (*it).emails();
    QString n = (*it).prefix() + " " +
		(*it).givenName() + " " +
		(*it).additionalName() + " " +
	        (*it).familyName() + " " +
		(*it).suffix();
    n = n.simplifyWhiteSpace();

    QRegExp needQuotes("[^ 0-9A-Za-z\\x0080-\\xFFFF]");
    QString endQuote = "\" ";
    QString empty = "";
    QStringList::ConstIterator mit;
    QString addr, email;

    for ( mit = emails.begin(); mit != emails.end(); ++mit ) {
      email = *mit;
      if (!email.isEmpty()) {
	if (n.isEmpty() || (email.find( '<' ) != -1))
	  addr = empty;
	else { /* do we really need quotes around this name ? */
          if (n.find(needQuotes) != -1)
	    addr = '"' + n + endQuote;
	  else
	    addr = n + ' ';
	}

	if (!addr.isEmpty() && (email.find( '<' ) == -1)
	    && (email.find( '>' ) == -1)
	    && (email.find( ',' ) == -1))
	  addr += '<' + email + '>';
	else
	  addr += email;
	addr = addr.stripWhiteSpace();
	result->append( addr );
      }
    }
  }
  KABC::DistributionListManager manager( addressBook );
  manager.load();

  QStringList names = manager.listNames();
  QStringList::Iterator jt;
  for ( jt = names.begin(); jt != names.end(); ++jt)
    result->append( *jt );
  result->sort();
}

//-----------------------------------------------------------------------------
QString KOabcBridge::expandDistributionLists(QString recipients)
{
  if (recipients.isEmpty())
    return "";
  KABC::AddressBook *addressBook = KABC::StdAddressBook::self();
  KABC::DistributionListManager manager( addressBook );
  manager.load();
  QStringList recpList, names = manager.listNames();
  QStringList::Iterator it, jt;
  QString receiver, expRecipients;
  unsigned int begin = 0, count = 0, quoteDepth = 0;
  for (; begin + count < recipients.length(); ++count) {
    if (recipients[begin + count] == '"')
      ++quoteDepth;
    if ((recipients[begin + count] == ',') && (quoteDepth % 2 == 0)) {
      recpList.append( recipients.mid( begin, count ) );
      begin += count + 1;
      count = 0;
    }
  }
  recpList.append( recipients.mid( begin ));

  for ( it = recpList.begin(); it != recpList.end(); ++it ) {
    if (!expRecipients.isEmpty())
      expRecipients += ", ";
    receiver = (*it).stripWhiteSpace();
    for ( jt = names.begin(); jt != names.end(); ++jt)
      if (receiver.lower() == (*jt).lower()) {
	QStringList el = manager.list( receiver )->emails();
	for ( QStringList::Iterator kt = el.begin(); kt != el.end(); ++kt ) {
	  if (!expRecipients.isEmpty())
	    expRecipients += ", ";
	  expRecipients += *kt;
	}
	break;
      }
    if ( jt == names.end() )
    {
      if (receiver.find('@') == -1)
      {
        KConfigGroup general( kapp->config(), "General" );
        QString defaultdomain = general.readEntry( "Default domain", "" );
        if( !defaultdomain.isEmpty() )
        {
          receiver += "@" + defaultdomain;
        }
        else
        {
          char hostname[100];
          gethostname(hostname, 100);
          QString username = receiver;
          receiver += "@";
          receiver += QCString(hostname, 100);

          passwd *pw = getpwnam(username.local8Bit());
          if (pw)
          {
              QString fn = QString::fromLocal8Bit(pw->pw_gecos);
	      int first_comma = fn.find(',');
	      if (first_comma > 0) {
		fn = fn.left(first_comma);
	      }
              if (fn.find(QRegExp("[^ 0-9A-Za-z\\x0080-\\xFFFF]")) != -1)
                receiver = "\"" + fn + "\" <" + receiver + ">";
              else
                receiver = fn + " <" + receiver + ">";
          }
        }
        expRecipients += receiver;
      }
      else
      {
        expRecipients += receiver;
      }
    }
  }
  return expRecipients;
}
