/*
 * This file is part of telepathy-presence-applet
 *
 * Copyright (C) 2011 Francesco Nwokeka <francesco.nwokeka@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/**
  This component shows text that, when clicked, can be edited
  */

import Qt 4.7
import org.kde.plasma.core 0.1 as PlasmaCore
import org.kde.plasma.graphicswidgets 0.1 as PlasmaWidgets

Item {
    id: textEditItem;

    property bool boldText: false;
    property bool italicText: false;
    property string clickMessage: i18n("Click to set text");
    property string editableText;

    signal textChanged(variant newText);

    // get color from current plasma theme
    property QtObject theme: PlasmaCore.Theme {}

    // show & edit
    state: "show";

    onEditableTextChanged: {
        console.log("editable text changed");
        if (editableText == "") {
            shownText.text = editText.clickMessage;
        } else {
            shownText.text = editableText;
        }
        editText.text = editableText;
    }

    // visible text
    Text {
        id: shownText;
        color: theme.textColor;
        elide: "ElideRight";
        wrapMode: Text.WrapAnywhere;
        anchors.fill: parent;

        font {
            bold: boldText;
            italic: italicText;
            pixelSize: textEditItem.height/2;
        }

        Component.onCompleted: {
            if (editableText == "") {
                text = editText.clickMessage;
            } else {
                text = editableText;
                editText.text = editableText;
            }
        }

        MouseArea {
            anchors.fill: parent;
            hoverEnabled: true;

            onClicked: {
                textEditItem.state = "edit";
            }

            // show edit button
            onEntered: {
                if (textEditItem.state == "show") {
                    editButton.opacity = 1;
                }
            }

            onExited: {
                editButton.opacity = 0;
            }
        }
    }

    // editable text
    PlasmaWidgets.LineEdit {
        id: editText;
        width: parent.width - 30;
        anchors.left: parent.left;
        clearButtonShown: true;
        clickMessage: textEditItem.clickMessage;

        onReturnPressed: {
            textEditItem.editableText = text;
            textEditItem.state = "show";
            parent.textChanged(editableText);
        }
    }

    // edit button
    PlasmaWidgets.IconWidget {
        id: editButton;
        height: parent.height;
        width: parent.height;
        opacity: 0;

        anchors {
            left: editText.right;
            leftMargin: 2;
            right: parent.right;
        }

        icon: QIcon("document-edit");

        onClicked: {
            textEditItem.state = "edit";
        }
    }

    // cancel "edit" button
    PlasmaWidgets.IconWidget {
        id: editCancelButton;
        height: parent.height;
        width: parent.height;

        anchors {
            left: editText.right;
            leftMargin: 2;
            right: parent.right;
        }

        icon: QIcon("edit-delete");

        onClicked: {
            textEditItem.state = "show";
        }
    }

    states: [
        State {
            name: "show";
            PropertyChanges {
                target: editText;
                opacity: 0;
            }
            PropertyChanges {
                target: shownText;
                opacity: 1;
            }
            PropertyChanges {
                target: editCancelButton;
                opacity: 0;
            }
        },
        State {
            name: "edit";
            PropertyChanges {
                target: editText;
                opacity: 1;
            }
            PropertyChanges {
                target: shownText;
                opacity: 0;
            }
            PropertyChanges {
                target: editButton;
                opacity: 0;
            }
            PropertyChanges {
                target: editCancelButton;
                opacity: 1;
            }
        }
    ]

    transitions: [
        Transition {
            PropertyAnimation {
                    properties: "color,opacity";
                    duration: 400;
                }
            }
    ]
}
