// Copyright 2025 The Hugo Authors. All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package hugolib

import (
	"testing"
)

// Before Hugo 0.49 we set the pseudo page kind RSS on the page when output to RSS.
// This had some unintended side effects, esp. when the only output format for that page
// was RSS.
// For the page kinds that can have multiple output formats, the Kind should be one of the
// standard home, page etc.
// This test has this single purpose: Check that the Kind is that of the source page.
// See https://github.com/gohugoio/hugo/issues/5138
func TestRSSKind(t *testing.T) {
	t.Parallel()

	files := `
-- hugo.toml --
baseURL = "http://example.com/"
-- layouts/index.rss.xml --
RSS Kind: {{ .Kind }}
`
	b := Test(t, files)

	b.AssertFileContent("public/index.xml", "RSS Kind: home")
}

func TestRSSCanonifyURLs(t *testing.T) {
	t.Parallel()

	files := `
-- hugo.toml --
baseURL = "http://example.com/"
-- layouts/index.rss.xml --
<rss>{{ range .Pages }}<item>{{ .Content | html }}</item>{{ end }}</rss>
-- content/page.md --
---
Title: My Page
---

Figure:

{{< figure src="/images/sunset.jpg" title="Sunset" >}}
`
	b := Test(t, files)

	b.AssertFileContent("public/index.xml", "img src=&#34;http://example.com/images/sunset.jpg")
}

// Issue 13332.
func TestRSSCanonifyURLsSubDir(t *testing.T) {
	t.Parallel()

	files := `
-- hugo.toml --
baseURL = 'https://example.org/subdir'
disableKinds = ['section','sitemap','taxonomy','term']
[markup.goldmark.renderHooks.image]
enableDefault = true
[markup.goldmark.renderHooks.link]
enableDefault = true
-- layouts/_markup/render-image.html --
{{- $u := urls.Parse .Destination -}}
{{- $src := $u.String | relURL -}}
<img srcset="{{ $src }}" src="{{ $src }} 2x">
<img src="{{ $src }}">
{{- /**/ -}}
-- layouts/home.html --
{{ .Content }}|
-- layouts/single.html --
{{ .Content }}|
-- layouts/rss.xml --
{{ with site.GetPage "/s1/p2" }}
  {{ .Content | transform.XMLEscape | safeHTML }}
{{ end }}
-- content/s1/p1.md --
---
title: p1
---
-- content/s1/p2/index.md --
---
title: p2
---
![alt](a.jpg)

[p1](/s1/p1)
-- content/s1/p2/a.jpg --
`

	b := Test(t, files)

	b.AssertFileContent("public/index.xml", "https://example.org/subdir/s1/p1/")
	b.AssertFileContent("public/index.xml",
		"img src=&#34;https://example.org/subdir/a.jpg",
		"img srcset=&#34;https://example.org/subdir/a.jpg&#34; src=&#34;https://example.org/subdir/a.jpg 2x")
}
