
/**
 * ListFormatter.java
 *
 * Copyright (c) 2006 Ying-Chun Liu (PaulLiu)
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are 
 * met:
 *
 *  1. Redistributions of source code must retain the copyright notice,
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in
 *     the documentation and/or other materials provided with the
 *     distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND
 * CONTRIBUTORS ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

package org.jlhafrontend;
import java.io.*;
import java.util.*;
import java.text.*;
import java.util.logging.*;
import jp.gr.java_conf.dangan.util.lha.*;

/**
 * Formatted output class for list/verbose list command
 */
public class ListFormatter {
    private LhaHeader lh=null;
    protected Logger mylogger=null;

    /**
     * constructor 
     */    
    public ListFormatter () {
	mylogger = Logger.getLogger("org.jlhafrontend");
    }

    /**
     * constructor with given LhaHeader
     * @param a the LhaHeader
     */
    public ListFormatter (LhaHeader a) {
	this();
	setLhaHeader(a);
    }
    
    /**
     * set an LhaHeader
     * The data will be retrived from this header. Formatted data can be get
     * by getX method.
     * @param lh the LhaHeader
     */
    public void setLhaHeader(LhaHeader lh) {
      this.lh = lh;
    }

    /**
     * padding spaces if a given string is not long enough
     * @param s1 the given string
     * @param fieldLen minimal length of the return string
     * @param leftAlign If true, the space will padding to the end of the string. If false, the space will padding to the head of the string.
     * @return padded string
     */
    public static String _toField(String s1,int fieldLen,boolean leftAlign) {
        String out1;
        int i;
        if (fieldLen <= s1.length()) {
            out1 = s1;
        } else {
            out1 = s1;
            for (i=0 ; i<fieldLen-s1.length(); i++) {
                if (leftAlign) {
                    out1 = out1 + " ";
                } else {
                    out1 = " " + out1;
                }
            }
        }
        return out1;
    }

    /**
     * padding spaces if a given string is not long enough
     * @param s1 the given string
     * @param fieldLen minimal length of the return string
     * @param leftAlign If true, the space will padding to the end of the string. If false, the space will padding to the head of the string.
     * @return padded string
     */
    public String toField(String s1,int fieldLen,boolean leftAlign) {
	return ListFormatter._toField(s1,fieldLen,leftAlign);
    }

    /**
     * get PERMSSN field
     * @return string of the field
     */
    public String getPERMSSN() {
	byte OSID;
	byte[] generic_table = {'2','9','K','3','H','U','C','F','m','R','T','X','w','W','J'};
	String ret="[unknown]";
	int i;
	OSID = lh.getOSID();
	if (OSID=='M') {
	    ret = "[MS-DOS]";
	} else if (OSID==0) {
	    ret = "[generic]";
	} else {	/* FIXME - should display correct OSID */
	    for (i=0 ; i<generic_table.length ; i++) {
		if (OSID == generic_table[i]) {
		    ret = "[generic]";
		}
	    }
	}
	return toField(ret,10,true);
    }

    /**
     * get PACKED field
     * @return string of the field
     */
    public String getPACKED() {
	return toField(Long.toString(lh.getCompressedSize()),7,false);
    }        

    /**
     * get METHOD field
     * @return string of the field
     */
    public String getMETHOD() {
        return lh.getCompressMethod();
    }
    
    /**
     * get CRC field
     * @return string of the field
     */
    public String getCRC() {
        String ret;
        ret = Integer.toHexString(lh.getCRC());
        while (ret.length()<4) {
          ret = "0"+ret;
        }
        return ret;
    }

    /**
     * get UIDGID field
     * @return string of the field
     */
    public String getUIDGID() {
	return toField("",11,true);
    }

    /**
     * get SIZE field
     * @return string of the field
     */
    public String getSIZE() {
	return toField(Long.toString(lh.getOriginalSize()),7,false);
    }

    /**
     * get RATIO field
     * @return string of the field
     */
    public String getRATIO() {
	
	long ratio_a,ratio_b;
	long osize;
	osize = lh.getOriginalSize();
	if (osize!=0) {
	    ratio_a = lh.getCompressedSize()*1000/osize;
	} else {
	    return (new String("******"));
	}
	ratio_b = ratio_a%10;
	ratio_a = ratio_a/10;
	return toField(Long.toString(ratio_a)+"."+Long.toString(ratio_b)+"%",6,false);
    }

    /**
     * get STAMP field
     * @return string of the field
     */
    public String getSTAMP() {
	SimpleDateFormat df1 = new SimpleDateFormat("MMM dd  yyyy",Locale.US);
	String ret;
	ret = df1.format(lh.getLastModified());
	
	return toField(ret,12,false);
    }

    /**
     * generate date string of a file
     * @param filename the name of the file
     * @return string of the date
     */
    static public String genFileDate(String filename) {
        File f1=null;
        try {
          f1 = new File(filename);
        } catch (Exception e) {
        }
        if (f1 != null) {
          return genFileDate(f1);
        }
        return "            ";
    }    

    /**
     * generate date string of a file
     * @param a the File instance
     * @return string of the date
     */
    static public String genFileDate(File a) {
        long lt=0;
        Date d1;
        SimpleDateFormat df1 = new SimpleDateFormat("MMM dd HH:mm",Locale.US);
        try {
          lt = a.lastModified();
        } catch (Exception e) {
        }
        d1 = new Date(lt);
        String ret;
        ret = df1.format(d1);
        return ret;
    }        
}
