/* NVTV Gui Philips -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id: gui_ph.c,v 1.23 2003/10/10 16:16:57 dthierbach Exp $
 *
 * Contents:
 *
 * The GTK graphical user interface. Philips part.
 */

#include "local.h" /* before everything else */

#include <math.h>
#include <gtk/gtk.h>

#include "gui.h"
#include "gui_ph.h"
#include "backend.h"
#include "data_ph.h"

#define two_to_21 2097152.0
#define two_to_20 1048576.0

/* Automatic minimum distance between overscan percentages */

static GtkAdjustment *update_ph;

static gint gui_timeout_ph_id = -1; /* status timeout */

static TVMode ph_calc =
  {{TV_SYSTEM_NONE, 0, 0, "Custom", "", 0.0, 0.0}};

#define FIELD(b,m) addr:&(b.m), size:sizeof(b.m)
#define FIELD_PH1(m) FIELD(gui_regs.enc.ph1,m)
#define FIELD_PH2(m) FIELD(gui_regs.enc.ph2,m)

/* -------- GUI Masks -------- */

struct mask_calc {
  GtkAdjustment *hdisp, *vdisp;
  GtkAdjustment *hsync;
  GtkAdjustment *htotal;
  GtkAdjustment *hoc, *voc;
  GtkAdjustment *hoffset, *voffset;
  GtkAdjustment *flicker;
  GtkAdjustment *yinc, *yskip, *yofse, *yofso;
  GtkLabel *hsize, *vsize;
  GtkLabel *vtotal, *vsync;
};

struct mask_calc ph_mask_calc;

struct mask_ph_freq {
  GtkLabel *clk, *hsyn, *vsyn, *hover, *vover, *aspect;
};

struct mask_ph_freq gui_mask_ph_freq;

struct mask_ph_status {
  GtkToggleButton *tvon, *over, *under;
};

struct mask_ph_status gui_mask_ph_status;

static GuiRegMask ph_mask1_twin [] = {
  {label:"fsc:",    bits:32, tick:1, FIELD_PH1(fsc)},
  {label:"pcl:",    bits:24, tick:1, FIELD_PH1(pcl)},
  {label:NULL}
};				    

/* -------- */

static GuiRegMask ph1_mask1_reg [] = {
  {label:"adwhs:",  bits:11, tick:1, FIELD_PH1(adwhs)},  
  {label:"adwhe:",  bits:11, tick:1, FIELD_PH1(adwhe)},  
  {label:"fal:",    bits:9,  tick:1, FIELD_PH1(fal)},    
  {label:"lal:",    bits:9,  tick:1, FIELD_PH1(lal)},    
  {label:"gy:",     bits:5,  tick:1, FIELD_PH1(gy)},     
  {label:"gcd:",    bits:5,  tick:1, FIELD_PH1(gcd)},    
  {label:"bs:",     bits:6,  tick:1, FIELD_PH1(bs)},     
  {label:"be:",     bits:6,  tick:1, FIELD_PH1(be)},     
  {label:"idel:",   bits:4,  tick:1, FIELD_PH1(idel)},   
  {label:"gainu:",  bits:9,  tick:1, FIELD_PH1(gainu)},  
  {label:"gainv:",  bits:9,  tick:1, FIELD_PH1(gainv)},  
  {label:"blckl:",  bits:6,  tick:1, FIELD_PH1(blckl)},  
  {label:"blnnl:",  bits:6,  tick:1, FIELD_PH1(blnnl)},  
  {label:"bsta:",   bits:6,  tick:1, FIELD_PH1(bsta)},   
  {label:"macro:",  bits:3,  tick:1, FIELD_PH1(macro)},
  {label:NULL}
};

static GuiRegMask ph1_mask2_reg [] = {
  {label:"xpix:",   bits:10, tick:1, FIELD_PH1(xpix)},   
  {label:"xofs:",   bits:10, tick:1, FIELD_PH1(xofs)},   
  {label:"xinc:",   bits:12, tick:1, FIELD_PH1(xinc)},   
  {label:"ypix:",   bits:10, tick:1, FIELD_PH1(ypix)},   
  {label:"yofso:",  bits:10, tick:1, FIELD_PH1(yofso)},  
  {label:"yofse:",  bits:10, tick:1, FIELD_PH1(yofse)},  
  {label:"yinc:",   bits:12, tick:1, FIELD_PH1(yinc)},   
  {label:"yskip:",  bits:12, tick:1, FIELD_PH1(yskip)},  
  {label:"yiwgto:", bits:12, tick:1, FIELD_PH1(yiwgto)}, 
  {label:"yiwgte:", bits:12, tick:1, FIELD_PH1(yiwgte)}, 
  {label:"hlen:",   bits:12, tick:1, FIELD_PH1(hlen)},   
  {label:"chps:",   bits:8,  tick:1, FIELD_PH1(chps)},   
  {label:"bcy:",    bits:8,  tick:1, FIELD_PH1(bcy)},    
  {label:"bcu:",    bits:8,  tick:1, FIELD_PH1(bcu)},    
  {label:"bcv:",    bits:8,  tick:1, FIELD_PH1(bcv)},    
  {label:"ccrs:",   bits:2,  tick:1, FIELD_PH1(ccrs)},   
  {label:"flc:",    bits:2,  tick:1, FIELD_PH1(flc)},   
  {label:"phres:",  bits:2,  tick:1, FIELD_PH1(phres)},   
  {label:NULL}
};				    

static GuiFlagMask ph1_mask1_flag [] = {
  {label:"fise",        mask:PH_FLAG1_FISE,     FIELD_PH1(flags1)},    
  {label:"pal",         mask:PH_FLAG1_PAL,      FIELD_PH1(flags1)},    
  {label:"ygs",         mask:PH_FLAG1_YGS,      FIELD_PH1(flags1)},    
  {label:NULL }
};

static GuiFlagMask ph1_mask2_flag [] = {
  {label:"scbw",        mask:PH_FLAG1_SCBW,     FIELD_PH1(flags1)}, 
  {label:"cen",         mask:PH_FLAG2_CEN,      FIELD_PH1(flags2)}, 
  {label:"cvbsen0",     mask:PH_FLAG2_CVBSEN0,  FIELD_PH1(flags2)}, 
  {label:"cvbsen1",     mask:PH_FLAG2_CVBSEN1,  FIELD_PH1(flags2)}, 
  {label:"vbsen",       mask:PH_FLAG2_VBSEN,    FIELD_PH1(flags2)}, 
  {label:"yfil",        mask:PH_FLAG1_YFIL,     FIELD_PH1(flags1)}, 
  {label:NULL }
};

/* -------- */

static GuiRegMask ph2_mask1_reg [] = {
  {label:"adwhs:",  bits:11, tick:1, FIELD_PH1(adwhs)},  
  {label:"adwhe:",  bits:11, tick:1, FIELD_PH1(adwhe)},  
  {label:"fal:",    bits:9,  tick:1, FIELD_PH1(fal)},    
  {label:"lal:",    bits:9,  tick:1, FIELD_PH1(lal)},    
  {label:"gy:",     bits:5,  tick:1, FIELD_PH1(gy)},     
  {label:"gcd:",    bits:5,  tick:1, FIELD_PH1(gcd)},    
  {label:"bs:",     bits:6,  tick:1, FIELD_PH1(bs)},     
  {label:"be:",     bits:6,  tick:1, FIELD_PH1(be)},     
  {label:"idel:",   bits:4,  tick:1, FIELD_PH1(idel)},   
  {label:"gainu:",  bits:9,  tick:1, FIELD_PH1(gainu)},  
  {label:"gainv:",  bits:9,  tick:1, FIELD_PH1(gainv)},  
  {label:"blckl:",  bits:6,  tick:1, FIELD_PH1(blckl)},  
  {label:"blnnl:",  bits:6,  tick:1, FIELD_PH1(blnnl)},  
  {label:"bsta:",   bits:6,  tick:1, FIELD_PH1(bsta)},   
  {label:"macro:",  bits:3,  tick:1, FIELD_PH1(macro)},
  {label:"fili:",   bits:4,  tick:1, FIELD_PH2(fili)}, 
  {label:"pcle:",   bits:2,  tick:1, FIELD_PH2(pcle)}, 
  {label:"pcli:",   bits:2,  tick:1, FIELD_PH2(pcli)}, 
  {label:NULL}
};

static GuiRegMask ph2_mask2_reg [] = {
  {label:"xpix:",   bits:10, tick:1, FIELD_PH1(xpix)},   
  {label:"xofs:",   bits:10, tick:1, FIELD_PH1(xofs)},   
  {label:"xinc:",   bits:12, tick:1, FIELD_PH1(xinc)},   
  {label:"ypix:",   bits:10, tick:1, FIELD_PH1(ypix)},   
  {label:"yofso:",  bits:10, tick:1, FIELD_PH1(yofso)},  
  {label:"yofse:",  bits:10, tick:1, FIELD_PH1(yofse)},  
  {label:"yinc:",   bits:12, tick:1, FIELD_PH1(yinc)},   
  {label:"yskip:",  bits:12, tick:1, FIELD_PH1(yskip)},  
  {label:"yiwgto:", bits:12, tick:1, FIELD_PH1(yiwgto)}, 
  {label:"yiwgte:", bits:12, tick:1, FIELD_PH1(yiwgte)}, 
  {label:"hlen:",   bits:12, tick:1, FIELD_PH1(hlen)},   
  {label:"chps:",   bits:8,  tick:1, FIELD_PH1(chps)},   
  {label:"bcy:",    bits:8,  tick:1, FIELD_PH1(bcy)},    
  {label:"bcu:",    bits:8,  tick:1, FIELD_PH1(bcu)},    
  {label:"bcv:",    bits:8,  tick:1, FIELD_PH1(bcv)},    
  {label:"ccrs:",   bits:2,  tick:1, FIELD_PH1(ccrs)},   
  {label:"flc:",    bits:2,  tick:1, FIELD_PH1(flc)},   
  {label:"phres:",  bits:2,  tick:1, FIELD_PH1(phres)},   
  {label:"yfil:",   bits:2,  tick:1, FIELD_PH2(yfil)}, 
  {label:NULL}
};				    

static GuiFlagMask ph2_mask1_flag [] = {
  {label:"fise",        mask:PH_FLAG1_FISE,     FIELD_PH1(flags1)},    
  {label:"pal",         mask:PH_FLAG1_PAL,      FIELD_PH1(flags1)},    
  {label:"ygs",         mask:PH_FLAG1_YGS,      FIELD_PH1(flags1)},    
  {label:"ifbp",        mask:PH_FLAG3_IFBP,     FIELD_PH2(flags3)}, 
  {label:"ifra",        mask:PH_FLAG3_IFRA,     FIELD_PH2(flags3)}, 
  {label:NULL }
};

static GuiFlagMask ph2_mask2_flag [] = {
  {label:"scbw",        mask:PH_FLAG1_SCBW,     FIELD_PH1(flags1)}, 
  {label:"cen",         mask:PH_FLAG2_CEN,      FIELD_PH1(flags2)}, 
  {label:"cvbsen0",     mask:PH_FLAG2_CVBSEN0,  FIELD_PH1(flags2)}, 
  {label:"cvbsen1",     mask:PH_FLAG2_CVBSEN1,  FIELD_PH1(flags2)}, 
  {label:"cvbsen2",     mask:PH_FLAG2_CVBSEN2,  FIELD_PH1(flags2)}, 
  {label:"vbsen",       mask:PH_FLAG2_VBSEN,    FIELD_PH1(flags2)}, 
  {label:"yfil",        mask:PH_FLAG1_YFIL,     FIELD_PH1(flags1)}, 
  {label:"xint",        mask:PH_FLAG3_XINT,     FIELD_PH2(flags3)}, 
  {label:"eidiv",       mask:PH_FLAG3_EIDIV,    FIELD_PH2(flags3)}, 
  {label:"yupsc",       mask:PH_FLAG3_YUPSC,    FIELD_PH2(flags3)}, 
  {label:NULL }
};

#if 0
#endif

/* -------- */

void ph_freq_calc_cb (GtkObject *obj, struct mask_ph_freq *m)
{
  double Fxtal = 27000000; /* Hz */
  double Fclk;
  double Fhsyn;
  double Fvsyn;
  double hoc, voc;
  int hclk;
  double ato, alo;
  char s [20];

  if ((gui_tv_chip & TV_COMPANY) != TV_PHILIPS) return;

  Fclk = Fxtal * gui_regs.enc.ph1.pcl / two_to_20;
  switch (gui_tv_chip & TV_ENCODER) {
    case TV_PHILIPS_MODEL1:
      Fclk /= 2;
      break;
    case TV_PHILIPS_MODEL2:
      Fclk /= (1 << gui_regs.enc.ph2.pcle);
      break;
  }
  sprintf (s, "%3.2f MHz", Fclk / 1e6);
  gtk_label_set_text (m->clk, s);

  Fhsyn = Fclk / (gui_regs.enc.ph1.hlen + 1);
  snprintf (s, 20, "%3.2f kHz", Fhsyn / 1e3);
  gtk_label_set_text (m->hsyn, s);

  if (gui_regs.enc.ph1.flags1 & PH_FLAG1_FISE) 
  { 
    Fvsyn = 60.0; /* NTSC etc. */
    hclk = 858;
    ato = 710 * 2.0; /* dclk */
    alo = 243.0; /* Ph-Data: 240.0; */
  } else {
    Fvsyn = 50.0; /* PAL etc. */
    hclk = 864;
    ato = 702 * 2.0; /* dclk */
    alo = 288.0; /* Ph_Data: 287.0; */
  }

  snprintf (s, 20, "%3.2f  Hz", Fvsyn);
  gtk_label_set_text (m->vsyn, s);

  voc = 1.0 - ((gui_regs.enc.ph1.lal - gui_regs.enc.ph1.fal) / alo);
  hoc = 1.0 - ((gui_regs.enc.ph1.adwhe - gui_regs.enc.ph1.adwhs) / ato);

  snprintf (s, 20, "%06.3f %%", hoc * 100.0);
  gtk_label_set_text (m->hover, s);
  snprintf (s, 20, "%06.3f %%", voc * 100.0);
  gtk_label_set_text (m->vover, s);
  gtk_label_set_text (m->aspect, "-");
}

gint check_ph_cb (struct mask_ph_status *m)
{
  int status;

  if ((gui_tv_chip & TV_COMPANY) != TV_PHILIPS) return TRUE;
  status = back_card->getStatus (0);
  if (status < 0) return TRUE;
  if (! (status & 0x100)) status = 0;
  gtk_toggle_button_set_active (m->tvon,  (status & 0x100) ? TRUE : FALSE);
  gtk_toggle_button_set_active (m->over,  (status & 0x008) ? TRUE : FALSE);
  gtk_toggle_button_set_active (m->under, (status & 0x004) ? TRUE : FALSE);
  return TRUE;
}

/* ---- Calc ---- */

void calc_ph_reset_cb (GtkWidget *widget, struct mask_calc *m)
{
  if ((gui_tv_chip & TV_COMPANY) != TV_PHILIPS) return;
  if (gui_list_mode) {
    gtk_adjustment_set_value (m->hdisp, 
      gui_list_mode->regs.crtc.nv.HDisplay);
    gtk_adjustment_set_value (m->vdisp, 
      gui_list_mode->regs.crtc.nv.VDisplay);
    gtk_adjustment_set_value (m->hsync, 
      gui_list_mode->regs.crtc.nv.HSyncStart);
    gtk_adjustment_set_value (m->htotal, 
      gui_list_mode->regs.crtc.nv.HTotal);
    gtk_adjustment_set_value (m->hoc, gui_list_mode->spec.hoc);
    gtk_adjustment_set_value (m->voc, gui_list_mode->spec.voc);
    gtk_adjustment_set_value (m->hoffset, 0);
    gtk_adjustment_set_value (m->voffset, 0);
    gtk_adjustment_set_value (m->flicker, 100);
    gtk_adjustment_set_value (m->yinc, 0);
    gtk_adjustment_set_value (m->yskip, 0);
    gtk_adjustment_set_value (m->yofse, 0);
    gtk_adjustment_set_value (m->yofso, 0);
    gtk_label_set_text (m->hsize, "-");
    gtk_label_set_text (m->vsize, "-");
    gtk_label_set_text (m->vsync, "-");
    gtk_label_set_text (m->vtotal, "-");
  }
}

void gui_xscale_cb (GtkWidget *widget, gpointer data)
{
}

void gui_yscale_cb (GtkWidget *widget, gpointer data)
{
}

/* The following are rough guesses for the border of the 'comfortable'
   PCL range. The docs hints at 40 MHz - 85 MHz, with 27 MHz crystal */

#define PCL_MIN ((40 << 21) / 27)
#define PCL_MAX ((85 << 21) / 27)

void calc_ph_cb (GtkButton *button, struct mask_calc *m)
{
  TVNvRegs *c = &ph_calc.regs.crtc.nv;
  TVPh1Regs *r = &ph_calc.regs.enc.ph1;
  TVPh2Regs *r2 = &ph_calc.regs.enc.ph2;
  int OutPix, OutLin;
  int OutOfsPix, OutOfsLin;
  int yinc_adj, yskip_adj, yofso_adj, yofse_adj;
  int InPix, InLin;
  int InPpl;
  int vtotal;
  int ratio; /* Clock ratio, RiePclk in docs */
  double hoc, voc;
  double flicker;
  double vfactor;
  double hfactor;
  double tlf; /* total lines/field */
  double tpl; /* total pixels/line */
  int alf; /* active lines/field */
  int flf; /* first line/field */
  int fpl; /* first pixel/line */
  int cpl; /* center pixel/line */
  char s[16];

  InPix = (int) m->hdisp->value;
  InLin = (int) m->vdisp->value;
  InPpl = (int) m->htotal->value;
  c->HSyncStart = (int) m->hsync->value;
  c->HTotal = (int) m->htotal->value;
  hoc = m->hoc->value;
  voc = m->voc->value;
  flicker = m->flicker->value;
  OutOfsPix = (int) m->hoffset->value;
  OutOfsLin = (int) m->voffset->value;
  yinc_adj  = (int) m->yinc->value;
  yskip_adj = (int) m->yskip->value;
  yofso_adj = (int) m->yofso->value;
  yofse_adj = (int) m->yofse->value;

  switch (gui_system) 
  {
    case TV_SYSTEM_NTSC: /* 60 Hz */
      tpl = 1716;   /* ntsc. = 63.555492 ms */
      fpl = 256;
      cpl = 710;
      tlf = 262.5;  /* ntsc */
      alf = 243;    /* Ph-Data: 240; */
      flf = 19;
      break;
    case TV_SYSTEM_PAL: /* 50 Hz */
      tpl = 1728;   /* pal.  = 63.999936 ms */
      fpl = 284;
      cpl = 702;
      tlf = 312.5;  /* pal */
      alf = 288;    /* Ph-Data: 287; */
      flf = 23;
      break;
    default:
      return; /* FIXME */
  }

  /* FIXME TODO */
  /* This is just a rough estimate according to the formulas in the
     doc. It needs to be adjusted. */

  OutPix = floor (cpl * (1.0 - hoc / 100.0));
  OutLin = floor (alf * (1.0 - voc / 100.0));
  
  vtotal  = (int) ((InLin + 2.0) / OutLin * tlf);
  vfactor = vtotal / tlf;
  ratio = 1; /* no x upscaling */

  ph_calc.spec.system = gui_system;
  ph_calc.spec.res_x = InPix;
  ph_calc.spec.res_y = InLin;
  ph_calc.spec.hoc = (1.0 - ((double) OutPix / cpl)) * 100.0;
  ph_calc.spec.voc = (1.0 - ((double) OutLin / alf)) * 100.0;

  c->HDisplay = InPix;
  c->VDisplay = InLin;
  c->HTotal = (InPpl + 7) & ~7;
  c->HSyncStart &= ~7;
  c->HSyncEnd = c->HTotal - (c->HSyncStart - c->HDisplay);
  if (c->HSyncEnd < c->HSyncStart)
    c->HSyncEnd = (c->HSyncStart + c->HTotal) / 2;

  switch (gui_tv_chip & TV_ENCODER) {
    case TV_PHILIPS_MODEL1: 
      data_init_ph1 (gui_system, r); 
      break;
    case TV_PHILIPS_MODEL2: 
      data_init_ph2 (gui_system, r2); 
      r2->pcli = 1;
      r2->pcle = 1;
      hfactor = 1.0 * OutPix / InPix;
      if (hfactor > 2.0) {
	r2->pcle = 3;
	ratio = 4;
      } else if (hfactor > 1.0) {
	r2->pcle = 2;
	ratio = 2;
      } else {
	ratio = 1;
      }
      /* FIXME set pcli/pcle according to frequency and upscaling */
      /* PH2 PLL between ca. 40 Mhz and 85 Mhz (before divide) */
      /* If YUPSC needed, divide YINC by 2 */
      break;
  }

  r->adwhs = fpl + cpl - OutPix + OutOfsPix;
  r->adwhe = r->adwhs + OutPix * 2;
  r->hlen = InPpl * ratio - 1;
  r->xpix = ceil (InPix * ratio / 2);
  r->xofs = c->HTotal - c->HSyncStart;
  r->xinc = ceil (OutPix * 4096.0 / (InPix * ratio));
  if (r->xinc >= 4096) r->xinc = 0;

  r->fal = flf + (alf - OutLin) / 2 + OutOfsLin;
  r->lal = r->fal + OutLin;
  switch (gui_tv_chip & TV_ENCODER) {
    case TV_PHILIPS_MODEL1: 
      r->pcl = floor ((vfactor * InPpl / tpl * two_to_21) + 0.5);
      break;
    case TV_PHILIPS_MODEL2: 
      /* Make pcl four times as big as base */
      r->pcl = floor ((vfactor * (InPpl << r2->pcle) / tpl * two_to_21) + 0.5);
      if (r2->pcle > 0 && r->pcl < PCL_MIN) {
	r2->pcle++; r2->pcli++; /* half freq */
      } else if (r->pcl > PCL_MAX) {
	r->pcl >>= 2; /* double freq */
	r2->pcle--; r2->pcli--;
      } else {
	r->pcl >>= 1; /* normal case */
      }
      break;
  }

  r->yofso = floor (r->fal * vfactor - 2.5) + yofso_adj;
  r->yofse = r->yofso + yofse_adj;
  r->ypix = InLin;
  r->yskip = floor ((100.0 - flicker) / 100.0 * 4095.0) + yskip_adj;
  r->yinc = floor (OutLin / (InLin + 2.0) * 
			   (1.0 + r->yskip / 4095.0) * 4096.0 - 0.5)
                    + yinc_adj;
  /* Check yinc max bound, reduce yskip if necessary */
  if (r->yinc > 4095) {
    r->yinc = 4095;
    r->yskip = floor ((r->yinc / 4096.0 / (OutLin / (InLin + 2.0)) - 1.0)
		      * 4095.0 - 0.5);
  }
  r->yiwgto = floor (r->yinc / 2.0 + 2048.0 - 0.5);
  r->yiwgte = floor ((r->yinc - r->yskip) / 2.0);
  while (r->yiwgte < 0) { 
    r->yiwgte += r->yinc; 
    r->yofse++; 
  }

  c->VTotal = vtotal;
  c->VSyncStart = c->VTotal - r->yofso;
  c->VSyncEnd   = c->VSyncStart + 2;

  gui_act_mode_set (&ph_calc); /* does setup */
  snprintf (s, 14, "%3i", OutPix);
  gtk_label_set_text (m->hsize, s);
  snprintf (s, 14, "%3i", OutLin);
  gtk_label_set_text (m->vsize, s);
  snprintf (s, 14, "%3i", vtotal);
  gtk_label_set_text (m->vtotal, s);
  snprintf (s, 14, "%3i", c->VSyncStart);
  gtk_label_set_text (m->vsync, s);
}

/* ---- ---- */

void reset_ph_cb (GtkObject *obj, gpointer data)
{
  if ((gui_tv_chip & TV_COMPANY) != TV_PHILIPS) return;
  if (gui_act_mode) {
    gui_regs.enc.ph1 = gui_act_mode->regs.enc.ph1;
    gui_tv_set ();
    gtk_signal_emit_by_name (GTK_OBJECT (update_ph), "changed");
    gtk_signal_emit_by_name (GTK_OBJECT (changed_all), "changed");
  }
}

void update_ph_cb (GtkObject *obj, gpointer data)
{
  if ((gui_tv_chip & TV_COMPANY) != TV_PHILIPS) return;
  gtk_signal_emit_by_name (GTK_OBJECT (update_ph), "changed");
}

void gui_map_ph_cb (GtkWidget *widget, gpointer data)
{
  RAISE (MSG_DEBUG, "gui_map_ph_cb");
#ifndef DISABLE_TIMEOUT
  if (gui_timeout_ph_id == -1) {
    gui_timeout_ph_id = gtk_timeout_add (500 /* ms */, 
      (GtkFunction) check_ph_cb, &gui_mask_ph_status);
  }
#endif
}

void gui_unmap_ph_cb (GtkWidget *widget, gpointer data)
{
  RAISE (MSG_DEBUG, "gui_unmap_ph_cb");
#ifndef DISABLE_TIMEOUT
  if (gui_timeout_ph_id != -1) {
    gtk_timeout_remove (gui_timeout_ph_id);
    gui_timeout_ph_id = -1;
  }
#endif
}

/* -------- GUI helper -------- */

GtkAdjustment *create_spin (GtkWidget *table, double min, double max,
  double inc, int decim, int x1, int x2, int y)
{
  GtkWidget *spin;   
  GtkAdjustment *adj;

  adj = (GtkAdjustment *) gtk_adjustment_new (0.0, min, max, 
           inc, inc * 10.0, 0.0);
  spin = gtk_spin_button_new (adj, 0.0, decim);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin), TRUE);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (spin), 
				     GTK_UPDATE_IF_VALID);
  gtk_table_attach (GTK_TABLE(table), spin, x1, x2, y,y+1,
    GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
  return adj;
}			    

/* -------- GUI Pages -------- */

GtkWidget *gui_ph_status_page (void)
{
  GtkWidget *page;
  GtkWidget *frame;
  GtkWidget *table;
  GtkWidget *button;

  page = gtk_vbox_new (FALSE, 5);
  gtk_signal_connect (GTK_OBJECT (page), "map",
    GTK_SIGNAL_FUNC (gui_map_ph_cb), NULL);
  gtk_signal_connect (GTK_OBJECT (page), "unmap",
    GTK_SIGNAL_FUNC (gui_unmap_ph_cb), NULL);
  gtk_container_set_border_width (GTK_CONTAINER (page), 5);

  /* Frequencies CRT / TV */

  frame = gtk_frame_new ("Frequencies");
  gtk_box_pack_start (GTK_BOX (page), frame, TRUE, TRUE, 0);

  table = gtk_table_new (3, 4, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 10);
  gtk_container_add (GTK_CONTAINER (frame), table);

  gui_mask_ph_freq.clk = 
    gui_mask_label (table, "Monitor dot clock:",   "---.-- MHz", 0,1,2, 0);
  gui_mask_ph_freq.hsyn = 
    gui_mask_label (table, "Monitor horiz. sync:", "---.-- kHz", 0,1,2, 1);
  gui_mask_ph_freq.vsyn = 
    gui_mask_label (table, "Monitor vert. sync:",  "---.--  Hz", 0,1,2, 2);
  gui_mask_ph_freq.hover = 
    gui_mask_label (table, "TV horiz. overscan:",  "--.--  %",   0,1,2, 3);
  gui_mask_ph_freq.vover = 
    gui_mask_label (table, "TV vert. overscan:",   "--.--  %",   0,1,2, 4);
  gui_mask_ph_freq.aspect = 
   gui_mask_label (table, "TV aspect ratio:",      "-.---  ",    0,1,2, 5);

  gtk_table_set_col_spacings (GTK_TABLE(table), 10);
  gtk_table_set_row_spacings (GTK_TABLE(table), 10);

  /* TV Status: PH Fifo overrun/underrun */

  frame = gtk_frame_new ("Philips Status");
  gtk_box_pack_start (GTK_BOX (page), frame, TRUE, TRUE, 0);

  table = gtk_table_new (1, 1, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 10);
  gtk_container_add (GTK_CONTAINER (frame), table);

  button = gtk_check_button_new_with_label ("TV on");
  gui_mask_ph_status.tvon = GTK_TOGGLE_BUTTON (button);
  gtk_table_attach (GTK_TABLE(table), button, 0,1, 0,1, GTK_FILL,GTK_FILL,0,0);

  button = gtk_check_button_new_with_label ("FIFO overrun");
  gui_mask_ph_status.over = (GtkToggleButton *) button;
  gtk_table_attach (GTK_TABLE(table), button, 0,1, 1,2, GTK_FILL,GTK_FILL,0,0);

  button = gtk_check_button_new_with_label ("FIFO underrun");
  gui_mask_ph_status.under = (GtkToggleButton *) button;
  gtk_table_attach (GTK_TABLE(table), button, 0,1, 2,3, GTK_FILL,GTK_FILL,0,0);

  gtk_table_set_col_spacings (GTK_TABLE(table), 5);
  gtk_table_set_row_spacings (GTK_TABLE(table), 5);

  /* FIXME to the two other FIFO bits */

  /* return */

  return page;
}

GtkWidget *gui_ph1_reg1_page (void)
{
  GtkAccelGroup *gui_ph_reg_accel_group;

  gui_ph_reg_accel_group = gtk_accel_group_new ();
  return gui_regs_page ("Philips Registers (Out)", 
    gui_ph_reg_accel_group, PRINT_CHIP_REGS,
    GTK_OBJECT (update_ph), GTK_SIGNAL_FUNC (reset_ph_cb),
    0, 0, 8, ph1_mask1_reg, 0, 8, ph_mask1_twin, 6, 0, 8, ph1_mask1_flag);
}

GtkWidget *gui_ph1_reg2_page (void)
{
  GtkAccelGroup *gui_ph_reg_accel_group;

  gui_ph_reg_accel_group = gtk_accel_group_new ();
  return gui_regs_page ("Philips Registers (In)", 
    gui_ph_reg_accel_group, PRINT_CHIP_REGS,
    GTK_OBJECT (update_ph), GTK_SIGNAL_FUNC (reset_ph_cb),
    0, 0, 10, ph1_mask2_reg, 0, 10, NULL, 6, 0, 11, ph1_mask2_flag);
}

GtkWidget *gui_ph2_reg1_page (void)
{
  GtkAccelGroup *gui_ph_reg_accel_group;

  gui_ph_reg_accel_group = gtk_accel_group_new ();
  return gui_regs_page ("Philips Registers (Out)", 
    gui_ph_reg_accel_group, PRINT_CHIP_REGS,
    GTK_OBJECT (update_ph), GTK_SIGNAL_FUNC (reset_ph_cb),
    0, 0, 9, ph2_mask1_reg, 0, 9, ph_mask1_twin, 6, 0, 8, ph2_mask1_flag);
}

GtkWidget *gui_ph2_reg2_page (void)
{
  GtkAccelGroup *gui_ph_reg_accel_group;

  gui_ph_reg_accel_group = gtk_accel_group_new ();
  return gui_regs_page ("Philips Registers (In)", 
    gui_ph_reg_accel_group, PRINT_CHIP_REGS,
    GTK_OBJECT (update_ph), GTK_SIGNAL_FUNC (reset_ph_cb),
    0, 0, 10, ph2_mask2_reg, 0, 10, NULL, 6, 0, 10, ph2_mask2_flag);
}

GtkWidget *gui_ph_calc_page (void)
{
  GtkWidget *page;
  GtkWidget *frame;
  GtkWidget *table;
  GtkWidget *label;
  GtkWidget *button;
  GtkWidget *pane;
  GtkAccelGroup *gui_ph_calc_accel_group;

  gui_ph_calc_accel_group = gtk_accel_group_new ();

  page = gtk_table_new (1,1,FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (page), 0);
  gtk_signal_connect (GTK_OBJECT (page), "map",
    GTK_SIGNAL_FUNC (gui_map_cb), (gpointer) gui_ph_calc_accel_group);
  gtk_signal_connect (GTK_OBJECT (page), "unmap",
    GTK_SIGNAL_FUNC (gui_unmap_cb), (gpointer) gui_ph_calc_accel_group);

  pane = gtk_table_new (1,1,FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (pane), 5);
  gtk_table_attach(GTK_TABLE(page), pane, 0,5, 0,1,
    GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 0, 0);

  /* Parameters */

  frame = gtk_frame_new ("Parameters");
  gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
  gtk_table_attach(GTK_TABLE(pane), frame, 0,1, 0,1,
    GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 0, 0);
  
  table = gtk_table_new (0, 0, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 5);
  gtk_container_add (GTK_CONTAINER (frame), table);

  label = gtk_label_new ("Horiz.");
  gtk_table_attach (GTK_TABLE(table), label, 1,3, 0,1,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  label = gtk_label_new ("Vert.");
  gtk_table_attach (GTK_TABLE(table), label, 3,5, 0,1,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  label = gtk_label_new ("Display:");
  gtk_table_attach (GTK_TABLE(table), label, 0,1, 1,2,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  ph_mask_calc.hdisp = create_spin (table, 0.0, 2048.0, 1.0, 0, 1,3, 1);
  ph_mask_calc.vdisp = create_spin (table, 0.0, 2048.0, 1.0, 0, 3,5, 1);

  label = gtk_label_new ("Sync:");
  gtk_table_attach (GTK_TABLE(table), label, 0,1, 2,3,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  ph_mask_calc.hsync = create_spin (table, 0.0, 2048.0, 1.0, 0, 1,3, 2);
  ph_mask_calc.vsync = GTK_LABEL (
    create_framed_label (table, NULL, "-", 0, 3,4, 2));

  label = gtk_label_new ("Total:");
  gtk_table_attach (GTK_TABLE(table), label, 0,1, 3,4,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  ph_mask_calc.htotal = create_spin (table, 0.0, 2048.0, 1.0, 0, 1,3, 3);
  ph_mask_calc.vtotal = GTK_LABEL (
    create_framed_label (table, NULL, "-", 0, 3,4, 3));

  label = gtk_label_new ("OC:");
  gtk_table_attach (GTK_TABLE(table), label, 0,1, 4,5,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  ph_mask_calc.hoc = create_spin (table, 0.0, 100.0, 0.1, 1, 1,3, 4);
  ph_mask_calc.voc = create_spin (table, 0.0, 100.0, 0.1, 1, 3,5, 4);

  label = gtk_label_new ("Offset:");
  gtk_table_attach (GTK_TABLE(table), label, 0,1, 5,6,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  ph_mask_calc.hoffset = create_spin (table, -100.0, 100.0, 1.0, 0, 1,3, 5);
  ph_mask_calc.voffset = create_spin (table,  -50.0,  50.0, 1.0, 0, 3,5, 5);

  label = gtk_label_new ("Flicker:");
  gtk_table_attach (GTK_TABLE(table), label, 0,1, 6,7,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  ph_mask_calc.flicker = create_spin (table, 0.0, 100.0, 0.1, 1, 3,5, 6);

  label = gtk_label_new ("Upscale:");
  gtk_table_attach (GTK_TABLE(table), label, 0,1, 7,8,
    GTK_FILL, GTK_FILL, 0, 0);

  /* Values */

  label = gtk_label_new ("Size:");
  gtk_table_attach (GTK_TABLE(table), label, 0,1, 10,11,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  ph_mask_calc.hsize = GTK_LABEL (
    create_framed_label (table, NULL, "-", 0, 1,2, 10));
  ph_mask_calc.vsize =  GTK_LABEL (
    create_framed_label (table, NULL, "-", 0, 3,4, 10));

  gtk_table_set_col_spacings (GTK_TABLE(table), 5);
  gtk_table_set_row_spacings (GTK_TABLE(table), 5);
  gtk_table_set_row_spacing (GTK_TABLE(table), 0, 10);

  /* Adjust */

  frame = gtk_frame_new ("Adjust");
  gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
  gtk_table_attach_defaults(GTK_TABLE(pane), frame, 1,2, 0,2);
  
  table = gtk_table_new (0, 0, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 5);
  gtk_container_add (GTK_CONTAINER (frame), table);

  label = gtk_label_new ("Offset");
  gtk_table_attach (GTK_TABLE(table), label, 1,2, 0,1,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  label = gtk_label_new ("YINC:");
  gtk_table_attach (GTK_TABLE(table), label, 0,1, 1,2,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  ph_mask_calc.yinc = create_spin (table, -4096.0, 4096.0, 1.0, 0, 1,2, 1);

  label = gtk_label_new ("YSKIP:");
  gtk_table_attach (GTK_TABLE(table), label, 0,1, 2,3,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  ph_mask_calc.yskip = create_spin (table, -4096.0, 4096.0, 1.0, 0, 1,2, 2);

  label = gtk_label_new ("YOFSO:");
  gtk_table_attach (GTK_TABLE(table), label, 0,1, 3,4,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  ph_mask_calc.yofso = create_spin (table, -4096.0, 4096.0, 1.0, 0, 1,2, 3);

  label = gtk_label_new ("YOFSE:");
  gtk_table_attach (GTK_TABLE(table), label, 0,1, 4,5,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  ph_mask_calc.yofse = create_spin (table, -4096.0, 4096.0, 1.0, 0, 1,2, 4);

  gtk_table_set_col_spacings (GTK_TABLE(table), 5);
  gtk_table_set_row_spacings (GTK_TABLE(table), 5);
  gtk_table_set_row_spacing (GTK_TABLE(table), 0, 10);

  /* Buttons */

  button = gtk_button_new_with_label ("Calc");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (calc_ph_cb), &ph_mask_calc);
  gtk_table_attach(GTK_TABLE(page), button, 0,1, 1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  button = gtk_button_new_with_label ("Print");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (gui_print_cb), 
    (gpointer) (PRINT_CHIP_REGS | PRINT_CRT_REGS));
  gtk_table_attach(GTK_TABLE(page), button, 3,4,1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  button = gtk_button_new_with_label ("Reset");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (calc_ph_reset_cb), &ph_mask_calc);
  gtk_widget_add_accelerator (GTK_WIDGET (button), "clicked", 
    gui_ph_calc_accel_group, 
    gui_accel[ACCEL_RESET].key, gui_accel[ACCEL_RESET].mods,
    GTK_ACCEL_VISIBLE);
  gtk_table_attach(GTK_TABLE(page), button, 4,5,1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  return page;
}

void gui_ph_init (void)
{
  update_ph = (GtkAdjustment *) gtk_adjustment_new (0, 0, 0, 0, 0, 0);
  gtk_signal_connect (GTK_OBJECT (update_chip), "changed", 
    GTK_SIGNAL_FUNC (update_ph_cb), NULL);
  gtk_signal_connect (GTK_OBJECT (update_mode), "changed",
    GTK_SIGNAL_FUNC (calc_ph_reset_cb), &ph_mask_calc);
  gtk_signal_connect (GTK_OBJECT (changed_all), "changed",
    GTK_SIGNAL_FUNC (ph_freq_calc_cb), &gui_mask_ph_freq);

}

