/*
 * Copyright (C) 2000-2025 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <dirent.h>
#include <sys/stat.h>
#include <errno.h>
#include <string.h>

#include "common.h"
#include "config_wrapper.h"
#include "skins.h"
#include "actions.h"
#include "acontrol.h"
#include "control.h"
#include "mrl_browser.h"
#include "event_sender.h"
#include "panel.h"
#include "playlist.h"
#include "errors.h"
#include "xine-toolkit/skin.h"

/* NOTE: User Shelby? shall be able to override an application supplied skin
 * with a downloaded different version of it, and then to switch back and forth there.
 * We do that by duplicating the name in list (application first).
 * libxine configfile stores the name not the index to deal with changing lists.
 * This will fall back to application whenever xine-ui restarts.
 * We can modify the duplicate name in list, but that leads to a compatibility issue.
 * Lets work around this and store the "user" mode flag in a separate hidden config
 * "gui.skin_user" instead. */

struct skins_locations_s {
  unsigned int name; /** << index into fullname. */
#define _SKIN_LOC_FLAG_USER 1 /** << user added in ~/.xine/skins. */
#define _SKIN_LOC_FLAG_FIRST 2 /** << freshly loaded, security clean needed. */
  unsigned int flags;
  char fullname[sizeof (DEFAULT_SKIN)]; /** << actual size will vary to fullname length. */
};

static int _skin_name_cmp (void *a, void *b) {
  skins_locations_t *d = (skins_locations_t *)a;
  skins_locations_t *e = (skins_locations_t *)b;
  int f = strcasecmp (d->fullname + d->name, e->fullname + e->name);
  /* sort by name (no case), then by application/user. */
  return f ? f : ((int)(d->flags & _SKIN_LOC_FLAG_USER) - (int)(e->flags & _SKIN_LOC_FLAG_USER));
}

static skins_locations_t *_skin_add_1 (gGui_t *gui, int *index,
  const char *fullname, const char *name, const char *end, uint32_t flags) {
  int i = -1;
  skins_locations_t *s = malloc (xitk_offsetof (skins_locations_t, fullname) + (end - fullname) + 1);
  if (s) {
    memcpy (s->fullname, fullname, end - fullname + 1);
    s->name = name - fullname;
    s->flags = flags;
    i = xine_sarray_add (gui->skins.a, s);
    if (i >= 0) {
      if (gui->verbosity >= 2)
        printf ("gui.skin.add (%s @ %d).\n", fullname, i);
    } else {
      free (s);
      s = xine_sarray_get (gui->skins.a, ~i);
      if (s) {
        s->flags |= flags & _SKIN_LOC_FLAG_FIRST;
        s = NULL;
      }
      if (gui->verbosity >= 2)
        printf ("gui.skin.found (%s @ %d).\n", fullname, ~i);
    }
  }
  *index = i;
  return s;
}

static const skins_locations_t _skin_default_l = {
  .fullname = DEFAULT_SKIN
};

static int _skin_get_names (gGui_t *gui, const char **names, int max) {
  const char **q;
  int i, n;
  q = names;
  n = xine_sarray_size (gui->skins.a);
  gui->skins.i[3] = n;
  if (n > max - 1)
    n = max - 1;
  for (i = 0; i < n; i++) {
    skins_locations_t *s = xine_sarray_get (gui->skins.a, i);
    *q++ = s->fullname + s->name;
  }
  *q = NULL;
  return q - names;
}

static int skin_get_names (gGui_t *gui, const char **names, int max) {
  return (!gui || !names) ? 0 : _skin_get_names (gui, names, max);
}

int skin_add_1 (gGui_t *gui, const char *fullname, const char *name, const char *end) {
  skins_locations_t *l;
  int index;
  if (!gui || !fullname || !name || !end)
    return -1;
  l = _skin_add_1 (gui, &index, fullname, name, end, _SKIN_LOC_FLAG_FIRST | _SKIN_LOC_FLAG_USER);
  if (l) {
    int v1, v2, v3;
    gui->skins.i[3]++;
    if (gui->skins.i[1] >= index)
      gui->skins.i[1]++;
    if (gui->skins.i[0] >= index)
      gui->skins.i[0]++;
    if (!gui->skins.s[0] && !strcasecmp (l->fullname + l->name, DEFAULT_SKIN))
      gui->skins.s[0] = l, gui->skins.i[0] = index;
    /* Attempt to update the existing config entry. Works with libxine 2020-07-23 (1.2.10hg).
     * Kills callback with old libxine. */
    xine_get_version (&v1, &v2, &v3);
    if (((v1 << 16) | (v2 << 8) | v3) >= 0x01020a) {
      const char *names[128];
      _skin_get_names (gui, names, sizeof (names) / sizeof (names[0]));
      xine_config_register_enum (gui->xine, "gui.skin", gui->skins.i[0], (char **)names,
        _("Skin theme"), CONFIG_NO_HELP, CONFIG_LEVEL_BEG, NULL, NULL);
    }
  }
  return index;
}

void skin_deinit (gGui_t *gui) {
  gui->skins.s[0] = NULL;
  gui->skins.s[1] = NULL;
  gui->skins.i[3] = 0;
  gui->skins.i[0] = -1;
  gui->skins.i[1] = -1;
  if (gui->skins.a) {
    int i, n = xine_sarray_size (gui->skins.a);
    for (i = 0; i < n; i++)
      free (xine_sarray_get (gui->skins.a, i));
    xine_sarray_delete (gui->skins.a);
    gui->skins.a = NULL;
  }
}

/*
 * Fill **skins_avail with available skins from path.
 */
static void _skin_add_dir (gGui_t *gui, char *path, char *pend, char *end, uint32_t flags) {
  DIR             *pdir;
  struct dirent   *pdirent;

  if (pend >= end)
    return;
  pdir = opendir (path);
  if (!pdir)
    return;
  *pend++ = '/';
  while ((pdirent = readdir (pdir))) {
    char *dend = pend + strlcpy (pend, pdirent->d_name, end - pend);
    int index;
    if (dend >= end)
      continue;
    if (pdirent->d_name[0] == '.') {
      if ((pdirent->d_name[1] == 0) || ((pdirent->d_name[1] == '.') && (pdirent->d_name[2] == 0)))
        continue;
    }
    if (xitk_filetype (path) != XITK_FILETYPE_DIR)
      continue;
    /* Check if a skinconfig file exist */
    memcpy (dend, "/skinconfig", 12);
    if (xitk_filetype (path) != XITK_FILETYPE_FILE) {
      if (gui->verbosity >= 2)
        printf ("gui.skin.configfile.missing (%s).\n", path);
      continue;
    }
    *dend = 0;
    _skin_add_1 (gui, &index, path, pend, dend, flags);
  }
  closedir (pdir);
}


/*
 * unload and reload the new config for a given
 * skin file. There is fallback if that fail.
 */
static int _skin_alter (gGui_t *gui, int index) {
  int                  version = 0;
  skins_locations_t   *new_location = xine_sarray_get (gui->skins.a, index), *old_location = gui->skins.s[1];
  const char          *old_name = old_location ? old_location->fullname + old_location->name : "";
  xitk_skin_config_t  *new_skin, *old_skin;

  if (!new_location) {
    gui_msg (gui, XUI_MSG_ERROR, _("Ooch, skin not found, use fallback '%s'.\n"), old_name);
    return gui->skins.i[1];
  }

  xitk_lock (gui->xitk, 1);
  new_skin = xitk_skin_new (gui->xitk);
  if (!new_skin) {
    xitk_lock (gui->xitk, 0);
    return gui->skins.i[1];
  }
  version = xitk_skin_load (new_skin, new_location->fullname, "skinconfig",
    xitk_bitmove (gui->flags, XUI_FLAG_trust_skin, XITK_SKIN_LOAD_TRUSTED) |
    xitk_bitmove (new_location->flags, _SKIN_LOC_FLAG_FIRST, XITK_SKIN_LOAD_FIRST));
  new_location->flags &= ~_SKIN_LOC_FLAG_FIRST;
  if (version < 0) {
    new_location = gui->skins.s[0];
    if (new_location && (!old_location || !strcmp (old_location->fullname, new_location->fullname))) {
      gui_msg (gui, XUI_MSG_ERROR, _("Failed to load %s/%s. Load fallback skin %s\n"),
        new_location->fullname, "skinconfig", DEFAULT_SKIN);
      version = xitk_skin_load (new_skin, new_location->fullname, "skinconfig",
        xitk_bitmove (gui->flags, XUI_FLAG_trust_skin, XITK_SKIN_LOAD_TRUSTED));
      if (version < SKIN_IFACE_VERSION) {
        xitk_skin_delete (&new_skin);
        xitk_lock (gui->xitk, 0);
        return gui->skins.i[1];
      }
    } else {
      xitk_skin_delete (&new_skin);
      xitk_lock (gui->xitk, 0);
      gui_msg (gui, XUI_MSG_ERROR, _("Failed to load %s/%s. Reload old skin '%s'.\n"),
        new_location->fullname, "skinconfig", old_name);
      return gui->skins.i[1];
    }
  }
  /* Check skin version */
  if (version < SKIN_IFACE_VERSION) {
    xitk_skin_delete (&new_skin);
    xitk_lock (gui->xitk, 0);
    gui_msg (gui, XUI_MSG_ERROR, _("Failed to load %s, wrong version. Load fallback skin '%s'.\n"),
      new_location->fullname, old_name);
    return gui->skins.i[1];
  }
  gui->skins.s[1] = new_location;
  gui->skins.i[1] = index;

  visual_anim_add (gui, xitk_skin_get_meta (new_skin, XITK_SKIN_animation), 0);

  old_skin = gui->skin_config;
  gui->skin_config = new_skin;

  /* Now, change skins for each window */
  panel_change_skins (gui->panel, 1);
  event_sender_change_skins (gui->eventer, 1);
  acontrol_change_skins (gui->actrl, 1);
  control_change_skins (gui->vctrl, 1);
  mrl_browser_change_skins (gui->mrlb, 1);

  playlist_change_skins (gui, 1);

  xitk_skin_delete (&old_skin);
  xitk_lock (gui->xitk, 0);

  return index;
}

static void gfx_saturation_cb (void *data, xine_cfg_entry_t *cfg) {
  gGui_t *gui = data;
  gui->gfx_saturation = cfg->num_value;
}

static void gfx_quality_cb (void *data, xine_cfg_entry_t *cfg) {
  gGui_t *gui = data;
  if (xitk_image_quality (gui->xitk, cfg->num_value)) {
    /* redraw skin. yes this does not need a full reload but this is rare,
     * and we dont want to add a whole new redraw API ;-) */
    _skin_alter (gui, gui->skins.i[1]);
  }
}

static void skin_trust_cb (void *data, xine_cfg_entry_t *cfg) {
  gGui_t *gui = data;
  gui->flags = (gui->flags & ~XUI_FLAG_trust_skin) | xitk_bitmove (cfg->num_value, 1, XUI_FLAG_trust_skin);
}

/*
 *
 */
static void skin_change_cb(void *data, xine_cfg_entry_t *cfg) {
  gGui_t *gui = data;
  int want, have;

  if (!gui->skins.a || gui->skins.i[2])
    return;
  /* no recursion, please. */
  gui->skins.i[2] += 1;
  /* First, try to see if the skin exist somewhere */
  want = cfg->num_value;
  have = _skin_alter (gui, want);
  if (have != want) {
    if (have < 0)
      have = gui->skins.i[0];
    cfg->num_value = have;
  }
  gui->skins.i[1] = have;
  if (gui->skins.s[1] && ((gui->skins.s[1]->flags ^ gui->skins.i[4]) & _SKIN_LOC_FLAG_USER))
    config_update_num (gui->xine, "gui.skin_user", gui->skins.i[4] ^= _SKIN_LOC_FLAG_USER);
  gui->skins.i[2] -= 1;
}

void skin_select (gGui_t *gui, int selected) {
  config_update_num (gui->xine, "gui.skin", selected);
}

/*
 * Initialize skin support.
 */
const char *skin_init (gGui_t *gui) {
  gui->skins.i[2] = 1;

  gui->skin_config = xitk_skin_new (gui->xitk);
  gui->skins.get_names = skin_get_names;
  gui->skins.a = xine_sarray_new (64, _skin_name_cmp);
  /* make selectable both supplied and user added skins. */
#if defined(XINE_SARRAY_MODE_UNIQUE)
  xine_sarray_set_mode (gui->skins.a, XINE_SARRAY_MODE_UNIQUE);
#endif
  gui->skins.s[0] = NULL;
  gui->skins.s[1] = NULL;
  gui->skins.i[0] = -1;
  gui->skins.i[1] = -1;
  gui->skins.i[3] = 0;

  /* Grab all available skins from $HOME/.xine/skins and XINE_SKINDIR locations. */
  if (gui->skins.a) {
    char buf[XITK_PATH_MAX + 2 * XITK_NAME_MAX + 2], *p = buf, *e = buf + sizeof (buf) - 16;

    p += strlcpy (buf, xine_get_homedir (), e - p);
    if (p < e) {
      memcpy (p, "/.xine/skins", 13);
      _skin_add_dir (gui, buf, p + 12, e, _SKIN_LOC_FLAG_FIRST | _SKIN_LOC_FLAG_USER);
    }
    memcpy (buf, XINE_SKINDIR, sizeof (XINE_SKINDIR));
    _skin_add_dir (gui, buf, buf + sizeof (XINE_SKINDIR) - 1, e, 0);
    /* # skins found */
    gui->skins.i[3] = xine_sarray_size (gui->skins.a);
    /* default skin... */
    gui->skins.i[0] = xine_sarray_binary_search (gui->skins.a, (void *)&_skin_default_l);
    if (gui->skins.i[0] >= 0) {
      /* ...found in application install as it should be. */
      gui->skins.s[0] = xine_sarray_get (gui->skins.a, gui->skins.i[0]);
    } else {
      /* ...found in user adds only ?? well, if so, it will be right here due to sort order. */
      gui->skins.s[0] = xine_sarray_get (gui->skins.a, gui->skins.i[0] = ~gui->skins.i[0]);
      if (!(gui->skins.s[0] && !strcasecmp (gui->skins.s[0]->fullname + gui->skins.s[0]->name, DEFAULT_SKIN)))
        gui->skins.s[0] = NULL, gui->skins.i[0] = -1;
    }
  }
  if (!gui->skins.i[3]) {
    fprintf (stderr, _("No available skin found. Say goodbye.\n"));
    exit (-1);
  }

  {
    const char *names[128];
    _skin_get_names (gui, names, sizeof (names) / sizeof (names[0]));
    gui->skins.i[1] = xine_config_register_enum (gui->xine, "gui.skin", gui->skins.i[0], (char **)names,
      _("gui skin theme"), CONFIG_NO_HELP, CONFIG_LEVEL_BEG, skin_change_cb, gui);
  }
  gui->skins.s[1] = xine_sarray_get (gui->skins.a, gui->skins.i[1]);
  gui->skins.i[4] = xine_config_register_bool (gui->xine, "gui.skin_user", 0,
    NULL, NULL, 0, NULL, NULL);
  if (gui->skins.s[1]) {
    int d = (int)(gui->skins.i[4] & _SKIN_LOC_FLAG_USER) - (int)(gui->skins.s[1]->flags & _SKIN_LOC_FLAG_USER);
    if (d) {
      skins_locations_t *l = xine_sarray_get (gui->skins.a, gui->skins.i[1] + d);
      if (l && !strcasecmp (l->fullname + l->name, gui->skins.s[1]->fullname + gui->skins.s[1]->name))
        gui->skins.s[1] = l, config_update_num (gui->xine, "gui.skin", gui->skins.i[1] += d);
    }
  }
  {
    int qmax = xitk_image_quality (gui->xitk, -2), v;
    v = xine_config_register_range (gui->xine, "gui.gfx_quality", qmax, 0, qmax,
      _("Graphics quality"),
      _("Trade speed for quality with some screen modes."),
      20, gfx_quality_cb, gui);
    xitk_image_quality (gui->xitk, v);
  }
  gui->gfx_saturation = xine_config_register_range (gui->xine, "gui.gfx_saturation", 55, 0, 255,
    _("Graphics color effects intensity"),
    _("Adjust the intensity of some GUI color effects."),
    0, gfx_saturation_cb, gui);
  {
    unsigned int v = xine_config_register_bool (gui->xine, "gui.skin_trust", 0,
      _("Run skin scripts as is"),
      _("Assume that skin supplied code is harmless, and use it without security filter."),
      30, skin_trust_cb, gui);
    gui->flags = (gui->flags & ~XUI_FLAG_trust_skin) | xitk_bitmove (v, 1, XUI_FLAG_trust_skin);
  }

  return gui->skins.s[1] ? gui->skins.s[1]->fullname : NULL;
}

void skin_load (gGui_t *gui) {
  skins_locations_t *l;
  int i, j, n;
  uint8_t list[128];

  if (!gui->skins.s[1]) {
    fprintf (stderr, _("Ooch, gui.skin config entry isn't registered. Say goodbye.\n"));
    exit (-1);
  }
  i = gui->skins.i[1];

  n = gui->skins.i[3];
  if (n <= 0) {
    fprintf (stderr, _("Failed to load fallback skin. Check your installed skins. Exiting.\n"));
    exit (-1);
  }
  if (!XITK_0_TO_MAX_MINUS_1 (i, n)) {
    gui_msg (gui, XUI_MSG_ERROR, _("Ooch, skin '%s' not found, use fallback '%s'.\n"), "", DEFAULT_SKIN);
    i = gui->skins.i[0];
  }

  if (n > (int)sizeof (list))
    n = sizeof (list);
  /* list set identity. */
  for (j = 0; j < n; j++)
    list[j] = j;
  /* oull up user selection. */
  if ((int)list[0] != i) {
    list[0] = i;
    list[i] = 0;
  }
  /* pull up default. */
  if ((gui->skins.i[0] != i) && (gui->skins.i[0] >= 0) && ((int)list[1] != gui->skins.i[0])) {
    list[1] = gui->skins.i[0];
    list[gui->skins.i[0]] = 1;
  }

  l = xine_sarray_get (gui->skins.a, list[j = 0]);
  while (1) {
    skins_locations_t *l2;
    int version = xitk_skin_load (gui->skin_config, l->fullname, "skinconfig",
      xitk_bitmove (gui->flags, XUI_FLAG_trust_skin, XITK_SKIN_LOAD_TRUSTED));

    if (version >= SKIN_IFACE_VERSION)
      break;
    if (++j >= n) {
      fprintf (stderr, _("Failed to load %s/%s. Exiting.\n"), l->fullname, "skinconfig");
      exit (-1);
    }
    l2 = xine_sarray_get (gui->skins.a, list[j]);
    if (version < 0) {
      fprintf (stderr, _("Failed to load %s/%s. Load fallback skin %s\n"),
        l->fullname, "skinconfig", l2->fullname);
    } else {
      fprintf (stderr, _("Failed to load %s skin, wrong version. Load fallback skin '%s'.\n"),
        l->fullname, l2->fullname);
    }
    l = l2;
  }
  if ((int)list[j] != gui->skins.i[1])
    config_update_num (gui->xine, "gui.skin", gui->skins.i[1] = list[j]);
  gui->skins.s[1] = l;

  gui->skins.i[2] = 0;

  visual_anim_add (gui, xitk_skin_get_meta (gui->skin_config, XITK_SKIN_animation), 0);
}
