/*****************************************************************************
 * sounds.c
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 *
 * Copyright (C) 2003-2005, Erica Andrews
 * (Phrozensmoke ['at'] yahoo.com)
 * http://phpaint.sourceforge.net/pyvoicechat/
 * 
 * Released under the terms of the GPL.
 * *NO WARRANTY*
 *
 * Erica Andrews, PhrozenSmoke ['at'] yahoo.com
 * added 8.31.2003, support for playing sound events on the ESound daemon
 *
 * Stefan Sikora, Hoshy ['at'] schrauberstube.de
 * 2006 added support for playing sound events with ALSA
 *****************************************************************************/

#include <sys/stat.h>
#include <sys/types.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <expat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>

#include <gtk/gtk.h>

#include "config.h"

#include "sounds.h"  /* added, PhrozenSmoke */
#include "util.h"
#include "gyach.h"
#include "users.h"
#include "bootprevent.h"

#include "gy_config.h"
#include "sound_plugin.h"


/* added, PhrozenSmoke, support for ESound sound events */

int xml_aud_counter=0;

GYAUDIBLE gyache_auds[] = {  /* current capacity is 75 */ 
 { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, 
 { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, 
 { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, 
 { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, 

 { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, 
 { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, 
 { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, 
 { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, 

 { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, 
 { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, 
 { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, 
 { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, 

 { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, 
 { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, 
 { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, 
 { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, 

 { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, 
 { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, 
 { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, 
 { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, 

 { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, 
 { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, 
 { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, 
 { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, 

 { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }, { NULL, NULL, NULL, NULL }
};

int auds_callback_enc(void *enc_data, const XML_Char *name, XML_Encoding *info) {return 0;}
void auds_callback_cdata(void *user_data, const XML_Char *data, int len) {return;}
void auds_callback_end(void *user_data, const XML_Char *name) {	xml_aud_counter++;}

void auds_callback_start(void *user_data, const XML_Char *name, const XML_Char **attrs) {
	if (xml_aud_counter>71) {return;}
	if (strncmp((char*)name,"audible",7)==0) {
		if (!attrs) {return;}
		if (! *attrs) {return;}
		else {
			GYAUDIBLE *sm_ptr;
			XML_Char **cptr=(XML_Char **) attrs;
			gyache_audibles=gyache_auds;
			sm_ptr = &gyache_audibles[xml_aud_counter];
			while (cptr && *cptr) {
				if (strncmp(cptr[0],"yname",5)==0) {sm_ptr->aud_file=strdup(cptr[1]);}
				if (strncmp(cptr[0],"quote",5)==0) {sm_ptr->aud_text=strdup(cptr[1]);}
				if (strncmp(cptr[0],"ahash",5)==0) {sm_ptr->aud_hash=strdup(cptr[1]);}
				if (strncmp(cptr[0],"filename",8)==0) {sm_ptr->aud_disk_name=strdup(cptr[1]);}
				cptr += 2;
			}
		}
	}
}

int load_xml_audibles() {
	char filename[256];
	int fd;
	int bytes;
	XML_Parser p;
	void *buff;
	snprintf(filename,254,"%s/audibles/gyaudibles.xml",  PACKAGE_DATA_DIR);
	fd = open( filename, O_RDONLY, 0600 );
	if ( fd == -1 ) {return( 0 );	}
	p = XML_ParserCreate(NULL);      /* XML_ParserCreate( "iso-8859-1"); */
	XML_SetElementHandler(p, auds_callback_start, auds_callback_end);
	XML_SetCharacterDataHandler(p, auds_callback_cdata);
	XML_SetUnknownEncodingHandler(p, auds_callback_enc, NULL);
	XML_SetUserData(p, "");
	buff = XML_GetBuffer(p, 18432);		
	if (buff == NULL) {	return -1;}
	bytes = read( fd, (char *)buff, 18400 );
	close( fd ); 
	xml_aud_counter=0;
	if (! XML_ParseBuffer(p, bytes, bytes == 0)) {return 1;}
	XML_ParserFree(p);	
	return 1;
}

void init_audibles() {
	GYAUDIBLE *sm_ptr;
	gyache_audibles=gyache_auds;
	sm_ptr = &gyache_audibles[0];
	if (! sm_ptr->aud_file) {load_xml_audibles(); }
}


int check_gy_audible( char *str ) {
	GYAUDIBLE *sm_ptr;
	if (!str) {return 0;}
	gyache_audibles=gyache_auds;
	sm_ptr = &gyache_audibles[0];
	while( sm_ptr->aud_file ) {
		if (!strncmp( str, sm_ptr->aud_file, strlen(sm_ptr->aud_file))) {
			return( 1 );
		}
		if (!strncmp( str, sm_ptr->aud_disk_name, strlen(sm_ptr->aud_disk_name))) {
			return( 1 );
		}
		sm_ptr++;
	}
	return( 0 );
}

char *play_audible(char *aud) {
	char audbuf[256];
	if (sounds_opening_too_fast()) {return NULL;}
	if ( capture_fp ) {	
		fprintf(capture_fp,"\n[%s] Y! AUDIBLE ANIMATION: %s [%s], Using MP3 Player Command for Audibles: '%s'\n",
			gyach_timestamp(),
			aud?aud:"None", 
			check_gy_audible(aud)?"Available":"Not Available",
			mp3_player?mp3_player:"mplayer");
		fflush( capture_fp );
	}
	if (!aud) {return NULL;}
	if (!check_gy_audible(aud)) {return NULL;}
	else {
		char *audy_file=NULL;
		audy_file=get_gy_audible_disk_name( aud );
		if (! audy_file) {return NULL;}
		snprintf(audbuf, sizeof(audbuf)-1, "%s %s/audibles/%s.mp3",
			 mp3_player?mp3_player:"mplayer",
			 PACKAGE_DATA_DIR,
			 audy_file);
		play_audible_command(strdup(audbuf));
		free(audy_file);
	}
	return NULL;
}

char *get_gy_audible_disk_name( char *str ) {
	GYAUDIBLE *sm_ptr;
	gyache_audibles=gyache_auds;
	sm_ptr = &gyache_audibles[0];
	while( sm_ptr->aud_file ) {
		if (!strncmp( str, sm_ptr->aud_file, strlen(sm_ptr->aud_file))) {
			return strdup(sm_ptr->aud_disk_name);
		}
		if (!strncmp( str, sm_ptr->aud_disk_name, strlen(sm_ptr->aud_disk_name))) {
			return strdup(sm_ptr->aud_disk_name);
		}
		sm_ptr++;
	}
	return NULL;
}


char *get_gy_audible_hash( char *str ) {
	GYAUDIBLE *sm_ptr;
	gyache_audibles=gyache_auds;
	sm_ptr = &gyache_audibles[0];
	while( sm_ptr->aud_file ) {
		if (!strncmp( str, sm_ptr->aud_file, strlen(sm_ptr->aud_file))) {
			return strdup(sm_ptr->aud_hash);
		}
		sm_ptr++;
	}
	return NULL;
}

char *get_gy_audible_text( char *str ) {
	GYAUDIBLE *sm_ptr;
	gyache_audibles=gyache_auds;
	sm_ptr = &gyache_audibles[0];
	while( sm_ptr->aud_file ) {
		if (!strncmp( str, sm_ptr->aud_file, strlen(sm_ptr->aud_file))) {
			return strdup(sm_ptr->aud_text);
		}
		sm_ptr++;
	}
	return NULL;
}



#ifdef  SUPPORT_SOUND_EVENTS
typedef struct gy_sounds {
        int   sound_value;
        char *sound_event;
        char *filename;
        int   file_size;
        char *file_content;
} GY_SOUND_FILE;

      
GY_SOUND_FILE sound_events[] = { {SOUND_EVENT_BUDDY_ON,  "buddon.raw",    0, 0, 0},
				 {SOUND_EVENT_BUDDY_OFF, "buddoff.raw",   0, 0, 0},
				 {SOUND_EVENT_MAIL,      "yahoomail.raw", 0, 0, 0},
				 {SOUND_EVENT_PM,        "pm.raw",        0, 0, 0},
				 {SOUND_EVENT_BUZZ,      "buzz.raw",      0, 0, 0},
				 {SOUND_EVENT_OTHER,     "other.raw",     0, 0, 0},
				 {SOUND_EVENT_REJECT,    "reject.raw",    0, 0, 0},
				 {-1,                    0,               0, 0, 0}};
  
/* forward declarations */
static GY_SOUND_FILE *find_sound_file(int sound_value);
static void gy_play_sound_event(GY_SOUND_FILE *gy_sound);

#ifdef G_THREADS_ENABLED
/* local thread related variables. Note, local to process, not local to thread */
GMutex  *play_sound_list_mutex;
GCond   *play_sound_list_wait_cond;
GThread *p_play_sound_thread;
int      play_sound_exit_flag = FALSE;

typedef struct play_sound_list_entry {
	GY_SOUND_FILE *gy_sound; /* these 2 are mutually exclusive */
	char *system_command;    /* i.e. either you have a sound event, or a system command to spawn off */
	struct play_sound_list_entry *next;
} GYACHI_PLAYLIST_ENTRY;
static GYACHI_PLAYLIST_ENTRY *play_sound_list;
static GYACHI_PLAYLIST_ENTRY *play_sound_list_freelist;

/* forward declarations */
static GY_SOUND_FILE *find_sound_file(int sound_value);
void *play_sound_thread(void *arg);

void play_sound_thread_init() {
	play_sound_list = NULL;
	play_sound_list_freelist = NULL;
	play_sound_list_mutex = g_mutex_new();
	play_sound_list_wait_cond = g_cond_new();
	play_sound_exit_flag = FALSE;
	p_play_sound_thread = g_thread_create(play_sound_thread, (gpointer)0, FALSE, NULL );

	/*
	  struct timespec initialization_sleep;

	  initialization_sleep.tv_sec=0;
	  initialization_sleep.tv_nsec=500;
	  // allow player thread to start up :)
	  nanosleep(&initialization_sleep, NULL);
	*/
}

void play_sound_thread_terminate() {
	play_sound_exit_flag = TRUE;
	g_cond_signal(play_sound_list_wait_cond);
}

/*  Note: *MUST* be called with play_sound_mutex unlocked */
void static add_item_to_play_sound_list(GYACHI_PLAYLIST_ENTRY *new_entry){
	GYACHI_PLAYLIST_ENTRY *next_entry;

        g_mutex_lock(play_sound_list_mutex);

	new_entry->next=NULL;

	if (play_sound_list == NULL) {
	        play_sound_list = new_entry;
	}
	else {
	        for (next_entry=play_sound_list;
		     next_entry->next != NULL;
		     next_entry=next_entry->next);
		next_entry->next = new_entry;
	}

	g_cond_signal(play_sound_list_wait_cond);
	g_mutex_unlock(play_sound_list_mutex);
}

void add_sound_event_command(GY_SOUND_FILE *gy_sound, char *system_command) {
        GYACHI_PLAYLIST_ENTRY *new_entry;

        g_mutex_lock(play_sound_list_mutex);
	if (play_sound_list_freelist) {
	        new_entry=play_sound_list_freelist;
		play_sound_list_freelist = new_entry->next;
	}
	else {
	        new_entry = malloc(sizeof(GYACHI_PLAYLIST_ENTRY));
	}
	g_mutex_unlock(play_sound_list_mutex);

	new_entry->gy_sound       = gy_sound;
	new_entry->system_command = system_command;
	new_entry->next           = NULL;
	add_item_to_play_sound_list(new_entry);
}

void play_sound_event(int sound_value) {
	GY_SOUND_FILE *gy_sound = find_sound_file(sound_value);

	if (!gy_sound) { return; }
	add_sound_event_command(gy_sound, NULL);
}

void play_audible_command(char *system_command) {
	if (!system_command) { return;}
	add_sound_event_command(NULL, system_command);	
}


/* the sound player thread */
void *play_sound_thread(void *arg) {
	GYACHI_PLAYLIST_ENTRY *sound_event;
	GY_SOUND_FILE *gy_sound;
	char *system_command;

        while (1) {
		if (play_sound_exit_flag) {
			g_mutex_free(play_sound_list_mutex);
			g_cond_free(play_sound_list_wait_cond);
			g_thread_exit(0);
		}

	        g_mutex_lock(play_sound_list_mutex);
		sound_event = play_sound_list;
		if (sound_event == NULL) {
		        g_cond_wait(play_sound_list_wait_cond, play_sound_list_mutex);
			g_mutex_unlock(play_sound_list_mutex);
			continue;
		}

	        play_sound_list = sound_event->next;
		gy_sound=sound_event->gy_sound;
		system_command = sound_event->system_command;
		/* recycle. link used sound event to top of freelist */
		sound_event->next = play_sound_list_freelist;
		play_sound_list_freelist = sound_event;
		g_mutex_unlock(play_sound_list_mutex);

		if (system_command) {
			system(system_command);
			free(system_command);
		}
		if (gy_sound) {
			gy_play_sound_event(gy_sound);
		}
	}
}
#else
/* stubs if no threading library */
void play_sound_thread_init() {
}

void play_sound_thread_terminate() {
}

void play_sound_event(int sound_value) {
        GY_SOUND_FILE *gy_sound = find_sound_file(sound_value);

	if (!gy_sound) { return; }
	gy_play_sound_event(gy_sound);
}

#endif

static GY_SOUND_FILE *find_sound_file(int sound_value)
{
	char           sound_f[256];
	GY_SOUND_FILE *gy_sound;
	struct stat    stat_buf;
	int            fd, needed;
	char          *buff;
	int            rv;

	if (sounds_opening_too_fast()) {return 0;}

	if (sound_value==SOUND_EVENT_PM) { if (!enable_sound_events_pm) {return 0;}}
	else {
		if (!enable_sound_events) {return 0;}
	}
	for (gy_sound=sound_events; gy_sound->sound_value != sound_value && gy_sound->sound_value != -1; gy_sound++);

	if (gy_sound->sound_value == -1) {
	        if ( capture_fp ) {
			fprintf(capture_fp, "\n[%s] Cannot locate sound file for sound event #%d\n",
				gyach_timestamp(), sound_value);
		        fflush( capture_fp );
		}
	        fprintf(stderr, "Cannot locate sound file for sound event #%d\n", sound_value);
		return 0;
	}

	if (gy_sound->filename == NULL) {
		snprintf(sound_f,254, "%s/sounds/%s", PACKAGE_DATA_DIR, gy_sound->sound_event);
		gy_sound->filename = strdup(sound_f);
		rv = stat(gy_sound->filename, &stat_buf);
		if (rv) {
		        if ( capture_fp ) {
				fprintf(capture_fp, "\n[%s] Cannot stat sound file %s. %s\n",
					gyach_timestamp(), gy_sound->filename, strerror(errno));
			        fflush( capture_fp );
			}
		        fprintf(stderr, "Cannot stat sound file %s. %s\n", gy_sound->filename, strerror(errno));
			return 0;
		}
		gy_sound->file_size = stat_buf.st_size;
		gy_sound->file_content = malloc(gy_sound->file_size);
		fd=open(gy_sound->filename, O_RDONLY);
		if (!fd) {
		        if ( capture_fp ) {
				fprintf(capture_fp, "\n[%s] Cannot open sound file %s. %s\n",
					gyach_timestamp(), gy_sound->filename, strerror(errno));
			        fflush( capture_fp );
			}
		        fprintf(stderr, "Cannot open sound file %s. %s\n", gy_sound->filename, strerror(errno));
			free(gy_sound->file_content);
			gy_sound->file_content=NULL;
			return 0;
		}
		needed = gy_sound->file_size;
		buff   = gy_sound->file_content;
		do {
		        rv=read(fd, buff, needed);
			if ((rv == -1) && (errno == EINTR)) {
			        continue;
			}
			if (rv == -1) {
			        if ( capture_fp ) {
				        fprintf(capture_fp, "\n[%s] Error reading sound file %s. %s\n",
						gyach_timestamp(), gy_sound->filename, strerror(errno));
				        fflush( capture_fp );
				}
			        fprintf(stderr, "Error reading sound file %s. %s\n",
					gy_sound->filename, strerror(errno));
				free(gy_sound->file_content);
				gy_sound->file_content=NULL;
				return 0;
			}
			needed -= rv;
			buff   += rv;
		} while (needed > 0);
		
		if ( capture_fp ) {
			fprintf(capture_fp,"\n[%s] SOUND EVENT File %s is now loaded.\n", gyach_timestamp(), gy_sound->filename);
			fflush( capture_fp );
		}
	}

	if (gy_sound->file_content == NULL) { return 0; };

	if ( capture_fp ) {	
		fprintf(capture_fp,"\n[%s] SOUND EVENT (ESound), File: %s\n", gyach_timestamp(), gy_sound->filename);
		fflush( capture_fp );
	}
	return(gy_sound);
}

static void gy_play_sound_event(GY_SOUND_FILE *gy_sound) {
	void *handle;
	GYACHI_FORMAT_TYPE format;
	int   channels;
	int   rate;
	int   total;
	const uint8_t *data;
	int   result;
	int   io_count;

	sync_sound_device();

	if (!selected_sound_plugin) {
		/* no sound plugin selected ... */
		return;
	}

	if (!selected_sound_plugin->play) {
		/* no sound player for this plugin ... */
		return;
	}

	if (gy_sound->sound_value==SOUND_EVENT_BUZZ) {
		channels = 1;
		rate     = 8000;
		format   = GY_SAMPLE_S16LE;
	} else if (gy_sound->sound_value==SOUND_EVENT_REJECT) {
		channels = 1;
		rate     = 8000;
		format   = GY_SAMPLE_U8;
	} else {
		channels = 1;
		rate     = 11025;
		format   = GY_SAMPLE_U8;
	}

	handle = selected_sound_plugin->open(GY_STREAM_PLAYBACK, format, channels, rate);
	if (handle <=0 ) {
		return;
	}

	data  = gy_sound->file_content;
	total = gy_sound->file_size;

	result = 0;
	while ((total > 0) && (result >= 0)) {
		io_count=(4096<total)?4096:total;
		result = selected_sound_plugin->play(handle, data, io_count, format);
		data  += io_count;
		total -= io_count;
	}

	if (selected_sound_plugin->drain) {
		selected_sound_plugin->drain(handle);
	}
	if (selected_sound_plugin->close) {
		selected_sound_plugin->close(handle);
	}
}

#else  /* sound events not compiled int */
	/* a callback that does nothing if sound events aren't supported */
	void play_sound_event(int sound_value) {}
#endif




