// SPDX-License-Identifier: GPL-2.0-or-later
/*
 * Copyright (c) 2024 Rockchip Electronics Co., Ltd.
 * Copyright (c) 2024 Collabora Ltd.
 * Author: Detlev Casanova <detlev.casanova@collabora.com>
 * Based on Sebastien Reichel's implementation for RK3588
 */

#include <linux/module.h>
#include <linux/of.h>
#include <dt-bindings/reset/rockchip,rk3562-cru.h>
#include "clk.h"

/* 0xff100000 + 0x0A00 */
#define RK3562_CRU_RESET_OFFSET(id, reg, bit) [id] = (0 + reg * 16 + bit)
/* 0xff110000 + 0x0A00 */
#define RK3562_PMU0CRU_RESET_OFFSET(id, reg, bit) [id] = (0x10000*4 + reg * 16 + bit)
/* 0xff118000 + 0x0A00 */
#define RK3562_PMU1CRU_RESET_OFFSET(id, reg, bit) [id] = (0x18000*4 + reg * 16 + bit)
/* 0xff120000 + 0x0A00 */
#define RK3562_DDRCRU_RESET_OFFSET(id, reg, bit) [id] = (0x20000*4 + reg * 16 + bit)
/* 0xff128000 + 0x0A00 */
#define RK3562_SUBDDRCRU_RESET_OFFSET(id, reg, bit) [id] = (0x28000*4 + reg * 16 + bit)
/* 0xff130000 + 0x0A00 */
#define RK3562_PERICRU_RESET_OFFSET(id, reg, bit) [id] = (0x30000*4 + reg * 16 + bit)

/* mapping table for reset ID to register offset */
static const int rk3562_register_offset[] = {
	/* SOFTRST_CON01 */
	RK3562_CRU_RESET_OFFSET(SRST_A_TOP_BIU, 1, 0),
	RK3562_CRU_RESET_OFFSET(SRST_A_TOP_VIO_BIU, 1, 1),
	RK3562_CRU_RESET_OFFSET(SRST_REF_PVTPLL_LOGIC, 1, 2),

	/* SOFTRST_CON03 */
	RK3562_CRU_RESET_OFFSET(SRST_NCOREPORESET0, 3, 0),
	RK3562_CRU_RESET_OFFSET(SRST_NCOREPORESET1, 3, 1),
	RK3562_CRU_RESET_OFFSET(SRST_NCOREPORESET2, 3, 2),
	RK3562_CRU_RESET_OFFSET(SRST_NCOREPORESET3, 3, 3),
	RK3562_CRU_RESET_OFFSET(SRST_NCORESET0, 3, 4),
	RK3562_CRU_RESET_OFFSET(SRST_NCORESET1, 3, 5),
	RK3562_CRU_RESET_OFFSET(SRST_NCORESET2, 3, 6),
	RK3562_CRU_RESET_OFFSET(SRST_NCORESET3, 3, 7),
	RK3562_CRU_RESET_OFFSET(SRST_NL2RESET, 3, 8),

	/* SOFTRST_CON04 */
	RK3562_CRU_RESET_OFFSET(SRST_DAP, 4, 9),
	RK3562_CRU_RESET_OFFSET(SRST_P_DBG_DAPLITE, 4, 10),
	RK3562_CRU_RESET_OFFSET(SRST_REF_PVTPLL_CORE, 4, 13),

	/* SOFTRST_CON05 */
	RK3562_CRU_RESET_OFFSET(SRST_A_CORE_BIU, 5, 0),
	RK3562_CRU_RESET_OFFSET(SRST_P_CORE_BIU, 5, 1),
	RK3562_CRU_RESET_OFFSET(SRST_H_CORE_BIU, 5, 2),

	/* SOFTRST_CON06 */
	RK3562_CRU_RESET_OFFSET(SRST_A_NPU_BIU, 6, 2),
	RK3562_CRU_RESET_OFFSET(SRST_H_NPU_BIU, 6, 3),
	RK3562_CRU_RESET_OFFSET(SRST_A_RKNN, 6, 4),
	RK3562_CRU_RESET_OFFSET(SRST_H_RKNN, 6, 5),
	RK3562_CRU_RESET_OFFSET(SRST_REF_PVTPLL_NPU, 6, 6),

	/* SOFTRST_CON08 */
	RK3562_CRU_RESET_OFFSET(SRST_A_GPU_BIU, 8, 3),
	RK3562_CRU_RESET_OFFSET(SRST_GPU, 8, 4),
	RK3562_CRU_RESET_OFFSET(SRST_REF_PVTPLL_GPU, 8, 5),
	RK3562_CRU_RESET_OFFSET(SRST_GPU_BRG_BIU, 8, 8),

	/* SOFTRST_CON09 */
	RK3562_CRU_RESET_OFFSET(SRST_RKVENC_CORE, 9, 0),
	RK3562_CRU_RESET_OFFSET(SRST_A_VEPU_BIU, 9, 3),
	RK3562_CRU_RESET_OFFSET(SRST_H_VEPU_BIU, 9, 4),
	RK3562_CRU_RESET_OFFSET(SRST_A_RKVENC, 9, 5),
	RK3562_CRU_RESET_OFFSET(SRST_H_RKVENC, 9, 6),

	/* SOFTRST_CON10 */
	RK3562_CRU_RESET_OFFSET(SRST_RKVDEC_HEVC_CA, 10, 2),
	RK3562_CRU_RESET_OFFSET(SRST_A_VDPU_BIU, 10, 5),
	RK3562_CRU_RESET_OFFSET(SRST_H_VDPU_BIU, 10, 6),
	RK3562_CRU_RESET_OFFSET(SRST_A_RKVDEC, 10, 7),
	RK3562_CRU_RESET_OFFSET(SRST_H_RKVDEC, 10, 8),

	/* SOFTRST_CON11 */
	RK3562_CRU_RESET_OFFSET(SRST_A_VI_BIU, 11, 3),
	RK3562_CRU_RESET_OFFSET(SRST_H_VI_BIU, 11, 4),
	RK3562_CRU_RESET_OFFSET(SRST_P_VI_BIU, 11, 5),
	RK3562_CRU_RESET_OFFSET(SRST_ISP, 11, 8),
	RK3562_CRU_RESET_OFFSET(SRST_A_VICAP, 11, 9),
	RK3562_CRU_RESET_OFFSET(SRST_H_VICAP, 11, 10),
	RK3562_CRU_RESET_OFFSET(SRST_D_VICAP, 11, 11),
	RK3562_CRU_RESET_OFFSET(SRST_I0_VICAP, 11, 12),
	RK3562_CRU_RESET_OFFSET(SRST_I1_VICAP, 11, 13),
	RK3562_CRU_RESET_OFFSET(SRST_I2_VICAP, 11, 14),
	RK3562_CRU_RESET_OFFSET(SRST_I3_VICAP, 11, 15),

	/* SOFTRST_CON12 */
	RK3562_CRU_RESET_OFFSET(SRST_P_CSIHOST0, 12, 0),
	RK3562_CRU_RESET_OFFSET(SRST_P_CSIHOST1, 12, 1),
	RK3562_CRU_RESET_OFFSET(SRST_P_CSIHOST2, 12, 2),
	RK3562_CRU_RESET_OFFSET(SRST_P_CSIHOST3, 12, 3),
	RK3562_CRU_RESET_OFFSET(SRST_P_CSIPHY0, 12, 4),
	RK3562_CRU_RESET_OFFSET(SRST_P_CSIPHY1, 12, 5),

	/* SOFTRST_CON13 */
	RK3562_CRU_RESET_OFFSET(SRST_A_VO_BIU, 13, 3),
	RK3562_CRU_RESET_OFFSET(SRST_H_VO_BIU, 13, 4),
	RK3562_CRU_RESET_OFFSET(SRST_A_VOP, 13, 6),
	RK3562_CRU_RESET_OFFSET(SRST_H_VOP, 13, 7),
	RK3562_CRU_RESET_OFFSET(SRST_D_VOP, 13, 8),
	RK3562_CRU_RESET_OFFSET(SRST_D_VOP1, 13, 9),

	/* SOFTRST_CON14 */
	RK3562_CRU_RESET_OFFSET(SRST_A_RGA_BIU, 14, 3),
	RK3562_CRU_RESET_OFFSET(SRST_H_RGA_BIU, 14, 4),
	RK3562_CRU_RESET_OFFSET(SRST_A_RGA, 14, 6),
	RK3562_CRU_RESET_OFFSET(SRST_H_RGA, 14, 7),
	RK3562_CRU_RESET_OFFSET(SRST_RGA_CORE, 14, 8),
	RK3562_CRU_RESET_OFFSET(SRST_A_JDEC, 14, 9),
	RK3562_CRU_RESET_OFFSET(SRST_H_JDEC, 14, 10),

	/* SOFTRST_CON15 */
	RK3562_CRU_RESET_OFFSET(SRST_B_EBK_BIU, 15, 2),
	RK3562_CRU_RESET_OFFSET(SRST_P_EBK_BIU, 15, 3),
	RK3562_CRU_RESET_OFFSET(SRST_AHB2AXI_EBC, 15, 4),
	RK3562_CRU_RESET_OFFSET(SRST_H_EBC, 15, 5),
	RK3562_CRU_RESET_OFFSET(SRST_D_EBC, 15, 6),
	RK3562_CRU_RESET_OFFSET(SRST_H_EINK, 15, 7),
	RK3562_CRU_RESET_OFFSET(SRST_P_EINK, 15, 8),

	/* SOFTRST_CON16 */
	RK3562_CRU_RESET_OFFSET(SRST_P_PHP_BIU, 16, 2),
	RK3562_CRU_RESET_OFFSET(SRST_A_PHP_BIU, 16, 3),
	RK3562_CRU_RESET_OFFSET(SRST_P_PCIE20, 16, 7),
	RK3562_CRU_RESET_OFFSET(SRST_PCIE20_POWERUP, 16, 8),
	RK3562_CRU_RESET_OFFSET(SRST_USB3OTG, 16, 10),

	/* SOFTRST_CON17 */
	RK3562_CRU_RESET_OFFSET(SRST_PIPEPHY, 17, 3),

	/* SOFTRST_CON18 */
	RK3562_CRU_RESET_OFFSET(SRST_A_BUS_BIU, 18, 3),
	RK3562_CRU_RESET_OFFSET(SRST_H_BUS_BIU, 18, 4),
	RK3562_CRU_RESET_OFFSET(SRST_P_BUS_BIU, 18, 5),

	/* SOFTRST_CON19 */
	RK3562_CRU_RESET_OFFSET(SRST_P_I2C1, 19, 0),
	RK3562_CRU_RESET_OFFSET(SRST_P_I2C2, 19, 1),
	RK3562_CRU_RESET_OFFSET(SRST_P_I2C3, 19, 2),
	RK3562_CRU_RESET_OFFSET(SRST_P_I2C4, 19, 3),
	RK3562_CRU_RESET_OFFSET(SRST_P_I2C5, 19, 4),
	RK3562_CRU_RESET_OFFSET(SRST_I2C1, 19, 6),
	RK3562_CRU_RESET_OFFSET(SRST_I2C2, 19, 7),
	RK3562_CRU_RESET_OFFSET(SRST_I2C3, 19, 8),
	RK3562_CRU_RESET_OFFSET(SRST_I2C4, 19, 9),
	RK3562_CRU_RESET_OFFSET(SRST_I2C5, 19, 10),

	/* SOFTRST_CON20 */
	RK3562_CRU_RESET_OFFSET(SRST_BUS_GPIO3, 20, 5),
	RK3562_CRU_RESET_OFFSET(SRST_BUS_GPIO4, 20, 6),

	/* SOFTRST_CON21 */
	RK3562_CRU_RESET_OFFSET(SRST_P_TIMER, 21, 0),
	RK3562_CRU_RESET_OFFSET(SRST_TIMER0, 21, 1),
	RK3562_CRU_RESET_OFFSET(SRST_TIMER1, 21, 2),
	RK3562_CRU_RESET_OFFSET(SRST_TIMER2, 21, 3),
	RK3562_CRU_RESET_OFFSET(SRST_TIMER3, 21, 4),
	RK3562_CRU_RESET_OFFSET(SRST_TIMER4, 21, 5),
	RK3562_CRU_RESET_OFFSET(SRST_TIMER5, 21, 6),
	RK3562_CRU_RESET_OFFSET(SRST_P_STIMER, 21, 7),
	RK3562_CRU_RESET_OFFSET(SRST_STIMER0, 21, 8),
	RK3562_CRU_RESET_OFFSET(SRST_STIMER1, 21, 9),

	/* SOFTRST_CON22 */
	RK3562_CRU_RESET_OFFSET(SRST_P_WDTNS, 22, 0),
	RK3562_CRU_RESET_OFFSET(SRST_WDTNS, 22, 1),
	RK3562_CRU_RESET_OFFSET(SRST_P_GRF, 22, 2),
	RK3562_CRU_RESET_OFFSET(SRST_P_SGRF, 22, 3),
	RK3562_CRU_RESET_OFFSET(SRST_P_MAILBOX, 22, 4),
	RK3562_CRU_RESET_OFFSET(SRST_P_INTC, 22, 5),
	RK3562_CRU_RESET_OFFSET(SRST_A_BUS_GIC400, 22, 6),
	RK3562_CRU_RESET_OFFSET(SRST_A_BUS_GIC400_DEBUG, 22, 7),

	/* SOFTRST_CON23 */
	RK3562_CRU_RESET_OFFSET(SRST_A_BUS_SPINLOCK, 23, 0),
	RK3562_CRU_RESET_OFFSET(SRST_A_DCF, 23, 1),
	RK3562_CRU_RESET_OFFSET(SRST_P_DCF, 23, 2),
	RK3562_CRU_RESET_OFFSET(SRST_F_BUS_CM0_CORE, 23, 3),
	RK3562_CRU_RESET_OFFSET(SRST_T_BUS_CM0_JTAG, 23, 5),
	RK3562_CRU_RESET_OFFSET(SRST_H_ICACHE, 23, 8),
	RK3562_CRU_RESET_OFFSET(SRST_H_DCACHE, 23, 9),

	/* SOFTRST_CON24 */
	RK3562_CRU_RESET_OFFSET(SRST_P_TSADC, 24, 0),
	RK3562_CRU_RESET_OFFSET(SRST_TSADC, 24, 1),
	RK3562_CRU_RESET_OFFSET(SRST_TSADCPHY, 24, 2),
	RK3562_CRU_RESET_OFFSET(SRST_P_DFT2APB, 24, 4),

	/* SOFTRST_CON25 */
	RK3562_CRU_RESET_OFFSET(SRST_A_GMAC, 25, 0),
	RK3562_CRU_RESET_OFFSET(SRST_P_APB2ASB_VCCIO156, 25, 1),
	RK3562_CRU_RESET_OFFSET(SRST_P_DSIPHY, 25, 5),
	RK3562_CRU_RESET_OFFSET(SRST_P_DSITX, 25, 8),
	RK3562_CRU_RESET_OFFSET(SRST_P_CPU_EMA_DET, 25, 9),
	RK3562_CRU_RESET_OFFSET(SRST_P_HASH, 25, 10),
	RK3562_CRU_RESET_OFFSET(SRST_P_TOPCRU, 25, 11),

	/* SOFTRST_CON26 */
	RK3562_CRU_RESET_OFFSET(SRST_P_ASB2APB_VCCIO156, 26, 0),
	RK3562_CRU_RESET_OFFSET(SRST_P_IOC_VCCIO156, 26, 1),
	RK3562_CRU_RESET_OFFSET(SRST_P_GPIO3_VCCIO156, 26, 2),
	RK3562_CRU_RESET_OFFSET(SRST_P_GPIO4_VCCIO156, 26, 3),
	RK3562_CRU_RESET_OFFSET(SRST_P_SARADC_VCCIO156, 26, 4),
	RK3562_CRU_RESET_OFFSET(SRST_SARADC_VCCIO156, 26, 5),
	RK3562_CRU_RESET_OFFSET(SRST_SARADC_VCCIO156_PHY, 26, 6),

	/* SOFTRST_CON27 */
	RK3562_CRU_RESET_OFFSET(SRST_A_MAC100, 27, 1),

	/* PMU0_SOFTRST_CON00 */
	RK3562_PMU0CRU_RESET_OFFSET(SRST_P_PMU0_CRU, 0, 0),
	RK3562_PMU0CRU_RESET_OFFSET(SRST_P_PMU0_PMU, 0, 1),
	RK3562_PMU0CRU_RESET_OFFSET(SRST_PMU0_PMU, 0, 2),
	RK3562_PMU0CRU_RESET_OFFSET(SRST_P_PMU0_HP_TIMER, 0, 3),
	RK3562_PMU0CRU_RESET_OFFSET(SRST_PMU0_HP_TIMER, 0, 4),
	RK3562_PMU0CRU_RESET_OFFSET(SRST_PMU0_32K_HP_TIMER, 0, 5),
	RK3562_PMU0CRU_RESET_OFFSET(SRST_P_PMU0_PVTM, 0, 6),
	RK3562_PMU0CRU_RESET_OFFSET(SRST_PMU0_PVTM, 0, 7),
	RK3562_PMU0CRU_RESET_OFFSET(SRST_P_IOC_PMUIO, 0, 8),
	RK3562_PMU0CRU_RESET_OFFSET(SRST_P_PMU0_GPIO0, 0, 9),
	RK3562_PMU0CRU_RESET_OFFSET(SRST_PMU0_GPIO0, 0, 10),
	RK3562_PMU0CRU_RESET_OFFSET(SRST_P_PMU0_GRF, 0, 11),
	RK3562_PMU0CRU_RESET_OFFSET(SRST_P_PMU0_SGRF, 0, 12),

	/* PMU0_SOFTRST_CON01 */
	RK3562_PMU0CRU_RESET_OFFSET(SRST_DDR_FAIL_SAFE, 1, 0),
	RK3562_PMU0CRU_RESET_OFFSET(SRST_P_PMU0_SCRKEYGEN, 1, 1),

	/* PMU0_SOFTRST_CON02 */
	RK3562_PMU0CRU_RESET_OFFSET(SRST_P_PMU0_I2C0, 2, 8),
	RK3562_PMU0CRU_RESET_OFFSET(SRST_PMU0_I2C0, 2, 9),

	/* PMU1_SOFTRST_CON00 */
	RK3562_PMU1CRU_RESET_OFFSET(SRST_P_PMU1_CRU, 0, 0),
	RK3562_PMU1CRU_RESET_OFFSET(SRST_H_PMU1_MEM, 0, 2),
	RK3562_PMU1CRU_RESET_OFFSET(SRST_H_PMU1_BIU, 0, 3),
	RK3562_PMU1CRU_RESET_OFFSET(SRST_P_PMU1_BIU, 0, 4),
	RK3562_PMU1CRU_RESET_OFFSET(SRST_P_PMU1_UART0, 0, 7),
	RK3562_PMU1CRU_RESET_OFFSET(SRST_S_PMU1_UART0, 0, 10),

	/* PMU1_SOFTRST_CON01 */
	RK3562_PMU1CRU_RESET_OFFSET(SRST_P_PMU1_SPI0, 1, 0),
	RK3562_PMU1CRU_RESET_OFFSET(SRST_PMU1_SPI0, 1, 1),
	RK3562_PMU1CRU_RESET_OFFSET(SRST_P_PMU1_PWM0, 1, 3),
	RK3562_PMU1CRU_RESET_OFFSET(SRST_PMU1_PWM0, 1, 4),

	/* PMU1_SOFTRST_CON02 */
	RK3562_PMU1CRU_RESET_OFFSET(SRST_F_PMU1_CM0_CORE, 2, 0),
	RK3562_PMU1CRU_RESET_OFFSET(SRST_T_PMU1_CM0_JTAG, 2, 2),
	RK3562_PMU1CRU_RESET_OFFSET(SRST_P_PMU1_WDTNS, 2, 3),
	RK3562_PMU1CRU_RESET_OFFSET(SRST_PMU1_WDTNS, 2, 4),
	RK3562_PMU1CRU_RESET_OFFSET(SRST_PMU1_MAILBOX, 2, 8),

	/* DDR_SOFTRST_CON00 */
	RK3562_DDRCRU_RESET_OFFSET(SRST_MSCH_BRG_BIU, 0, 4),
	RK3562_DDRCRU_RESET_OFFSET(SRST_P_MSCH_BIU, 0, 5),
	RK3562_DDRCRU_RESET_OFFSET(SRST_P_DDR_HWLP, 0, 6),
	RK3562_DDRCRU_RESET_OFFSET(SRST_P_DDR_PHY, 0, 8),
	RK3562_DDRCRU_RESET_OFFSET(SRST_P_DDR_DFICTL, 0, 9),
	RK3562_DDRCRU_RESET_OFFSET(SRST_P_DDR_DMA2DDR, 0, 10),

	/* DDR_SOFTRST_CON01 */
	RK3562_DDRCRU_RESET_OFFSET(SRST_P_DDR_MON, 1, 0),
	RK3562_DDRCRU_RESET_OFFSET(SRST_TM_DDR_MON, 1, 1),
	RK3562_DDRCRU_RESET_OFFSET(SRST_P_DDR_GRF, 1, 2),
	RK3562_DDRCRU_RESET_OFFSET(SRST_P_DDR_CRU, 1, 3),
	RK3562_DDRCRU_RESET_OFFSET(SRST_P_SUBDDR_CRU, 1, 4),

	/* SUBDDR_SOFTRST_CON00 */
	RK3562_SUBDDRCRU_RESET_OFFSET(SRST_MSCH_BIU, 0, 1),
	RK3562_SUBDDRCRU_RESET_OFFSET(SRST_DDR_PHY, 0, 4),
	RK3562_SUBDDRCRU_RESET_OFFSET(SRST_DDR_DFICTL, 0, 5),
	RK3562_SUBDDRCRU_RESET_OFFSET(SRST_DDR_SCRAMBLE, 0, 6),
	RK3562_SUBDDRCRU_RESET_OFFSET(SRST_DDR_MON, 0, 7),
	RK3562_SUBDDRCRU_RESET_OFFSET(SRST_A_DDR_SPLIT, 0, 8),
	RK3562_SUBDDRCRU_RESET_OFFSET(SRST_DDR_DMA2DDR, 0, 9),

	/* PERI_SOFTRST_CON01 */
	RK3562_PERICRU_RESET_OFFSET(SRST_A_PERI_BIU, 1, 3),
	RK3562_PERICRU_RESET_OFFSET(SRST_H_PERI_BIU, 1, 4),
	RK3562_PERICRU_RESET_OFFSET(SRST_P_PERI_BIU, 1, 5),
	RK3562_PERICRU_RESET_OFFSET(SRST_P_PERICRU, 1, 6),

	/* PERI_SOFTRST_CON02 */
	RK3562_PERICRU_RESET_OFFSET(SRST_H_SAI0_8CH, 2, 0),
	RK3562_PERICRU_RESET_OFFSET(SRST_M_SAI0_8CH, 2, 3),
	RK3562_PERICRU_RESET_OFFSET(SRST_H_SAI1_8CH, 2, 5),
	RK3562_PERICRU_RESET_OFFSET(SRST_M_SAI1_8CH, 2, 8),
	RK3562_PERICRU_RESET_OFFSET(SRST_H_SAI2_2CH, 2, 10),
	RK3562_PERICRU_RESET_OFFSET(SRST_M_SAI2_2CH, 2, 13),

	/* PERI_SOFTRST_CON03 */
	RK3562_PERICRU_RESET_OFFSET(SRST_H_DSM, 3, 1),
	RK3562_PERICRU_RESET_OFFSET(SRST_DSM, 3, 2),
	RK3562_PERICRU_RESET_OFFSET(SRST_H_PDM, 3, 4),
	RK3562_PERICRU_RESET_OFFSET(SRST_M_PDM, 3, 5),
	RK3562_PERICRU_RESET_OFFSET(SRST_H_SPDIF, 3, 8),
	RK3562_PERICRU_RESET_OFFSET(SRST_M_SPDIF, 3, 11),

	/* PERI_SOFTRST_CON04 */
	RK3562_PERICRU_RESET_OFFSET(SRST_H_SDMMC0, 4, 0),
	RK3562_PERICRU_RESET_OFFSET(SRST_H_SDMMC1, 4, 2),
	RK3562_PERICRU_RESET_OFFSET(SRST_H_EMMC, 4, 8),
	RK3562_PERICRU_RESET_OFFSET(SRST_A_EMMC, 4, 9),
	RK3562_PERICRU_RESET_OFFSET(SRST_C_EMMC, 4, 10),
	RK3562_PERICRU_RESET_OFFSET(SRST_B_EMMC, 4, 11),
	RK3562_PERICRU_RESET_OFFSET(SRST_T_EMMC, 4, 12),
	RK3562_PERICRU_RESET_OFFSET(SRST_S_SFC, 4, 13),
	RK3562_PERICRU_RESET_OFFSET(SRST_H_SFC, 4, 14),

	/* PERI_SOFTRST_CON05 */
	RK3562_PERICRU_RESET_OFFSET(SRST_H_USB2HOST, 5, 0),
	RK3562_PERICRU_RESET_OFFSET(SRST_H_USB2HOST_ARB, 5, 1),
	RK3562_PERICRU_RESET_OFFSET(SRST_USB2HOST_UTMI, 5, 2),

	/* PERI_SOFTRST_CON06 */
	RK3562_PERICRU_RESET_OFFSET(SRST_P_SPI1, 6, 0),
	RK3562_PERICRU_RESET_OFFSET(SRST_SPI1, 6, 1),
	RK3562_PERICRU_RESET_OFFSET(SRST_P_SPI2, 6, 3),
	RK3562_PERICRU_RESET_OFFSET(SRST_SPI2, 6, 4),

	/* PERI_SOFTRST_CON07 */
	RK3562_PERICRU_RESET_OFFSET(SRST_P_UART1, 7, 0),
	RK3562_PERICRU_RESET_OFFSET(SRST_P_UART2, 7, 1),
	RK3562_PERICRU_RESET_OFFSET(SRST_P_UART3, 7, 2),
	RK3562_PERICRU_RESET_OFFSET(SRST_P_UART4, 7, 3),
	RK3562_PERICRU_RESET_OFFSET(SRST_P_UART5, 7, 4),
	RK3562_PERICRU_RESET_OFFSET(SRST_P_UART6, 7, 5),
	RK3562_PERICRU_RESET_OFFSET(SRST_P_UART7, 7, 6),
	RK3562_PERICRU_RESET_OFFSET(SRST_P_UART8, 7, 7),
	RK3562_PERICRU_RESET_OFFSET(SRST_P_UART9, 7, 8),
	RK3562_PERICRU_RESET_OFFSET(SRST_S_UART1, 7, 11),
	RK3562_PERICRU_RESET_OFFSET(SRST_S_UART2, 7, 14),

	/* PERI_SOFTRST_CON08 */
	RK3562_PERICRU_RESET_OFFSET(SRST_S_UART3, 8, 1),
	RK3562_PERICRU_RESET_OFFSET(SRST_S_UART4, 8, 4),
	RK3562_PERICRU_RESET_OFFSET(SRST_S_UART5, 8, 7),
	RK3562_PERICRU_RESET_OFFSET(SRST_S_UART6, 8, 10),
	RK3562_PERICRU_RESET_OFFSET(SRST_S_UART7, 8, 13),

	/* PERI_SOFTRST_CON09 */
	RK3562_PERICRU_RESET_OFFSET(SRST_S_UART8, 9, 0),
	RK3562_PERICRU_RESET_OFFSET(SRST_S_UART9, 9, 3),

	/* PERI_SOFTRST_CON10 */
	RK3562_PERICRU_RESET_OFFSET(SRST_P_PWM1_PERI, 10, 0),
	RK3562_PERICRU_RESET_OFFSET(SRST_PWM1_PERI, 10, 1),
	RK3562_PERICRU_RESET_OFFSET(SRST_P_PWM2_PERI, 10, 3),
	RK3562_PERICRU_RESET_OFFSET(SRST_PWM2_PERI, 10, 4),
	RK3562_PERICRU_RESET_OFFSET(SRST_P_PWM3_PERI, 10, 6),
	RK3562_PERICRU_RESET_OFFSET(SRST_PWM3_PERI, 10, 7),

	/* PERI_SOFTRST_CON11 */
	RK3562_PERICRU_RESET_OFFSET(SRST_P_CAN0, 11, 0),
	RK3562_PERICRU_RESET_OFFSET(SRST_CAN0, 11, 1),
	RK3562_PERICRU_RESET_OFFSET(SRST_P_CAN1, 11, 2),
	RK3562_PERICRU_RESET_OFFSET(SRST_CAN1, 11, 3),

	/* PERI_SOFTRST_CON12 */
	RK3562_PERICRU_RESET_OFFSET(SRST_A_CRYPTO, 12, 0),
	RK3562_PERICRU_RESET_OFFSET(SRST_H_CRYPTO, 12, 1),
	RK3562_PERICRU_RESET_OFFSET(SRST_P_CRYPTO, 12, 2),
	RK3562_PERICRU_RESET_OFFSET(SRST_CORE_CRYPTO, 12, 3),
	RK3562_PERICRU_RESET_OFFSET(SRST_PKA_CRYPTO, 12, 4),
	RK3562_PERICRU_RESET_OFFSET(SRST_H_KLAD, 12, 5),
	RK3562_PERICRU_RESET_OFFSET(SRST_P_KEY_READER, 12, 6),
	RK3562_PERICRU_RESET_OFFSET(SRST_H_RK_RNG_NS, 12, 7),
	RK3562_PERICRU_RESET_OFFSET(SRST_H_RK_RNG_S, 12, 8),
	RK3562_PERICRU_RESET_OFFSET(SRST_H_TRNG_NS, 12, 9),
	RK3562_PERICRU_RESET_OFFSET(SRST_H_TRNG_S, 12, 10),
	RK3562_PERICRU_RESET_OFFSET(SRST_H_CRYPTO_S, 12, 11),

	/* PERI_SOFTRST_CON13 */
	RK3562_PERICRU_RESET_OFFSET(SRST_P_PERI_WDT, 13, 0),
	RK3562_PERICRU_RESET_OFFSET(SRST_T_PERI_WDT, 13, 1),
	RK3562_PERICRU_RESET_OFFSET(SRST_A_SYSMEM, 13, 2),
	RK3562_PERICRU_RESET_OFFSET(SRST_H_BOOTROM, 13, 3),
	RK3562_PERICRU_RESET_OFFSET(SRST_P_PERI_GRF, 13, 4),
	RK3562_PERICRU_RESET_OFFSET(SRST_A_DMAC, 13, 5),
	RK3562_PERICRU_RESET_OFFSET(SRST_A_RKDMAC, 13, 6),

	/* PERI_SOFTRST_CON14 */
	RK3562_PERICRU_RESET_OFFSET(SRST_P_OTPC_NS, 14, 0),
	RK3562_PERICRU_RESET_OFFSET(SRST_SBPI_OTPC_NS, 14, 1),
	RK3562_PERICRU_RESET_OFFSET(SRST_USER_OTPC_NS, 14, 2),
	RK3562_PERICRU_RESET_OFFSET(SRST_P_OTPC_S, 14, 3),
	RK3562_PERICRU_RESET_OFFSET(SRST_SBPI_OTPC_S, 14, 4),
	RK3562_PERICRU_RESET_OFFSET(SRST_USER_OTPC_S, 14, 5),
	RK3562_PERICRU_RESET_OFFSET(SRST_OTPC_ARB, 14, 6),
	RK3562_PERICRU_RESET_OFFSET(SRST_P_OTPPHY, 14, 7),
	RK3562_PERICRU_RESET_OFFSET(SRST_OTP_NPOR, 14, 8),

	/* PERI_SOFTRST_CON15 */
	RK3562_PERICRU_RESET_OFFSET(SRST_P_USB2PHY, 15, 0),
	RK3562_PERICRU_RESET_OFFSET(SRST_USB2PHY_POR, 15, 4),
	RK3562_PERICRU_RESET_OFFSET(SRST_USB2PHY_OTG, 15, 5),
	RK3562_PERICRU_RESET_OFFSET(SRST_USB2PHY_HOST, 15, 6),
	RK3562_PERICRU_RESET_OFFSET(SRST_P_PIPEPHY, 15, 7),

	/* PERI_SOFTRST_CON16 */
	RK3562_PERICRU_RESET_OFFSET(SRST_P_SARADC, 16, 4),
	RK3562_PERICRU_RESET_OFFSET(SRST_SARADC, 16, 5),
	RK3562_PERICRU_RESET_OFFSET(SRST_SARADC_PHY, 16, 6),
	RK3562_PERICRU_RESET_OFFSET(SRST_P_IOC_VCCIO234, 16, 12),

	/* PERI_SOFTRST_CON17 */
	RK3562_PERICRU_RESET_OFFSET(SRST_P_PERI_GPIO1, 17, 0),
	RK3562_PERICRU_RESET_OFFSET(SRST_P_PERI_GPIO2, 17, 1),
	RK3562_PERICRU_RESET_OFFSET(SRST_PERI_GPIO1, 17, 2),
	RK3562_PERICRU_RESET_OFFSET(SRST_PERI_GPIO2, 17, 3),
};

void rk3562_rst_init(struct device_node *np, void __iomem *reg_base)
{
	rockchip_register_softrst_lut(np,
				      rk3562_register_offset,
				      ARRAY_SIZE(rk3562_register_offset),
				      reg_base + RK3562_SOFTRST_CON(0),
				      ROCKCHIP_SOFTRST_HIWORD_MASK);
}
