/*
 * Copyright (c) 2022-2023 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#if defined(ARM_COMPUTE_ENABLE_SME2)

#include <algorithm>
#include <cstddef>
#include "arm_gemm.hpp"

using arm_gemm::Requantize32;

namespace arm_conv {
namespace depthwise {

void sme2_u8s8u8q_planar_5x5_s1_4rows_dot_za_impl(
  const uint8_t *inptr,
  size_t ld_in_row,
  size_t ld_in_col,
  size_t ld_in_vl,
  unsigned int pad_top,
  unsigned int valid_input_rows,
  unsigned int pad_left,
  unsigned int valid_input_cols,
  const int8_t *weights,
  uint8_t **outptrs,
  const size_t *outlds,
  const size_t *outvllds,
  unsigned int output_cols,
  unsigned int start_channel,
  unsigned int valid_channels,
  const arm_gemm::Requantize32 &qp
)
{
  struct Args
  {
    const uint8_t *inptr;
    size_t ld_in_vl;
    long unsigned int pad_top, pad_bottom, pad_left;
    const int8_t *weights;
    long unsigned int input_cols, output_cols;
    uint8_t **outptrs;
    const size_t *ld_out_cols;
    const size_t *ld_out_vls;
    long unsigned int current_channel, n_channels;
  };

  Args args = { inptr, ld_in_vl, pad_top, 8u - std::min(8u, pad_top + valid_input_rows), pad_left, weights, valid_input_cols, output_cols, outptrs, outlds, outvllds, start_channel, valid_channels };

  __asm__ __volatile__(
    ".inst 0xd503477f  // SMSTART ZA\n"
    "ldr x5, [%x[args], %[offsetof_Args_pad_bottom]]\n"
    "ptrue p2.b\n"
    "mov x20, #0x8\n"
    "ldr x6, [%x[args], %[offsetof_Args_pad_top]]\n"
    "ld1rh { z17.h }, p2/Z, [%x[qp], %[offsetof_Requantize32_a_offset]]\n"
    "sub x20, x20, x5\n"
    ".inst 0x25207812  // ptrue pn10.b\n"
    "ldr x7, [%x[args], %[offsetof_Args_n_channels]]\n"
    "whilelt p1.s, XZR, x7\n"
    "whilelt p9.s, XZR, x20\n"
    "ld1rw { z12.s }, p2/Z, [%x[qp], %[offsetof_Requantize32_c_offset]]\n"
    "whilelt p8.s, XZR, x6\n"
    "addvl SP, SP, #-30\n"
    "ldr x17, [%x[args], %[offsetof_Args_current_channel]]\n"
    "neg z17.h, p2/M, z17.h\n"
    "eor p8.b, p2/Z, p8.b, p9.b\n"
    "ld1rw { z7.s }, p2/Z, [%x[qp], %[offsetof_Requantize32_per_layer_mul]]\n"
    "ld1rw { z4.s }, p2/Z, [%x[qp], %[offsetof_Requantize32_per_layer_right_shift]]\n"
    "ld1rw { z24.s }, p2/Z, [%x[qp], %[offsetof_Requantize32_minval]]\n"
    "ld1rw { z16.s }, p2/Z, [%x[qp], %[offsetof_Requantize32_maxval]]\n"
    "1:"  // Channel loop
    "ldr x20, [%x[qp], %[offsetof_Requantize32_bias]]\n"
    "mov z18.s, #0x0\n"
    "cbz x20, 2f\n"
    "ld1w { z18.s }, p1/Z, [x20, x17, LSL #2]\n"
    "2:"  // Load bias: Done
    "ldr x23, [%x[args], %[offsetof_Args_weights]]\n"
    "mov x20, x23\n"
    "ld1sb { z2.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "ld1rh { z3.h }, p2/Z, [%x[qp], %[offsetof_Requantize32_b_offset]]\n"
    "mov z15.h, #0x0\n"
    "sub z2.h, z2.h, z3.h\n"
    "incw x23\n"
    "ld1sb { z13.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "sub z13.h, z13.h, z3.h\n"
    "trn1 z11.h, z15.h, z2.h\n"
    "ld1sb { z27.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "sub z27.h, z27.h, z3.h\n"
    "trn1 z0.h, z2.h, z13.h\n"
    "ld1sb { z19.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "sub z19.h, z19.h, z3.h\n"
    "trn1 z26.h, z13.h, z27.h\n"
    "ld1sb { z14.s }, p2/Z, [x20]\n"
    "sub z14.h, z14.h, z3.h\n"
    "mov x20, x23\n"
    "trn1 z10.h, z27.h, z19.h\n"
    "ld1sb { z9.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "trn1 z19.h, z19.h, z14.h\n"
    "trn1 z1.h, z14.h, z15.h\n"
    "ld1sb { z5.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "sub z9.h, z9.h, z3.h\n"
    "sub z5.h, z5.h, z3.h\n"
    "ld1sb { z29.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "sub z29.h, z29.h, z3.h\n"
    "addvl x22, SP, #30\n"
    "ld1sb { z2.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "incw x23\n"
    "sub z2.h, z2.h, z3.h\n"
    "ld1sb { z23.s }, p2/Z, [x20]\n"
    "addvl x22, x22, #-6\n"
    "sub z23.h, z23.h, z3.h\n"
    "mov x20, x23\n"
    "st1h { z11.h }, p2, [x22]\n"
    "trn1 z20.h, z15.h, z9.h\n"
    "incw x23\n"
    "ldr x21, [%x[qp], %[offsetof_Requantize32_per_channel_muls]]\n"
    "st1h { z0.h }, p2, [x22, #1, MUL VL]\n"
    "trn1 z22.h, z9.h, z5.h\n"
    "ld1sb { z25.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "st1h { z26.h }, p2, [x22, #2, MUL VL]\n"
    "trn1 z9.h, z5.h, z29.h\n"
    "ld1sb { z21.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "st1h { z10.h }, p2, [x22, #3, MUL VL]\n"
    "trn1 z26.h, z29.h, z2.h\n"
    "ld1sb { z0.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "st1h { z19.h }, p2, [x22, #4, MUL VL]\n"
    "trn1 z28.h, z2.h, z23.h\n"
    "ld1sb { z19.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "st1h { z1.h }, p2, [x22, #5, MUL VL]\n"
    "trn1 z2.h, z23.h, z15.h\n"
    "sub z25.h, z25.h, z3.h\n"
    "addvl x22, x22, #-6\n"
    "sub z21.h, z21.h, z3.h\n"
    "ld1sb { z6.s }, p2/Z, [x20]\n"
    "sub z0.h, z0.h, z3.h\n"
    "mov x20, x23\n"
    "sub z19.h, z19.h, z3.h\n"
    "sub z6.h, z6.h, z3.h\n"
    "st1h { z20.h }, p2, [x22]\n"
    "incw x23\n"
    "st1h { z22.h }, p2, [x22, #1, MUL VL]\n"
    "trn1 z11.h, z15.h, z25.h\n"
    "trn1 z10.h, z25.h, z21.h\n"
    "ld1sb { z5.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "st1h { z9.h }, p2, [x22, #2, MUL VL]\n"
    "trn1 z14.h, z21.h, z0.h\n"
    "ld1sb { z23.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "st1h { z26.h }, p2, [x22, #3, MUL VL]\n"
    "trn1 z21.h, z0.h, z19.h\n"
    "ld1sb { z27.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "st1h { z28.h }, p2, [x22, #4, MUL VL]\n"
    "trn1 z19.h, z19.h, z6.h\n"
    "ld1sb { z29.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "st1h { z2.h }, p2, [x22, #5, MUL VL]\n"
    "trn1 z13.h, z6.h, z15.h\n"
    "sub z5.h, z5.h, z3.h\n"
    "sub z23.h, z23.h, z3.h\n"
    "ld1sb { z1.s }, p2/Z, [x20]\n"
    "addvl x22, x22, #-6\n"
    "sub z27.h, z27.h, z3.h\n"
    "sub z29.h, z29.h, z3.h\n"
    "mov x20, x23\n"
    "st1h { z11.h }, p2, [x22]\n"
    "sub z1.h, z1.h, z3.h\n"
    "st1h { z10.h }, p2, [x22, #1, MUL VL]\n"
    "trn1 z30.h, z15.h, z5.h\n"
    "trn1 z26.h, z5.h, z23.h\n"
    "ld1sb { z11.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "st1h { z14.h }, p2, [x22, #2, MUL VL]\n"
    "trn1 z22.h, z23.h, z27.h\n"
    "ld1sb { z5.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "st1h { z21.h }, p2, [x22, #3, MUL VL]\n"
    "trn1 z28.h, z27.h, z29.h\n"
    "ld1sb { z8.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "st1h { z19.h }, p2, [x22, #4, MUL VL]\n"
    "trn1 z27.h, z29.h, z1.h\n"
    "ld1sb { z9.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "st1h { z13.h }, p2, [x22, #5, MUL VL]\n"
    "trn1 z2.h, z1.h, z15.h\n"
    "ld1sb { z14.s }, p2/Z, [x20]\n"
    "sub z11.h, z11.h, z3.h\n"
    "addvl x22, x22, #-6\n"
    "sub z5.h, z5.h, z3.h\n"
    "sub z8.h, z8.h, z3.h\n"
    "st1h { z30.h }, p2, [x22]\n"
    "sub z9.h, z9.h, z3.h\n"
    "sub z14.h, z14.h, z3.h\n"
    "st1h { z26.h }, p2, [x22, #1, MUL VL]\n"
    "st1h { z22.h }, p2, [x22, #2, MUL VL]\n"
    "mov z19.d, z18.d\n"
    "trn1 z22.h, z15.h, z11.h\n"
    "st1h { z28.h }, p2, [x22, #3, MUL VL]\n"
    "trn1 z1.h, z11.h, z5.h\n"
    "trn1 z31.h, z5.h, z8.h\n"
    "st1h { z27.h }, p2, [x22, #4, MUL VL]\n"
    "trn1 z8.h, z8.h, z9.h\n"
    "trn1 z21.h, z9.h, z14.h\n"
    "st1h { z2.h }, p2, [x22, #5, MUL VL]\n"
    "addvl x22, x22, #-6\n"
    "trn1 z15.h, z14.h, z15.h\n"
    "st1h { z22.h }, p2, [x22]\n"
    "st1h { z1.h }, p2, [x22, #1, MUL VL]\n"
    "st1h { z31.h }, p2, [x22, #2, MUL VL]\n"
    "st1h { z8.h }, p2, [x22, #3, MUL VL]\n"
    "st1h { z21.h }, p2, [x22, #4, MUL VL]\n"
    "st1h { z15.h }, p2, [x22, #5, MUL VL]\n"
    "cbz x21, 3f\n"
    "ld1w { z7.s }, p1/Z, [x21, x17, LSL #2]\n"
    "3:"  // Load mul: End
    "ldr x20, [%x[qp], %[offsetof_Requantize32_per_channel_right_shifts]]\n"
    "cbz x20, 4f\n"
    "ld1w { z4.s }, p1/Z, [x20, x17, LSL #2]\n"
    "4:"  // Load right_shift: End
    "ldr x25, [%x[args], %[offsetof_Args_input_cols]]\n"
    "sub x20, x25, #0x1\n"
    "orr x23, x20, %x[ld_in_col], LSL #16\n"
    "ldr x16, [%x[args], %[offsetof_Args_inptr]]\n"
    "orr x23, x7, x23, LSL #22\n"
    "mov x22, #0x8\n"
    "add x21, x6, x5\n"
    "lsl x20, %x[ld_in_row], #0x0\n"
    "ldr x15, [%x[args], %[offsetof_Args_output_cols]]\n"
    "mov x11, #0x0\n"
    "mov x8, #0x8\n"
    "lsl x23, x23, #0x0\n"
    "sub x22, x22, x21\n"
    "madd x20, x20, x6, x16\n"
    "5:"  // Issue prefetches
    "subs x22, x22, #0x1\n"
    ".inst 0xf8b74a9c  // rprfm pldstrm, x23, [x20]\n"
    "add x20, x20, %x[ld_in_col]\n"
    "bgt 5b\n"
    "ldr x23, [%x[args], %[offsetof_Args_outptrs]]\n"
    "lsl x20, %x[ld_in_row], #0x0\n"
    "msub x16, x6, x20, x16\n"
    ".inst 0xc0046a40  // mova za.d[x11, #0], { z18.d-z19.d }\n"
    "ldr x20, [%x[args], %[offsetof_Args_ld_out_cols]]\n"
    ".inst 0xc0046a41  // mova za.d[x11, #1], { z18.d-z19.d }\n"
    "mov x22, #0x4\n"
    "ldp x14, x13, [x23], #0x10\n"
    ".inst 0xc0046a42  // mova za.d[x11, #2], { z18.d-z19.d }\n"
    "ldp x4, x10, [x20], #0x10\n"
    ".inst 0xc0046a43  // mova za.d[x11, #3], { z18.d-z19.d }\n"
    "ldr x21, [%x[args], %[offsetof_Args_pad_left]]\n"
    ".inst 0xc0046a44  // mova za.d[x11, #4], { z18.d-z19.d }\n"
    "ldp x9, x28, [x23], #0x10\n"
    ".inst 0xc0046a45  // mova za.d[x11, #5], { z18.d-z19.d }\n"
    "ldp x27, x26, [x20], #0x10\n"
    ".inst 0xc0046a46  // mova za.d[x11, #6], { z18.d-z19.d }\n"
    ".inst 0xc0046a47  // mova za.d[x11, #7], { z18.d-z19.d }\n"
    ".inst 0xc0040a40  // mova za.d[x8, #0], { z18.d-z19.d }\n"
    ".inst 0xc0040a41  // mova za.d[x8, #1], { z18.d-z19.d }\n"
    "cbz x21, 7f\n"
    "cmp x21, x22\n"
    "csel x20, x21, x22, LT\n"
    "sub x21, x21, x20\n"
    "sub x22, x22, x20\n"
    "cbz x21, 7f\n"
    ".inst 0xc0066814  // mova { z20.d-z21.d }, za.d[x11, #0]\n"
    "sub x15, x15, x21\n"
    ".inst 0xc0066836  // mova { z22.d-z23.d }, za.d[x11, #1]\n"
    ".inst 0xc1a7ac14  // sqdmulh { z20.s-z23.s }, { z20.s-z23.s }, z7.s\n"
    ".inst 0xc1a4aa34  // srshl { z20.s-z23.s }, { z20.s-z23.s }, z4.s\n"
    ".inst 0xc1acab14  // add { z20.s-z23.s }, { z20.s-z23.s }, z12.s\n"
    ".inst 0xc1b0cf14  // sclamp { z20.s-z23.s }, z24.s, z16.s\n"
    "6:"  // Left padding
    "subs x21, x21, #0x1\n"
    "st1b { z20.s }, p1, [x14]\n"
    "add x14, x14, x4\n"
    "st1b { z22.s }, p1, [x13]\n"
    "add x13, x13, x10\n"
    "st1b { z21.s }, p1, [x9]\n"
    "add x9, x9, x27\n"
    "st1b { z23.s }, p1, [x28]\n"
    "add x28, x28, x26\n"
    "bgt 6b\n"
    "7:"  // Left padding: End
    "adds XZR, x6, x5\n"
    "bne 14f\n"
    "cbz x22, 12f\n"
    "cmp x22, #0x1\n"
    "sub x25, x25, x22\n"
    "beq 11f\n"
    "cmp x22, #0x2\n"
    "beq 10f\n"
    "cmp x22, #0x3\n"
    "beq 9f\n"
    "8:"  // Unpadded: 4 priming loads
    "add x21, x16, %x[ld_in_row]\n"
    "ld1b { z1.s }, p1/Z, [x16]\n"
    "addvl x20, SP, #24\n"
    "ld1b { z28.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z27.h, z1.h, z28.h\n"
    "add z27.h, z27.h, z17.h\n"
    "ld1b { z1.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "add x16, x16, %x[ld_in_col]\n"
    "ld1b { z2.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z28.h, z1.h, z2.h\n"
    "add z28.h, z28.h, z17.h\n"
    "ld1b { z13.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "ld1b { z6.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z29.h, z13.h, z6.h\n"
    "add z29.h, z29.h, z17.h\n"
    "ld1b { z30.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    ".inst 0xa1402a82  // ld1h { z2.h, z10.h }, pn10.b/Z, [x20]\n"
    ".inst 0xc16a7768  // sdot za.s[x11, 0], { z27.h-z28.h }, z10.h\n"
    "ld1b { z20.s }, p1/Z, [x21]\n"
    "trn1 z30.h, z30.h, z20.h\n"
    ".inst 0xc1627769  // sdot za.s[x11, 1], { z27.h-z28.h }, z2.h\n"
    ".inst 0xa1412a81  // ld1h { z1.h, z9.h }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    "add z30.h, z30.h, z17.h\n"
    ".inst 0xc1697788  // sdot za.s[x11, 0], { z28.h-z29.h }, z9.h\n"
    ".inst 0xc1617789  // sdot za.s[x11, 1], { z28.h-z29.h }, z1.h\n"
    ".inst 0xa0422a8a  // ld1h { z10.h-z11.h }, pn10.b/Z, [x20, #0x4, MUL VL]\n"
    ".inst 0xc16b77a8  // sdot za.s[x11, 0], { z29.h-z30.h }, z11.h\n"
    ".inst 0xc16a77a9  // sdot za.s[x11, 1], { z29.h-z30.h }, z10.h\n"
    "9:"  // Unpadded: 3 priming loads
    "add x22, x16, %x[ld_in_row]\n"
    "ld1b { z2.s }, p1/Z, [x16]\n"
    "addvl x21, SP, #18\n"
    "ld1b { z28.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "trn1 z20.h, z2.h, z28.h\n"
    "add z20.h, z20.h, z17.h\n"
    "ld1b { z31.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "addvl x20, SP, #24\n"
    "ld1b { z11.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "trn1 z21.h, z31.h, z11.h\n"
    "add z21.h, z21.h, z17.h\n"
    "ld1b { z25.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "add x16, x16, %x[ld_in_col]\n"
    "ld1b { z8.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "trn1 z22.h, z25.h, z8.h\n"
    "add z22.h, z22.h, z17.h\n"
    "ld1b { z8.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    ".inst 0xa1402aa6  // ld1h { z6.h, z14.h }, pn10.b/Z, [x21]\n"
    ".inst 0xc16e7688  // sdot za.s[x11, 0], { z20.h-z21.h }, z14.h\n"
    "ld1b { z3.s }, p1/Z, [x22]\n"
    "trn1 z23.h, z8.h, z3.h\n"
    ".inst 0xc1667689  // sdot za.s[x11, 1], { z20.h-z21.h }, z6.h\n"
    ".inst 0xa0402a80  // ld1h { z0.h-z1.h }, pn10.b/Z, [x20]\n"
    ".inst 0xc161768a  // sdot za.s[x11, 2], { z20.h-z21.h }, z1.h\n"
    "add z23.h, z23.h, z17.h\n"
    ".inst 0xa1412aa1  // ld1h { z1.h, z9.h }, pn10.b/Z, [x21, #0x2, MUL VL]\n"
    ".inst 0xc160768b  // sdot za.s[x11, 3], { z20.h-z21.h }, z0.h\n"
    ".inst 0xc16976a8  // sdot za.s[x11, 0], { z21.h-z22.h }, z9.h\n"
    ".inst 0xa0422aae  // ld1h { z14.h-z15.h }, pn10.b/Z, [x21, #0x4, MUL VL]\n"
    ".inst 0xc16176a9  // sdot za.s[x11, 1], { z21.h-z22.h }, z1.h\n"
    ".inst 0xa1412a81  // ld1h { z1.h, z9.h }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    ".inst 0xc16976aa  // sdot za.s[x11, 2], { z21.h-z22.h }, z9.h\n"
    ".inst 0xc16176ab  // sdot za.s[x11, 3], { z21.h-z22.h }, z1.h\n"
    ".inst 0xc16f76c8  // sdot za.s[x11, 0], { z22.h-z23.h }, z15.h\n"
    ".inst 0xc16e76c9  // sdot za.s[x11, 1], { z22.h-z23.h }, z14.h\n"
    ".inst 0xa0422a8a  // ld1h { z10.h-z11.h }, pn10.b/Z, [x20, #0x4, MUL VL]\n"
    ".inst 0xc16b76ca  // sdot za.s[x11, 2], { z22.h-z23.h }, z11.h\n"
    ".inst 0xc16a76cb  // sdot za.s[x11, 3], { z22.h-z23.h }, z10.h\n"
    "10:"  // Unpadded: 2 priming loads
    "add x23, x16, %x[ld_in_row]\n"
    "ld1b { z2.s }, p1/Z, [x16]\n"
    "addvl x22, SP, #12\n"
    "ld1b { z22.s }, p1/Z, [x23]\n"
    "add x23, x23, %x[ld_in_row]\n"
    "trn1 z0.h, z2.h, z22.h\n"
    "add z0.h, z0.h, z17.h\n"
    "ld1b { z14.s }, p1/Z, [x23]\n"
    "add x23, x23, %x[ld_in_row]\n"
    "addvl x21, SP, #18\n"
    "ld1b { z6.s }, p1/Z, [x23]\n"
    "add x23, x23, %x[ld_in_row]\n"
    "trn1 z1.h, z14.h, z6.h\n"
    "add z1.h, z1.h, z17.h\n"
    "ld1b { z15.s }, p1/Z, [x23]\n"
    "add x23, x23, %x[ld_in_row]\n"
    "addvl x20, SP, #24\n"
    "ld1b { z6.s }, p1/Z, [x23]\n"
    "add x23, x23, %x[ld_in_row]\n"
    "trn1 z2.h, z15.h, z6.h\n"
    "add z2.h, z2.h, z17.h\n"
    "ld1b { z21.s }, p1/Z, [x23]\n"
    "add x23, x23, %x[ld_in_row]\n"
    "add x16, x16, %x[ld_in_col]\n"
    ".inst 0xa0402ace  // ld1h { z14.h-z15.h }, pn10.b/Z, [x22]\n"
    ".inst 0xc16f7408  // sdot za.s[x11, 0], { z0.h-z1.h }, z15.h\n"
    "ld1b { z30.s }, p1/Z, [x23]\n"
    "trn1 z3.h, z21.h, z30.h\n"
    ".inst 0xc16e7409  // sdot za.s[x11, 1], { z0.h-z1.h }, z14.h\n"
    ".inst 0xa1402aa5  // ld1h { z5.h, z13.h }, pn10.b/Z, [x21]\n"
    ".inst 0xc16d740a  // sdot za.s[x11, 2], { z0.h-z1.h }, z13.h\n"
    "add z3.h, z3.h, z17.h\n"
    ".inst 0xa0412ace  // ld1h { z14.h-z15.h }, pn10.b/Z, [x22, #0x2, MUL VL]\n"
    ".inst 0xc165740b  // sdot za.s[x11, 3], { z0.h-z1.h }, z5.h\n"
    ".inst 0xa0402a8a  // ld1h { z10.h-z11.h }, pn10.b/Z, [x20]\n"
    ".inst 0xc16f7428  // sdot za.s[x11, 0], { z1.h-z2.h }, z15.h\n"
    ".inst 0xc16e7429  // sdot za.s[x11, 1], { z1.h-z2.h }, z14.h\n"
    ".inst 0xa0412aae  // ld1h { z14.h-z15.h }, pn10.b/Z, [x21, #0x2, MUL VL]\n"
    ".inst 0xa0422ac8  // ld1h { z8.h-z9.h }, pn10.b/Z, [x22, #0x4, MUL VL]\n"
    ".inst 0xc16b740c  // sdot za.s[x11, 4], { z0.h-z1.h }, z11.h\n"
    ".inst 0xc16a740d  // sdot za.s[x11, 5], { z0.h-z1.h }, z10.h\n"
    ".inst 0xc16f742a  // sdot za.s[x11, 2], { z1.h-z2.h }, z15.h\n"
    ".inst 0xc16e742b  // sdot za.s[x11, 3], { z1.h-z2.h }, z14.h\n"
    ".inst 0xa0412a8e  // ld1h { z14.h-z15.h }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    ".inst 0xc1697448  // sdot za.s[x11, 0], { z2.h-z3.h }, z9.h\n"
    ".inst 0xc1687449  // sdot za.s[x11, 1], { z2.h-z3.h }, z8.h\n"
    ".inst 0xa0422aaa  // ld1h { z10.h-z11.h }, pn10.b/Z, [x21, #0x4, MUL VL]\n"
    ".inst 0xc16f742c  // sdot za.s[x11, 4], { z1.h-z2.h }, z15.h\n"
    ".inst 0xc16e742d  // sdot za.s[x11, 5], { z1.h-z2.h }, z14.h\n"
    ".inst 0xc16b744a  // sdot za.s[x11, 2], { z2.h-z3.h }, z11.h\n"
    ".inst 0xc16a744b  // sdot za.s[x11, 3], { z2.h-z3.h }, z10.h\n"
    ".inst 0xa0422a80  // ld1h { z0.h-z1.h }, pn10.b/Z, [x20, #0x4, MUL VL]\n"
    ".inst 0xc161744c  // sdot za.s[x11, 4], { z2.h-z3.h }, z1.h\n"
    ".inst 0xc160744d  // sdot za.s[x11, 5], { z2.h-z3.h }, z0.h\n"
    "11:"  // Unpadded: 1 priming loads
    "add x24, x16, %x[ld_in_row]\n"
    "ld1b { z0.s }, p1/Z, [x16]\n"
    "addvl x23, SP, #6\n"
    "ld1b { z3.s }, p1/Z, [x24]\n"
    "add x24, x24, %x[ld_in_row]\n"
    "trn1 z28.h, z0.h, z3.h\n"
    "add z28.h, z28.h, z17.h\n"
    "ld1b { z6.s }, p1/Z, [x24]\n"
    "add x24, x24, %x[ld_in_row]\n"
    "addvl x22, SP, #12\n"
    "ld1b { z30.s }, p1/Z, [x24]\n"
    "add x24, x24, %x[ld_in_row]\n"
    "trn1 z29.h, z6.h, z30.h\n"
    "add z29.h, z29.h, z17.h\n"
    "ld1b { z1.s }, p1/Z, [x24]\n"
    "add x24, x24, %x[ld_in_row]\n"
    "addvl x21, SP, #18\n"
    "ld1b { z25.s }, p1/Z, [x24]\n"
    "add x24, x24, %x[ld_in_row]\n"
    "trn1 z30.h, z1.h, z25.h\n"
    "add z30.h, z30.h, z17.h\n"
    "ld1b { z3.s }, p1/Z, [x24]\n"
    "add x24, x24, %x[ld_in_row]\n"
    "addvl x20, SP, #24\n"
    ".inst 0xa0402ae0  // ld1h { z0.h-z1.h }, pn10.b/Z, [x23]\n"
    ".inst 0xc1617788  // sdot za.s[x11, 0], { z28.h-z29.h }, z1.h\n"
    "add x16, x16, %x[ld_in_col]\n"
    "ld1b { z5.s }, p1/Z, [x24]\n"
    "trn1 z31.h, z3.h, z5.h\n"
    ".inst 0xc1607789  // sdot za.s[x11, 1], { z28.h-z29.h }, z0.h\n"
    ".inst 0xa1402ac6  // ld1h { z6.h, z14.h }, pn10.b/Z, [x22]\n"
    ".inst 0xc16e778a  // sdot za.s[x11, 2], { z28.h-z29.h }, z14.h\n"
    "add z31.h, z31.h, z17.h\n"
    ".inst 0xa1412ae2  // ld1h { z2.h, z10.h }, pn10.b/Z, [x23, #0x2, MUL VL]\n"
    ".inst 0xc166778b  // sdot za.s[x11, 3], { z28.h-z29.h }, z6.h\n"
    ".inst 0xa0402aae  // ld1h { z14.h-z15.h }, pn10.b/Z, [x21]\n"
    ".inst 0xc16a77a8  // sdot za.s[x11, 0], { z29.h-z30.h }, z10.h\n"
    ".inst 0xc16277a9  // sdot za.s[x11, 1], { z29.h-z30.h }, z2.h\n"
    ".inst 0xa0412ac8  // ld1h { z8.h-z9.h }, pn10.b/Z, [x22, #0x2, MUL VL]\n"
    ".inst 0xa1422ae2  // ld1h { z2.h, z10.h }, pn10.b/Z, [x23, #0x4, MUL VL]\n"
    ".inst 0xc16f778c  // sdot za.s[x11, 4], { z28.h-z29.h }, z15.h\n"
    ".inst 0xc16e778d  // sdot za.s[x11, 5], { z28.h-z29.h }, z14.h\n"
    ".inst 0xa1402a86  // ld1h { z6.h, z14.h }, pn10.b/Z, [x20]\n"
    ".inst 0xc16977aa  // sdot za.s[x11, 2], { z29.h-z30.h }, z9.h\n"
    ".inst 0xc16877ab  // sdot za.s[x11, 3], { z29.h-z30.h }, z8.h\n"
    ".inst 0xa1412aa5  // ld1h { z5.h, z13.h }, pn10.b/Z, [x21, #0x2, MUL VL]\n"
    ".inst 0xc16a77c8  // sdot za.s[x11, 0], { z30.h-z31.h }, z10.h\n"
    ".inst 0xc16277c9  // sdot za.s[x11, 1], { z30.h-z31.h }, z2.h\n"
    ".inst 0xa1422ac2  // ld1h { z2.h, z10.h }, pn10.b/Z, [x22, #0x4, MUL VL]\n"
    ".inst 0xc16e778e  // sdot za.s[x11, 6], { z28.h-z29.h }, z14.h\n"
    ".inst 0xc166778f  // sdot za.s[x11, 7], { z28.h-z29.h }, z6.h\n"
    ".inst 0xc16d77ac  // sdot za.s[x11, 4], { z29.h-z30.h }, z13.h\n"
    ".inst 0xc16577ad  // sdot za.s[x11, 5], { z29.h-z30.h }, z5.h\n"
    ".inst 0xa1412a86  // ld1h { z6.h, z14.h }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    ".inst 0xc16a77ca  // sdot za.s[x11, 2], { z30.h-z31.h }, z10.h\n"
    ".inst 0xc16277cb  // sdot za.s[x11, 3], { z30.h-z31.h }, z2.h\n"
    ".inst 0xa0422aa8  // ld1h { z8.h-z9.h }, pn10.b/Z, [x21, #0x4, MUL VL]\n"
    ".inst 0xc16e77ae  // sdot za.s[x11, 6], { z29.h-z30.h }, z14.h\n"
    ".inst 0xc16677af  // sdot za.s[x11, 7], { z29.h-z30.h }, z6.h\n"
    ".inst 0xc16977cc  // sdot za.s[x11, 4], { z30.h-z31.h }, z9.h\n"
    ".inst 0xc16877cd  // sdot za.s[x11, 5], { z30.h-z31.h }, z8.h\n"
    ".inst 0xa1422a86  // ld1h { z6.h, z14.h }, pn10.b/Z, [x20, #0x4, MUL VL]\n"
    ".inst 0xc16e77ce  // sdot za.s[x11, 6], { z30.h-z31.h }, z14.h\n"
    ".inst 0xc16677cf  // sdot za.s[x11, 7], { z30.h-z31.h }, z6.h\n"
    "12:"  // Unpadded: 0 priming loads
    ".inst 0xa0402be0  // ld1h { z0.h-z1.h }, pn10.b/Z, [SP]\n"
    ".inst 0xa1412be5  // ld1h { z5.h, z13.h }, pn10.b/Z, [SP, #0x2, MUL VL]\n"
    ".inst 0xa0422be2  // ld1h { z2.h-z3.h }, pn10.b/Z, [SP, #0x4, MUL VL]\n"
    "cbz x25, 22f\n"
    "add x20, x16, %x[ld_in_row]\n"
    "ld1b { z26.s }, p1/Z, [x16]\n"
    "sub x25, x25, #0x1\n"
    "ld1b { z28.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "trn1 z25.h, z26.h, z28.h\n"
    "sub x15, x15, #0x1\n"
    "ld1b { z31.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "cmp x25, x15\n"
    "add z25.h, z25.h, z17.h\n"
    "ld1b { z15.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "trn1 z26.h, z31.h, z15.h\n"
    "csel x25, x25, x15, LT\n"
    "ld1b { z22.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "add z26.h, z26.h, z17.h\n"
    "add x16, x16, %x[ld_in_col]\n"
    "ld1b { z8.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "trn1 z27.h, z22.h, z8.h\n"
    "add z27.h, z27.h, z17.h\n"
    "ld1b { z21.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "sub x15, x15, x25\n"
    "ld1b { z20.s }, p1/Z, [x20]\n"
    "trn1 z28.h, z21.h, z20.h\n"
    "add z28.h, z28.h, z17.h\n"
    "cbz x25, 21f\n"
    "13:"  // Unpadded: Main loop
    "addvl x24, SP, #6\n"
    ".inst 0xc1617728  // sdot za.s[x11, 0], { z25.h-z26.h }, z1.h\n"
    "addvl x23, SP, #12\n"
    "ld1b { z21.s }, p1/Z, [x16]\n"
    ".inst 0xc1607729  // sdot za.s[x11, 1], { z25.h-z26.h }, z0.h\n"
    ".inst 0xa0402b0e  // ld1h { z14.h-z15.h }, pn10.b/Z, [x24]\n"
    "addvl x22, SP, #18\n"
    "addvl x21, SP, #24\n"
    ".inst 0xc16f772a  // sdot za.s[x11, 2], { z25.h-z26.h }, z15.h\n"
    "add x20, x16, %x[ld_in_row]\n"
    "ld1b { z0.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0xc16e772b  // sdot za.s[x11, 3], { z25.h-z26.h }, z14.h\n"
    ".inst 0xa1402ae6  // ld1h { z6.h, z14.h }, pn10.b/Z, [x23]\n"
    "subs x25, x25, #0x1\n"
    "add x16, x16, %x[ld_in_col]\n"
    ".inst 0xc16d7748  // sdot za.s[x11, 0], { z26.h-z27.h }, z13.h\n"
    "ld1b { z20.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0xc1657749  // sdot za.s[x11, 1], { z26.h-z27.h }, z5.h\n"
    ".inst 0xa1412b05  // ld1h { z5.h, z13.h }, pn10.b/Z, [x24, #0x2, MUL VL]\n"
    ".inst 0xc16e772c  // sdot za.s[x11, 4], { z25.h-z26.h }, z14.h\n"
    "ld1b { z31.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0xc166772d  // sdot za.s[x11, 5], { z25.h-z26.h }, z6.h\n"
    ".inst 0xa0402ace  // ld1h { z14.h-z15.h }, pn10.b/Z, [x22]\n"
    ".inst 0xc16d774a  // sdot za.s[x11, 2], { z26.h-z27.h }, z13.h\n"
    "ld1b { z29.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0xc165774b  // sdot za.s[x11, 3], { z26.h-z27.h }, z5.h\n"
    ".inst 0xa1412ae5  // ld1h { z5.h, z13.h }, pn10.b/Z, [x23, #0x2, MUL VL]\n"
    ".inst 0xc1637768  // sdot za.s[x11, 0], { z27.h-z28.h }, z3.h\n"
    "ld1b { z22.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0xc1627769  // sdot za.s[x11, 1], { z27.h-z28.h }, z2.h\n"
    ".inst 0xa1422b02  // ld1h { z2.h, z10.h }, pn10.b/Z, [x24, #0x4, MUL VL]\n"
    ".inst 0xc16f772e  // sdot za.s[x11, 6], { z25.h-z26.h }, z15.h\n"
    "ld1b { z30.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0xc16e772f  // sdot za.s[x11, 7], { z25.h-z26.h }, z14.h\n"
    ".inst 0xa0402aae  // ld1h { z14.h-z15.h }, pn10.b/Z, [x21]\n"
    ".inst 0xc16d774c  // sdot za.s[x11, 4], { z26.h-z27.h }, z13.h\n"
    "ld1b { z6.s }, p1/Z, [x20]\n"
    ".inst 0xc165774d  // sdot za.s[x11, 5], { z26.h-z27.h }, z5.h\n"
    ".inst 0xa1412ac5  // ld1h { z5.h, z13.h }, pn10.b/Z, [x22, #0x2, MUL VL]\n"
    ".inst 0xc16a776a  // sdot za.s[x11, 2], { z27.h-z28.h }, z10.h\n"
    ".inst 0xc162776b  // sdot za.s[x11, 3], { z27.h-z28.h }, z2.h\n"
    ".inst 0xa1422ae2  // ld1h { z2.h, z10.h }, pn10.b/Z, [x23, #0x4, MUL VL]\n"
    ".inst 0xc16d774e  // sdot za.s[x11, 6], { z26.h-z27.h }, z13.h\n"
    ".inst 0xc165774f  // sdot za.s[x11, 7], { z26.h-z27.h }, z5.h\n"
    ".inst 0xa1412aa5  // ld1h { z5.h, z13.h }, pn10.b/Z, [x21, #0x2, MUL VL]\n"
    ".inst 0xc16a776c  // sdot za.s[x11, 4], { z27.h-z28.h }, z10.h\n"
    ".inst 0xc162776d  // sdot za.s[x11, 5], { z27.h-z28.h }, z2.h\n"
    ".inst 0xa1422ac1  // ld1h { z1.h, z9.h }, pn10.b/Z, [x22, #0x4, MUL VL]\n"
    ".inst 0xc169776e  // sdot za.s[x11, 6], { z27.h-z28.h }, z9.h\n"
    ".inst 0xc161776f  // sdot za.s[x11, 7], { z27.h-z28.h }, z1.h\n"
    ".inst 0xa0422aaa  // ld1h { z10.h-z11.h }, pn10.b/Z, [x21, #0x4, MUL VL]\n"
    ".inst 0xc16f1728  // sdot za.s[x8, 0], { z25.h-z26.h }, z15.h\n"
    ".inst 0xc16e1729  // sdot za.s[x8, 1], { z25.h-z26.h }, z14.h\n"
    "trn1 z25.h, z21.h, z0.h\n"
    ".inst 0xa0402be0  // ld1h { z0.h-z1.h }, pn10.b/Z, [SP]\n"
    ".inst 0xc16d1748  // sdot za.s[x8, 0], { z26.h-z27.h }, z13.h\n"
    "add z25.h, z25.h, z17.h\n"
    ".inst 0xc1651749  // sdot za.s[x8, 1], { z26.h-z27.h }, z5.h\n"
    "trn1 z26.h, z20.h, z31.h\n"
    ".inst 0xa1412be5  // ld1h { z5.h, z13.h }, pn10.b/Z, [SP, #0x2, MUL VL]\n"
    ".inst 0xc16b1768  // sdot za.s[x8, 0], { z27.h-z28.h }, z11.h\n"
    "add z26.h, z26.h, z17.h\n"
    ".inst 0xc16a1769  // sdot za.s[x8, 1], { z27.h-z28.h }, z10.h\n"
    "trn1 z27.h, z29.h, z22.h\n"
    "trn1 z28.h, z30.h, z6.h\n"
    "add x8, x8, #0x2\n"
    ".inst 0xc0066808  // mova { z8.d-z9.d }, za.d[x11, #0]\n"
    ".inst 0xa0422be2  // ld1h { z2.h-z3.h }, pn10.b/Z, [SP, #0x4, MUL VL]\n"
    "add z27.h, z27.h, z17.h\n"
    ".inst 0xc006682a  // mova { z10.d-z11.d }, za.d[x11, #1]\n"
    ".inst 0xc1a7ac08  // sqdmulh { z8.s-z11.s }, { z8.s-z11.s }, z7.s\n"
    "add x11, x11, #0x2\n"
    ".inst 0xc1a4aa28  // srshl { z8.s-z11.s }, { z8.s-z11.s }, z4.s\n"
    ".inst 0xc0040a40  // mova za.d[x8, #0], { z18.d-z19.d }\n"
    ".inst 0xc1acab08  // add { z8.s-z11.s }, { z8.s-z11.s }, z12.s\n"
    ".inst 0xc0040a41  // mova za.d[x8, #1], { z18.d-z19.d }\n"
    ".inst 0xc1b0cf08  // sclamp { z8.s-z11.s }, z24.s, z16.s\n"
    "st1b { z8.s }, p1, [x14]\n"
    "add x14, x14, x4\n"
    "add z28.h, z28.h, z17.h\n"
    "st1b { z10.s }, p1, [x13]\n"
    "add x13, x13, x10\n"
    "st1b { z9.s }, p1, [x9]\n"
    "add x9, x9, x27\n"
    "st1b { z11.s }, p1, [x28]\n"
    "add x28, x28, x26\n"
    "bgt 13b\n"
    "b 21f\n"
    "14:"  // Padded
    "cbz x22, 19f\n"
    "cmp x22, #0x1\n"
    "sub x25, x25, x22\n"
    "beq 18f\n"
    "cmp x22, #0x2\n"
    "beq 17f\n"
    "cmp x22, #0x3\n"
    "beq 16f\n"
    "15:"  // Padded: 4 priming loads
    "mov x12, #0x0\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1b { z9.s }, p0/Z, [x16]\n"
    "add z9.h, p0/M, z9.h, z17.h\n"
    "add x21, x16, %x[ld_in_row]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1b { z22.s }, p0/Z, [x21]\n"
    "add z22.h, p0/M, z22.h, z17.h\n"
    "add x21, x21, %x[ld_in_row]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1b { z21.s }, p0/Z, [x21]\n"
    "add z21.h, p0/M, z21.h, z17.h\n"
    "add x21, x21, %x[ld_in_row]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "ld1b { z20.s }, p0/Z, [x21]\n"
    "add z20.h, p0/M, z20.h, z17.h\n"
    "mov x12, #0x4\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z31.h, z9.h, z22.h\n"
    "trn1 z0.h, z21.h, z20.h\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1b { z22.s }, p0/Z, [x21]\n"
    "add z22.h, p0/M, z22.h, z17.h\n"
    "add x21, x21, %x[ld_in_row]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1b { z20.s }, p0/Z, [x21]\n"
    "add z20.h, p0/M, z20.h, z17.h\n"
    "add x21, x21, %x[ld_in_row]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1b { z21.s }, p0/Z, [x21]\n"
    "addvl x20, SP, #24\n"
    "add z21.h, p0/M, z21.h, z17.h\n"
    "add x21, x21, %x[ld_in_row]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    ".inst 0xa1402a82  // ld1h { z2.h, z10.h }, pn10.b/Z, [x20]\n"
    "trn1 z1.h, z22.h, z20.h\n"
    "ld1b { z20.s }, p0/Z, [x21]\n"
    "add z20.h, p0/M, z20.h, z17.h\n"
    ".inst 0xc16a77e8  // sdot za.s[x11, 0], { z31.h-z0.h }, z10.h\n"
    "add x16, x16, %x[ld_in_col]\n"
    ".inst 0xc16277e9  // sdot za.s[x11, 1], { z31.h-z0.h }, z2.h\n"
    ".inst 0xa1412a85  // ld1h { z5.h, z13.h }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    "trn1 z2.h, z21.h, z20.h\n"
    ".inst 0xc16d7408  // sdot za.s[x11, 0], { z0.h-z1.h }, z13.h\n"
    ".inst 0xa0422a88  // ld1h { z8.h-z9.h }, pn10.b/Z, [x20, #0x4, MUL VL]\n"
    ".inst 0xc1657409  // sdot za.s[x11, 1], { z0.h-z1.h }, z5.h\n"
    ".inst 0xc1697428  // sdot za.s[x11, 0], { z1.h-z2.h }, z9.h\n"
    ".inst 0xc1687429  // sdot za.s[x11, 1], { z1.h-z2.h }, z8.h\n"
    "16:"  // Padded: 3 priming loads
    "mov x12, #0x0\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1b { z5.s }, p0/Z, [x16]\n"
    "add z5.h, p0/M, z5.h, z17.h\n"
    "add x20, x16, %x[ld_in_row]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1b { z22.s }, p0/Z, [x20]\n"
    "add z22.h, p0/M, z22.h, z17.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1b { z21.s }, p0/Z, [x20]\n"
    "add z21.h, p0/M, z21.h, z17.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "ld1b { z20.s }, p0/Z, [x20]\n"
    "add z20.h, p0/M, z20.h, z17.h\n"
    "mov x12, #0x4\n"
    "add x20, x20, %x[ld_in_row]\n"
    "trn1 z28.h, z5.h, z22.h\n"
    "trn1 z29.h, z21.h, z20.h\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1b { z22.s }, p0/Z, [x20]\n"
    "add z22.h, p0/M, z22.h, z17.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1b { z20.s }, p0/Z, [x20]\n"
    "add z20.h, p0/M, z20.h, z17.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1b { z21.s }, p0/Z, [x20]\n"
    "addvl x21, SP, #18\n"
    "add z21.h, p0/M, z21.h, z17.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    ".inst 0xa0402aa0  // ld1h { z0.h-z1.h }, pn10.b/Z, [x21]\n"
    "trn1 z30.h, z22.h, z20.h\n"
    "ld1b { z20.s }, p0/Z, [x20]\n"
    "addvl x20, SP, #24\n"
    "add z20.h, p0/M, z20.h, z17.h\n"
    ".inst 0xc1617788  // sdot za.s[x11, 0], { z28.h-z29.h }, z1.h\n"
    ".inst 0xc1607789  // sdot za.s[x11, 1], { z28.h-z29.h }, z0.h\n"
    ".inst 0xa1402a81  // ld1h { z1.h, z9.h }, pn10.b/Z, [x20]\n"
    "trn1 z31.h, z21.h, z20.h\n"
    "add x16, x16, %x[ld_in_col]\n"
    ".inst 0xa0412aae  // ld1h { z14.h-z15.h }, pn10.b/Z, [x21, #0x2, MUL VL]\n"
    ".inst 0xc169778a  // sdot za.s[x11, 2], { z28.h-z29.h }, z9.h\n"
    ".inst 0xc161778b  // sdot za.s[x11, 3], { z28.h-z29.h }, z1.h\n"
    ".inst 0xa1422aa3  // ld1h { z3.h, z11.h }, pn10.b/Z, [x21, #0x4, MUL VL]\n"
    ".inst 0xc16f77a8  // sdot za.s[x11, 0], { z29.h-z30.h }, z15.h\n"
    ".inst 0xc16e77a9  // sdot za.s[x11, 1], { z29.h-z30.h }, z14.h\n"
    ".inst 0xa1412a81  // ld1h { z1.h, z9.h }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    ".inst 0xc16977aa  // sdot za.s[x11, 2], { z29.h-z30.h }, z9.h\n"
    ".inst 0xc16177ab  // sdot za.s[x11, 3], { z29.h-z30.h }, z1.h\n"
    ".inst 0xc16b77c8  // sdot za.s[x11, 0], { z30.h-z31.h }, z11.h\n"
    ".inst 0xc16377c9  // sdot za.s[x11, 1], { z30.h-z31.h }, z3.h\n"
    ".inst 0xa0422a8e  // ld1h { z14.h-z15.h }, pn10.b/Z, [x20, #0x4, MUL VL]\n"
    ".inst 0xc16f77ca  // sdot za.s[x11, 2], { z30.h-z31.h }, z15.h\n"
    ".inst 0xc16e77cb  // sdot za.s[x11, 3], { z30.h-z31.h }, z14.h\n"
    "17:"  // Padded: 2 priming loads
    "mov x12, #0x0\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1b { z29.s }, p0/Z, [x16]\n"
    "add z29.h, p0/M, z29.h, z17.h\n"
    "add x20, x16, %x[ld_in_row]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1b { z22.s }, p0/Z, [x20]\n"
    "add z22.h, p0/M, z22.h, z17.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1b { z21.s }, p0/Z, [x20]\n"
    "add z21.h, p0/M, z21.h, z17.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "ld1b { z20.s }, p0/Z, [x20]\n"
    "add z20.h, p0/M, z20.h, z17.h\n"
    "mov x12, #0x4\n"
    "add x20, x20, %x[ld_in_row]\n"
    "trn1 z8.h, z29.h, z22.h\n"
    "trn1 z9.h, z21.h, z20.h\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1b { z22.s }, p0/Z, [x20]\n"
    "add z22.h, p0/M, z22.h, z17.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1b { z20.s }, p0/Z, [x20]\n"
    "add z20.h, p0/M, z20.h, z17.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1b { z21.s }, p0/Z, [x20]\n"
    "addvl x22, SP, #12\n"
    "add z21.h, p0/M, z21.h, z17.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    ".inst 0xa0402ace  // ld1h { z14.h-z15.h }, pn10.b/Z, [x22]\n"
    "trn1 z10.h, z22.h, z20.h\n"
    "ld1b { z20.s }, p0/Z, [x20]\n"
    "addvl x21, SP, #18\n"
    "add z20.h, p0/M, z20.h, z17.h\n"
    ".inst 0xc16f7508  // sdot za.s[x11, 0], { z8.h-z9.h }, z15.h\n"
    ".inst 0xc16e7509  // sdot za.s[x11, 1], { z8.h-z9.h }, z14.h\n"
    ".inst 0xa1402aa6  // ld1h { z6.h, z14.h }, pn10.b/Z, [x21]\n"
    "addvl x20, SP, #24\n"
    "trn1 z11.h, z21.h, z20.h\n"
    ".inst 0xa1412ac5  // ld1h { z5.h, z13.h }, pn10.b/Z, [x22, #0x2, MUL VL]\n"
    ".inst 0xc16e750a  // sdot za.s[x11, 2], { z8.h-z9.h }, z14.h\n"
    "add x16, x16, %x[ld_in_col]\n"
    ".inst 0xc166750b  // sdot za.s[x11, 3], { z8.h-z9.h }, z6.h\n"
    ".inst 0xa0402a8e  // ld1h { z14.h-z15.h }, pn10.b/Z, [x20]\n"
    ".inst 0xc16d7528  // sdot za.s[x11, 0], { z9.h-z10.h }, z13.h\n"
    ".inst 0xa0422ac0  // ld1h { z0.h-z1.h }, pn10.b/Z, [x22, #0x4, MUL VL]\n"
    ".inst 0xc1657529  // sdot za.s[x11, 1], { z9.h-z10.h }, z5.h\n"
    ".inst 0xa1412aa5  // ld1h { z5.h, z13.h }, pn10.b/Z, [x21, #0x2, MUL VL]\n"
    ".inst 0xc16f750c  // sdot za.s[x11, 4], { z8.h-z9.h }, z15.h\n"
    ".inst 0xc16e750d  // sdot za.s[x11, 5], { z8.h-z9.h }, z14.h\n"
    ".inst 0xc16d752a  // sdot za.s[x11, 2], { z9.h-z10.h }, z13.h\n"
    ".inst 0xc165752b  // sdot za.s[x11, 3], { z9.h-z10.h }, z5.h\n"
    ".inst 0xa1412a86  // ld1h { z6.h, z14.h }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    ".inst 0xc1617548  // sdot za.s[x11, 0], { z10.h-z11.h }, z1.h\n"
    ".inst 0xc1607549  // sdot za.s[x11, 1], { z10.h-z11.h }, z0.h\n"
    ".inst 0xa0422aa0  // ld1h { z0.h-z1.h }, pn10.b/Z, [x21, #0x4, MUL VL]\n"
    ".inst 0xc16e752c  // sdot za.s[x11, 4], { z9.h-z10.h }, z14.h\n"
    ".inst 0xc166752d  // sdot za.s[x11, 5], { z9.h-z10.h }, z6.h\n"
    ".inst 0xc161754a  // sdot za.s[x11, 2], { z10.h-z11.h }, z1.h\n"
    ".inst 0xc160754b  // sdot za.s[x11, 3], { z10.h-z11.h }, z0.h\n"
    ".inst 0xa0422a8e  // ld1h { z14.h-z15.h }, pn10.b/Z, [x20, #0x4, MUL VL]\n"
    ".inst 0xc16f754c  // sdot za.s[x11, 4], { z10.h-z11.h }, z15.h\n"
    ".inst 0xc16e754d  // sdot za.s[x11, 5], { z10.h-z11.h }, z14.h\n"
    "18:"  // Padded: 1 priming loads
    "mov x12, #0x0\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1b { z1.s }, p0/Z, [x16]\n"
    "add z1.h, p0/M, z1.h, z17.h\n"
    "add x20, x16, %x[ld_in_row]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1b { z22.s }, p0/Z, [x20]\n"
    "add z22.h, p0/M, z22.h, z17.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1b { z21.s }, p0/Z, [x20]\n"
    "add z21.h, p0/M, z21.h, z17.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "ld1b { z20.s }, p0/Z, [x20]\n"
    "add z20.h, p0/M, z20.h, z17.h\n"
    "mov x12, #0x4\n"
    "add x20, x20, %x[ld_in_row]\n"
    "trn1 z26.h, z1.h, z22.h\n"
    "trn1 z27.h, z21.h, z20.h\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1b { z22.s }, p0/Z, [x20]\n"
    "add z22.h, p0/M, z22.h, z17.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1b { z20.s }, p0/Z, [x20]\n"
    "add z20.h, p0/M, z20.h, z17.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1b { z21.s }, p0/Z, [x20]\n"
    "addvl x23, SP, #6\n"
    "add z21.h, p0/M, z21.h, z17.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    ".inst 0xa0402aee  // ld1h { z14.h-z15.h }, pn10.b/Z, [x23]\n"
    "trn1 z28.h, z22.h, z20.h\n"
    "ld1b { z20.s }, p0/Z, [x20]\n"
    "addvl x22, SP, #12\n"
    "add z20.h, p0/M, z20.h, z17.h\n"
    ".inst 0xc16f7748  // sdot za.s[x11, 0], { z26.h-z27.h }, z15.h\n"
    ".inst 0xc16e7749  // sdot za.s[x11, 1], { z26.h-z27.h }, z14.h\n"
    ".inst 0xa0402ac0  // ld1h { z0.h-z1.h }, pn10.b/Z, [x22]\n"
    "addvl x21, SP, #18\n"
    "trn1 z29.h, z21.h, z20.h\n"
    ".inst 0xa0412aea  // ld1h { z10.h-z11.h }, pn10.b/Z, [x23, #0x2, MUL VL]\n"
    ".inst 0xc161774a  // sdot za.s[x11, 2], { z26.h-z27.h }, z1.h\n"
    "addvl x20, SP, #24\n"
    "add x16, x16, %x[ld_in_col]\n"
    ".inst 0xc160774b  // sdot za.s[x11, 3], { z26.h-z27.h }, z0.h\n"
    ".inst 0xa1402aa6  // ld1h { z6.h, z14.h }, pn10.b/Z, [x21]\n"
    ".inst 0xc16b7768  // sdot za.s[x11, 0], { z27.h-z28.h }, z11.h\n"
    ".inst 0xa0422ae8  // ld1h { z8.h-z9.h }, pn10.b/Z, [x23, #0x4, MUL VL]\n"
    ".inst 0xc16a7769  // sdot za.s[x11, 1], { z27.h-z28.h }, z10.h\n"
    ".inst 0xa0412aca  // ld1h { z10.h-z11.h }, pn10.b/Z, [x22, #0x2, MUL VL]\n"
    ".inst 0xc16e774c  // sdot za.s[x11, 4], { z26.h-z27.h }, z14.h\n"
    ".inst 0xc166774d  // sdot za.s[x11, 5], { z26.h-z27.h }, z6.h\n"
    ".inst 0xa1402a85  // ld1h { z5.h, z13.h }, pn10.b/Z, [x20]\n"
    ".inst 0xc16b776a  // sdot za.s[x11, 2], { z27.h-z28.h }, z11.h\n"
    ".inst 0xc16a776b  // sdot za.s[x11, 3], { z27.h-z28.h }, z10.h\n"
    ".inst 0xa1412aa6  // ld1h { z6.h, z14.h }, pn10.b/Z, [x21, #0x2, MUL VL]\n"
    ".inst 0xc1697788  // sdot za.s[x11, 0], { z28.h-z29.h }, z9.h\n"
    ".inst 0xc1687789  // sdot za.s[x11, 1], { z28.h-z29.h }, z8.h\n"
    ".inst 0xa1422ac2  // ld1h { z2.h, z10.h }, pn10.b/Z, [x22, #0x4, MUL VL]\n"
    ".inst 0xc16d774e  // sdot za.s[x11, 6], { z26.h-z27.h }, z13.h\n"
    ".inst 0xc165774f  // sdot za.s[x11, 7], { z26.h-z27.h }, z5.h\n"
    ".inst 0xc16e776c  // sdot za.s[x11, 4], { z27.h-z28.h }, z14.h\n"
    ".inst 0xc166776d  // sdot za.s[x11, 5], { z27.h-z28.h }, z6.h\n"
    ".inst 0xa1412a86  // ld1h { z6.h, z14.h }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    ".inst 0xc16a778a  // sdot za.s[x11, 2], { z28.h-z29.h }, z10.h\n"
    ".inst 0xc162778b  // sdot za.s[x11, 3], { z28.h-z29.h }, z2.h\n"
    ".inst 0xa0422aa0  // ld1h { z0.h-z1.h }, pn10.b/Z, [x21, #0x4, MUL VL]\n"
    ".inst 0xc16e776e  // sdot za.s[x11, 6], { z27.h-z28.h }, z14.h\n"
    ".inst 0xc166776f  // sdot za.s[x11, 7], { z27.h-z28.h }, z6.h\n"
    ".inst 0xc161778c  // sdot za.s[x11, 4], { z28.h-z29.h }, z1.h\n"
    ".inst 0xc160778d  // sdot za.s[x11, 5], { z28.h-z29.h }, z0.h\n"
    ".inst 0xa1422a82  // ld1h { z2.h, z10.h }, pn10.b/Z, [x20, #0x4, MUL VL]\n"
    ".inst 0xc16a778e  // sdot za.s[x11, 6], { z28.h-z29.h }, z10.h\n"
    ".inst 0xc162778f  // sdot za.s[x11, 7], { z28.h-z29.h }, z2.h\n"
    "19:"  // Padded: 0 priming loads
    ".inst 0xa0402be0  // ld1h { z0.h-z1.h }, pn10.b/Z, [SP]\n"
    ".inst 0xa1412be5  // ld1h { z5.h, z13.h }, pn10.b/Z, [SP, #0x2, MUL VL]\n"
    ".inst 0xa0422be2  // ld1h { z2.h-z3.h }, pn10.b/Z, [SP, #0x4, MUL VL]\n"
    "cbz x25, 22f\n"
    "mov x12, #0x0\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1b { z6.s }, p0/Z, [x16]\n"
    "add z6.h, p0/M, z6.h, z17.h\n"
    "add x20, x16, %x[ld_in_row]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1b { z30.s }, p0/Z, [x20]\n"
    "add z30.h, p0/M, z30.h, z17.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1b { z27.s }, p0/Z, [x20]\n"
    "add z27.h, p0/M, z27.h, z17.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "ld1b { z26.s }, p0/Z, [x20]\n"
    "add z26.h, p0/M, z26.h, z17.h\n"
    "mov x12, #0x4\n"
    "add x20, x20, %x[ld_in_row]\n"
    "trn1 z25.h, z6.h, z30.h\n"
    "trn1 z26.h, z27.h, z26.h\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1b { z8.s }, p0/Z, [x20]\n"
    "add z8.h, p0/M, z8.h, z17.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1b { z9.s }, p0/Z, [x20]\n"
    "add z9.h, p0/M, z9.h, z17.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1b { z21.s }, p0/Z, [x20]\n"
    "add z21.h, p0/M, z21.h, z17.h\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "ld1b { z29.s }, p0/Z, [x20]\n"
    "add z29.h, p0/M, z29.h, z17.h\n"
    "sub x25, x25, #0x1\n"
    "sub x15, x15, #0x1\n"
    "cmp x25, x15\n"
    "trn1 z27.h, z8.h, z9.h\n"
    "trn1 z28.h, z21.h, z29.h\n"
    "csel x25, x25, x15, LT\n"
    "add x16, x16, %x[ld_in_col]\n"
    "sub x15, x15, x25\n"
    "cbz x25, 21f\n"
    "20:"  // Padded: Main loop
    "mov x12, #0x0\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1b { z8.s }, p0/Z, [x16]\n"
    "add z8.h, p0/M, z8.h, z17.h\n"
    "add x24, x16, %x[ld_in_row]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1b { z21.s }, p0/Z, [x24]\n"
    ".inst 0xc1617728  // sdot za.s[x11, 0], { z25.h-z26.h }, z1.h\n"
    "addvl x23, SP, #6\n"
    ".inst 0xc1607729  // sdot za.s[x11, 1], { z25.h-z26.h }, z0.h\n"
    ".inst 0xa0402ae0  // ld1h { z0.h-z1.h }, pn10.b/Z, [x23]\n"
    "addvl x22, SP, #12\n"
    "add z21.h, p0/M, z21.h, z17.h\n"
    "add x24, x24, %x[ld_in_row]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    ".inst 0xc161772a  // sdot za.s[x11, 2], { z25.h-z26.h }, z1.h\n"
    ".inst 0xc160772b  // sdot za.s[x11, 3], { z25.h-z26.h }, z0.h\n"
    ".inst 0xa1402ac6  // ld1h { z6.h, z14.h }, pn10.b/Z, [x22]\n"
    "addvl x21, SP, #18\n"
    "addvl x20, SP, #24\n"
    "ld1b { z29.s }, p0/Z, [x24]\n"
    ".inst 0xc16d7748  // sdot za.s[x11, 0], { z26.h-z27.h }, z13.h\n"
    "add z29.h, p0/M, z29.h, z17.h\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    ".inst 0xc1657749  // sdot za.s[x11, 1], { z26.h-z27.h }, z5.h\n"
    ".inst 0xa1412ae5  // ld1h { z5.h, z13.h }, pn10.b/Z, [x23, #0x2, MUL VL]\n"
    "mov x12, #0x4\n"
    "add x24, x24, %x[ld_in_row]\n"
    ".inst 0xc16e772c  // sdot za.s[x11, 4], { z25.h-z26.h }, z14.h\n"
    "ld1b { z30.s }, p0/Z, [x24]\n"
    "add z30.h, p0/M, z30.h, z17.h\n"
    "add x24, x24, %x[ld_in_row]\n"
    ".inst 0xc166772d  // sdot za.s[x11, 5], { z25.h-z26.h }, z6.h\n"
    ".inst 0xa1402aa6  // ld1h { z6.h, z14.h }, pn10.b/Z, [x21]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "subs x25, x25, #0x1\n"
    ".inst 0xc16d774a  // sdot za.s[x11, 2], { z26.h-z27.h }, z13.h\n"
    "ld1b { z15.s }, p0/Z, [x24]\n"
    "add z15.h, p0/M, z15.h, z17.h\n"
    "add x24, x24, %x[ld_in_row]\n"
    ".inst 0xc165774b  // sdot za.s[x11, 3], { z26.h-z27.h }, z5.h\n"
    ".inst 0xa0412aca  // ld1h { z10.h-z11.h }, pn10.b/Z, [x22, #0x2, MUL VL]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x16, x16, %x[ld_in_col]\n"
    ".inst 0xc1637768  // sdot za.s[x11, 0], { z27.h-z28.h }, z3.h\n"
    "ld1b { z20.s }, p0/Z, [x24]\n"
    "add z20.h, p0/M, z20.h, z17.h\n"
    "add x24, x24, %x[ld_in_row]\n"
    ".inst 0xc1627769  // sdot za.s[x11, 1], { z27.h-z28.h }, z2.h\n"
    ".inst 0xa1422ae1  // ld1h { z1.h, z9.h }, pn10.b/Z, [x23, #0x4, MUL VL]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    ".inst 0xc16e772e  // sdot za.s[x11, 6], { z25.h-z26.h }, z14.h\n"
    "ld1b { z31.s }, p0/Z, [x24]\n"
    "add z31.h, p0/M, z31.h, z17.h\n"
    "add x24, x24, %x[ld_in_row]\n"
    ".inst 0xc166772f  // sdot za.s[x11, 7], { z25.h-z26.h }, z6.h\n"
    ".inst 0xa0402a82  // ld1h { z2.h-z3.h }, pn10.b/Z, [x20]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    ".inst 0xc16b774c  // sdot za.s[x11, 4], { z26.h-z27.h }, z11.h\n"
    "ld1b { z22.s }, p0/Z, [x24]\n"
    "add z22.h, p0/M, z22.h, z17.h\n"
    ".inst 0xc16a774d  // sdot za.s[x11, 5], { z26.h-z27.h }, z10.h\n"
    ".inst 0xa1412aa6  // ld1h { z6.h, z14.h }, pn10.b/Z, [x21, #0x2, MUL VL]\n"
    ".inst 0xc169776a  // sdot za.s[x11, 2], { z27.h-z28.h }, z9.h\n"
    ".inst 0xc161776b  // sdot za.s[x11, 3], { z27.h-z28.h }, z1.h\n"
    ".inst 0xa0422ac0  // ld1h { z0.h-z1.h }, pn10.b/Z, [x22, #0x4, MUL VL]\n"
    ".inst 0xc16e774e  // sdot za.s[x11, 6], { z26.h-z27.h }, z14.h\n"
    ".inst 0xc166774f  // sdot za.s[x11, 7], { z26.h-z27.h }, z6.h\n"
    ".inst 0xa1412a86  // ld1h { z6.h, z14.h }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    ".inst 0xc161776c  // sdot za.s[x11, 4], { z27.h-z28.h }, z1.h\n"
    ".inst 0xc160776d  // sdot za.s[x11, 5], { z27.h-z28.h }, z0.h\n"
    ".inst 0xa1422aa1  // ld1h { z1.h, z9.h }, pn10.b/Z, [x21, #0x4, MUL VL]\n"
    ".inst 0xc169776e  // sdot za.s[x11, 6], { z27.h-z28.h }, z9.h\n"
    ".inst 0xc161776f  // sdot za.s[x11, 7], { z27.h-z28.h }, z1.h\n"
    ".inst 0xa0422a8a  // ld1h { z10.h-z11.h }, pn10.b/Z, [x20, #0x4, MUL VL]\n"
    ".inst 0xc1631728  // sdot za.s[x8, 0], { z25.h-z26.h }, z3.h\n"
    ".inst 0xc1621729  // sdot za.s[x8, 1], { z25.h-z26.h }, z2.h\n"
    ".inst 0xa0402be0  // ld1h { z0.h-z1.h }, pn10.b/Z, [SP]\n"
    "trn1 z25.h, z8.h, z21.h\n"
    ".inst 0xc16e1748  // sdot za.s[x8, 0], { z26.h-z27.h }, z14.h\n"
    ".inst 0xc1661749  // sdot za.s[x8, 1], { z26.h-z27.h }, z6.h\n"
    ".inst 0xa1412be5  // ld1h { z5.h, z13.h }, pn10.b/Z, [SP, #0x2, MUL VL]\n"
    "trn1 z26.h, z29.h, z30.h\n"
    ".inst 0xc16b1768  // sdot za.s[x8, 0], { z27.h-z28.h }, z11.h\n"
    ".inst 0xc16a1769  // sdot za.s[x8, 1], { z27.h-z28.h }, z10.h\n"
    "add x8, x8, #0x2\n"
    ".inst 0xa0422be2  // ld1h { z2.h-z3.h }, pn10.b/Z, [SP, #0x4, MUL VL]\n"
    "trn1 z27.h, z15.h, z20.h\n"
    ".inst 0xc0066808  // mova { z8.d-z9.d }, za.d[x11, #0]\n"
    "trn1 z28.h, z31.h, z22.h\n"
    ".inst 0xc006682a  // mova { z10.d-z11.d }, za.d[x11, #1]\n"
    ".inst 0xc1a7ac08  // sqdmulh { z8.s-z11.s }, { z8.s-z11.s }, z7.s\n"
    "add x11, x11, #0x2\n"
    ".inst 0xc1a4aa28  // srshl { z8.s-z11.s }, { z8.s-z11.s }, z4.s\n"
    ".inst 0xc0040a40  // mova za.d[x8, #0], { z18.d-z19.d }\n"
    ".inst 0xc1acab08  // add { z8.s-z11.s }, { z8.s-z11.s }, z12.s\n"
    ".inst 0xc0040a41  // mova za.d[x8, #1], { z18.d-z19.d }\n"
    ".inst 0xc1b0cf08  // sclamp { z8.s-z11.s }, z24.s, z16.s\n"
    "st1b { z8.s }, p1, [x14]\n"
    "add x14, x14, x4\n"
    "st1b { z10.s }, p1, [x13]\n"
    "add x13, x13, x10\n"
    "st1b { z9.s }, p1, [x9]\n"
    "add x9, x9, x27\n"
    "st1b { z11.s }, p1, [x28]\n"
    "add x28, x28, x26\n"
    "bgt 20b\n"
    "21:"  // Main loop tail
    "addvl x23, SP, #6\n"
    ".inst 0xc1617728  // sdot za.s[x11, 0], { z25.h-z26.h }, z1.h\n"
    "addvl x22, SP, #12\n"
    ".inst 0xc1607729  // sdot za.s[x11, 1], { z25.h-z26.h }, z0.h\n"
    ".inst 0xa0402ae0  // ld1h { z0.h-z1.h }, pn10.b/Z, [x23]\n"
    "addvl x21, SP, #18\n"
    "addvl x20, SP, #24\n"
    ".inst 0xc161772a  // sdot za.s[x11, 2], { z25.h-z26.h }, z1.h\n"
    ".inst 0xc160772b  // sdot za.s[x11, 3], { z25.h-z26.h }, z0.h\n"
    ".inst 0xa1402ac6  // ld1h { z6.h, z14.h }, pn10.b/Z, [x22]\n"
    ".inst 0xc16d7748  // sdot za.s[x11, 0], { z26.h-z27.h }, z13.h\n"
    ".inst 0xc1657749  // sdot za.s[x11, 1], { z26.h-z27.h }, z5.h\n"
    ".inst 0xa1412ae1  // ld1h { z1.h, z9.h }, pn10.b/Z, [x23, #0x2, MUL VL]\n"
    ".inst 0xc16e772c  // sdot za.s[x11, 4], { z25.h-z26.h }, z14.h\n"
    ".inst 0xc166772d  // sdot za.s[x11, 5], { z25.h-z26.h }, z6.h\n"
    ".inst 0xa1402aa6  // ld1h { z6.h, z14.h }, pn10.b/Z, [x21]\n"
    ".inst 0xc169774a  // sdot za.s[x11, 2], { z26.h-z27.h }, z9.h\n"
    ".inst 0xc161774b  // sdot za.s[x11, 3], { z26.h-z27.h }, z1.h\n"
    ".inst 0xa1412ac1  // ld1h { z1.h, z9.h }, pn10.b/Z, [x22, #0x2, MUL VL]\n"
    ".inst 0xc1637768  // sdot za.s[x11, 0], { z27.h-z28.h }, z3.h\n"
    ".inst 0xc1627769  // sdot za.s[x11, 1], { z27.h-z28.h }, z2.h\n"
    ".inst 0xa0422aea  // ld1h { z10.h-z11.h }, pn10.b/Z, [x23, #0x4, MUL VL]\n"
    ".inst 0xc16e772e  // sdot za.s[x11, 6], { z25.h-z26.h }, z14.h\n"
    ".inst 0xc166772f  // sdot za.s[x11, 7], { z25.h-z26.h }, z6.h\n"
    ".inst 0xa0402a8e  // ld1h { z14.h-z15.h }, pn10.b/Z, [x20]\n"
    ".inst 0xc169774c  // sdot za.s[x11, 4], { z26.h-z27.h }, z9.h\n"
    ".inst 0xc161774d  // sdot za.s[x11, 5], { z26.h-z27.h }, z1.h\n"
    ".inst 0xa1412aa5  // ld1h { z5.h, z13.h }, pn10.b/Z, [x21, #0x2, MUL VL]\n"
    ".inst 0xc16b776a  // sdot za.s[x11, 2], { z27.h-z28.h }, z11.h\n"
    ".inst 0xc16a776b  // sdot za.s[x11, 3], { z27.h-z28.h }, z10.h\n"
    ".inst 0xa0422ac2  // ld1h { z2.h-z3.h }, pn10.b/Z, [x22, #0x4, MUL VL]\n"
    ".inst 0xc16d774e  // sdot za.s[x11, 6], { z26.h-z27.h }, z13.h\n"
    ".inst 0xc165774f  // sdot za.s[x11, 7], { z26.h-z27.h }, z5.h\n"
    ".inst 0xa0412a88  // ld1h { z8.h-z9.h }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    ".inst 0xc163776c  // sdot za.s[x11, 4], { z27.h-z28.h }, z3.h\n"
    ".inst 0xc162776d  // sdot za.s[x11, 5], { z27.h-z28.h }, z2.h\n"
    ".inst 0xa1422aa2  // ld1h { z2.h, z10.h }, pn10.b/Z, [x21, #0x4, MUL VL]\n"
    ".inst 0xc16a776e  // sdot za.s[x11, 6], { z27.h-z28.h }, z10.h\n"
    ".inst 0xc162776f  // sdot za.s[x11, 7], { z27.h-z28.h }, z2.h\n"
    ".inst 0xa0422a80  // ld1h { z0.h-z1.h }, pn10.b/Z, [x20, #0x4, MUL VL]\n"
    ".inst 0xc16f1728  // sdot za.s[x8, 0], { z25.h-z26.h }, z15.h\n"
    ".inst 0xc16e1729  // sdot za.s[x8, 1], { z25.h-z26.h }, z14.h\n"
    ".inst 0xc1691748  // sdot za.s[x8, 0], { z26.h-z27.h }, z9.h\n"
    ".inst 0xc1681749  // sdot za.s[x8, 1], { z26.h-z27.h }, z8.h\n"
    ".inst 0xc1611768  // sdot za.s[x8, 0], { z27.h-z28.h }, z1.h\n"
    ".inst 0xc1601769  // sdot za.s[x8, 1], { z27.h-z28.h }, z0.h\n"
    "add x8, x8, #0x2\n"
    ".inst 0xc0066808  // mova { z8.d-z9.d }, za.d[x11, #0]\n"
    ".inst 0xc006682a  // mova { z10.d-z11.d }, za.d[x11, #1]\n"
    ".inst 0xc1a7ac08  // sqdmulh { z8.s-z11.s }, { z8.s-z11.s }, z7.s\n"
    "add x11, x11, #0x2\n"
    ".inst 0xc1a4aa28  // srshl { z8.s-z11.s }, { z8.s-z11.s }, z4.s\n"
    ".inst 0xc0040a40  // mova za.d[x8, #0], { z18.d-z19.d }\n"
    ".inst 0xc1acab08  // add { z8.s-z11.s }, { z8.s-z11.s }, z12.s\n"
    ".inst 0xc0040a41  // mova za.d[x8, #1], { z18.d-z19.d }\n"
    ".inst 0xc1b0cf08  // sclamp { z8.s-z11.s }, z24.s, z16.s\n"
    "st1b { z8.s }, p1, [x14]\n"
    "add x14, x14, x4\n"
    "st1b { z10.s }, p1, [x13]\n"
    "add x13, x13, x10\n"
    "st1b { z9.s }, p1, [x9]\n"
    "add x9, x9, x27\n"
    "st1b { z11.s }, p1, [x28]\n"
    "add x28, x28, x26\n"
    "22:"  // Main loop skip tail
    "cbz x15, 24f\n"
    "23:"  // Right padding loop
    ".inst 0xc0066808  // mova { z8.d-z9.d }, za.d[x11, #0]\n"
    "add x8, x8, #0x2\n"
    "subs x15, x15, #0x1\n"
    ".inst 0xc006682a  // mova { z10.d-z11.d }, za.d[x11, #1]\n"
    ".inst 0xc1a7ac08  // sqdmulh { z8.s-z11.s }, { z8.s-z11.s }, z7.s\n"
    "add x11, x11, #0x2\n"
    ".inst 0xc1a4aa28  // srshl { z8.s-z11.s }, { z8.s-z11.s }, z4.s\n"
    ".inst 0xc0040a40  // mova za.d[x8, #0], { z18.d-z19.d }\n"
    ".inst 0xc1acab08  // add { z8.s-z11.s }, { z8.s-z11.s }, z12.s\n"
    ".inst 0xc0040a41  // mova za.d[x8, #1], { z18.d-z19.d }\n"
    ".inst 0xc1b0cf08  // sclamp { z8.s-z11.s }, z24.s, z16.s\n"
    "st1b { z8.s }, p1, [x14]\n"
    "add x14, x14, x4\n"
    "st1b { z10.s }, p1, [x13]\n"
    "add x13, x13, x10\n"
    "st1b { z9.s }, p1, [x9]\n"
    "add x9, x9, x27\n"
    "st1b { z11.s }, p1, [x28]\n"
    "add x28, x28, x26\n"
    "bgt 23b\n"
    "24:"  // End
    "ldr x20, [%x[args], %[offsetof_Args_weights]]\n"
    "incw x20, ALL, MUL #16\n"
    "incw x20, ALL, MUL #9\n"
    "str x20, [%x[args], %[offsetof_Args_weights]]\n"
    "ldr x21, [%x[args], %[offsetof_Args_ld_in_vl]]\n"
    "incw x17\n"
    "whilelt p1.s, x17, x7\n"
    "ldr x20, [%x[args], %[offsetof_Args_inptr]]\n"
    "add x20, x20, x21\n"
    "str x20, [%x[args], %[offsetof_Args_inptr]]\n"
    "ldr x25, [%x[args], %[offsetof_Args_outptrs]]\n"
    "ldr x24, [%x[args], %[offsetof_Args_ld_out_vls]]\n"
    "ldp x23, x22, [x25, #0x0]\n"
    "ldp x21, x20, [x24, #0x0]\n"
    "add x23, x23, x21\n"
    "add x22, x22, x20\n"
    "stp x23, x22, [x25, #0x0]\n"
    "ldp x23, x22, [x25, #0x10]\n"
    "ldp x21, x20, [x24, #0x10]\n"
    "add x23, x23, x21\n"
    "add x22, x22, x20\n"
    "stp x23, x22, [x25, #0x10]\n"
    "b.any 1b\n"
    "addvl SP, SP, #30\n"
    ".inst 0xd503467f  // SMSTOP\n"
    :
    : [args] "r" (&args), [ld_in_col] "r" (ld_in_col), [ld_in_row] "r" (ld_in_row), [offsetof_Args_current_channel] "I" (offsetof(Args, current_channel)), [offsetof_Args_inptr] "I" (offsetof(Args, inptr)), [offsetof_Args_input_cols] "I" (offsetof(Args, input_cols)), [offsetof_Args_ld_in_vl] "I" (offsetof(Args, ld_in_vl)), [offsetof_Args_ld_out_cols] "I" (offsetof(Args, ld_out_cols)), [offsetof_Args_ld_out_vls] "I" (offsetof(Args, ld_out_vls)), [offsetof_Args_n_channels] "I" (offsetof(Args, n_channels)), [offsetof_Args_outptrs] "I" (offsetof(Args, outptrs)), [offsetof_Args_output_cols] "I" (offsetof(Args, output_cols)), [offsetof_Args_pad_bottom] "I" (offsetof(Args, pad_bottom)), [offsetof_Args_pad_left] "I" (offsetof(Args, pad_left)), [offsetof_Args_pad_top] "I" (offsetof(Args, pad_top)), [offsetof_Args_weights] "I" (offsetof(Args, weights)), [offsetof_Requantize32_a_offset] "I" (offsetof(arm_gemm::Requantize32, a_offset)), [offsetof_Requantize32_b_offset] "I" (offsetof(arm_gemm::Requantize32, b_offset)), [offsetof_Requantize32_bias] "I" (offsetof(arm_gemm::Requantize32, bias)), [offsetof_Requantize32_c_offset] "I" (offsetof(arm_gemm::Requantize32, c_offset)), [offsetof_Requantize32_maxval] "I" (offsetof(arm_gemm::Requantize32, maxval)), [offsetof_Requantize32_minval] "I" (offsetof(arm_gemm::Requantize32, minval)), [offsetof_Requantize32_per_channel_muls] "I" (offsetof(arm_gemm::Requantize32, per_channel_muls)), [offsetof_Requantize32_per_channel_right_shifts] "I" (offsetof(arm_gemm::Requantize32, per_channel_right_shifts)), [offsetof_Requantize32_per_layer_mul] "I" (offsetof(arm_gemm::Requantize32, per_layer_mul)), [offsetof_Requantize32_per_layer_right_shift] "I" (offsetof(arm_gemm::Requantize32, per_layer_right_shift)), [qp] "r" (&qp)
    : "cc", "memory", "p0", "p1", "p2", "p3", "p4", "p5", "p6", "p7", "p8", "p9", "p10", "p11", "p12", "p13", "p14", "p15", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "x14", "x15", "x16", "x17", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31"
  );
}

}  // namespace depthwise
}  // namespace arm_conv

#endif  // defined(ARM_COMPUTE_ENABLE_SME2)
