/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2012 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "cdw_file_manager.h"
#include "gettext.h"
#include "cdw_create_image.h"
#include "cdw_utils.h"
#include "cdw_debug.h"
#include "cdw_ext_tools.h"
#include "cdw_mkisofs.h"
#include "cdw_xorriso.h"
#include "cdw_main_window.h"
#include "cdw_processwin.h"
#include "cdw_iso9660_wizard.h"


//static cdw_rv_t cdw_create_image_check_boot_image(void);
//static cdw_rv_t cdw_create_image_get_target_data_image_path(void);


/**
   \brief Check preconditions for creating image and invoke process creating image

   Check if there are any files selected by user, do any other
   necessary checks and then call run_command_*() to create ISO
   image from the files.

   \return CDW_NO if some of preconditions were not met
   \return CDW_ERROR on errors
   \return value returned by cdw_mkisofs_run_task()
*/
cdw_rv_t cdw_create_image(void)
{
	cdw_task_t *task = cdw_task_new(CDW_TASK_CREATE_IMAGE, (cdw_disc_t *) NULL);
	if (task == (cdw_task_t *) NULL) {
		cdw_vdm ("ERROR: failed to create a task\n");
		return CDW_ERROR;
	}

	/* create graftpoints file for mkisofs; first checks if there
	   are any files that will be used as source data */
	cdw_rv_t crv = cdw_file_manager_create_graftpoints_file();
	if (crv != CDW_OK) {
		cdw_task_delete(&task);
		return CDW_NO;
	}


	crv = cdw_iso9660_wizard(task);
	if (crv != CDW_OK) {
		cdw_vdm ("INFO: cancelled in wizard\n");
		cdw_task_delete(&task);
		cdw_file_manager_delete_graftpoints_file();
		return CDW_NO;
	}

#if 0
	/* check if user specified boot image, and if so,
	   if the boot image exists */
	crv = cdw_create_image_check_boot_image();
	if (crv != CDW_OK) {
		return CDW_NO;
	}
#endif

	/* 2TRANS: this is title of dialog window */
	cdw_processwin_create(_("Create image"),
			      /* 2TRANS: this is message in dialog
				 window - creating iso image is in
				 progress */
			      _("Creating iso image..."), true);

	/* go! */
	if (task->create_image.tool.id == CDW_TOOL_MKISOFS) {
		cdw_vdm ("INFO: path at call to run task: %s\n", task->create_image.iso9660_file_fullpath);
		crv = cdw_mkisofs_run_task(task, (cdw_disc_t *) NULL);
	} else if (task->create_image.tool.id == CDW_TOOL_XORRISO) {
		crv = cdw_xorriso_run_task(task, (cdw_disc_t *) NULL);
	} else {
		cdw_vdm ("ERROR: tool id is invalid: %lld\n", task->create_image.tool.id);
	}

	/* 2TRANS: this is message in dialog window:
	   operation finished with unknown result */
	cdw_processwin_destroy(_("Finished"), true);

	cdw_file_manager_delete_graftpoints_file();

	/* this function sets task.success according to task->tool_status,
	   and resets task->tool_status; displays error messages in case
	   of (some) problems reported by mkisofs */
	cdw_task_check_tool_status(task);

	if (crv == CDW_OK) {
		cdw_task_save(task);
		/* 2TRANS: this is title of dialog window, window shows
		   messages from program writing selected files to iso
		   image (creating iso image from selected files) */
		after_event(_("\"Create image\" log"), 1);
	} else {
		cdw_vdm ("ERROR: failed to create image\n");
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window:
				      probably some malloc() call failed when
				      program was preparing call of mkisofs;
				      'command' is command-line command. 'Image'
				      means iso image file. */
				   _("System error occurred when preparing command. Image not created."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
	}

	cdw_task_delete(&task);

	return crv;
}




#if 0
cdw_rv_t cdw_create_image_check_boot_image(void)
{
	if (strlen(global_config.boot_image_path)) {
		int rv = cdw_fs_check_existing_path(global_config.boot_image_path, R_OK, CDW_FS_FILE);
		if (rv != 0) {
			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Boot image error"),
					   /* 2TRANS: this is message in dialog window */
					   _("Boot image file doesn't exist or has wrong permissions! Check \"Boot image\" option in Configuration."),
					   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
			return CDW_NO;
		}
	}
	return CDW_OK;
}





cdw_rv_t cdw_create_image_get_target_data_image_path(void)
{
	/* 2TRANS: this is title of dialog window */
	cdw_rv_t crv = cdw_fs_ui_file_picker(_("Path to iso image"),
					     /* 2TRANS: this is message in dialog window;
					     below it there is an input field where user can
					     enter path to ISO image file */
					     _("Please enter FULL path to new iso image file:"),
					     &(global_config.iso_image_full_path),
					     CDW_FS_FILE, R_OK | W_OK, CDW_FS_NEW | CDW_FS_EXISTING);

	cdw_vcrv ("cdw_fs_ui_file_picker", crv);
	if (crv == CDW_OK) {
		cdw_vdm ("INFO: file picker returns \"%s\"\n", global_config.iso_image_full_path);
	} else if (crv == CDW_CANCEL) {
		cdw_vdm ("INFO: pressed escape in file picker, fullpath is \"%s\"\n", global_config.iso_image_full_path);
	} else {
		cdw_vdm ("INFO: file picker returns CDW_ERROR, fullpath is \"%s\"\n", global_config.iso_image_full_path);
	}
	cdw_main_ui_main_window_wrefresh();

	return crv;
}


#endif
