/* $Id: kmo_stats-test.c,v 1.9 2013-10-08 11:18:57 aagudo Exp $
 *
 * This file is part of the KMOS Library
 * Copyright (C) 2002-2006 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: aagudo $
 * $Date: 2013-10-08 11:18:57 $
 * $Revision: 1.9 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <sys/stat.h>

#include <cpl.h>

#include "kmo_dfs.h"

const char  *test_global_path_test_data;
float   test_global_seed_data, test_global_seed_noise;
int     test_global_size_x, test_global_size_y, test_global_size_z, test_global_nr_frames;
void kmo_get_pipe_command(char*, const char*, const char*, int);
int kmo_test_file_exists(const char*);
void kmo_test_verbose_off();
void kmo_test_verbose_on();
double kmo_test_esorex_data(const char*, int);
const char* kmo_test_cat_strings(char*, const char*, const char*);
int  kmo_test_create_RAW_data(int, int, const char*, char**, char**, int*, int, char**, char**, int*, int, float, float);
int  kmo_test_create_RAW_data_zero(int, int, const char*);
int  kmo_test_create_F2D_data(int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2D_data_noise(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2I_data(int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2I_data_noise(int, int, int, const char*, char *, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data(int, int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data_infinite(int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data2(int, int, int, int, int*, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data_noise(int, int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F1I_data(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F1I_data_noise(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_B2D_data(int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);

double          scalar              = 6.7;
int             ifu_nr              = 4;

const char      *path_recipe        = "kmo_stats/";

const char      *valid_files[]      = {"v_data_F3I_all.fits",
                                       "v_data_noise_F3I_all.fits",
                                       "v_data_F3I_all_infinite.fits",
                                       "v_mask_F2I.fits",
                                       "v_data_F2I.fits",
                                       "v_data_F1I.fits",
                                       "v_data_RAW.fits",
                                       "v_mask_F2D_large.fits",
                                       "v_data_F2D.fits",
                                       "v_data_B2D.fits",
                                       "v_data_noise_F2D.fits",
                                       "v_data_noise_F2I.fits",
                                       "v_data_noise_F1I.fits",
                                       "v_mask_F2I_single.fits",
                                       "v_mask_F2D_single.fits"};

const char      *invalid_files[]    = {"i_mask.fits"};

/**
    @defgroup kmo_stats_test   kmo_stats unit tests

    @{
*/

/**
 * @brief   test with two frames
 */
static void test_stats(const char *path_op1,
                       double pos_thres, double neg_thres,
                       int ret_val, int mask, const char *path_op2)
{
    // create sof-file
    const char *sof_path = "test_stats.sof";
    FILE *fh = fopen(sof_path, "w");
    fprintf (fh, "%s\n", path_op1);
    if (mask == TRUE) {
        fprintf (fh, "%s\n", path_op2);
    }
    fclose(fh);

    // create esorex-command
    char esorex_command[512];
    sprintf(esorex_command,
            "esorex --suppress-prefix=TRUE kmo_stats --cpos_rej=%g  --cneg_rej=%g %s",
            pos_thres, neg_thres, sof_path);

    kmo_get_pipe_command(esorex_command,
                         "log_kmo_stats.txt", sof_path, TRUE);

    // execute esorex-command
    if (ret_val == 0) {
        cpl_test_eq(0, system(esorex_command));
    } else {
        cpl_test_noneq(0, system(esorex_command));
    }
}

static void test_stats2(const char *path_op1,
                       double pos_thres, double neg_thres,
                       int ret_val, int mask, const char *path_op2, int ext)
{
    // create sof-file
    const char *sof_path = "test_stats2.sof";
    FILE *fh = fopen(sof_path, "w");
    fprintf (fh, "%s\n", path_op1);
    if (mask == TRUE) {
        fprintf (fh, "%s\n", path_op2);
    }
    fclose(fh);

    // create esorex-command
    char esorex_command[512];

    sprintf(esorex_command,
            "esorex --suppress-prefix=TRUE kmo_stats --cpos_rej=%g --cneg_rej=%g --ext=%d %s",
            pos_thres, neg_thres, ext, sof_path);

    kmo_get_pipe_command(esorex_command,
                         "log_kmo_stats.txt", sof_path, TRUE);

    // execute esorex-command
    if (ret_val == 0) {
        cpl_test_eq(0, system(esorex_command));
    } else {
        cpl_test_noneq(0, system(esorex_command));
    }
}

/**
    @brief
        Generates test data for kmo_stats.

    @param path Path where the generated test date should be saved to.
*/
static int kmo_generate_test_data_stats(const char *path)
{
    char        file_path[256];

    /* ----- valid test data ----- */
    // F3I
    cpl_test_eq(0,
                kmo_test_create_F3I_data(test_global_size_x, test_global_size_y,
                          test_global_size_z, FALSE, test_global_nr_frames,
                          kmo_test_cat_strings(file_path, path, valid_files[0]),
                          NULL, NULL, NULL, 0,
                          NULL, NULL, NULL, 0));

    // F3I with noise
    cpl_test_eq(0,
               kmo_test_create_F3I_data_noise(test_global_size_x, test_global_size_y,
                          test_global_size_z, FALSE, test_global_nr_frames,
                          kmo_test_cat_strings(file_path, path, valid_files[1]),
                          NULL, NULL, NULL, 0,
                          NULL, NULL, NULL, 0));

    // F3I with infinite
    cpl_test_eq(0,
            kmo_test_create_F3I_data_infinite(test_global_size_x, test_global_size_y,
                          test_global_size_z, test_global_nr_frames,
                          kmo_test_cat_strings(file_path, path, valid_files[2]),
                          NULL, NULL, NULL, 0,
                          NULL, NULL, NULL, 0));

    cpl_test_eq(0,
            kmo_test_create_F3I_data_infinite(test_global_size_x, test_global_size_y,
                          test_global_size_z, test_global_nr_frames,
                          kmo_test_cat_strings(file_path, path, valid_files[2]),
                          NULL, NULL, NULL, 0,
                          NULL, NULL, NULL, 0));

    // mask
    cpl_test_eq(0,
            kmo_test_create_F2I_data(test_global_size_x, test_global_size_y, TRUE,
                          test_global_nr_frames,
                          kmo_test_cat_strings(file_path, path, valid_files[3]),
                          NULL, NULL, NULL, 0,
                          NULL, NULL, NULL, 0));

    cpl_test_eq(0,
            kmo_test_create_F2D_data(test_global_size_x, test_global_size_y, TRUE,
                          kmo_test_cat_strings(file_path, path, valid_files[7]),
                          NULL, NULL, NULL, 0,
                          NULL, NULL, NULL, 0));

    // mask, single extension
    cpl_test_eq(0,
            kmo_test_create_F2I_data(test_global_size_x, test_global_size_y, TRUE,
                          1,
                          kmo_test_cat_strings(file_path, path, valid_files[13]),
                          NULL, NULL, NULL, 0,
                          NULL, NULL, NULL, 0));

    // F2I
    cpl_test_eq(0,
                kmo_test_create_F2I_data(test_global_size_x, test_global_size_y, FALSE,
                          test_global_nr_frames,
                          kmo_test_cat_strings(file_path, path, valid_files[4]),
                          NULL, NULL, NULL, 0,
                          NULL, NULL, NULL, 0));

    // F1I
    cpl_test_eq(0,
                kmo_test_create_F1I_data(test_global_size_x,
                          test_global_nr_frames,
                          kmo_test_cat_strings(file_path, path, valid_files[5]),
                          NULL, NULL, NULL, 0,
                          NULL, NULL, NULL, 0));

    // RAW
    cpl_test_eq(0,
                kmo_test_create_RAW_data(test_global_size_x, test_global_size_y,
                          kmo_test_cat_strings(file_path, path, valid_files[6]),
                          NULL, NULL, NULL, 0,
                          NULL, NULL, NULL, 0,
                          1.0, 0.1));

    // F2D
    cpl_test_eq(0,
                kmo_test_create_F2D_data(test_global_size_x, test_global_size_y, FALSE,
                          kmo_test_cat_strings(file_path, path, valid_files[8]),
                          NULL, NULL, NULL, 0,
                          NULL, NULL, NULL, 0));

    // B2D
    cpl_test_eq(0,
                kmo_test_create_B2D_data(test_global_size_x, test_global_size_y, TRUE,
                          kmo_test_cat_strings(file_path, path, valid_files[9]),
                          NULL, NULL, NULL, 0,
                          NULL, NULL, NULL, 0));

    // F2D with noise
    cpl_test_eq(0,
               kmo_test_create_F2D_data_noise(test_global_size_x, test_global_size_y,
                          kmo_test_cat_strings(file_path, path, valid_files[10]),
                          NULL, NULL, NULL, 0,
                          NULL, NULL, NULL, 0));

    // F2I with noise
    cpl_test_eq(0,
                kmo_test_create_F2I_data_noise(test_global_size_x, test_global_size_y,
                          test_global_nr_frames,
                          kmo_test_cat_strings(file_path, path, valid_files[11]),
                          NULL, NULL, NULL, 0,
                          NULL, NULL, NULL, 0));

    // F1I with noise
    cpl_test_eq(0,
                kmo_test_create_F1I_data_noise(test_global_size_x,
                          test_global_nr_frames,
                          kmo_test_cat_strings(file_path, path, valid_files[12]),
                          NULL, NULL, NULL, 0,
                          NULL, NULL, NULL, 0));

    /* ----- invalid test data ----- */
    // to small mask
    cpl_test_eq(0,
            kmo_test_create_F2I_data(test_global_size_x, test_global_size_y-1, TRUE,
                          test_global_nr_frames,
                          kmo_test_cat_strings(file_path, path, invalid_files[0]),
                          NULL, NULL, NULL, 0,
                          NULL, NULL, NULL, 0));

    return 0;
}


/**
  @brief	Test of kmo_stats recipe.
  @param	argc   the number of parameters
  @param    argv   the parameter list

  Call @c kmo_stats-test @c --generate when only the test data for this
  recipe should be generated.
  Call @c kmo_stats-test @c --no-gen when only the tests should be
  executed and no data should be generated.
FALSE
 */
int main(int argc, char *argv[])
{
    float   tol = 0.01;

    char    test_path[256],
            op1_file[256],
            op2_file[256];

    cpl_test_init("<kmos-spark@mpe.mpg.de>", CPL_MSG_WARNING);

    strcpy(test_path, test_global_path_test_data);
    strcat(test_path, path_recipe);

    if (kmo_test_file_exists(test_path) == FALSE) {
        mkdir(test_path, 0777);
    }

    if ((argc == 1) || ((argc > 1) && (strcmp(argv[1], "--no-gen") != 0))) {
        cpl_test_zero(kmo_generate_test_data_stats(test_path));
    }

    if ((argc > 1) && (strcmp(argv[1], "--generate") == 0)) {
        // just generate the data
        return cpl_test_end(0);
    }

    /* ----- run the tests ----- */
    FILE *fd = fopen("log_kmo_stats.txt", "w");
    fprintf(fd, "%s\n", " ");
    fclose(fd);

    /* ----- valid tests ----- */

    //
    // w/o mask, w/o noise
    //

    // F3I all finite
    kmo_test_cat_strings(op1_file, test_path, valid_files[0]);
    test_stats(op1_file, 3.0, 3.0, 0, FALSE, NULL);
    cpl_test_abs(kmo_test_esorex_data(STATS, 0), 167.798, tol);

    // F3I with infinite
    kmo_test_cat_strings(op1_file, test_path, valid_files[2]);
    test_stats(op1_file, 3.0, 3.0, 0, FALSE, NULL);
    cpl_test_abs(kmo_test_esorex_data(STATS, 0), 167.353, tol);

    // F3I all finite, with rejection
    kmo_test_cat_strings(op1_file, test_path, valid_files[0]);
    test_stats(op1_file, 1.0, 1.0, 0, FALSE, NULL);
    cpl_test_abs(kmo_test_esorex_data(STATS, 0), 165.66, tol);

    // F2I
    kmo_test_cat_strings(op1_file, test_path, valid_files[4]);
    test_stats(op1_file, 3.0, 3.0, 0, FALSE, NULL);
    cpl_test_abs(kmo_test_esorex_data(STATS, 0), 51.9522, tol);

    // F1I
    kmo_test_cat_strings(op1_file, test_path, valid_files[5]);
    test_stats(op1_file, 3.0, 3.0, 0, FALSE, NULL);
    cpl_test_abs(kmo_test_esorex_data(STATS, 0), 10.3756, tol);

    // RAW
    kmo_test_cat_strings(op1_file, test_path, valid_files[6]);
    test_stats(op1_file, 3.0, 3.0, 0, FALSE, NULL);
    cpl_test_abs(kmo_test_esorex_data(STATS, 0), 49.0105, tol);

    // F2D
    kmo_test_cat_strings(op1_file, test_path, valid_files[8]);
    test_stats(op1_file, 3.0, 3.0, 0, FALSE, NULL);
    cpl_test_abs(kmo_test_esorex_data(STATS, 0), 51.9522, tol);

    // B2D
    kmo_test_cat_strings(op1_file, test_path, valid_files[9]);
    test_stats(op1_file, 3.0, 3.0, 0, FALSE, NULL);
    cpl_test_abs(kmo_test_esorex_data(STATS, 0), 17.1263, tol);

    //
    // w/o mask, with noise
    //

    // F3I
    kmo_test_cat_strings(op1_file, test_path, valid_files[1]);
    test_stats(op1_file, 3.0, 3.0, 0, FALSE, NULL);
    cpl_test_abs(kmo_test_esorex_data(STATS, 0), 303.162, tol);

    // F2I
    kmo_test_cat_strings(op1_file, test_path, valid_files[11]);
    test_stats(op1_file, 3.0, 3.0, 0, FALSE, NULL);
    cpl_test_abs(kmo_test_esorex_data(STATS, 0), 73.0388, tol);

    // F1I
    kmo_test_cat_strings(op1_file, test_path, valid_files[12]);
    test_stats(op1_file, 3.0, 3.0, 0, FALSE, NULL);
    cpl_test_abs(kmo_test_esorex_data(STATS, 0), 16.2775, tol);

    // F2D
    kmo_test_cat_strings(op1_file, test_path, valid_files[10]);
    test_stats(op1_file, 3.0, 3.0, 0, FALSE, NULL);
    cpl_test_abs(kmo_test_esorex_data(STATS, 0), 73.0388, tol);

    //
    // with mask, w/o noise
    //

    // F3I all finite with mask
    kmo_test_cat_strings(op1_file, test_path, valid_files[0]);
    kmo_test_cat_strings(op2_file, test_path, valid_files[3]);
    test_stats(op1_file, 3.0, 3.0, 0, TRUE, op2_file);
    cpl_test_abs(kmo_test_esorex_data(STATS, 0), 94.7123, tol);

    // F2I with mask
    kmo_test_cat_strings(op1_file, test_path, valid_files[4]);
    kmo_test_cat_strings(op2_file, test_path, valid_files[3]);
    test_stats(op1_file, 3.0, 3.0, 0, TRUE, op2_file);
    cpl_test_abs(kmo_test_esorex_data(STATS, 0), 39.3168, tol);

    // RAW with mask (F2D), specific extension
    kmo_test_cat_strings(op1_file, test_path, valid_files[6]);
    kmo_test_cat_strings(op2_file, test_path, valid_files[7]);
    test_stats2(op1_file, 3.0, 3.0, 0, TRUE, op2_file, 1);
    cpl_test_abs(kmo_test_esorex_data(STATS, 0), 11.6712, tol);

    // F2D with mask (F2D)
    kmo_test_cat_strings(op1_file, test_path, valid_files[8]);
    kmo_test_cat_strings(op2_file, test_path, valid_files[7]);
    test_stats(op1_file, 3.0, 3.0, 0, TRUE, op2_file);
    cpl_test_abs(kmo_test_esorex_data(STATS, 0), 39.3168, tol);

    // B2D with mask (F2D)
    kmo_test_cat_strings(op1_file, test_path, valid_files[9]);
    kmo_test_cat_strings(op2_file, test_path, valid_files[7]);
    test_stats(op1_file, 3.0, 3.0, 0, TRUE, op2_file);
    cpl_test_abs(kmo_test_esorex_data(STATS, 0), 7.90909, tol);

    // RAW with mask (B2D), specific extension
    kmo_test_cat_strings(op1_file, test_path, valid_files[6]);
    kmo_test_cat_strings(op2_file, test_path, valid_files[9]);
    test_stats2(op1_file, 3.0, 3.0, 0, TRUE, op2_file, 1);
    cpl_test_abs(kmo_test_esorex_data(STATS, 0), 11.6712, tol);

    // F2D with mask (B2D)
    kmo_test_cat_strings(op1_file, test_path, valid_files[8]);
    kmo_test_cat_strings(op2_file, test_path, valid_files[9]);
    test_stats(op1_file, 3.0, 3.0, 0, TRUE, op2_file);
    cpl_test_abs(kmo_test_esorex_data(STATS, 0), 39.3168, tol);

    // B2D with mask (B2D)
    kmo_test_cat_strings(op1_file, test_path, valid_files[9]);
    kmo_test_cat_strings(op2_file, test_path, valid_files[9]);
    test_stats(op1_file, 3.0, 3.0, 0, TRUE, op2_file);
    cpl_test_abs(kmo_test_esorex_data(STATS, 0), 7.90909, tol);

    // F2I with mask (single extension in mask)
    kmo_test_cat_strings(op1_file, test_path, valid_files[4]);
    kmo_test_cat_strings(op2_file, test_path, valid_files[13]);
    test_stats2(op1_file, 3.0, 3.0, 0, TRUE, op2_file, 2);

    // no output created, since ext=2 is empty
    test_stats2(op1_file, 3.0, 3.0, 0, TRUE, op2_file, 3);
    cpl_test_abs(kmo_test_esorex_data(STATS, 0), 13.1056, tol);

    //
    // with mask, with noise
    //
    // F3I
    kmo_test_cat_strings(op1_file, test_path, valid_files[1]);
    kmo_test_cat_strings(op2_file, test_path, valid_files[3]);
    test_stats(op1_file, 3.0, 3.0, 0, TRUE, op2_file);
    cpl_test_abs(kmo_test_esorex_data(STATS, 0), 230.077, tol);

    // F2I
    kmo_test_cat_strings(op1_file, test_path, valid_files[11]);
    kmo_test_cat_strings(op2_file, test_path, valid_files[3]);
    test_stats(op1_file, 3.0, 3.0, 0, TRUE, op2_file);
    cpl_test_abs(kmo_test_esorex_data(STATS, 0), 60.4033, tol);

    // F2D
    kmo_test_cat_strings(op1_file, test_path, valid_files[10]);
    kmo_test_cat_strings(op2_file, test_path, valid_files[9]);
    test_stats(op1_file, 3.0, 3.0, 0, TRUE, op2_file);
    cpl_test_abs(kmo_test_esorex_data(STATS, 0), 60.4033, tol);

    /* ----- invalid tests ----- */
    // F1I with mask
    kmo_test_cat_strings(op1_file, test_path, valid_files[5]);
    kmo_test_cat_strings(op2_file, test_path, valid_files[5]);
    test_stats(op1_file, 3.0, 3.0, -1, TRUE, op2_file);

    // F1I with mask with noise
    kmo_test_cat_strings(op1_file, test_path, valid_files[12]);
    kmo_test_cat_strings(op2_file, test_path, valid_files[5]);
    test_stats(op1_file, 3.0, 3.0, -1, TRUE, op2_file);

    // RAW with mask (F2I)
    kmo_test_cat_strings(op1_file, test_path, valid_files[6]);
    kmo_test_cat_strings(op2_file, test_path, valid_files[3]);
    test_stats(op1_file, 3.0, 3.0, -1, TRUE, op2_file);

    // F2D with mask (RAW)
    kmo_test_cat_strings(op1_file, test_path, valid_files[8]);
    kmo_test_cat_strings(op2_file, test_path, valid_files[6]);
    test_stats(op1_file, 3.0, 3.0, -1, TRUE, op2_file);

    // F2D with mask (F2I)
    kmo_test_cat_strings(op1_file, test_path, valid_files[8]);
    kmo_test_cat_strings(op2_file, test_path, valid_files[3]);
    test_stats(op1_file, 3.0, 3.0, -1, TRUE, op2_file);

    // B2D with mask (F2I)
    kmo_test_cat_strings(op1_file, test_path, valid_files[9]);
    kmo_test_cat_strings(op2_file, test_path, valid_files[3]);
    test_stats(op1_file, 3.0, 3.0, -1, TRUE, op2_file);

    // F3I all finite with too small mask
    kmo_test_cat_strings(op1_file, test_path, valid_files[0]);
    kmo_test_cat_strings(op2_file, test_path, invalid_files[0]);
    test_stats(op1_file, 3.0, 3.0, -1, TRUE, op2_file);

    return cpl_test_end(0);
}

/** @} */
