/* $Id$ 
 *
 * GetParam: Get a parameter in the transfer region.
 *
 * Copyright (C) 2008-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */
#ifndef __GET_PARAM_HPP_INCLUDED
#define __GET_PARAM_HPP_INCLUDED

#include "intermediate/opcodes/OpCode.hpp"
#include "intermediate/operands/Reference.hpp"

namespace intermediate {

//! Copy slot in the transfer region of a container to dest.
/** This opcode will copy source in the transfer slot of container
 *  container into the dst operand.
 *
 *  Read Operands: cnt, src
 *  Write operands: dst
 *
 *  Operation:
 *      copy src residing in the slot defined by cnt/src to dst.
 *
 *  Errors:
 *      GetParam must be evaluated after a corresponding BeginTransfer
 *      statement in the same text segment before a corresponding
 *      EndTransfer statement.
 *
 *      The Reference to the container must match the Reference to 
 *      BeginTransfer.
 *
 *  Hints:
 *      Useful for OUT/INOUT variable parameters, to access the result.
 *      Should not be annotated foreign, as this would be very hard to
 *      map to a native C call for example.
 */
class GetParam : public OpCode {
public:
	//! c'tor
	/** @param source source operand living in container
	 *  @param cnt Reference to the container
	 *  @param dest destination operand.
	 */
	GetParam(
		Reference *source, 
		Reference *cnt, 
		Operand *dest
		) :	src(source),
			container(cnt),
			dst(dest) {}

	//! Accept a Visitor.
 	/** All intermediate code nodes need to implement this method.
         *
         *  @param v the Visitor that can visit this node.
         */
	virtual void accept(Visitor& v) {
		v.visit(*this);
	}

	//! source operand (item in transfer segment of container)
	Reference *src;
	//! referenced container
	Reference *container;
	//! destination operand.
	Operand *dst;

protected:
	virtual ~GetParam() {
		util::MiscUtil::terminate(this->src);
		util::MiscUtil::terminate(this->container);
		util::MiscUtil::terminate(this->dst);
	}
};

}; /* namespace intermediate */

#endif /* __GET_PARAM_HPP_INCLUDED */
