/* $Id: sig_match.h,v 1.19 2012-03-06 14:46:49 siflkres Exp $ 
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifndef __SIG_MATCH_H_INCLUDED
#define __SIG_MATCH_H_INCLUDED

#include <inttypes.h>
#include <stdbool.h>

#include "sig_gen.h"

/** callback functions for sig_match */
struct sig_match_funcs {

	/** get's called every time an image get's matched 
	 *  @param s instance 
	 *  @param visible is the match visible?
	 *  @param x x coordinate of match (or 0 if not visible)
	 *  @param y y coordinate of match (or 0 if not visible)
	 *  @param w width of the matched pattern (or 0 if not visible)
	 *  @param h height of the matched pattern (or 0 if not visible)
	 *  (TODO: x,y offset)
	 */
	void (*event)(void *s, bool visible, uint16_t x, uint16_t y, 
	                                     uint16_t w, uint16_t h);

	/** callback to add a fullscreen match for the pattern matcher.
	 *  @param s instance.
	 *  @param image path to the image.
	 */
	void (*add_match)(void *s, const char* image);

	/** callback for adding a match for a limited area to the pattern
	 *  matcher.
	 *  @param s instance.
	 *  @param image path to the image.
	 *  @param x x part of the top/left corner.
	 *  @param y y part of the top/left corner.
	 *  @param w width.
	 *  @param h height.
	 */
	void (*add_match_partial)(void *s, const char* image, uint16_t x, 
				  uint16_t y, uint16_t w, uint16_t h);

	/** disable a match. 
	 *  @param s instance.
	 */
	void (*remove_match)(void *s);
};

/** structure with callbacks
 *  FIXME: partial rectangle of screen! */
struct sig_match {
	enum sig_gen_type type;
	/** members participating on signal */
	struct {
		/** object data. */
		void *s;
		/** callback functions */
		const struct sig_match_funcs *f;
	} member[2];
	/** number of members on signal. */
	unsigned int nmembers;
};

/** Emit a match event.
 *  @param m signal instance
 *  @param s sender instance
 *  @param x x coordinate
 *  @param y y coordinate
 *  @param w width of matched pattern
 *  @param h height of matched pattern
 */
extern void
sig_match_event(const struct sig_match* m, void *s, uint16_t x, uint16_t y,
                                                    uint16_t w, uint16_t h);

/** Emit a no longer matching event.
 *  @param m signal instance
 *  @param s sender instance
 */
extern void
sig_match_invisible(const struct sig_match* m, void *s);

/** Add a fullscreen match for the pattern matcher.
 *  @param m signal instance.
 *  @param s sender instance.
 *  @param image path to the image.
 */
extern void
sig_match_add_match(const struct sig_match* m, void *s, const char* image);

/** Add a match for a limited area to the pattern matcher.
 *  @param m signal instance.
 *  @param s sender instance.
 *  @param image path to the image.
 *  @param x x part of the top/left corner.
 *  @param y y part of the top/left corner.
 *  @param w width.
 *  @param h height.
 */
extern void
sig_match_add_match_partial(
	const struct sig_match* m,
	void *s, 
	const char* image,
	uint16_t x, 
	uint16_t y,
	uint16_t w,
	uint16_t h
);

/** Disable a registered match again.
 *  @param m signal instance.
 *  @param s sender instance.
 */
extern void
sig_match_remove_match(const struct sig_match* m, void *s);

/** connect to match signal.
 *  @param m signal instance.
 *  @param s sender instance.
 *  @param f callback functions.
 */
extern void
sig_match_connect(
	struct sig_match* m,
	void *s, 
	const struct sig_match_funcs *f
);

/** create a match signal.
 *  @param name name of signal.
 *  @param nr number of signal.
 *  @return initialized sig_match signal. */
extern struct sig_match*
sig_match_create(const char *name);

/** destroy a match signal.
 *  @sig signal to be destroyed.
 */
extern void
sig_match_destroy(struct sig_match *sig);

extern void
sig_match_suspend(struct sig_match *b, FILE *fSig);
extern void
sig_match_resume(struct sig_match *b, FILE *fSig);

#endif /* __SIG_MATCH_H_INCLUDED */
