/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | Copyright (C) 1991-2010 OpenCFD Ltd.
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Application
    mshToFoam

Description
    Converts .msh file generated by the Adventure system.

Note
    The .msh format does not contain any boundary information. It is
    purely a description of the internal mesh.

    Can read both linear-tet format (i.e. 4 verts per tet) and linear-hex
    format (8 verts per hex) (if provided with the -hex option)

    Will bomb out if not supplied with the correct option for the
    file format

    Not extensively tested.

Usage

    - mshToFoam [OPTIONS] \<.msh file\>

    @param \<.msh file\> \n
    @todo Detailed description of argument.

    @param -hex \n
    Read hex cells.

    @param -help \n
    Display help message.

    @param -doc \n
    Display Doxygen API documentation page for this application.

    @param -srcDoc \n
    Display Doxygen source documentation page for this application.

\*---------------------------------------------------------------------------*/

#include <OpenFOAM/argList.H>
#include <OpenFOAM/Time.H>
#include <OpenFOAM/polyMesh.H>
#include <OpenFOAM/IFstream.H>
#include <OpenFOAM/polyPatch.H>
#include <OpenFOAM/ListOps.H>
#include <OpenFOAM/cellModeller.H>

#include <fstream>

using namespace Foam;

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

// Main program:

int main(int argc, char *argv[])
{
    argList::noParallel();
    argList::validArgs.append(".msh file");
    argList::validOptions.insert("hex", "");

#   include <OpenFOAM/setRootCase.H>
#   include <OpenFOAM/createTime.H>

    bool readHex = args.optionFound("hex");

    fileName mshFile(args.additionalArgs()[0]);
    IFstream mshStream(mshFile);

    label nCells;
    mshStream >> nCells;

    if (readHex)
    {
        Info<< "Trying to read " << nCells << " hexes." << endl << endl;
    }
    else
    {
        Info<< "Trying to read " << nCells << " tets." << endl << endl;
    }

    cellShapeList cells(nCells);

    const cellModel& tet = *(cellModeller::lookup("tet"));
    const cellModel& hex = *(cellModeller::lookup("hex"));

    labelList tetPoints(4);
    labelList hexPoints(8);

    if (readHex)
    {
        for (label cellI = 0; cellI < nCells; cellI++)
        {
            for (label cp = 0; cp < 8; cp++)
            {
                mshStream >> hexPoints[cp];
            }
            cells[cellI] = cellShape(hex, hexPoints);
        }
    }
    else
    {
        for (label cellI = 0; cellI < nCells; cellI++)
        {
            for (label cp = 0; cp < 4; cp++)
            {
                mshStream >> tetPoints[cp];
            }
            cells[cellI] = cellShape(tet, tetPoints);
        }
    }


    label nPoints;

    mshStream >> nPoints;

    Info<< "Trying to read " << nPoints << " points." << endl << endl;

    pointField points(nPoints);


    for (label pointI = 0; pointI < nPoints; pointI++)
    {
        scalar x, y, z;

        mshStream >> x >> y >> z;

        points[pointI] = point(x, y, z);
    }


    polyMesh mesh
    (
        IOobject
        (
            polyMesh::defaultRegion,
            runTime.constant(),
            runTime
        ),
        xferMove(points),
        cells,
        faceListList(0),
        wordList(0),
        wordList(0),
        "defaultFaces",
        polyPatch::typeName,
        wordList(0)
    );

    Info<< "Writing mesh ..." << endl;

    mesh.write();


    Info<< "End\n" << endl;

    return 0;
}


// ************************ vim: set sw=4 sts=4 et: ************************ //
