/*
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 */

/**
   \defgroup LcmapsGlobusInterface  The LCMAPS Globus dependent interfaces
   
    \brief These interface functions require Globus data types
    
    This part of the API defines types and functions for the Globus
    GSI interface.  This includes all of the interfaces found in
    lcmaps_openssl.h, and lcmaps_basic.h.

    \ingroup LcmapsInterface
*/

/**
    \file   lcmaps_globus.h
    \brief  GSS dependent functions of the LCMAPS library.
    \author Martijn Steenbakkers for the EU DataGrid.
    \author Oscar Koeroo for the EGEE project.

    This header contains the declarations of the LCMAPS library functions:
    -# lcmaps_run():
        To do the user mapping
    -# lcmaps_run_and_return_username():
        To do the user mapping and return the user name
    -# lcmaps_run_and_return_poolindex():
        To do the user mapping and return the poolindex

    And the following
    -# lcmaps_return_poolindex()
        Returns the poolindex based on fixed arguments:
                security context, buffer and buffer length
    -# lcmaps_return_poolindex_from_gss_cred()
        Returns the poolindex based on fixed arguments:
                gss credential, buffer and buffer length

    \ingroup LcmapsGlobusInterface
*/

#ifndef LCMAPS_GLOBUS_H
#define LCMAPS_GLOBUS_H


/******************************************************************************
                             Include header files
******************************************************************************/
#include <gssapi.h>
#include "lcmaps_openssl.h"

/******************************************************************************
 *                 Module definition
 *****************************************************************************/


/*!
    \brief let LCMAPS handle the user mapping

    Do the user mapping based on the user's gss (gsi) credential and the job
    request.
    As a back-up for empty credentials the user DN may be specified as well.
    This is the legacy lcmaps interface and is used by
        the \b gatekeeper.

    \param user_dn_tmp user DN
    \param user_cred   GSS/GSI user credential
    \param request     authorization request as RSL string

    \retval 0 mapping succeeded.
    \retval 1 mapping failed.
*/
typedef int lcmaps_run_t(
        char * user_dn_tmp,
        gss_cred_id_t user_cred,
        lcmaps_request_t request
);


/*!
    \brief let LCMAPS handle the user mapping and return user name

    do the user mapping based on the provided list of policies (first successful
    policy found in the lcmaps policy file (lcmaps.db) will result in the user
    mapping) and return user name.
    As a back-up for empty credentials the user DN may be specified as well.
    For this \b ALLOW_EMPTY_CREDENTIALS should be defined.
    This interface is used by the
        \b GridFTP server.

    \param user_dn_tmp user DN
    \param user_cred   GSS/GSI user credential
    \param request     authorization request as RSL string
    \param usernamep   pointer to user name (to be freed by calling application).
                       Note: usernamep should be non-NULL at the start !
    \param npols       number of policies to be considered for evaluation
    \param policynames the names of the policies to be considered for evaluation

    \retval 0 mapping succeeded.
    \retval 1 mapping failed.
*/
typedef int lcmaps_run_and_return_username_t(
        char * user_dn_tmp,
        gss_cred_id_t user_cred,
        lcmaps_request_t request,
        char ** usernamep,
        int npols,
        char ** policynames
);


/*!
    \brief let LCMAPS handle the user mapping and return a poolindex

    do the user mapping based on the provided list of policies (first successful
    policy found in the lcmaps policy file (lcmaps.db) will result in the user
    mapping) and return the poolindex
    As a back-up for empty credentials the user DN may be specified as well.
    For this \b ALLOW_EMPTY_CREDENTIALS should be defined.
    This interface was intended to be used by a wrapper function for the 
        \b WorkSpace \b Service (WSS, f.q.a. Dynamic Account Service - DAS).

    \param user_dn_tmp user DN
    \param user_cred   GSS/GSI user credential
    \param request     authorization request as RSL string
    \param poolindexp  pointer to poolindex (to be freed by calling application).
                       Note: poolindex should be non-NULL at the start !
    \param npols       number of policies to be considered for evaluation
    \param policynames the names of the policies to be considered for evaluation

    \retval 0 mapping succeeded.
    \retval 1 mapping failed.
*/
typedef int lcmaps_run_and_return_poolindex_t(
        char * user_dn_tmp,
        gss_cred_id_t user_cred,
        lcmaps_request_t request,
        char ** poolindexp,
        int npols,
        char ** policynames
);



# ifndef LCMAPS_USE_DLOPEN
lcmaps_run_t lcmaps_run;
lcmaps_run_and_return_username_t lcmaps_run_and_return_username;
lcmaps_run_and_return_poolindex_t lcmaps_run_and_return_poolindex;
# endif /* LCMAPS_USE_DLOPEN */

#endif /* LCMAPS_GLOBUS_H */
