package tests::ReportParserTest;

use base qw/Lire::Test::TestCase/;

use IO::Scalar;

use Lire::ReportParser;
use Lire::Report::TableInfo;
use Lire::Test::Mock;

sub set_up {
    my $self = $_[0];
    $self->SUPER::set_up();

    return;
}

sub tear_down {
    my $self = $_[0];
    $self->SUPER::tear_down();

    return;
}

sub test_column_info_start {
    my $self = $_[0];

    my $parser = new Lire::ReportParser();
    my $info = new Lire::Report::TableInfo();
    $parser->init_stack( 'lrp_group-info' );
    $parser->stack_push( 'lrp_group-info', $info );

    my $attr = {
                'name' => 'file',
                'class' => 'categorical',
                'type' => 'string',
                'label' => 'A File',
                'col-start' => 1,
                'col-end' => 1,
                'col-width' => 20,
                'max-chars' => 20,
                'avg-chars' => 8,
                };
    $parser->column_info_start( 'lire:column-info', $attr );

    my $col_info = $info->column_info_by_name( 'file' );
    $self->assert_not_null( $col_info,
                            "no ColumnInfo named 'file' created" );
    $self->assert_equals( "categorical", $col_info->class() );
    $self->assert_equals( "string", $col_info->type() );
    $self->assert_equals( "A File", $col_info->label() );
    $self->assert_equals( 1, $col_info->col_start() );
    $self->assert_equals( 1, $col_info->col_end() );
    $self->assert_equals( 20, $col_info->col_width() );
    $self->assert_equals( 20, $col_info->max_chars() );
    $self->assert_equals( 8, $col_info->avg_chars() );
}

sub test_timespan {
    my $self = $_[0];

    my $parser = new_proxy Lire::Test::Mock( 'Lire::ReportParser' );
    my $now = time;
    $self->assert_null( $parser->{'lrp_curr_timespan'} );
    my $infos = { 'timespan' => '',
                  'period' => 'daily',
                  'start' => $now,
                  'end' => $now + 86400 };
    $parser->timespan_start( 'lire:timespan', { 'period' => $infos->{'period'},
                                                'start' => $infos->{'start'},
                                                'end' => $infos->{'end'}  } );
    $self->assert_deep_equals( $infos, $parser->{'lrp_curr_timespan'} );
    $self->assert_deep_equals( $infos, $parser->current_timespan() );
    $parser->timespan_char( 'My ' );
    $self->assert_str_equals( 'My ',
                              $parser->{'lrp_curr_timespan'}{'timespan'} );
    $parser->timespan_char( 'timespan' );
    $self->assert_str_equals( 'My timespan',
                              $parser->{'lrp_curr_timespan'}{'timespan'} );
    $parser->timespan_end( 'lire:timespan' );
    $self->assert_deep_equals( [ $parser, 'My timespan', $now, $now + 86400,
                                 'daily' ],
                               $parser->get_invocation( 'handle_timespan' ) );
    $infos->{'timespan'} = 'My timespan';
    $self->assert_deep_equals( $infos, $parser->current_timespan() );
}

sub test_date {
    my $self = $_[0];

    my $parser = new_proxy Lire::Test::Mock( 'Lire::ReportParser' );
    my $now = time;
    $self->assert_null( $parser->{'lrp_curr_date'} );
    my $infos = { 'date' => '',
                  'time' => $now };
    $parser->date_start( 'lire:date', { 'time' => $now } );
    $self->assert_deep_equals( $infos, $parser->{'lrp_curr_date'} );
    $self->assert_deep_equals( $infos, $parser->current_date() );
    $parser->date_char( 'My ' );
    $self->assert_str_equals( 'My ', $parser->{'lrp_curr_date'}{'date'} );
    $parser->date_char( 'date' );
    $self->assert_str_equals( 'My date', $parser->{'lrp_curr_date'}{'date'} );
    $parser->date_end( 'lire:date' );
    $self->assert_deep_equals( [ $parser, 'My date', $now ],
                               $parser->get_invocation( 'handle_date' ) );
    $infos->{'date'} = 'My date';
    $self->assert_deep_equals( $infos, $parser->current_date() );
}

sub test_report_start {
    my $self = $_[0];

    my $parser = new_proxy Lire::Test::Mock( 'Lire::ReportParser' );
    $parser->set_result( 'error' => sub { shift; die @_ } );
    $self->assert_dies( qr/missing 'version' attribute/,
                        sub { $parser->report_start( 'lire:report' ) } );
    $self->assert_dies( qr/'version' attribute should be 2.0 or 2.1: 1.0 /,
                        sub { $parser->report_start( 'lire:report',
                                                     { 'version' => '1.0' } ) } );
    $parser->report_start( 'lire:report', { 'version' => '2.1' } );
    $self->assert_num_equals( 0, $parser->{'lrp_subreport_count'} );
}

sub test_subreport_start {
    my $self = $_[0];

    my $parser = new_proxy Lire::Test::Mock( 'Lire::ReportParser' );
    $parser->set_result( 'error' => sub { shift; die @_ } );
    $self->assert_dies( qr/missing 'superservice' attribute/,
                        sub { $parser->subreport_start( 'lire:subreport' ) } );
    $self->assert_dies( qr/missing 'type' attribute/,
                        sub { $parser->subreport_start( 'lire:subreport', 
                                                        { 'superservice' => 'test' } ) } );
    $parser->{'lrp_section_subreport_count'} = 0;
    $parser->{'lrp_subreport_count'} = 0;
    $parser->subreport_start( 'lire:subreport', { 'superservice' => 'test',
                                                  'type' => 'top-dirs' } );
    $self->assert_num_equals( 1, $parser->{'lrp_section_subreport_count'} );
    $self->assert_num_equals( 1, $parser->current_section_subreport_count() );
    $self->assert_num_equals( 1, $parser->is_stack_empty( 'lrp_group')  );
    $self->assert_num_equals( 1, $parser->is_stack_empty( 'lrp_entry')  );
    $self->assert_deep_equals( { 'superservice' => 'test',
                                 'type' => 'top-dirs' },
                               $parser->{'lrp_subreport'} );
    $self->assert_str_equals( 'top-dirs', $parser->current_type() );
    $self->assert_str_equals( 'test', $parser->current_superservice() );
}

1;
