//
//  Lynkeos
//  $Id: MyImageView.m,v 1.3 2004/08/09 22:41:48 j-etienne Exp $
//
//  Created by Jean-Etienne LAMIAUD on Wed Sep 24 2003.
//  Copyright (c) 2003-2004. Jean-Etienne LAMIAUD
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

#include <math.h>

#import "MyImageView.h"

#ifdef GNUSTEP
#include "processing_core.h"
#endif

#define K_MAX_ZOOM 4.0

@interface MyImageView(Zoom)
- (void) applyZoom :(double)newZoom from:(id)sender ;
- (void) stepZoom :(double)step;
@end

@implementation MyImageView(Zoom)

- (void) applyZoom :(double)newZoom from:(id)sender
{
    NSSize newFrameSize,
    newBoundsSize = _imageSize;
    NSRect visible = [self visibleRect];

    // Apply zoom
    if ( _imageSize.width != 0 )
    {
        newFrameSize.width = _imageSize.width*newZoom;
        newBoundsSize.width = _imageSize.width;
    }
    else
    {	// NSView don't like null sizes
        newFrameSize.width = 1;
        newBoundsSize.width = 1;
    }

    if ( _imageSize.height != 0 )
    {
        newFrameSize.height = _imageSize.height*newZoom;
        newBoundsSize.height = _imageSize.height;
    }
    else
    {	// NSView don't like null sizes
        newFrameSize.height = 1;
        newBoundsSize.height = 1;
    }

    [self setFrameSize:newFrameSize];
    [self setBoundsSize:newBoundsSize];

    // Adjust scroll to keep center still
    if ( _imageSize.width != 0 && _imageSize.height != 0 )
    {
        visible.origin.x += (1 - _zoom/newZoom)*visible.size.width/2;
        if ( visible.origin.x < 0 )
            visible.origin.x = 0;
        visible.origin.y += (1 - _zoom/newZoom)*visible.size.height/2;
        if ( visible.origin.y < 0 )
            visible.origin.y = 0;
        visible.size.width *= _zoom/newZoom;
        visible.size.height *= _zoom/newZoom;

        [self scrollRectToVisible:visible];
    }

    if ( newZoom != _zoom )
    {
        _zoom = newZoom;
        if ( sender != _zoomField )
            [_zoomField setDoubleValue:_zoom*100.0];
        if ( sender != _zoomSlider )
            [_zoomSlider setDoubleValue:log(_zoom)/log(K_MAX_ZOOM)];
    }

    // Make me redraw
    [self setNeedsDisplay:YES];
}

- (void) stepZoom :(double)step
{
    double z = log(_zoom)/log(2);

    // Get the wanted half integer value
    if ( step < 0.0 && z > 0.0 )
        z += 0.49999999;
    else if ( step > 0.0 && z < 0.0 )
        z -= 0.49999999;
    z = (double)((int)((z+step)*2))/2.0;
    z = exp(log(2)*z);
    [self applyZoom: z from:nil];
}

@end

@implementation MyImageView

// Initializations and allocation stuff
- (id)initWithFrame:(NSRect)frameRect
{
    MyIntegerRect nowhere = { {0,0}, {0,0} };

    [super initWithFrame:frameRect];

    _offset = NSMakePoint(0.0,0.0);
    _imageRep = nil;

    _zoom = 1.0;

    _selectionOrigin = nowhere.origin;
    _lastPoint = nowhere.origin;
    _selection = nowhere;

    [self initCursors];

    return self;
}

- (void) dealloc
{
    if ( _imageRep != nil )
        [_imageRep release];

    [_crossCursor release];
    [_leftCursor release];
    [_rightCursor release];
    [_topCursor release];
    [_bottomCursor release];
    [_topLeftCursor release];
    [_topRightCursor release];
    [_bottomLeftCursor release];
    [_bottomRightCursor release];
    [_insideCursor release];

    [super dealloc];
}

// Image and zoom
- (IBAction)doZoom:(id)sender
{
    double z;

    if ( sender == _zoomSlider )
        z = exp(log(4)*[sender doubleValue]);
    else
        z = [sender doubleValue]/100.0;
    [self applyZoom: z from:sender];
}

- (IBAction)moreZoom:(id)sender
{
    if ( _zoom < K_MAX_ZOOM )
        [self stepZoom:0.5];
}

- (IBAction)lessZoom:(id)sender
{
    if ( _zoom > 1.0/K_MAX_ZOOM )
        [self stepZoom:-0.5];	// To be improved in stepZoom
}

- (void) setImage :(NSImage*)image offset:(NSPoint)offset
{
#ifdef GNUSTEP
  //ugly thing to display 24 bits picture ...
  NSImage* displayedImage;
  NSBitmapImageRep*  bitMap = [image bestRepresentationForDevice:nil]; // get a rep from your image, or grab from a view

  // NSLog(@"bits/pixel: %d\n",[bitMap bitsPerPixel]);

  if([bitMap bitsPerPixel] == 24){
    displayedImage = image;
  }
  else{
    NSMutableData *data;
    
    u_char *plane[5];
    u_short x, y, w, h, rowSize, pixelSize;
    bool planar;
    RGB color;
    NSSize size;
    int nbBytes;
    char* header;
    char buffer[3];
    int pixelOffset;
    
    [bitMap getBitmapDataPlanes:plane];
    rowSize = [bitMap bytesPerRow];
    pixelSize = [bitMap bitsPerPixel]/8;
    size = [image size];
    
    planar = [bitMap isPlanar];
    w = size.width;
    h = size.height;

    data = [NSMutableData new];
    [data autorelease];
    [data setLength: 0];
    nbBytes = asprintf(&header, "P6\n%d %d\n255\n", w, h);
    [data appendBytes:header length:nbBytes];
    free(header);

    nbBytes = 3;
    for ( y = 0; y < h; y++ ) {
      for ( x = 0; x < w; x++ ) {
        // Read the data in the bitmap
        if ( planar ) {
          pixelOffset = (y*rowSize)+x;
          color.red = plane[0][pixelOffset];
          color.green = plane[1][pixelOffset];
          color.blue = plane[2][pixelOffset];
        }
        else {
          pixelOffset = (y*rowSize)+(x*pixelSize);
          color.red = plane[0][pixelOffset+1];
          color.green = plane[0][pixelOffset+3];
          color.blue = plane[0][pixelOffset+5];
        }
        
        buffer[0] = (char)(color.red);
        buffer[1] = (char)(color.green);
        buffer[2] = (char)(color.blue);
      [data appendBytes: buffer length:nbBytes];
      }
    }
    
    //data = [bits representationUsingType: NSJPEGFileType properties: nil];
    //[data writeToFile: @"/path/to/wherever/test.png" atomically: NO];
    displayedImage = [[NSImage alloc] initWithData:data];
    image = displayedImage;
  }
#endif
    // Cleanup before else
    _imageSize.width = 0.0;
    _imageSize.height = 0.0;

    if ( _imageRep != nil )
    {
        [_imageRep release];
        _imageRep = nil;
    }

    // Get that new image
    _offset = offset;
    if ( image != nil )
    {
        _imageRep = [[image bestRepresentationForDevice:nil] retain];
        if ( _imageRep != nil )
        {
            _imageSize.width = [_imageRep pixelsWide];
            _imageSize.height = [_imageRep pixelsHigh];
            [_imageRep setSize:_imageSize];
        }
    }

    [self applyZoom:_zoom from:nil];
    [[self window] resetCursorRects];
}

// Drawing
- (void)drawRect:(NSRect)rect
{
   if ( _imageRep != nil )
   {
      NSRect r = [self bounds];
      r.origin.x += _offset.x;
      r.origin.y += _offset.y;
      [_imageRep drawInRect:r];
   }

   if ( _selection.size.width != 0 && _selection.size.height != 0 )
   {
      NSGraphicsContext *g = [NSGraphicsContext currentContext];

      [g saveGraphicsState];
      [[NSColor redColor] set];
      [NSBezierPath strokeRect:NSRectFromIntegerRect(_selection)];
      [g restoreGraphicsState];
   }
}

- (NSSize) imageSize
{
    return( _imageSize );
}

@end
