///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/utilities/ProgressIndicator.h>
#include "POSCARWriter.h"
#include <atomviz/atoms/AtomsObject.h>
#include <atomviz/atoms/datachannels/AtomTypeDataChannel.h>

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(POSCARWriter, MultiFileWriter)

/******************************************************************************
* Writes the output file.
******************************************************************************/
bool POSCARWriter::writeAtomsFile(const QString& filepath, DataSet* dataset, const QVector<TimeTicks>& exportFrames, bool suppressDialogs)
{
	if(exportFrames.size() != 1)
		MsgLogger() << "Warning: The POSCAR file writer can only write single frames." << endl;

	TimeTicks time = exportFrames[0];
	int frame = time / dataset->animationSettings()->ticksPerFrame();

	// Extract the atoms to be exported from the scene.
	PipelineFlowState flowState = retrieveAtoms(dataset, time);
	AtomsObject* atoms = dynamic_object_cast<AtomsObject>(flowState.result());
	if(atoms == NULL || atoms->atomsCount() == 0)
		throw Exception(tr("The scene does not contain any atoms that could be exported (at animation frame %1).").arg(frame));

	DataChannel* posChannel = atoms->getStandardDataChannel(DataChannel::PositionChannel);
	AtomTypeDataChannel* atomTypeChannel = static_object_cast<AtomTypeDataChannel>(atoms->getStandardDataChannel(DataChannel::AtomTypeChannel));
	DataChannel* velocityChannel = atoms->getStandardDataChannel(DataChannel::VelocityChannel);
	if(atomTypeChannel && atomTypeChannel->atomTypes().empty()) atomTypeChannel = NULL;

	if(posChannel == NULL)
		throw Exception(tr("Cannot export atoms to POSCAR file because there is no position data channel present in the source data."));

	MsgLogger() << "Opening POSCAR file" << filepath << "for writing." << endl;

	QFile stream(filepath);
	if(!stream.open(QIODevice::WriteOnly|QIODevice::Text))
		throw Exception(tr("Failed to open the file %1 for writing: %2").arg(filepath, stream.errorString()));
	QTextStream textStream(&stream);
	textStream.setRealNumberPrecision(12);

	textStream << "POSCAR file written by OVITO" << endl;
	textStream << "1" << endl;
	AffineTransformation cell = atoms->simulationCell()->cellMatrix();
	for(size_t i=0; i<3; i++) {
		textStream << cell(0, i) << " " << cell(1, i) << " " << cell(2, i) << endl;
	}

	// Count number of atoms for each atom type.
	QVector<int> atomCounts;
	if(atomTypeChannel) {
		const int* atype = atomTypeChannel->constDataInt();
		const int* atype_end = atype + atomTypeChannel->size();
		for(; atype != atype_end; ++atype) {
			OVITO_ASSERT(*atype >= 0);
			if(*atype < 0) continue;
			if(*atype >= atomCounts.size())
				atomCounts.resize(*atype+1);
			atomCounts[*atype]++;
		}
	}
	else atomCounts.push_back(atoms->atomsCount());
	for(int i=0; i<atomCounts.size(); i++) {
		if(atomCounts[i] == 0 && i == 0) continue;
		textStream << atomCounts[i] << " ";
	}
	textStream << endl;

	// Write atomic positions.
	textStream << "Cartesian" << endl;
	for(int atype=0; atype<atomCounts.size(); atype++) {
		const Point3* p = posChannel->constDataPoint3();
		for(size_t i = 0; i < posChannel->size(); i++, ++p) {
			if(atomTypeChannel) {
				if(atomTypeChannel->getInt(i) != atype) continue;
			}
			textStream << p->X << " " << p->Y << " " << p->Z << endl;
		}
	}

	// Write atomic velocities.
	if(velocityChannel) {
		textStream << "Cartesian" << endl;
		for(int atype=0; atype<atomCounts.size(); atype++) {
			const Vector3* v = velocityChannel->constDataVector3();
			for(size_t i = 0; i < velocityChannel->size(); i++, ++v) {
				if(atomTypeChannel) {
					if(atomTypeChannel->getInt(i) != atype) continue;
				}
				textStream << v->X << " " << v->Y << " " << v->Z << endl;
			}
		}
	}

	return true;
}

};	// End of namespace AtomViz
